;;; -*- Mode: LISP; Syntax: Common-lisp; Package: COMMON-MUSIC; Base: 10 -*-
;;; **********************************************************************
;;; Copyright (c) 1989, 1990 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  No warranty is made about this software, its performance
;;; or its conformity to any specification.  Any distribution of this 
;;; software must comply with all applicable United States export control
;;; laws. Suggestions, comments and bug reports are welcome.  Please 
;;; address electronic correspondence to: hkt@ccrma.stanford.edu
;;; **********************************************************************

(in-package "COMMON-MUSIC")

(defparameter *coordinates-are-x-y-pairs* T
  "If true, function coordinates are parsed as X Y pairs, otherwise as Y X pairs")

(defun interpolation (&rest coords)
  (unless (and coords (evenp (length coords)))
    (error "Malformed interpolation list: ~S" coords))
  (let ((l (if *coordinates-are-x-y-pairs*
	       coords
	     (loop for (y x) on coords by #'cddr nconc (list x y)))))
    (loop with x2 for x1 in l by #'cddr
          unless (or (null x2)(< x2 x1))
          do (error "X value ~S not < ~S in ~S" x2 x1 l)
          do (setf x2 x1))
    l))


(defun step-function (&rest coords)
  (declare (ignore coords))
  (error "STEP-FUNCTION has been removed. See documentation on LOOKUP instead."))


;;;
;;; Interp is a low level function used by interpl and function-value. Coords
;;; are assumed to be in x,y order.

(defun interp (x coords)
  (let ((sav coords))
    (macrolet ((getcoord (l)
		 `(or (pop ,l)
		      (error "Malformed interpolation list: ~S" sav))))
      (coerce (let* ((x1 (getcoord coords))
		     (y1 (getcoord coords))
		     (x2 x1)
		     (y2 y1))
		(loop when (or (null coords)
			       (> x2 x))
		      return nil
		      do
		  (setf x1 x2 y1 y2 
			x2 (getcoord coords) y2 (getcoord coords)))
		(cond ((>= x x2)
		       y2)
		      ((<= x x1)
		       y1)
		      (t
		       (+ y1 (* (- x x1) 
				(/ (- y2 y1)
				   (- x2 x1)))))))
	      'float))))

;;;
;;; Interpl returns the interpolated value of x in coords.
;;;

(defun interpl (x &rest coords)
  (declare (inline interp))
  (interp x coords))

;;;
;;; Function-value returns an interpolated value for x in the function
;;; coords,  and optionally scales and offsets the result.
;;;

(defun function-value (x coords &optional scale offset)
  (declare (inline interp))
  (let ((y (interp x coords)))
    (when scale
      (setf y (* y scale)))
    (when offset
      (setf y (+ y offset)))
    y))

;;;
;;; Lookup returns the uninterpolated y value of x.  Since the y value is
;;; not examined it can be any lisp object, not just numbers.
;;;

(defun lookup (x coords)
  (macrolet ((getcoord (l)
	       `(or (pop ,l)
		    (error "Malformed lookup list: ~S" sav))))
    (let* ((sav coords)
	   (x1 (getcoord coords))
	   (y1 (getcoord coords))
	   (x2 x1)
	   (y2 y1))
      (loop when (or (null coords) (> x2 x))
            return nil
            do
	(setq x1 x2 y1 y2
	      x2 (getcoord coords) y2 (getcoord coords)))
      (if (>= x x2) y2 y1))))
