;;;-*- Mode: Lisp; Package: STELLA -*-
;;;
;;; File: mcl-stella.lisp
;;;
;;; 
;;; Input to stella from arbitrary fred windows:
;;;
;;; o  at the stella prompt, meta-enter evaluates one of:
;;;      o  the current selection, if any, or else
;;;      o  the current {}-delimited multiline form, if any, or else
;;;      o  the current line 
;;;    as a stella command.
;;; o  meta-return pastes the current selection, if any, or a newline into the
;;;    listener. this is fine to answer subsequential prompts produced by
;;;    tl:ask-user, which calls read-line.
;;; o  meta-control-return is the same as meta-return, except that it tries to
;;;    paste the current symbolic expression, if there isn't already a 
;;;    selection.
;;;
;;; Scripting utility functions: qs ("QuickScript"), script-stella
;;;
;;; See also the examples at the end of this file.
;;;
;;;
;;;   1-6-93  works with the new tl.lisp. moved the remaining stuff to stella.
;;;   5-6-93  new script-stella function: accepts new echo keyword and uses 
;;;           both tl:eval-from-file and tl:eval-from-string.
;;;           meta-enter now supports the {} multiline stella command syntax.
;;;  10-6-93  eliminated tl:*external-input*: ed-input-to-stella now uses
;;;           tl:top-level's parser.
;;;           ed-input-to-stella now no longer reads 'random' {} pairs
;;;           when a buffer contains multiple nested levels of curly braces.
;;;   9-8-93  added qs function
;;;  25-8-93  script-stella now shields probe-file from strings containing
;;;           wildcard characters.

(in-package stella)

(export '(qs script-stella) :stella)


(defun script-stella (output echo &rest strings)
  (if (every #'stringp strings)
    (progn
      (when (eq output t) (setf output *standard-output*))
      (if (and (not (wild-pathname-p (first strings)))
               (probe-file (first strings)))
        (tl:eval-from-file (first strings) :output output 
                           :commands *commands* :echo-input echo)
        (progn 
          (setf strings (format nil "~{~a~%~}" strings))
          (tl:eval-from-string strings :output output :commands *commands*
                             :echo-input echo))))
    (error "wrong argument type: data or pathname string(s) expexcted.")))

;;;
;;; "QuickScript": a quick interface to script-stella. 
;;; try for example (qs "edit top-level~%?~%q" t t)

(defun qs (format-string &optional (output nil) (echo nil)) 
  (script-stella output echo (format nil format-string)))


;;;
;;; Fred window and command stuff.

(defun input-to-listener (fred-mixin &key (select-cur-sexp nil))
  (multiple-value-bind (start end) (ccl:selection-range fred-mixin)
    (flet ((paste-it (fred-mixin start end)
             (let ((stream (ccl::window-selection-stream 
                            fred-mixin start end)))
               (loop for line = (read-line stream nil nil nil)
                     while line
                     do (write-line line *standard-input*)))))
      (if (eql start end)
        (if select-cur-sexp
          (let ((buf (ccl:fred-buffer fred-mixin)))
            (multiple-value-bind (s e) (ccl:buffer-current-sexp-bounds buf)
              (when s 
                (setq start s end e)
                (paste-it fred-mixin start end))))
          (write-char #\newline *standard-input*))
        (paste-it fred-mixin start end)))))

(defun ed-input-to-stella (fred-mixin)
  (multiple-value-bind (start end) (ccl:selection-range fred-mixin)
    (when (eql start end)
      (let* ((buf (ccl:fred-buffer fred-mixin)) 
             (pos (ccl:buffer-position buf))
             closing-}-pos opening-{-pos next-closing-}-pos)
        (if (if (setf closing-}-pos 
                      (ccl::buffer-forward-search 
                                     buf "}" (if (= pos 0)
                                               pos
                                               (1- pos)) t))
              (if (setf opening-{-pos 
                        (ccl::buffer-backward-search  
                         buf "{" closing-}-pos 0))
                (if (<= opening-{-pos pos)
                  (if (setf next-closing-}-pos 
                            (ccl::buffer-backward-search  
                             buf "}" (1- closing-}-pos) 0))
                    (if (< next-closing-}-pos opening-{-pos)
                      t
                      nil)
                    t)
                  nil)
                nil)
              nil)
          (setf start opening-{-pos
                end closing-}-pos)
          (setf start (ccl:buffer-line-start buf pos)
                end (ccl:buffer-line-start buf pos 1)))))
    (let ((stream (ccl::window-selection-stream fred-mixin start end)))
      (ccl:eval-enqueue `(:read-input ,stream)))))

(defun ed-paste-input-to-listener (fred-mixin)
  (input-to-listener fred-mixin :select-cur-sexp nil))

(defun ed-paste-sexp-to-listener (fred-mixin)
  (input-to-listener fred-mixin :select-cur-sexp t))

(ccl::def-fred-command (:meta #\return) ed-paste-input-to-listener)
(ccl::def-fred-command (:meta :control #\return) ed-paste-sexp-to-listener)
(ccl::def-fred-command (:meta #\enter) ed-input-to-stella)



#|
;;; scripting stella
;;;  
;;; syntax: script-stella <output-stream> <echo-input> {string}*
;;; 
;;; output-stream t is the same as *standard-output*, echo-input determines
;;; whether or not the input form should be printed. 
;;;
;;; example 1: direct data strings

(script-stella t t "(expt 23 34)" "help help" "parameters" "" "" ""
               "(inspector::universal-time-string (get-universal-time))")

;;;
;;; example 2: using format to construct one single string. note, that 
;;  there is no need for a final ~%. 

(script-stella t t (format nil "(print \"hi!\")~%help write~%~s" 
                       '(cm::date-string)))

;;;
;;; example 3: multiline syntax

(script-stella nil nil "{map 1 set amplitude .5 
                         when (= $channel 0) set amplitude .3 
                         when (= (mod (degree $note) 5) 0) scale duration 2}")

;;;
;;; example 4: executing a script file.

(script-stella t t "core:desktop folder:foo.tl")

(script-stella t t "import .midi:test.midi
test")
(script-stella t t "listen"
"^")
(script-stella t t "{listen}	    
^")
(script-stella t t "{list
1[1:5]} 
list")
(script-stella t t "edit 1" "?" "start" "" "quit")


;;;
;;; QuickScript: qs <format-string> &optional (output nil) (echo nil)
;;; 
;;; the first parameter must be a string acceptable to format. Both output
;;; and echo parameters are optional and default to nil.
;;;
;;; Examples:

(qs "list Top-Level" t)                 ; output must be t here
(qs "parameters~%containers~%" t)       ;   and here
(qs "listen 1~%^")
(qs "unfreeze 1~%listen 1~%1" t t)

|#

;;;
;;; -*- EOF -*-
