;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; item stream classes are indexed by their data and pattern types.
;;;

(defprop items :class-index 0)
(defprop notes :class-index 1)
(defprop pitches :class-index 2)
(defprop degrees :class-index 3)
(defprop intervals :class-index 4)
(defprop steps :class-index 5)
(defprop rhythms :class-index 6)
(defprop amplitudes :class-index 7)
(defprop series :class-index 8)
(defprop numbers :class-index 9)

(defprop cycle :class-index 0)
(defprop random :class-index 1)
(defprop heap :class-index 2)
(defprop palindrome :class-index 3)
(defprop graph  :class-index 4)
(defprop sequence :class-index 5)
(defprop function :class-index 6)
(defprop rotation :class-index 7)
(defprop accumulation :class-index 8)

;;;
;;; constructors macroexpand any data form that is consp with a symbol in the
;;; car with an :item-expand property of t. otherwise the form is treated as
;;; basic data and not macroexpanded.
;;;

(defprop items :item-expand t)
(defprop notes :item-expand t)
(defprop pitches :item-expand t)
(defprop degrees :item-expand t)
(defprop rhythms :item-expand t)
(defprop intervals :item-expand t)
(defprop steps :item-expand t)
(defprop voicings :item-expand t)  ; remove next release
(defprop numbers :item-expand t)

(defprop function :disallow-items t)
(defprop numbers  :disallow-items t)

;;;
;;; constructor options are symbols whose :item-option property holds a
;;; parsing list: (keyword quoted? patterns datatypes). keyword is the
;;; option's make-instance keyword argument, quoted? is a flag that, if
;;; true, causes the option value to be quoted during macroexpansion.
;;; patterns and datatypes are the lists of patterns and datatypes that
;;; accept the option, or t if the option applies to all. 
;;;

(defprop ABOVE
  :item-option '(:above nil (cycle random) (numbers)))
(defprop ARGS 
  :item-option '(:args nil (function) t))
(defprop BELOW
  :item-option '(:below nil (cycle random) (numbers)))
(defprop BY
  :item-option '(:by nil (cycle random) (numbers)))
(defprop CHANGE 
  :item-option '(:change nil (rotation) t))
(defprop COUNTING 
  :item-option '(:flags nil t t))
(defprop DOWNTO
  :item-option '(:downto nil (cycle random)(numbers)))
(defprop FOR 
  :item-option '(:for nil t t))
(defprop ELIDED :item-option
  '(:flags nil (palindrome) t))
(defprop forming 
  :item-option '(:forming nil (cycle) (series)))
(defprop FROM 
  :item-option '(:from nil t (intervals steps series numbers)))
(defprop IN :item-option t)           ; system option not passed on
(defprop INITIALLY-FROM
  :item-option '(:initially-from nil t (intervals steps series)))
(defprop LINKED-TO 
  :item-option '(:linked-to t t (intervals steps series)))
(defprop LOUDEST
  :item-option '(:loudest nil t (amplitudes)))
(defprop MODULUS
   :item-option  '(:modulus nil (cycle) (series)))
(defprop MULTIPLE
  :item-option '(:multiple nil (cycle) (series)))
(defprop NAMED
  :item-option '(:name t t t))
(defprop OF
  :item-option '(:of nil t (notes pitches degrees intervals steps series)))
(defprop ON
  :item-option '(:on nil t (intervals steps series)))
(defprop POWER
  :item-option '(:power nil t (amplitudes)))
(defprop PREVIOUS
  :item-option '(:last t (graph) t))
(defprop PROPS
  :item-option '(:props nil (graph) t))
(defprop RETURNING
  :item-option '(:flags nil t '(intervals steps series)))
(defprop SOFTEST
  :item-option '(:softest nil t (amplitudes)))
(defprop TEMPO
  :item-option '(:tempo nil t (rhythms)))
(defprop TO
  :item-option '(:to nil (cycle random) (numbers)))
(defprop TRAVERSING 
  :item-option '(:flags nil t t))
(defprop USING
  :item-option '(:using nil (heap random) t))
(defprop WITH 
  :item-option '(:function nil (graph function) t))

;;;
;;; expand-constructor expands item stream macros into calls to make-instance
;;; with the data selectively evaluated and options mapped keyword arguments.
;;;

(defun expand-constructor (datatype forced-pattern forms env)
  (let ((flags 0) data options class pattern)
    (setf data
      (loop while forms
            for thing = (car forms)
            until (and (symbolp thing) (get thing ':item-option))
            collect (pop forms)))
    (when (and data (get datatype ':items-disallowed))
      (error "~A does not allow explicit items: ~S" datatype data))
    (loop with saved = forms
          while forms
          for option = (pop forms)
          for parser = (or (and (symbolp option)
                                (get option ':item-option))
                           (error "~S is not a constructor option."
                                  option))
          for value = (if forms (pop forms)
                          (error "Malformed (uneven) options list: ~S" 
                                 saved))
          do (cond ((eq option 'in) (setf pattern value))
                   ((eq (first parser) ':flags)
                    (setf flags (logior flags (parse-flag option value))))
                   (t
                    (push value options)
                    (push option options))))
    (if pattern
       (unless (or (null forced-pattern) (eq forced-pattern pattern))
         (error "~S pattern must be ~A." datatype forced-pattern))
      (setf pattern 'cycle))
    (setf class (lookup-class datatype pattern))
    (if data
        (if (or (get pattern ':disallow-items) (get datatype ':disallow-items))
            (error "~A in ~A does not allow explicit items: ~S" 
                   datatype pattern data)
          (setf data (list ':items (relist-for-eval data env))))
      (unless (or (get pattern ':disallow-items) (get datatype ':disallow-items))
        (error "Missing items for ~A in ~A." datatype pattern)))
    ;; destructively substitute keywords for option names and quote
    ;; values when necessary.
    (loop for tail on options by #'cddr
          for option = (car tail)
          for info = (or (get option ':item-option)
                         (error "~S is not an item stream option."
                                option))
          do 
          (unless (or (eq (third info) t) (find pattern (third info)))
            (error "Option ~S not applicable to ~S." option pattern))
          (unless (or (eq (fourth info) t) (find datatype (fourth info)))
            (error "Option ~S not applicable to ~S." option datatype))
          (setf (car tail) (first info))
          when (second info)
          do (setf (cadr tail) `(quote ,(cadr tail))))
    (when (> flags 0) 
      (push flags options)
      (push ':flags options))
    ;; cobble up the make-instance form with data rewritten for selective 
    ;; evaluation: non-constant atoms are quoted and (...) is replaced by
    ;; (LIST ...) unless its car is a symbol with :item-expand property t.
    `(make-instance ',class ,.data ,.options)))

(defun relist-for-eval (form env)
  (cond ((null form) nil)
        ((listp form) 
         (if (and (symbolp (car form)) (get (car form) ':item-expand))
             (macroexpand form env)
          `(list ,@(loop for f in form 
                         collect (relist-for-eval f env)))))
        ((constantp form) form)
        (t `(quote ,form))))

(defvar *flag-values*
  `((returning
     (,+coerce-to-note+ note) (,+coerce-to-pitch+ pitch) (0 degree))
    (counting
     (,+count-values+ values) (0 periods))
    (traversing
     (,+breadth-first+ breadth) (0 depth))
    (elided
     (,+elide-left+ left start) (,+elide-right+ right end)
     (,(logior +elide-left+ +elide-right+) yes t) (0 no nil) )))

(defun parse-flag (option value)
  (let ((set (cdr (assoc option *flag-values*)))
        val)
    (unless (setf val (find value set :key #'cdr :test #'member))
      (error "~S value ~S not in ~S" option value
             (loop for x in (cdr set) append (cdr x))))
    (first val)))

(defvar *item-stream-classes* 
  (make-array '(10 9)
    :initial-contents
    '((cyclic-item-stream random-item-stream heap-item-stream
       palindromic-item-stream graph-item-stream sequential-item-stream
       functional-item-stream rotational-item-stream accumulating-item-stream)
      (cyclic-note-stream random-note-stream heap-note-stream
       palindromic-note-stream graph-note-stream sequential-note-stream
       functional-note-stream rotational-note-stream accumulating-note-stream)
      (cyclic-pitch-stream random-pitch-stream heap-pitch-stream
       palindromic-pitch-stream graph-pitch-stream sequential-pitch-stream
       functional-pitch-stream rotational-pitch-stream 
       accumulating-pitch-stream)
      (cyclic-degree-stream random-degree-stream heap-degree-stream
       palindromic-degree-stream graph-degree-stream sequential-degree-stream
       functional-degree-stream rotational-degree-stream 
       accumulating-degree-stream)
      (cyclic-interval-stream random-interval-stream heap-interval-stream
       palindromic-interval-stream graph-interval-stream 
       sequential-interval-stream functional-interval-stream
       rotational-interval-stream  accumulating-interval-stream)
      (cyclic-step-stream random-step-stream heap-step-stream
       palindromic-step-stream graph-step-stream sequential-step-stream
       functional-step-stream rotational-step-stream accumulating-step-stream)
      (cyclic-rhythm-stream random-rhythm-stream heap-rhythm-stream
       palindromic-rhythm-stream graph-rhythm-stream sequential-rhythm-stream
       functional-rhythm-stream rotational-rhythm-stream 
       accumulating-rhythm-stream)
      (cyclic-amplitude-stream random-amplitude-stream heap-amplitude-stream
       palindromic-amplitude-stream graph-amplitude-stream
       sequential-amplitude-stream functional-amplitude-stream
       rotational-amplitude-stream accumulating-amplitude-stream)
      (series-stream nil nil nil nil nil nil nil nil)
      (linear-number-stream random-number-stream nil nil nil nil nil nil nil))))

(defun lookup-class (datatype pattern)
  ;; return the item stream class for the two symbolic indicies
  (let ((row (or (get datatype ':class-index)
                 (error "~S is not an item stream type." datatype)))
        (col (or (get pattern ':class-index)
                 (error "~S is not an item pattern type." pattern))))
    (or (aref *item-stream-classes* row col)
        (error "No item stream for ~S and ~S!" datatype pattern))))
;;;
;;;
;;;

(defmacro items (&body body &environment env)
  (expand-constructor 'items nil body env))

(defmacro notes (&body body &environment env)
  (expand-constructor 'notes nil body env))

(defmacro pitches (&body body &environment env)
  (expand-constructor 'pitches nil body env))
            
(defmacro degrees (&body body &environment env)
  (expand-constructor 'degrees nil body env))

(defmacro intervals (&body body &environment env)
  (expand-constructor 'intervals nil body env))
            
(defmacro steps (&body body &environment env)
  (expand-constructor 'steps nil body env))

(defmacro rhythms (&body body &environment env)
  (expand-constructor 'rhythms nil body env))

(defmacro amplitudes (&body body &environment env)
  (expand-constructor 'amplitudes nil body env))

(defmacro series (&body body &environment env)
  (expand-constructor 'series 'cycle body env))

(defmacro numbers (&body body &environment env)
  (expand-constructor 'numbers nil body env))

(defmacro voicings (&body body)
  (declare (ignore body))
  (error "VOICINGS is no longer necessary and has been removed. ~
          Use the INTERVALS macro with the ON option instead. ~
          (VOICINGS ... FROM x) = (INTERVALS ... ON x)"))

(defun make-item-stream (datatype pattern items &rest args)
  (apply #'make-instance (lookup-class datatype pattern) 
        :items items args))

;;;
;;; chord (also the [] read macro)
;;;

(defprop chord :item-expand t)

(defmacro chord (&body body &environment env)
  `(make-chord ,(relist-for-eval body env)))

(defun make-chord (items)
  (if (find-if #'symbolp items)
      (make-instance 'note-chord-stream :items items)
    (make-instance 'chord-stream :items items)))

;;;
;;; crescendo and diminuendo
;;;

(defprop crescendo :item-expand t)
(defprop diminuendo :item-expand t)
(defprop decrescendo :item-expand t)

(defmacro crescendo (&rest body)
  `(make-dynamics ,(relist-for-eval body nil)))

(defmacro diminuendo (&rest body)
  `(make-dynamics ,(relist-for-eval body nil)))
    
(defmacro decrescendo (&rest body)
  (warn "DECRESCENDO works but has been renamed DIMINUENDO")
  `(diminuendo ,@body))

(defun make-dynamics (args)
  (let (from to in inc)
    (loop for (arg val) on args by #'cddr
          do 
      (case arg
        (from (setf from (parse-amplitude val)))
        (to (setf to (parse-amplitude val)))
        ((in for) (setf in val))))
    (unless from (error "Missing FROM starting amplitude.~A"))
    (unless to (error "Missing TO ending amplitude."))
    (unless in (error "Missing IN stepping value."))
    (setf inc (/ (- to from) (1- in)))
    (make-instance 'cyclic-amplitude-stream
      :items (loop for i below in collect from do (incf from inc))
      :for in)))

;;;
;;;  mirror returns a period followed by its strict retrograde
;;;

(defprop mirror :item-expand t)

(defmacro mirror (stream &rest rest)
  `(items in function with
     (let ((items '()) 
           (states '()))
       #'(lambda (s)
           (values
             (if items
                 (prog1 (nreverse items)
                        (setf items nil))
                 (multiple-value-setq (items states) (read-items s t t)))
             states)))
     args (list ,stream) ,@rest))

;;;
;;; repeat repeats the current period of a stream n times. n can 
;;; be an item stream.
;;;
#+lispworks
(eval-when (:load-toplevel
	    :compile-toplevel
	    :execute)
  (shadow 'common-lisp:repeat))

(defprop repeat :item-expand t)

(defmacro repeat (stream n &rest rest)
  `(items in function with
      (let ((items '())
            (states '())
            (limit most-positive-fixnum)
            (count 0))
         #'(lambda (s rep)
             (unless (< count limit)
               (setf items nil count 0))
             (values
               (if items
                   (prog1 items (incf count))
                   (prog1
                     (multiple-value-setq (items states) 
                       (read-items s t t)) 
                     (setf limit (item rep))))
               states)))
      args (list ,stream ,n) ,@rest))

;;;
;;;  motive (also the read macro #@ )
;;;

(defprop motive :item-expand t)

(defmacro motive (name)
  `(find-stream ',name))    

;;;
;;; tempo
;;;

(defprop tempo :item-expand t)

(defmacro tempo (&body body)
  (let ((len (length body)))
    (if (< len 3)
      (if (= len 1)
        (car body)    ; number
        (* (car body) (rhythm (cadr body) 60)))			
      (let ((coords
             (loop for tail on body while (numberp (car tail))
                   collect (pop body)))
            (args ())
            (old body)
            from to in)
        (loop with prop and value
              while body
              do
              (setf prop (pop body))
              (unless body
                (error "Malformed (uneven) tempo specification: ~S." 
                       old))
              (setf value (pop body))
              (ecase prop
                (from (setf from value))
                (to (setf to value))
                (in (setf in value))
                (update
                 (push (quote-if-necessary value) args)
                 (push ':mode args))
                ((pulse beat)
                 (push (quote-if-necessary value) args)
                 (push ':beat args))))
        (when (or from to in)
	  (unless from
	    (error "Missing starting tempo: ~S." old))
	  (unless to
	    (error "Missing ending tempo: ~S." old))
	  (unless in
	    (error "Missing number of beats: ~S." old))
	  (setf coords
                (if *coordinates-are-x-y-pairs*
                  (list 0 from in to)
                  (list from 0 to in))))		   
        
        `(make-instance 'tempo-function :coords ',coords ,.args)))))

;;;
;;; idsel make a selection table used in for markov proceeses (graph stream)
;;;

(defprop idsel :item-expand t)

(defmacro idsel (&rest args &environment env)
  `(list :idsel ., (loop for (ids val) on args by #'cddr
                         collect `(quote ,(if (consp ids) ids (list ids)))
                         collect (relist-for-eval val env))))

;;;
;;; changes constucts one or more rotational changes.
;;;

(defprop changes :item-expand t)

(defmacro changes (&rest args)
  (let ((parsed '()))
    (loop with tail = args
          while tail
          for arg = (pop tail)
          do 
          (push (if tail (pop tail)
                  (error "Malformed (uneven) options list: ~S" args))
                parsed)
          (ecase arg
           ((start :start) (push ':start parsed))
           ((end :end) (push ':end parsed))
           ((step :step) (push ':step parsed))
           ((width :width) (push ':width parsed))))
    `(make-change ,.parsed)))

(defun make-change (&rest args &key start end step width)
  (flet ((chng (start step width end)
                (if end
                  (list (or start 0) (or step 1) (or width 1) end)
                  (if width
                    (list (or start 0) (or step 1) width)
                    (if step 
                      (list (or start 0) step)
                      start)))))
    (macrolet ((next (x) 
               `(if (consp ,x) 
                    (if (cdr ,x) (pop ,x) (car ,x)) 
                  ,x)))
      (if (or (consp start) (consp end) (consp step) (consp width))
          (let ((max (loop for x in args when (consp x) maximize (length x))))
            (make-instance 'cyclic-item-stream
               :items
               (loop repeat max
                     collect (chng (next start) (next step)
                                   (next width) (next end)))))
        (chng start step width end)))))

