;;; **********************************************************************
;;; Copyright (c) 89-93, 94 Heinrich Taube.  All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: hkt@zkm.de
;;; **********************************************************************

(in-package :common-music)

;;;
;;; ampitude streams map logical amplitudes to actual amplitude values.
;;; logical amplitudes are numbers between 0 and 1.0 or symbols in
;;; the set *amplitude-symbols*.
;;;

(defparameter *amplitude-minimum* 0.0
  "Minimum amplitude when logical amplitude is 0.0")

(defparameter *amplitude-maximum* 1.0 
  "Maximum amplitude when logical amplitude is 1.0")

(defparameter *amplitude-power* 1 "Amplitude power curve.")

(defprop niente :amplitude-weight 0)
(defprop pppp :amplitude-weight .1)
(defprop ppp :amplitude-weight .2)
(defprop pp :amplitude-weight .3)
(defprop p :amplitude-weight .4)
(defprop mp :amplitude-weight .5)
(defprop mf :amplitude-weight .6)
(defprop f :amplitude-weight .7)
(defprop ff :amplitude-weight .8)
(defprop fff :amplitude-weight .9)
(defprop ffff :amplitude-weight 1)

(defclass amplitude-stream (item-stream) 
  ((power :initform *amplitude-power* :initarg :power)
   (softest :initform *amplitude-minimum* :initarg :softest)
   (loudest :initform *amplitude-maximum* :initarg :loudest)))


(defmethod initialize-instance :after ((object amplitude-stream)
                                       &rest args)
  (declare (ignore args))
  ;; loudest is converted to loudest-softest to avoid recomputation
  ;; each time item is called.
  (setf (slot-value object 'loudest)
    (- (slot-value object 'loudest)
       (slot-value object 'softest))))

(defmethod item :after ((object amplitude-stream))
  (setf (slot-value object 'value)
     (amplitude-value (slot-value object 'loudest)
                      (slot-value object 'value)
                      (slot-value object 'softest) 
                      (slot-value object 'power))))

(defmethod amplitude-value ((range integer) amp min power)
  (values (floor (+ min (* range (expt amp power))))))

(defmethod amplitude-value ((range number) amp min power)
  (+ min (* range (expt amp power))))

(defmethod datum-parser ((object amplitude-stream))
  #'parse-amplitude)

(defmethod parse-amplitude ((amp symbol))
  (or (get amp ':amplitude-weight)
      (error "~S is not an amplitude." amp)))

(defmethod parse-amplitude ((amp number))
  (unless (and (>= amp 0.0) (<= amp 1.0))
    (error "~S is not an amplitude." amp))
  amp)

(defmethod parse-amplitude ((amp item-stream))
  amp)

(defmethod parse-amplitude ((amp t))
  (error "~S is not an amplitude." amp))

;;;
;;;
;;;

(defclass cyclic-amplitude-stream (amplitude-stream cycle-pattern) ())
(defclass seqeuntial-amplitude-stream (amplitude-stream sequence-pattern) ())
(defclass palindromic-amplitude-stream (amplitude-stream palindrome-pattern)
  ())
(defclass random-amplitude-stream (amplitude-stream random-pattern) ())
(defclass graph-amplitude-stream (amplitude-stream graph-pattern) ())
(defclass heap-amplitude-stream (amplitude-stream heap-pattern) ())
(defclass rotational-amplitude-stream (amplitude-stream rotation-pattern) ())
(defclass functional-amplitude-stream (amplitude-stream function-pattern) ())
(defclass accumulating-amplitude-stream (amplitude-stream accumulation-pattern)
  ())

;;;
;;; user functions
;;;

(defmethod amplitude ((amp t) &key (softest *amplitude-minimum*) min max
		      (loudest *amplitude-maximum*) (power *amplitude-power*))
  (setf min (or min softest))
  (setf max (or max loudest))
  (amplitude-value (- max min) (parse-amplitude amp) min power))

(defmethod amplitude ((amp item-stream) &key min max power softest loudest)
  (declare (ignore min max power loudest softest))
  amp)
