;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-Defs.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	Compile- and load-time macros, defstructs, etc.
;;;	Last Edit:	3/03/89 05:26:45
;;;
;;;	Copyright (c) 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;
;;;	This file contains definitions for all TimeLogic structs, vars,
;;;	macros, etc.  It needs to be loaded for compilation of every other
;;;	TimeLogic file.

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))

;;; Public macros

;;; The interval structure.

;;; Intervals are global internal structures (mapped by name, which can be any
;;; Lisp object, as long as EQ test applies).  They contain the name, an index
;;; (which is more like a creation stamp), a clink pointer to a context-dependent
;;; linkage structure, the context for which the clink is correct, two endpoint
;;; marks for sorting purposes, and a flag which is t if this interval was
;;; generated by the system.

(defstruct (interval (:print-function tl-print-interval))
  name			; Unique name of interval
  index			; N'th created interval
  clink			; may be inherited
  (context NIL)		; context for which clink is ok
  headpoint		; Used for sorting
  tailpoint		; Used for sorting
  generated-p		; T iff generated by TimeLogic
)

;;; Relations between intervals are context-dependent.  Contexts form a tree,
;;; and contain a table mapping intervals to their clinks appropriate for the
;;; context (may be inherited).  The context "nil" refers to the current
;;; context, the context "t" refers to the root context.

(defstruct (context (:print-function tl-print-context))
  (name NIL)		; Unique (EQ) name of context or NIL
  (parent NIL)		; parent context or NIL
  (children NIL)	; child contexts or NIL
  (clinks-hash		; hashtable of int->clink
    (make-hash-table :size 997 :test #'eq))
)

;;; The clink structure records the way an interval is related to other
;;; intervals within a certain context.  An interval may inherit a clink in
;;; the current context from an ancestral context.  If modifications to the
;;; clink are required, a copy specific to the current context must be created
;;; first.  CAVEAT!!!
;;; The clink contains fields for storing the home context of the clink, a
;;; list of reference intervals, a list of intervals for which it is a referent,
;;; two lists of ilink structures, storing relational and durational
;;; information, and two hash tables for fast mapping of intervals to the
;;; corresponding ilink.

(defstruct clink
  context		; home context of this clink
  referents		; List of intervals
  referrals		; List of intervals
  rel-ilinks-hash	; Hash table of int->ilinks
  rel-ilinks-list	; List of unbroken ilinks
  rel-ilinks-list-all	; List of all ilinks
  dur-ilinks-hash	; Hash table of int->ilinks
  dur-ilinks-list	; List of unbroken ilinks
  dur-ilinks-list-all	; List of all ilinks
)

;;; Intervals are related with one another through ilinks, which contain fields
;;; for the type of ilink (relational or durational), the source and target
;;; intervals, the current constraint, previous constraints, and a flag
;;; indicating whether this ilink has been broken (to eliminate unnecessary
;;; propagation -- cf.  auto-reference)

(defstruct (ilink (:print-function tl-print-ilink))
  type			; oneof {:rel,:dur}
  (class NIL)		; oneof {nil,:up,:down,:side}
  source		; link from interval
  target		; link to interval
  current		; Encoded relation or duration
  (previous NIL)	; PDL of previous current's
  (broken-p NIL)	; T iff no propagation across this link
  (unique-p NIL)	; T iff current is unique
)

;;; Relations are asserted relative to a backtracking point
(defvar *btpoint-id* 0)

(defstruct (btpoint (:print-function tl-print-btpoint))
  kind			; What kind of event is backtracked
  tx			; Primary interval involved
  (ty NIL)		; Secondary interval involved or NIL
  (prev NIL)		; Previous backtracking point
  (used-p NIL)                  ; Already used for backtracking
  (id (incf *btpoint-id*))      ; for tracing
)

;;; Endpoints are used for sorting and displaying intervals

(defstruct (endpoint)
  ordinal		; number or endpoint it is equal to
  (related NIL)		; t if related to some other interval
  (leaders NIL)		; endpoints ahead of this one
)


;;; General system variables


(defvar *tl-auto-backtrack-p* nil
  "Automatically backtrack on current propagations if an error occurs?")

(defvar *tl-auto-define-p* nil
  "Automatically define an interval if undefined and adding constraint?")

(defvar *tl-auto-reference-p* nil
  "Generate a reference hierarchy automatically?")

(defvar *tl-backtrack-in-progress* nil
  "T if currently backtracking")

(defvar *tl-backtrack-point* nil
  "The most recently defined backtracking point")

(defvar *tl-clean-p* t
  "After initialization but before database changes made?")

(defvar *tl-defined-intervals-hash* (make-hash-table :size 997 :test #'eq)
  "A hash table mapping interval names to interval structs, for fast access")

(defvar *tl-defined-intervals-list* nil
  "A list of all defined interval structs, for fast mapping")

(defvar *tl-posting-runtime* nil
  "Stats counter of internal time units spent in constraint posting")

(defvar *tl-fetching-runtime* nil
  "Stats counter of internal time units spent in constraint fetching")

(defvar *tl-contexts* nil
  "Global list of all TimeLogic contexts")

(defvar *tl-context-leaves-only* nil
  "T or :WARN or NIL.  If T, don't allow updates to non-leaf contexts")

(defvar *tl-current-context* nil
  "Free var containing current TimeLogic context")

(defvar *tl-dump-version* 1 "Dump format version number")

(defvar *tl-durations-enabled-p* nil
  "Propagate durational information?")

(defvar *tl-error-occurred-p* nil
  "Did an error occur while asserting relations?")

(defvar *tl-interval-count* 0
  "How many interval structs have been created since last INIT")

(defvar *tl-props* nil
  "AList containing current TimeLogic property settings")

(defvar *tl-props-reset-p* nil
  "T if TimeLogic property default settings have been applied")

(defvar *tl-propagate-depth-first-p* nil
  "Propagate information depth-first or through LIFO queue?")

(defvar *tl-propagation-level* 1
  "Debugging: Shows depth of propagation")

(defvar *tl-relations-enabled-p* nil
  "Relations between intervals enabled?")

(defvar *tl-rlink-to-dlink-map* nil
  "AList mapping rlinks to dlinks")

(defvar *tl-root-context* nil
  "Global var containing root TimeLogic context")

(defvar *tl-search-all-paths-p* nil
  "If T, compute R- or D-link across all paths thru ref hierarchy between TX and TY")

(defvar *tl-stats-p* nil
  "Debugging: Keep statistics?")


;;; Relational-specific system constants

;;; Note: The encoding order is critical! cf.  TR-INVERT-RLINK

(defconstant *tlr-nil* 0)
(defconstant *tlr-b*   (ash 1 0))
(defconstant *tlr-a*   (ash 1 1))
(defconstant *tlr-d*   (ash 1 2))
(defconstant *tlr-c*   (ash 1 3))
(defconstant *tlr-f*   (ash 1 4))
(defconstant *tlr-fi*  (ash 1 5))
(defconstant *tlr-m*   (ash 1 6))
(defconstant *tlr-mi*  (ash 1 7))
(defconstant *tlr-o*   (ash 1 8))
(defconstant *tlr-oi*  (ASH 1 9))
(defconstant *tlr-s*   (ash 1 10))
(defconstant *tlr-si*  (ash 1 11))
(defconstant *tlr-e*   (ash 1 12))
(defconstant *tlr-dur* (logior *tlr-d* *tlr-f* *tlr-s*))
(defconstant *tlr-con* (logior *tlr-c* *tlr-fi* *tlr-si*))
(defconstant *tlr-dis* (logior *tlr-a* *tlr-b* *tlr-m* *tlr-mi*))
(defconstant *tlr-int* (logior *tlr-c* *tlr-d* *tlr-e* *tlr-f* *tlr-fi*
			       *tlr-o* *tlr-oi* *tlr-s* *tlr-si*))
(defconstant *tlr-neq* (logior *tlr-a* *tlr-b* *tlr-c* *tlr-d*
			       *tlr-f* *tlr-fi* *tlr-m* *tlr-mi*
			       *tlr-o* *tlr-oi* *tlr-s* *tlr-si*))
(eval-when (compile load eval)
  (defconstant *tlr-all* (logior *tlr-a* *tlr-b* *tlr-c* *tlr-d* *tlr-e*
                                 *tlr-f* *tlr-fi* *tlr-m* *tlr-mi*
                                 *tlr-o* *tlr-oi* *tlr-s* *tlr-si*)))

(deftype rlink () `(integer 0 ,*tlr-all*))

;;; Relation-specific variables

(defvar *tlr-assert-count* nil
  "How many assertions done sofar")

(defvar *tlr-assert-hook* nil
  "NIL or a function to be invoked when TimeLogic attempts to add or
update a relational constraint.  The hook is called with arglist
(INT1 CONSTRAINT INT2) in encoded form.  If the hook returns NIL an
incompatibility error is raised.")

(defvar *tlr-autoref-unique-uplinks-p* nil
  "Should AutoRef UpLinks be unique relations?")

(defvar *tlr-autoref-unique-sidelinks-p* t
  "Should AutoRef SideLinks be unique relations?" )

(defvar *tlr-constraint-count* nil
  "How many constraints have been propagated")

(defvar *tlr-constraint-queue* nil
  "List of recently added relational constraints needing propagation")

(defvar *tlr-hook-constraints* nil
  "List of added relational constraints to be given to *tlr-assert-hook*")

(defvar *tlr-uplink-queue* nil
  "List of recently added uplinks")

(defvar *tlr-sidelink-queue* nil
  "List of recently added sidelinks")

(defvar *tlr-rlink-decoding-map* nil
  "AList mapping encoded RLinks to decoded RLinks")

(defvar *tlr-multiply-count* nil
  "How many RLink multiplications have occurred sofar")

(defvar *tlr-try-add-count* nil
  "How many additional constraints have been attempted due to propagation")



;;; Duration-specific variables

(defvar *tld-assert-count* nil
  "How many durational constraints have been asserted sofar")

(defvar *tld-assert-hook* nil
  "NIL or a function to be invoked when TimeLogic attempts to add or
update a durational constraint.  The hook is called with arglist
(INT1 CONSTRAINT INT2) in encoded form.  If the hook returns NIL an
incompatibility error is raised.")

(defvar *tld-constraint-count* nil
  "How many constraints have been attempted from without")

(defvar *tld-constraint-queue* nil
  "List of DLinks recently constrained needing propagation")

(defvar *tld-floats-enabled-p* nil
  "If NIL, Floatp duration specs are rationalized")

(defvar *tld-hook-constraints* nil
  "List of added durational constraints to be given to *tld-assert-hook*")

(defvar *tld-multiply-count* nil
  "How many durational multiplications have been done sofar")

(defvar *tld-tolerance* nil
  "New constraint must be at least this factor better than old one in order to propagate")

(defvar *tld-try-add-count* nil
  "How many additional durational constraints have been attempted thru propagation")



;;; Trace-specific variables

(defvar *tl-clear-display-form* nil)

(defvar *tl-display-enabled-p* nil)

(defvar *tl-display-initialized-p* nil)

(defvar *tl-display-stream* t)

(defvar *tl-report-rlinks-p* nil)

(defvar *tl-segment-width* nil)

(defvar *tl-sort-ints-before-display-p* nil)

(defvar *tl-trace-enabled-p* nil)

(defvar *tl-trace-mode* nil)

(defvar *tl-trace-wait* nil)

(defvar *tl-traced-intervals* nil)

(defvar *tl-display-width* nil)


;;; Exportable macros

(defmacro addintconq (txname constraint tyname &key context (type :rel))
  `(add-interval-constraint ',txname ',constraint ',tyname :context ',context :type ',type))

(defmacro defintq (&optional intname refname)
  `(define-interval ',intname ',refname))

(defmacro getintconq (txname tyname &key context (type :rel))
  `(get-interval-constraint ',txname ',tyname :context ',context :type ',type))

(defmacro traceintq (int &optional withints)
  `(trace-interval ',int ',withints))

(defmacro untraceintq (int)
  `(untrace-interval ',int))


;;; System macros

(defmacro tl-interval-defined-p (txname)
  `(gethash ,txname *tl-defined-intervals-hash*))

(defmacro tl-older-interval-p (tx ty)
  `(< (interval-index ,tx)
      (interval-index ,ty)))

(defmacro tl-time-it (accumulator &body code)
  (let ((clock (gensym "TL-Clock-")))
    `(let ((,clock (if ,accumulator (get-internal-run-time))))
       (prog1 (progn ,@code)
	      (if ,clock (incf ,accumulator
			       (- (get-internal-run-time) ,clock))))))
)


;; Relational-specific system macros

;; apparent bug in 4.2b for allegro logical tests, so temporarily changes these from macros to funcionts.
;; return to macro - sent patch 5/6/93

(defmacro tlr-all-rlink-p (rlink)
  `(tlr-same-rlink-p (the rlink ,rlink) *tlr-all*))

(defmacro tlr-card (rlink)
  `(logcount (the rlink ,rlink)))

;;(defun tlr-card (rlink)
;;  (logcount (the rlink rlink)))

(defmacro tlr-const (&rest rels)		; Note:  compile-time evaluation!!!!!
  (tlr-encode-rlink rels))

;;(defun tlr-const (&rest rels)		
;;  (tlr-encode-rlink rels))

(defmacro tlr-intersect-p (rlink1 rlink2)
  `(logtest (the rlink ,rlink1) (the rlink ,rlink2)))

;;(defun tlr-intersect-p (rlink1 rlink2)
;;  (logtest (the rlink rlink1) (the rlink rlink2)))

(defmacro tlr-intersect-rlinks (rlink1 rlink2)
  `(the rlink (logand (the rlink ,rlink1) (the rlink ,rlink2))))
;;(defun tlr-intersect-rlinks (rlink1 rlink2)
;;  (the rlink (logand (the rlink rlink1) (the rlink rlink2))))

(defmacro tlr-no-rlink-p (rlink)
  `(tlr-same-rlink-p (the rlink ,rlink) *tlr-nil*))

(defmacro tlr-same-rlink-p (rlink1 rlink2)
  `(= (the rlink ,rlink1) (the rlink ,rlink2)))

;;(defun tlr-same-rlink-p (rlink1 rlink2)
;;  (= (the rlink rlink1) (the rlink rlink2)))

(defmacro tlr-unite-rlinks (&rest rlinks)
  `(the rlink (logior ,@(mapcar #'(lambda (rl) `(the rlink ,rl)) rlinks))))

;;(defun tlr-unite-rlinks (&rest rlinks)
;;(the rlink (apply #'logior rlinks)))

(defmacro tlr-subset-p (rlink1 rlink2)
  (cond ((and (symbolp rlink1)
	      (symbolp rlink2))
	 `(tlr-same-rlink-p ,rlink2 (tlr-unite-rlinks ,rlink2 ,rlink1)))
	(t `(prog ((l1 ,rlink1)
		   (l2 ,rlink2))
		  (return (tlr-same-rlink-p l2 (tlr-unite-rlinks l2 l1)))))))


;;; Duration-specific system macros

(defmacro tld-no-dlink-p (dlink)
  `(null ,dlink))

(defmacro tld-all-dlink-p (dlink)
  `(tld-all-range-p (car ,dlink)))

(defmacro tld-same-dlink-p (dlink1 dlink2)
  `(equal ,dlink1 ,dlink2))

(defmacro tld-bound-value (bound)
  `(car ,bound))

(defmacro tld-const (&rest drels)		; Note:  compile-time evaluation!!!!!
  `(quote ,(tld-encode-dlink (if (cdr drels)
				 (cons 'or drels)
				 (car drels)))))

(defmacro tld-hi-bound (range)
  `(cdr ,range))

(defmacro tld-inf-bound-p (bound)
  `(eql *tld-inf-value* (tld-bound-value ,bound)))

(defmacro tld-nul-bound-p (bound)
  `(eql *tld-nul-value* (tld-bound-value ,bound)))

(defmacro tld-lo-bound (range)
  `(car ,range))

(defmacro tld-make-bound (n &optional (openp nil))
  `(cons ,n ,openp))

(defmacro tld-make-range (lobound hibound)
  `(cons ,lobound ,hibound))

(defmacro tld-open-bound-p (bound)
  `(cdr ,bound))


;;; Duration-specific variables (should be constants but lists don't work well
;;; as constants)

(defvar *tld-nul-value* 0)

(defvar *tld-nul-bound* (tld-make-bound *tld-nul-value* t)
  "A zero bound is open and has 0 value")

(defvar *tld-inf-value* nil)

(defvar *tld-inf-bound* (tld-make-bound *tld-inf-value* t)
  "An infinite bound is open and has infinite value")

(defconstant *tld-maxint* 32767
  "The square root of the maximum positive FIXNUM")

(defvar *tld-lt* (list (tld-make-range *tld-nul-bound* (tld-make-bound 1 t)))
  "Same as (tld-const (0 (1)))")

(defvar *tld-gt* (list (tld-make-range (tld-make-bound 1 t) *tld-inf-bound*))
  "Same as (tld-const ((1) :INF))")

(defvar *tld-le* (list (tld-make-range *tld-nul-bound* (tld-make-bound 1 nil)))
  "Same as (tld-const (0 1))")

(defvar *tld-ge* (list (tld-make-range (tld-make-bound 1 nil) *tld-inf-bound*))
  "Same as (tld-const (1 :INF))")

(defvar *tld-eq* (list (tld-make-range (tld-make-bound 1 nil)
				       (tld-make-bound 1 nil)))
  "Same as (tld-const (1 1))")

(defvar *tld-ne* (append *tld-lt* *tld-gt*)
  "Same as (tld-const (0 (1)) ((1) :INF))")



;;; End of file TL-DEFS
