;;; -*- Package: Timelogic; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10. -*-
;;;
;;;	File:		TL-Contexts.lisp
;;;	Author:		Johannes A. G. M. Koomen
;;;	Purpose:	TimeLogic support for contextual reasoning
;;;	Last Edit:	3/03/89 01:18:35
;;;
;;;	Copyright (c) 1989  University of Rochester
;;;
;;;	The TimeLogic System is being made available by the University of
;;;	Rochester for research purposes.  No commercial use or distribution to
;;;	third parties is allowed without the explicit written permission of
;;;	the University of Rochester.
;;;
;;;	The University of Rochester will have a non-exclusive right, at no
;;;	expense, to the derivative works, modifications and enhancements made
;;;	to or resulting from the TimeLogic System, and the University of
;;;	Rochester shall be informed of such development and furnished with the
;;;	source codes to such works, modifications and enhancements when
;;;	available.  The University of Rochester will accept such derivative
;;;	works, modifications and enhancements "as is."
;;;
;;;	For documentation on this implementation see Technical Report #231,
;;;	Department of Computer Science, University of Rochester.
;;;
;;;	This file implements the functionality for contextual reasoning in
;;;	TimeLogic.  Instead of maintaining relations between global interval
;;;	structures, store relations in context-dependent clink structure,
;;;	and make global interval structure point to appropriate clink.

(eval-when (compile load eval)
  (in-package "TIMELOGIC"))

;;; Public functions

(defun create-context (&optional context parent)
  (context-name (tl-create-context context parent))
)


(defun define-context (&optional context parent)
  "
    (DEFINE-CONTEXT &optional CONTEXT PARENT)
	Defines a new context if necessary, with name = CONTEXT and
   parent = PARENT, which defaults to the current context."
  (context-name (tl-create-context context parent))
)


(defun defined-contexts ()
  "
    (DEFINED-CONTEXTS)
	Returns a list of defined TimeLogic contexts.
  "
  (mapcar #'context-name *tl-contexts*)
)


(defun context-defined-p (context)
  "
    (CONTEXT-DEFINED-P CONTEXT)
	If CONTEXT is defined returns its name, otherwise NIL.
  "
  (let ((c (tl-find-context context t)))
    (if c (context-name c)))
)


(defun delete-context (context)
  "
    (DELETE-CONTEXT CONTEXT)
	Deletes the given CONTEXT and all its children.
	Error if CONTEXT = T (root).
	Returns a tree of conses reflecting deleted context structure.
  "
  (tl-delete-context context t nil)
)


(defun switch-context (context)
  "
    (SWITCH-CONTEXT CONTEXT)
	Makes CONTEXT the current context.
  "
  (context-name (tl-switch-context (tl-find-context context)))
)


(defun current-context ()
  "
    (CURRENT-CONTEXT)
	Returns the current context, or NIL if none exists.
  "
  (if (context-p *tl-current-context*)
      (context-name *tl-current-context*))
)


(defun push-context ()
  "
    (PUSH-CONTEXT)
	Gensyms a new context with the current context as its parent,
	and makes it the current context.
	Same as (SWITCH-CONTEXT (DEFINE-CONTEXT))
  "
  (context-name (tl-push-context))
)


(defun pop-context (&optional dont-delete-p)
  "
    (POP-CONTEXT &optional DONT-DELETE-P)
	Makes the parent of the current context the new current context,
	deleting the old current context if DONT-DELETE-P = NIL.
  "
  (context-name (tl-pop-context dont-delete-p))
)


(defun context-tree (&optional (context t))
  "
    (CONTEXT-TREE &optional (CONTEXT T))
	Returns a tree of contexts headed by CONTEXT (defaults to root).
	A tree node is either the name of a leaf context or a list whose
	CAR is the name of a context and whose CDR is a list of tree nodes,
	one for each of the context's children.
  "
  (tl-context-tree context)
)

;;; Implementation

;;; Context creation, deletion, etc.

(defun tl-create-context (context parent)
  (prog ((cc (and context (tl-find-context context t)))
	 (pc (tl-find-context parent)))
	(cond ((not (context-p cc))
	       (setf cc (make-context :name (or context (gensym "TLC"))
				      :parent pc
				      :CHILDREN NIL))
	       (if (context-p pc)
		   (push cc (context-children pc)))
	       (push cc *tl-contexts*)
	       (if *tl-trace-enabled-p* (tl-trace :newcon cc (context-parent cc))))
	      ((not (eq (context-parent cc) pc))
	       (tl-error "Can't create existing context ~S with different parent ~S!"
			 context parent)))
	(return cc))
)

(defun tl-delete-context (context collectp rootokp)
  "Removes context CC and all its offspring,
   returning a list of deleted context names if COLLECTP"
  (prog* ((cc (tl-find-context context))
	  (pc (and cc (context-parent cc))))
	 (cond ((context-p pc)
		(setf (context-children pc)
		      (delete cc (context-children pc))))
	       ((not rootokp)
		(tl-error "Can't delete root context!")
		(return nil)))
	 (return
	   (labels ((del0 (cc)			; Actually deletes context
		      (if *tl-trace-enabled-p* (tl-trace :delcon cc (context-parent cc)))
		      (setf *tl-contexts* (delete cc *tl-contexts*))
		      (clrhash (context-clinks-hash cc))
						; in case GC chokes on circularities
		      (if (eq cc *tl-current-context*)
			  (tl-switch-context (context-parent cc))))
		    (del1 (cc)			; Recursively deletes context,
						; returning tree of names deleted
		      (prog1 (if (context-children cc)
				 (cons (context-name cc)
				       (mapcar #'del1 (context-children cc)))
				 (context-name cc))
			     (del0 cc)))
		    (del2 (cc)			; Recursively deletes context, quickly
		      (mapc #'del2 (context-children cc))
		      (del0 cc)))
	     (if collectp
		 (del1 cc)
		 (del2 cc)))))
)

(defun tl-switch-context (context)
  (unless (eq *tl-current-context* context)
    (setf *tl-current-context* context)
    (if (and *tl-trace-enabled-p* (context-p context))
	(tl-trace :switch context (context-parent context))))
  *tl-current-context*
)

(defun tl-push-context ()
  (tl-switch-context (tl-create-context nil nil))
)

(defun tl-pop-context (dontdeletep)
  (let ((pc (context-parent *tl-current-context*)))
    (cond ((null pc)
	   (tl-error "Can't pop from root context!"))
	  (t (unless dontdeletep (tl-delete-context *tl-current-context* nil nil))
	     (tl-switch-context pc))))
)

(defun tl-reset-context (cc)
 ;; Remove circularities, for refcounting gc
 (setf (context-parent cc) nil)
  (setf (context-children cc) nil)
  (clrhash (context-clinks-hash cc))
)

(defun tl-init-contexts ()
  (setf *tl-root-context* nil)			; so it won't be found for sure
  (setf *tl-current-context* nil)		; so it won't be found for sure
  (tl-switch-context (setf *tl-root-context* (tl-create-context t nil)))
)

(defun tl-find-context (context &optional noerrorp)
  (cond ((null context) *tl-current-context*)
	((eq context t) *tl-root-context*)
	((context-p context) context)
	((dolist (c *tl-contexts*)
	   (when (eq context (context-name c))
	     (return c))))
	(noerrorp nil)
	(t (tl-error "Undefined context: ~S" context)))
)

(defun tl-context-tree (context)
  (labels ((tree (cc)
	     (if (context-children cc)
		 (cons (context-name cc)
		       (mapcar #'tree (context-children cc)))
		 (context-name cc))))
    (tree (tl-find-context context)))
)

(defun tl-print-context (context stream level)
  ;;  Primarily for debugging purposes.  Users are not intended to ever
  ;;  see these.
  (write `(context :name ,(context-name context)
		   :parent ,(context-parent context))
	 :stream stream
	 :level level)
)


;;; Interval clinks

(defun tl-get-readable-clink (tx)
  (if (eq (interval-context tx) *tl-current-context*)
      (interval-clink tx)
    (setf (interval-clink tx)
	  (labels
	    ((findit (context)
	       (when context
		 (or (gethash tx (context-clinks-hash context))
		     (setf (gethash tx (context-clinks-hash context))
			   (findit (context-parent context)))))))
	    (findit (setf (interval-context tx) *tl-current-context*)))))
)

(defun tl-get-writable-clink (tx)
  (cond ((null (context-children *tl-current-context*)))	; 's kosher
	((null *tl-context-leaves-only*))	; don't mind, apparently
	((eq *tl-context-leaves-only* :warn)
	 (tl-warn "Changing info on ~S in non-leaf context ~S might cause inconsistencies!"
		  (interval-name tx)
		  (context-name *tl-current-context*)))
	(t (tl-error "Can't change info on ~S in non-leaf context ~S!"
		     (interval-name tx)
		     (context-name *tl-current-context*))
	   (return-from tl-get-writable-clink nil)))
  (let ((clink (tl-get-readable-clink tx)))
    (if (and clink (eq (clink-context clink) *tl-current-context*))
	clink
	(setf (interval-clink tx)
	      (setf (gethash tx (context-clinks-hash
				  (setf (interval-context tx)
					*tl-current-context*)))
		    (tl-copy-clink clink)))))
)

(defun tl-copy-clink (clink)
  (let* ((old-rel-ilinks (if clink (clink-rel-ilinks-list-all clink)))
	 (new-rel-ilinks (mapcar #'tl-copy-ilink old-rel-ilinks))
	 (old-dur-ilinks (if clink (clink-dur-ilinks-list-all clink)))
	 (new-dur-ilinks (mapcar #'tl-copy-ilink old-dur-ilinks))
	 (new-rel-ilinks-hash (tl-copy-ilinks-hash new-rel-ilinks))
	 (new-dur-ilinks-hash (tl-copy-ilinks-hash new-dur-ilinks)))
    (make-clink :context *tl-current-context*
		:rel-ilinks-hash new-rel-ilinks-hash
		:rel-ilinks-list-all new-rel-ilinks
		:rel-ilinks-list (remove-if #'(lambda (x) (ilink-broken-p x))
					    new-rel-ilinks)
		:dur-ilinks-hash new-dur-ilinks-hash
		:dur-ilinks-list-all new-dur-ilinks
		:dur-ilinks-list (remove-if #'(lambda (x) (ilink-broken-p x))
					    new-dur-ilinks)
		:referents (if clink (copy-list (clink-referents clink)))
		:referrals (if clink (copy-list (clink-referrals clink)))))
)

(defun tl-copy-ilink (ilink)
  (make-ilink :type (ilink-type ilink)
	      :class (ilink-class ilink)
	      :source (ilink-source ilink)
	      :target (ilink-target ilink)
	      :current (ilink-current ilink)
	      ;; no need to copy previous, as backtracking
	      ;; across contexts is not allowed
	      :unique-p (ilink-unique-p ilink)
	      :broken-p (ilink-broken-p ilink))
)

(defun tl-copy-ilinks-hash (newilinks-list)
  (let ((newilinks-hash
	  (make-hash-table :test #'eq
			   :size (min 997 (* 3 (length newilinks-list))))))
    (dolist (ilink newilinks-list)
      (setf (gethash (ilink-target ilink) newilinks-hash) ilink))
    newilinks-hash)
)


;;; interval/clink field access semi-transparent to context

(defun tl-get-referents (tx)
  (let ((clink (tl-get-readable-clink tx)))
    (if clink (clink-referents clink)))
)

(defun tl-add-referent (tx rx)
  (unless (member rx (tl-get-referents tx))
    (push rx (clink-referents (tl-get-writable-clink tx)))
    (push tx (clink-referrals (tl-get-writable-clink rx)))
    rx)
)

(defun tl-del-referent (tx rx)
  ;; Does a non-destructive removal so current loops over referents list
  ;; continue to work properly
  (when (member rx (tl-get-referents tx))
    (let ((clink (tl-get-writable-clink tx)))
      (setf (clink-referents clink)
	    (remove rx (clink-referents clink))))
    (let ((clink (tl-get-writable-clink rx)))
      (setf (clink-referrals clink)
	    (remove tx (clink-referrals clink))))
    rx)
)

;;; Referrals

(defun tl-get-referrals (tx)
  (let ((clink (tl-get-readable-clink tx)))
    (if clink (clink-referrals clink)))
)

;;; no tl-add-referrals or tl-del-referrals !!!


;;; Rel ILinks

(defun tlr-get-ilinks (tx &optional all-p)
  (let ((clink (tl-get-readable-clink tx)))
    (when clink
      (if all-p
	  (clink-rel-ilinks-list-all clink)
	  (clink-rel-ilinks-list clink))))
)

(defun tlr-get-readable-ilink (tx ty)
  (let ((clink (tl-get-readable-clink tx)))
    (if clink (gethash ty (clink-rel-ilinks-hash clink))))
)

(defun tlr-get-writable-ilink (tx ty)
  (let ((clink (tl-get-writable-clink tx)))
    (or (gethash ty (clink-rel-ilinks-hash clink))
	(let ((ilink (make-ilink :type :rel
				 :source tx
				 :target ty
				 :current (tlr-const :all))))
	  (push ilink (clink-rel-ilinks-list clink))
	  (push ilink (clink-rel-ilinks-list-all clink))
	  (setf (gethash ty (clink-rel-ilinks-hash clink))
		ilink))))
)


;;; Dur ILinks

(defun tld-get-ilinks (tx &optional all-p)
  (let ((clink (tl-get-readable-clink tx)))
    (when clink
      (if all-p
	  (clink-dur-ilinks-list-all clink)
	  (clink-dur-ilinks-list clink))))
)

(defun tld-get-readable-ilink (tx ty)
  (let ((clink (tl-get-readable-clink tx)))
    (if clink (gethash ty (clink-dur-ilinks-hash clink))))
)

(defun tld-get-writable-ilink (tx ty)
  (let ((clink (tl-get-writable-clink tx)))
    (or (gethash ty (clink-dur-ilinks-hash clink))
	(let ((ilink (make-ilink :type :dur
				 :source tx
				 :target ty
				 :current (tld-const (0 :INF)))))
	  (push ilink (clink-dur-ilinks-list clink))
	  (setf (gethash ty (clink-dur-ilinks-hash clink))
		ilink))))
)



;;; End of file TL-CONTEXTS
