;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? temporal reasoning component

;;;    
;;;
;;;   propositions are (inst x y t) (isa x y t) (attr .... t)
;;;   all calls to time reasoner would use the time-int in place of the proposition
;;;
;;;   shared time-ints: if not shared flag is nil, there is by definition no structure in time graph
;;
;;;   two functions (time-assert int1 op int2 belief)
;;;;                (time-test int1 op int2 belief)
;;;
;;;; NOTE about assert: previously shared intervals may become unshared so propositions
;;;;    will have to change their intervals.... we will return a list (int1 int2) that
;;;        can be blindly set


;;; belief times in queries are time intervals that must be checked against belief-time constants
;;; belief times in asserts are belief-time constants







(defun time-intersect-conventional (time1 time2 belief make)
  (let ((start (time-max (time-start time1) (time-start time2)))
	(end (time-min (time-end time1) (time-end time2))))
    (if (not (time> start end))
      (cond ((and (eq start (time-start time1))
		  (eq end (time-end time1)))
	     time1)
	    ((and time2
		  (eq start (time-start time2))
		  (eq end (time-end time2)))
	     time2)
	    (make
	     (make-time-int :start start
			    :end end))
	    (t t))      
     nil))
  )


(defun time-intersect (time1 time2 belief &optional (make t))
  (cond ((eq time1 time2) time2)
	((and (time-conventional-p time1) (time-conventional-p time2))
	 (time-intersect-conventional time1 time2 belief make))
	((and (not make) (eq (time-not-shared time1) :unconstrained)) time2)
	((and (not make) (eq (time-not-shared time2) :unconstrained)) time1)
	((time-test time1 :during time2 belief) time1)
	((time-test time2 :during time1 belief) time2)
	(make (time-create-intersection time1 time2 belief))
	(t (time-test time1 :intersects time2 belief)))
  )

	 
(defmacro time-intersect-p (time1 time2 belief)
  `(time-intersect ,time1 ,time2 ,belief nil)
  )


;; constraint is in the query, context is on the link
;; this means that the constraint must contain context if the check is satisfied
;; a null context is true in every context

;; if the constraint is <during int>, belief-time satisfies it if its boundaries are not inside

(defun belief-satisfies (constraint-context belief-time-context)
  (let ((constraint (get-context-belief constraint-context))
	(c-context (get-context-context constraint-context))
	(belief-time (get-context-belief belief-time-context))
	(b-context (get-context-context belief-time-context)))
    (and (or (null b-context) (context-includes c-context b-context))
	 (case (belief-time-constraint constraint)
	   (:intersects (time> (time-max (time-start constraint) (time-start belief-time))
			       (time-min (time-end constraint) (time-end belief-time))))
	   (:during (and (not (time> (belief-time-start belief-time) (belief-time-start constraint)))
			 (not (time> (belief-time-end constraint) (belief-time-end belief-time)))))
	   (t nil)
	   )))
  )



(defun belief-during (time during-time)
  (and (not (time> (belief-time-start during-time) (belief-time-start time)))
       (not (time> (belief-time-end time) (belief-time-end during-time)))
       )
  )


(defun get-time-int (interval belief)
  (let* ((int (make-time-int :not-shared t))
	 (start (if (or (null interval) (time-point-p (time-start interval)))
		  (make-time-point :start :- :end :+ :interval int)
		  (time-start interval)))
	 (end (if (or (null interval) (time-point-p (time-end interval)))
		(make-time-point :start :- :end :+ :interval int)
		(time-end interval)))
	 )
    (setf (time-start int) start)
    (setf (time-end int) end)
    (if interval
      (progn
	(if (not (eq start (time-start interval)))
	  (time-assert-same start (time-start interval) belief))
	(if (not (eq end (time-end interval)))
	  (time-assert-same end (time-end interval) belief))
	(if (or (not (eq start (time-start interval)))
		(not (eq end (time-end interval))))
	  (time-assert-precedes start end belief))
	)
      (time-assert-precedes start end belief)
      )
    int
    )
  )


(defun compose-time-int (start end belief &optional (not-shared t))
  (let* ((int (make-time-int :not-shared not-shared))
	 (actual-start (if start start (make-time-point :start :- :end :+ :interval int)))
	 (actual-end (if end end (make-time-point :start :- :end :+ :interval int))))
    (setf (time-start int) actual-start)
    (setf (time-end int) actual-end)
    (time-assert-precedes actual-start actual-end belief)
    int
    )
  )





(defmacro time-check-consistent-precedes (p1 p2 belief)
  `(not (time-point-precedes ,p2 ,p1 ,belief))
  )




;; add all items from list2 to list1 and make sure that it is on all nodes in the equivalence class
;; list1 is never null

(defun time-merge-same (list1 list2)
  (lconc list1 (car list2))
  (doloop (link (car list1))
   :vars ((item1 nil (point-link-source link))
	  (item2 nil (point-link-dest link)))
    (when (and (time-point-p item1) (not (eq (point-same-as item1) list1)))
      (setf (point-same-as item1) list1))
    (when (and (time-point-p item2) (not (eq (point-same-as item2) list1)))
      (setf (point-same-as item2) list1))
    )
  list1
  )

(defvar *new-same-links* nil)
(defvar *new-precedes-links* nil)

(defun time-assert-same (p1 p2 belief)
  (if (not (time-point-p p1))
    (if (not (time-point-p p2))
     nil				; useless to say two conventional times are equal
      (time-assert-same p2 p1 belief)
      )
    (let ((link (list belief p2 p1))
	  (same1 (point-same-as p1))	; p1 is always a time point
	  same)
      (push link *new-same-links*)
      (if (null same1)
	(if (and (time-point-p p2) (point-same-as p2))
	  (setf same (point-same-as p2))
	  (setf same (tconc nil))
	  )
	(if (and (time-point-p p2) (point-same-as p2))
	  (setf same (time-merge-same same1 (point-same-as p2)))
	  (setf same same1)
	  )
	)
      (tconc same link)
      (when (not (eq (point-same-as p1) same))
	(setf (point-same-as p1) same))
      (when (and (time-point-p p2) (not (eq (point-same-as p2) same)))
	(setf (point-same-as p2) same)
	)
      )
    )
  )


;; have to check for cycles; could just check time-point-precedes but can be more efficient for new points

(defun time-assert-precedes (p1 p2 belief)
  (let ((link (list belief p2 p1)))
    (push link *new-precedes-links*)
    (when (time-point-p p1) (push link (point-precedes p1)))
    (when (time-point-p p2) (push link (point-preceded-by p2)))
    )
  )
     




(defmacro time-assert-meets (int1 int2 belief)
  `(time-assert-same (time-end ,int1) (time-start ,int2) ,belief)
  )


(defmacro time-assert-during (int1 int2 belief)
  `(progn
     (time-assert-precedes (time-start ,int2) (time-start ,int1) ,belief)
     (time-assert-precedes (time-end ,int1) (time-end ,int2) ,belief)
     )
  )


(defmacro time-assert-startsbefore (int1 int2 belief)
  `(time-assert-precedes (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-assert-endsbefore (int1 int2 belief)
  `(time-assert-precedes (time-end ,int1) (time-end ,int2) ,belief)
  )

(defmacro time-assert-rightbefore (int1 int2 belief)
  `(time-assert-same (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-assert-before (int1 int2 belief)
  `(time-assert-precedes (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-assert-costarts (int1 int2 belief)
  `(time-assert-same (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-assert-coends (int1 int2 belief)
  `(time-assert-same (time-end ,int1) (time-end ,int2) ,belief)
  )


(defmacro time-assert-overlaps (int1 int2 belief)
  `(progn
     (time-assert-precedes (time-end ,int1) (time-end ,int2) ,belief)
     (time-assert-precedes (time-start ,int2) (time-end ,int1) ,belief)
     )
  )


(defmacro time-assert-equals (int1 int2 belief)
  `(progn
     (time-assert-same (time-start ,int1) (time-start ,int2) ,belief)
     (time-assert-same (time-end ,int1) (time-end ,int2) ,belief)
     )
  )



(defun time-assert (input-int1 op input-int2 belief belief1 belief2 &optional errors)
  (let ((int1 (if (or (null (time-not-shared input-int1)) (eq (time-not-shared input-int1) :unconstrained))
		(get-time-int input-int1 belief1)
		input-int1))
	(int2 (if (or (null (time-not-shared input-int2)) (eq (time-not-shared input-int2) :unconstrained))
		(get-time-int input-int2 belief2)
		input-int2))
	(ok t))
    (setf ok (time-check-consistent int1 op int2 belief))
    (when (and (not ok) errors) (tconc errors `(inconsistent ,int1 ,op ,int2 ,belief)))
    (when (and (eq :bad-op ok) errors) (tconc errors `(bad-op ,int1 ,op ,int2 ,belief)))
    (if (and ok
	     (not (eq ok :bad-op))
	     (not (eq 'error
		      (case op
			(:meets (time-assert-meets int1 int2 belief))
			((:e := :equals :at) (time-assert-equals int1 int2 belief))
			((:d :during) (time-assert-during int1 int2 belief))
			((:o :over :includes :inc) (time-assert-during int2 int1 belief))
			((:sb :startsbefore) (time-assert-startsbefore int1 int2 belief))
			((:sa :startsafter) (time-assert-startsbefore int2 int1 belief))
			((:eb :endsbefore) (time-assert-endsbefore int1 int2 belief))
			((:ea :endsafter) (time-assert-endsbefore int2 int1 belief))
			((:rb :rightbefore) (time-assert-rightbefore int1 int2 belief))
			((:ra :rightafter) (time-assert-rightbefore int2 int1 belief))
			((:b :before) (time-assert-before int1 int2 belief))
			((:a :after) (time-assert-before int2 int1 belief))
			((:cs :costarts) (time-assert-costarts int1 int2 belief))
			((:ce :coends) (time-assert-coends int1 int2 belief))
			((:ov :overlaps) (time-assert-overlaps int1 int2 belief))
			((:ov-by :overlapped-by) (time-assert-overlaps int2 int1 belief))
			(t 'error)))))
      (list int1 int2)
     nil)
    )
  )


(defun time-create-intersection (int1 int2 belief)
  (let ((inter (compose-time-int nil nil belief (cons int1 int2))))
    (time-assert-precedes (time-start int1) (time-start inter) belief)
    (time-assert-precedes (time-start int2) (time-start inter) belief)
    (if (time-check-consistent-precedes (time-end inter) (time-end int1) belief)
      (progn
	(time-assert-precedes (time-end inter) (time-end int1) belief)
	(if (time-check-consistent-precedes (time-end inter) (time-end int2) belief)
	  (progn
	    (time-assert-precedes (time-end inter) (time-end int2) belief)
	    inter)
	  nil))
      nil)
    )
  )

     
(defun time-context-reset (&optional save-contexts)
  (doloop (link *new-same-links*)
    (if! (doloop (save save-contexts)
	  :some (or (null (get-context-context (point-link-belief link)))
		    (context-includes (get-context-context save) (get-context-context (point-link-belief link)))))
      (setf (point-link-belief link) (get-context-belief (point-link-belief link)))
     else
      (let* ((pair (if (time-point-p (point-link-dest link))
		     (point-same-as (point-link-dest link))
		     (point-same-as (point-link-source link))))
	     (new-list (remove link (car pair))))
	(setf (car pair) new-list)
	(setf (cdr pair) (last new-list))
	)
      )
    )
  (doloop (link *new-precedes-links*)
    (if! (doloop (save save-contexts)
	  :some (or (null (get-context-context (point-link-belief link)))
		    (context-includes (get-context-context save) (get-context-context (point-link-belief link)))))
      (setf (point-link-belief link) (get-context-belief (point-link-belief link)))
     else
      (let ((p2 (point-link-dest link))
	    (p1 (point-link-source link)))
	(if (time-point-p p1) (setf (point-precedes p1) (remove link (point-precedes p1))))
	(if (time-point-p p2) (setf (point-preceded-by p2) (remove link (point-preceded-by p2))))
	)
      )
    )
  (setf *new-same-links* nil)
  (setf *new-precedes-links* nil)
  (setq *next-time-context* 0)
  )


(defun time-point-same (p1 p2 belief)
  (cond ((eq p1 p2) t)
	((eq p1 :+) nil)
	((not (time-point-p p1))
	 (if (time-point-p p2)
	   (time-point-same p2 p1 belief)
	  nil))
	((or (null (point-same-as p1)) (and (time-point-p p2) (null (point-same-as p2)))) nil)
	(t (doloop (link (car (point-same-as p1)))
	    :when (and (belief-satisfies belief (point-link-belief link))
		       (or (eq (point-link-dest link) p2)
			   (eq (point-link-source link) p2)))
	    :return t
	     ))
	)
  )



;; like forward except no check of same equivalence class

(defun time-point-precedes-forward-forward (p1 p2 belief)
  (cond ((eq p1 p2) nil)
	((eq p1 :+) nil)
	((not (time-point-p p1))
	 (if (not (time-point-p p2))
	   (time> p2 p1)
	   (time-point-precedes-backward p1 p2 belief)))
	(t (doloop (link (point-precedes p1))
	    :when (and (belief-satisfies belief (point-link-belief link))
		       (time-point-precedes-forward (point-link-dest link) p2 belief))
	    :return t
	     )
	   ))
  )

(defun time-point-precedes-forward (p1 p2 belief)
  (cond ((eq p1 p2) t)
	((eq p2 :-) nil)
	((not (time-point-p p1))
	 (if (not (time-point-p p2))
	   (time> p2 p1)
	   (time-point-precedes-backward p1 p2 belief)))
	(t (or (doloop (link (car (point-same-as p1)))
		:when (and (belief-satisfies belief (point-link-belief link))
			   (time-point-precedes-forward-forward (if (eq (point-link-dest link) p1)
								  (point-link-source link)
								  (point-link-dest link))
								p2 belief))
		:return t
		 )
	       (doloop (link (point-precedes p1))
		:when (and (belief-satisfies belief (point-link-belief link))
			   (time-point-precedes-forward (point-link-dest link) p2 belief))
		:return t
		 ))
	   ))
  )


;; like backward but don't look at same equivalence class

(defun time-point-precedes-backward-backward (p1 p2 belief)
  (cond ((eq p1 p2) t)
	((eq p2 :-) nil)
	((not (time-point-p p2))
	 (if (not (time-point-p p1))
	   (progn (time> p2 p1))
	   (time-point-precedes-forward p1 p2 belief)))
	(t (doloop (link (point-preceded-by p2))
		:when (and (belief-satisfies belief (point-link-belief link))
			   (time-point-precedes-backward p1 (point-link-source link) belief))
		:return t
		 )
	   ))
  )



(defun time-point-precedes-backward (p1 p2 belief)
  (cond ((eq p1 p2) t)
	((eq p2 :-) nil)
	((not (time-point-p p2))
	 (if (not (time-point-p p1))
	   (progn (time> p2 p1))
	   (time-point-precedes-forward p1 p2 belief)))
	(t (or (doloop (link (car (point-same-as p2)))
		:when (and (belief-satisfies belief (point-link-belief link))
			   (time-point-precedes-backward-backward p1
								  (if (eq (point-link-dest link) p2)
								    (point-link-source link)
								    (point-link-dest link))
								  belief))
		:return t
		 )
	       (doloop (link (point-preceded-by p2))
		:when (and (belief-satisfies belief (point-link-belief link))
			   (time-point-precedes-backward p1 (point-link-source link) belief))
		:return t
		 ))
	   ))
  )


(defun time-point-precedes (p1 p2 belief)
  (cond ((eq p1 p2) nil)
	((or (eq p1 :-) (eq p2 :+)) t)
	((time-point-p p1)
	 (if (and (time-point-p p2) (null (point-preceded-by p2)) (null (point-same-as p2)))
	  nil
	   (if (or (point-precedes p1) (point-same-as p1))
	     (time-point-precedes-forward p1 p2 belief)
	    nil
	     )
	   ))
	((time-point-p p2)
	 (if (or (point-preceded-by p2) (point-same-as p2))
	   (time-point-precedes-backward p1 p2 belief)
	  nil
	   ))
	(t (time> p2 p1)))
  )

		       


(defmacro time-check-meets (int1 int2 belief)
  `(time-point-same (time-end ,int1) (time-start ,int2) ,belief)     
  )


(defmacro time-check-during (int1 int2 belief)
  `(and
     (time-point-precedes (time-start ,int2) (time-start ,int1) ,belief)
     (time-point-precedes (time-end ,int1) (time-end ,int2) ,belief)
     )
  )


(defmacro time-check-startsbefore (int1 int2 belief)
  `(time-point-precedes (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-endsbefore (int1 int2 belief)
  `(time-point-precedes (time-end ,int1) (time-end ,int2) ,belief)
  )

(defmacro time-check-rightbefore (int1 int2 belief)
  `(time-point-same (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-before (int1 int2 belief)
  `(time-point-precedes (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-costarts (int1 int2 belief)
  `(time-point-same (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-coends (int1 int2 belief)
  `(time-point-same (time-end ,int1) (time-end ,int2) ,belief)
  )

(defmacro time-check-overlaps (int1 int2 belief)
  `(and
     (time-point-precedes (time-end ,int1) (time-end ,int2) ,belief)
     (time-point-precedes (time-start ,int2) (time-end ,int1) ,belief)
     )
  )


(defmacro time-check-equals (int1 int2 belief)
  `(and
     (time-point-same (time-start ,int1) (time-start ,int2) ,belief)
     (time-point-same (time-end ,int1) (time-end ,int2) ,belief)
     )
  )



(defmacro time-check-intersect (int1 int2 belief)
  `(or
    (time-check-overlaps ,int1 ,int2 ,belief)
    (time-check-overlaps ,int2 ,int1 ,belief)
    )
  )


(defmacro time-intersection-consistent (int1 int2 belief)
  (macro-setup (iv1 iv2 ib)
    `(let ((,iv1 ,int1)
	   (,iv2 ,int2)
	   (,ib ,belief))
       (and
	(not (time-point-precedes (time-end ,iv1) (time-start ,iv2) ,ib))
	(not (time-point-precedes (time-end ,iv2) (time-start ,iv1) ,ib))
	)
       )
    ))


(defun time-test (int1 op int2 belief)
  (case op
    ((:i :intersects) (time-check-intersect int1 int2 belief))
    ((:m :meets) (time-check-meets int1 int2 belief))
    ((:e := :equals :at) (time-check-equals int1 int2 belief))
    ((:d :during) (time-check-during int1 int2 belief))
    ((:o :over :includes :inc) (time-check-during int2 int1 belief))
    ((:sb :startsbefore) (time-check-startsbefore int1 int2 belief))
    ((:sa :startsafter) (time-check-startsbefore int2 int1 belief))
    ((:eb :endsbefore) (time-check-endsbefore int1 int2 belief))
    ((:ea :endsafter) (time-check-endsbefore int2 int1 belief))
    ((:rb :rightbefore) (time-check-rightbefore int1 int2 belief))
    ((:ra :rightafter) (time-check-rightbefore int2 int1 belief))
    ((:b :before) (time-check-before int1 int2 belief))
    ((:a :after) (time-check-before int2 int1 belief))
    ((:cs :costarts) (time-check-costarts int1 int2 belief))
    ((:ce :coends) (time-check-coends int1 int2 belief))
    ((:ol :overlaps) (time-check-overlaps int1 int2 belief))
    ((:ol-by :overlapped-by) (time-check-overlaps int2 int1 belief))
    (t 'error))
  )
		 






(defun normal-graph (i &optional (seen (make-hash-table)))
  (cond ((listp i)
	 (doloop (item i) :collect (normal-graph item)))
	((not (time-point-p i)) i)
	((gethash i seen) (normal-hash i "tp"))
	(t (setf (gethash i seen) t)
	   (list (normal-hash i "tp")
		 :same
		 (doloop (link (car (point-same-as i)))
		  :collect (normal-hash (if (eq (point-link-dest link) i)
					  (point-link-source link)
					  (point-link-dest link))
					"tp"))
		 :precedes
		 (doloop (link (point-precedes i))
		  :collect (normal-graph (point-link-dest link) seen))))
	))





(defun time-point-links (i)
  (cond ((listp i)
	 (doloop (item i) :collect (normal-graph item)))
	((not (time-point-p i)) i)
	(t (list (normal-hash i "tp")
		 :same
		 (doloop (link (car (point-same-as i)))
		  :collect (normal-hash (if (eq (point-link-dest link) i)
					  (point-link-source link)
					  (point-link-dest link))
					"tp"))
		 :precedes
		 (doloop (link (point-precedes i))
		  :collect (normal-hash (point-link-dest link) "tp"))))
	)
  )


(defun time-interval-links (i)
  (cond ((listp i) (doloop (item i) :collect (time-interval-links item)))
	((time-int-p i) (list (normal-clause i) (time-point-links (time-start i)) (time-point-links (time-end i))))
	(t i))
  )
	    


(defun dotest nil
  (setq b (std-belief))
  (setq i- (compose-time-int :- 20 b))
  (setq i+ (compose-time-int 10 :+ b))
  (setq i0 (compose-time-int 10 20 b))
  (setq i1 (get-time-int nil b))
  (setq i2 (get-time-int nil b))
  (setq i3 (get-time-int nil b))	 
  (setq i4 (get-time-int nil b))	 
  (setq i5 (get-time-int nil b))
  (setq i6 (compose-time-int nil 40 b))
  (setq i7 (get-time-int nil b))
  (setq i8 (get-time-int nil b))
  (setq i9 (get-time-int nil b))	 
  (setq i10 (get-time-int nil b))	 
  (setq i11 (get-time-int nil b))
  (setq i12 (compose-time-int 20 40 b))
  (setq i13 (compose-time-int 30 50 b))
  (dbgn-print 
   '1b2 (time-assert i1 :before i2 b b b)
   '2b3 (time-assert i2 :before i3 b b b)
   '6b3 (time-assert i6 :before i3 b b b)
   '3eq4 (time-assert i3 :equals i4 b b b)
   '4b5 (time-assert i4 :before i5 b b b)
   '7m2 (time-assert i7 :meets i2 b b b)


   '8b9 (time-assert i8 :before i9 b b b)
   '10ov9 (time-assert i10 :ov i9 b b b)
   '11ov10 (time-assert i11 :ov i10 b b b)
   '8ea11 (time-assert i8 :ea i11 b b b)
   )
  
  (dbgn-print 'test
	      '1b5 (eq t (time-test i1 :before i5 b))
	      '7m2 (eq t (time-test i7 :meets i2 b))
	      '7b5 (eq t (time-test i7 :before i5 b))
	      '8i10 (eq t (time-test i8 :intersects i10 b))
	      '12i13 (eq t (time-test i12 :intersects i13 b))
	      )
  )





(defun time-unassert-same (p1 p2 belief) ; belief time is (x,+)
  (if (not (time-point-p p1))
    (if (not (time-point-p p2))
     nil				; useless to say two conventional times are equal
      (time-unassert-same p2 p1 belief)
      )
    (doloop (link (car (point-same-as p1)))
      :when (and (belief-satisfies belief (point-link-belief link))
		 (or (and (eq (point-link-source link) p1)
			  (eq (point-link-dest link) p2))
		     (and (eq (point-link-source link) p2)
			  (eq (point-link-dest link) p1)))
		 )
      (setf (point-link-belief link) (make-belief-time :start (belief-time-start (point-link-belief link)) :end (belief-time-start belief)))
      )
    )
  )


(defun time-unassert-precedes (p1 p2 belief) ; belief time is (x,+)
  (when (time-point-p p1)
    (doloop (link (point-precedes p1))
      :when (and (belief-satisfies belief (point-link-belief link))
		 (and (eq (point-link-source link) p1)
		      (eq (point-link-dest link) p2))
		 )
      (setf (point-link-belief link) (make-belief-time :start (belief-time-start (point-link-belief link)) :end (belief-time-start belief)))
      )    
    )
  (when (time-point-p p2)
    (doloop (link (point-preceded-by p2))
      :when (and (belief-satisfies belief (point-link-belief link))
		 (and (eq (point-link-source link) p1)
		      (eq (point-link-dest link) p2))
		 )
      (setf (point-link-belief link) (make-belief-time :start (belief-time-start (point-link-belief link)) :end (belief-time-start belief)))
      )    
    )
  )
     


(defmacro time-unassert-meets (int1 int2 belief)
  `(time-unassert-same (time-end ,int1) (time-start ,int2) ,belief)
  )


(defmacro time-unassert-during (int1 int2 belief)
  `(progn
     (time-unassert-precedes (time-start ,int2) (time-start ,int1) ,belief)
     (time-unassert-precedes (time-end ,int1) (time-end ,int2) ,belief)
     )
  )


(defmacro time-unassert-startsbefore (int1 int2 belief)
  `(time-unassert-precedes (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-unassert-endsbefore (int1 int2 belief)
  `(time-unassert-precedes (time-end ,int1) (time-end ,int2) ,belief)
  )

(defmacro time-unassert-rightbefore (int1 int2 belief)
  `(time-unassert-same (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-unassert-before (int1 int2 belief)
  `(time-unassert-precedes (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-unassert-costarts (int1 int2 belief)
  `(time-unassert-same (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-unassert-coends (int1 int2 belief)
  `(time-unassert-same (time-end ,int1) (time-end ,int2) ,belief)
  )

(defmacro time-unassert-overlaps (int1 int2 belief)
  `(progn
     (time-unassert-precedes (time-end ,int1) (time-end ,int2) ,belief)
     (time-unassert-precedes (time-start ,int2) (time-end ,int1) ,belief)
     )
  )


(defmacro time-unassert-equals (int1 int2 belief)
  `(progn
     (time-unassert-same (time-start ,int1) (time-start ,int2) ,belief)
     (time-unassert-same (time-end ,int1) (time-end ,int2) ,belief)
     )
  )



(defun time-unassert (int1 op int2 belief)
 (if (not (eq 'error
		 (case op
		   (:meets (time-unassert-meets int1 int2 belief))
		   ((:equals :at) (time-unassert-equals int1 int2 belief))
		   ((:during) (time-unassert-during int1 int2 belief))
		   ((:over :inc :o :includes) (time-unassert-during int2 int1 belief))
		   ((:startsbefore) (time-unassert-startsbefore int1 int2 belief))
		   ((:startsafter) (time-unassert-startsbefore int2 int1 belief))
		   ((:endsbefore) (time-unassert-endsbefore int1 int2 belief))
		   ((:endsafter) (time-unassert-endsbefore int2 int1 belief))
		   ((:rightbefore) (time-unassert-rightbefore int1 int2 belief))
		   ((:rightafter) (time-unassert-rightbefore int2 int1 belief))
		   ((:before) (time-unassert-before int1 int2 belief))
		   ((:after) (time-unassert-before int2 int1 belief))
		   ((:costarts) (time-unassert-costarts int1 int2 belief))
		   ((:coends) (time-unassert-coends int1 int2 belief))
		   ((:overlaps) (time-unassert-overlaps int1 int2 belief))
		   ((:overlapped-by) (time-unassert-overlaps int2 int1 belief))
		   (t 'error))))
      (list int1 int2)
     nil)
  )






(defun time-check-consistent-same (p1 p2 belief)
  (if (not (time-point-p p1))
    (if (not (time-point-p p2))
      (eq p1 p2)
      (or (eq p1 :+)
	  (eq p1 :-)
	  (time-check-consistent-same p2 p1 belief))
      )
    (or (eq p2 :+)
	(eq p2 :-)
	(and (not (time-point-precedes p1 p2 belief))
	     (not (time-point-precedes p2 p1 belief))))
    )
  )


     


(defmacro time-check-consistent-meets (int1 int2 belief)
  `(time-check-consistent-same (time-end ,int1) (time-start ,int2) ,belief)
  )


(defmacro time-check-consistent-during (int1 int2 belief)
  `(and
     (time-check-consistent-precedes (time-start ,int2) (time-start ,int1) ,belief)
     (time-check-consistent-precedes (time-end ,int1) (time-end ,int2) ,belief)
     )
  )


(defmacro time-check-consistent-startsbefore (int1 int2 belief)
  `(time-check-consistent-precedes (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-consistent-endsbefore (int1 int2 belief)
  `(time-check-consistent-precedes (time-end ,int1) (time-end ,int2) ,belief)
  )

(defmacro time-check-consistent-rightbefore (int1 int2 belief)
  `(and (time-check-consistent-same (time-end ,int1) (time-start ,int2) ,belief)
	(time-check-consistent-precedes (time-start ,int1) (time-start ,int2) ,belief))
  )

(defmacro time-check-consistent-before (int1 int2 belief)
  `(time-check-consistent-precedes (time-end ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-consistent-costarts (int1 int2 belief)
  `(time-check-consistent-same (time-start ,int1) (time-start ,int2) ,belief)
  )

(defmacro time-check-consistent-coends (int1 int2 belief)
  `(time-check-consistent-same (time-end ,int1) (time-end ,int2) ,belief)
  )


(defmacro time-check-consistent-overlaps (int1 int2 belief)
  `(and
     (time-check-consistent-precedes (time-end ,int1) (time-end ,int2) ,belief)
     (time-check-consistent-precedes (time-start ,int2) (time-end ,int1) ,belief)
     )
  )


(defmacro time-check-consistent-equals (int1 int2 belief)
  `(and
     (time-check-consistent-same (time-start ,int1) (time-start ,int2) ,belief)
     (time-check-consistent-same (time-end ,int1) (time-end ,int2) ,belief)
     )
  )



(defun time-check-consistent (int1 op int2 belief)
  (case op
    (:meets (time-check-consistent-meets int1 int2 belief))
    ((:e := :equals :at) (time-check-consistent-equals int1 int2 belief))
    ((:d :during) (time-check-consistent-during int1 int2 belief))
    ((:o :over :inc :includes) (time-check-consistent-during int2 int1 belief))
    ((:sb :startsbefore) (time-check-consistent-startsbefore int1 int2 belief))
    ((:sa :startsafter) (time-check-consistent-startsbefore int2 int1 belief))
    ((:eb :endsbefore) (time-check-consistent-endsbefore int1 int2 belief))
    ((:ea :endsafter) (time-check-consistent-endsbefore int2 int1 belief))
    ((:rb :rightbefore) (time-check-consistent-rightbefore int1 int2 belief))
    ((:ra :rightafter) (time-check-consistent-rightbefore int2 int1 belief))
    ((:b :before) (time-check-consistent-before int1 int2 belief))
    ((:a :after) (time-check-consistent-before int2 int1 belief))
    ((:cs :costarts) (time-check-consistent-costarts int1 int2 belief))
    ((:ce :coends) (time-check-consistent-coends int1 int2 belief))
    ((:ov :overlaps) (time-check-consistent-overlaps int1 int2 belief))
    ((:ov-by :overlapped-by) (time-check-consistent-overlaps int2 int1 belief))
    (t :bad-op))
  )

(defun time-lower-bound-point-back (p belief bound)
  (cond ((not (time-point-p p))
	 (setf (car bound) (time-max (car bound) p))
	 bound)
	(t (doloop (link (point-preceded-by p))
		:when (belief-satisfies belief (point-link-belief link))
		 (time-lower-bound-point (point-link-source link) belief bound)
		 )
	   bound
	   ))
  )



(defun time-lower-bound-point (p belief bound)
  (cond ((not (time-point-p p))
	 (setf (car bound) (time-max (car bound) p))
	 bound)
	(t (doloop (link (car (point-same-as p)))
	    :when (belief-satisfies belief (point-link-belief link))
	     (time-lower-bound-point-back (if (eq (point-link-dest link) p)
					    (point-link-source link)
					    (point-link-dest link))
					  belief bound)
	     )
	   (doloop (link (point-preceded-by p))
	    :when (belief-satisfies belief (point-link-belief link))
	     (time-lower-bound-point (point-link-source link) belief bound)
	     )
	   bound))
  )




(defun time-upper-bound-point-back (p belief bound)
  (cond ((not (time-point-p p))
	 (setf (car bound) (time-min (car bound) p))
	 bound)
	(t (doloop (link (point-precedes p))
		:when (belief-satisfies belief (point-link-belief link))
		 (time-upper-bound-point (point-link-dest link) belief bound)
		 )
	   bound
	   ))
  )



(defun time-upper-bound-point (p belief bound)
  (cond ((not (time-point-p p))
	 (setf (car bound) (time-min (car bound) p))
	 bound)
	(t (doloop (link (car (point-same-as p)))
	    :when (belief-satisfies belief (point-link-belief link))
	     (time-upper-bound-point-back (if (eq (point-link-dest link) p)
					    (point-link-source link)
					    (point-link-dest link))
					  belief bound)
	     )
	   (doloop (link (point-precedes p))
	    :when (belief-satisfies belief (point-link-belief link))
	     (time-upper-bound-point (point-link-dest link) belief bound)
	     )
	   bound))
  )



(defun time-interval-bounds (interval belief)
  (let ((lowers (list :-))
	(uppers (list :+))
	(lowere (list :-))
	(uppere (list :+)))
    (time-upper-bound-point (time-start interval) belief uppers)
    (time-lower-bound-point (time-start interval) belief lowers)
    (time-upper-bound-point (time-end interval) belief uppere)
    (time-lower-bound-point (time-end interval) belief lowere)
    (list (list (car lowers) (car uppers)) (list (car lowere) (car uppere)))
    )
  )
