;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         sticks.l
; Description:  Expert system to play NIM.
; Author:       Eric Muehle
; Created:      18-Aug-87
; Package:      USER
; RCS $Header: sticks.l,v 1.3 88/03/19 15:58:10 jed Exp $
;
; (c) Copyright 1987, University of Utah, all rights reserved
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Expert system to play the game of NIM.  The game is
;;;; played by two players who take 1,2 or 3 sticks from
;;;; a set of sticks.  The player who takes the last stick 
;;;; LOSES.

;;; The game state which keeps track of the number of 
;;; sticks and whose turn it is.  
(def-class state nil
  :slots (turn       ; whose turn is it? p1 or p2
          p1         ; type of player 1
          p2         ; type of player 2
          num-left)) ; number of sticks left

;;; Sets up the game and starts the system.
(def-method (state start)(sticks p1 p2)
  (setf (num-left $self) sticks)
  (setf (p1 $self) p1)
  (setf (p2 $self) p2)
  (erase $self 'turn)
  (assert-val $self 'turn 'p1)
  nil)

;;; This needs to be defined at compile time so
;;; new-instance is used instead of new-instance*
(new-instance {class state} :name game)


;;; The game is over when there are no more sticks left.  
;;; Whichever player takes the last stick loses.  This 
;;; rule should always be considered before any other rules.
(def-rule game-over 
  :value :best
  :prem ((num-left {game} 0))
  :conc (progn
          (format t "~S won!!!!~%" (turn {game}))
          ;; lets stop any further rules from firing
          (stop-fc)))

;;; Determine how many sticks the computer should remove.
(def-rule comp-move 
  :local (?n ?move ?turn)
  :prem ((turn {game} ?turn)
         (?turn {game} computer)
         (num-left {game} ?n)
         (eval (format t "There are ~D sticks left.~%" ?n))
         (bind ?move (mod (+ 3 (mod ?n 4)) 4)))
  :conc (progn
          (when (zerop ?move)
            (format t "~S is in a losing state!~%" ?turn)
            (setf ?move 1))
          (format t "~S removes ~D sticks.~%" ?turn ?move)
          (setf (turn {game}) (if (eq ?turn 'p1) 'p2 'p1))
          (assert-val {game} 'num-left (- ?n ?move))))

;;; The human player is asked to remove a number of sticks.
(def-rule human-move 
  :local (?n ?move ?turn)
  :prem ((turn {game} ?turn)
         (?turn {game} human)
         (num-left {game} ?n)
         (eval (format t "There are ~D sticks left.~%" ?n))
         (bind ?move (get-move ?n)))
  :conc (progn
          (setf (turn {game}) (if (eq ?turn 'p1) 'p2 'p1))
          (assert-val {game} 'num-left (- ?n ?move))))

;;; Returns the human player input.
(defun get-move (n)
  (let ((big  (min n 3))
        move)
    (format t "Please enter an integer between 1 and ~D~%" big)
    (setf move (read))
    (if (and (integerp move)
             (> move 0)
             (<= move big))
      move
      (get-move n))))

;;; The invoking call
;(start {game} 10 'computer 'human)

;; End of file.
