;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         daemons.l
; Description:  contains the daemon system
; Author:       Eric Muehle
; Created:      11-May-87
; Package:      FROBS
; RCS $Header: daemons.l,v 2.3 88/04/08 09:58:41 kessler Exp $
;
; (c) Copyright 1987, University of Utah, all rights reserved
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(provide 'daemons)

(in-package 'frobs)

(def-class init-daemon nil 
  :generic ((%%init-daemon nil)))

(defmacro define-init-daemon ((class name) &body code)
  `(if (slot? ,class '%%init-daemon)
     (progn
       (defun ,name ($self $parent) ,@code)
       (pushnew ',name (%%init-daemon ,class)))
     (error "Class ~S does not inherit from {CLASS INIT-DAEMON}~%" ',class)))


(define-init-daemon ({class init-daemon} i-d-init)
  (dolist (daemon (%%init-daemon $parent))
    (pushnew daemon (%%init-daemon $self))))

(def-sys 'init-daemon)
(def-sys-meth '(%%init-daemon set-%%init-daemon))

;;; Runs the init time daemons
(defun run-init-time-daemons (parents class-frob)
  (when (slot? class-frob '%%init-daemon)
    (setf (%%init-daemon class-frob) nil))
  (dolist (p parents)
    (when (slot? p '%%init-daemon)
      (dolist (fn (%%init-daemon p))
	(pushnew fn (%%init-daemon class-frob))
	(funcall fn class-frob p)))))

;;; The daemon class definition
(def-class daemon {class init-daemon}
   :generic ((before-assert nil)
	     (after-assert nil)
	     (before-ask nil)
	     (after-ask nil)
	     (before-remove nil)
	     (after-remove nil)))

;;; Pushes the daemons from a parent daemon class onto a child
;;; daemon class.
(defmacro daemon-init-aux (slot class parent)
  `(let (slot fns)
     (do ((pair (,slot ,parent) (cddr pair)))
	 ((null pair))
	 (setf slot (car pair))
	 (setf fns (second pair))
	 (dolist (fn fns)
	   (pushnew fn (getf (,slot ,class) slot))))))

;;; Inherits the daemons from a parent daemon class to a child daemon
;;; class.
(define-init-daemon ({class daemon} daemon-init)
  (daemon-init-aux before-assert $self $parent)
  (daemon-init-aux after-assert $self $parent)
  (daemon-init-aux before-ask $self $parent)
  (daemon-init-aux after-ask $self $parent)
  (daemon-init-aux before-remove $self $parent)
  (daemon-init-aux after-remove $self $parent))

(def-sys 'daemon)
(def-sys-meth '(before-assert set-before-assert after-assert set-after-assert
	        before-ask set-before-ask after-ask set-after-ask
		before-remove set-before-remove after-remove set-after-remove))

;;; Form that defines daemons.  Expects a class frob, daemon name, a slot,
;;; and a type.   Type must be one of :before-assert :after-assert :before-ask
;;; :after-ask :before-remove :after-remove
(defmacro define-daemon ((class name slot type) &rest code)
  (when (symbolp slot)(setf slot (list slot)))
  `(eval-when (load compile eval)
     (defun ,name ($self $slot $value) ,@code)
     (dolist (sl ',slot)
       (install-daemon ,class ',name sl ,type))
     ',name))

;;; Installs the daemon onto the class frob.
(defun install-daemon (class name slot type)
  (cond ((or (gethash (class-frob-name class) *daemons*)
	     (and (member 'daemon (class-frob-generic class) :key #'car)
		  (setf (gethash (class-frob-name class) *daemons*) t)))
	 (case type
	   (:before-assert (pushnew name (getf (before-assert class) slot)))
	   (:after-assert (pushnew name (getf (after-assert class) slot)))
	   (:before-ask (pushnew name (getf (before-ask class) slot)))
	   (:after-ask (pushnew name (getf (after-ask class) slot)))
	   (:before-remove (pushnew name (getf (before-remove class) slot)))
	   (:after-remove (pushnew name (getf (after-remove class) slot)))
	   (otherwise (error "Bad daemon type ~S~%" type))))
	(t (error "~S does not inherit from {class daemon}~%" class)))
  name)

;;; Removes a daemon
(defun remove-daemon (class slot type name)
  (cond ((gethash (class-frob-name class) *daemons*)
	 (case type
	   (:before-assert 
	    (setf (getf (before-assert class) slot)
		  (delete name (getf (before-assert class) slot))))
	   (:after-assert 
	    (setf (getf (after-assert class) slot)
		  (delete name (getf (after-assert class) slot))))
	   (:before-ask 
	    (setf (getf (before-ask class) slot)
		  (delete name (getf (before-ask class) slot))))
	   (:after-ask
	    (setf (getf (after-ask class) slot)
		  (delete name (getf (after-ask class) slot))))
	   (:before-remove
	    (setf (getf (before-remove class) slot)
		  (delete name (getf (before-remove class) slot))))
	   (:after-remove
	    (setf (getf (after-remove class) slot)
		  (delete name (getf (after-remove class) slot))))
	   (otherwise (error "Bad daemon type ~S~%" type))))
	(t (error "~S does not inherit from {CLASS DAEMON}~%" class))))

;;; The real version of check daemons that works!
(defun check-daemons (time frob slot class value)
  (let ((daemons 
	 (getf 
	  (case time
	    (:before-assert (before-assert class))
	    (:after-assert (after-assert class))
	    (:before-ask (before-ask class))
	    (:after-ask (after-ask class))
	    (:before-remove (before-remove class))
	    (:after-remove (after-remove class)))
	  slot)))
    (dolist (fn daemons)
      (funcall fn frob slot value))))

;; End of file.
