/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <klic/basic.h>
#include <klic/struct.h>
#include <klic/primitives.h>
#include <nlist.h>
#include <strings.h>
#include <ctype.h>
#include <stdio.h>

#define MAXARGSTRACED		99
#define TRACEGOALEXTBULK	64

char *program_name;

static struct mod_table_entry {
  module func;
  char *name;
} *mod_table = 0, **mod_index;

static struct pred_table_entry {
  struct predicate *pred;
  struct mod_table_entry *mte;
  char *name;
  char spied;
  char default_trace;
} *pred_table = 0;

enum trace_port {
  Call, Susp, Commit, Fail, AllPorts, NoMorePort, UnknownPort };
static int enabled, leashed, leaping, spying;

#define Enabled(port)	(enabled >> (port) & 1)
#define Leashed(port)	(leashed >> (port) & 1)

/* For remembering goals resumed during reduction */

static int max_resumed_goals = 0;
static int n_resumed = 0;
static struct goalrec **resumed_goals;

/****************************************
  Predicate and Module Tables
****************************************/

static unsigned int n_mod = 0, n_pred = 0;

static mod_compare_names(x, y)
     struct mod_table_entry *x, *y;
{
  return strcmp(x->name, y->name);
}

static mod_compare_addr(x, y)
     struct mod_table_entry **x, **y;
{
  return (*x)->func - (*y)->func;
}

static pred_compare_names(x, y)
     struct pred_table_entry *x, *y;
{
  int result;
  if (!(result = strcmp(x->mte->name, y->mte->name)))
    if (!(result = strcmp(x->name, y->name)))
      result = x->pred->arity - y->pred->arity;
  return result;
}  

static struct mod_table_entry *get_mte(pred)
     struct predicate *pred;
{
  module mod = pred->func;
  unsigned int low = 0, high = n_mod, mid;
  while (low != high) {
    mid = ((high+low) >> 1);
    if (mod_index[mid]->func == mod) {
      return mod_index[mid];
    } else if (mod_index[mid]->func > mod) {
      high = mid;
    } else {
      low = mid + 1;
    }
  }
  return 0;
}

static struct pred_hash_entry {
  struct predicate *pred;
  struct pred_table_entry *pte;
} *pred_hash;

static unsigned int pred_hash_mask;

static enter_pred_hash(pred, pte)
     struct predicate *pred;
     struct pred_table_entry *pte;
{
  int h = ((((unsigned int) pred) >> 2) & pred_hash_mask);
  while (pred_hash[h].pred != 0) {
    h = (h + 1) & pred_hash_mask;
  }
  pred_hash[h].pred = pred;
  pred_hash[h].pte = pte;
}  

static struct pred_table_entry *get_pte(pred)
     struct predicate *pred;
{
  int h = ((((unsigned int) pred) >> 2) & pred_hash_mask);
  while (pred_hash[h].pred != pred) {
    h = (h + 1) & pred_hash_mask;
  }
  return pred_hash[h].pte;
}  

static make_name_tables()
{
  char tempname[] = "/tmp/klicXXXXXX";
  char command[100];
  FILE *nmout;
  int k;
  extern char *malloc(), *calloc();
  static int compress_ul();
  struct mod_table_entry *mte, **mtxe;
  struct pred_table_entry *pte, **ptxe;

  mktemp(tempname);
  sprintf(command, "nm -gp %s >%s", program_name, tempname);
  system(command);
  nmout = fopen(tempname, "r");
  for (;;) {
    unsigned int adr, len;
    char typ, *cp;
    char nam[500];
    if (fscanf(nmout, "%x %c %s", &adr, &typ, &nam) == EOF) break;
    if (nam[0] == '_' &&
	nam[1] != '_') {
      if ((cp = rindex(nam, '_')) != &nam[1] &&
	  strcmp(cp+1, "module") == 0 &&
	  strlen(cp+1) == 6) {
	n_mod++;
      } else if (strncmp(nam+1, "predicate_", 10) == 0) {
	n_pred++;
      }
    }
  }
  mod_table =
    (struct mod_table_entry *)
      calloc(n_mod, sizeof(struct mod_table_entry));
  mod_index =
    (struct mod_table_entry **)
      calloc(n_mod, sizeof(struct mod_table_entry *));
  pred_table =
    (struct pred_table_entry *)
      calloc(n_pred, sizeof(struct pred_table_entry));
  {
    unsigned int hash_size = 1;
    while (hash_size <= (3*n_pred/2)) hash_size *= 2;
    pred_hash =
      (struct pred_hash_entry *)
	calloc(hash_size, sizeof(struct pred_hash_entry));
    pred_hash_mask = hash_size - 1;
    for (k = 0; k < hash_size; k++) {
      pred_hash[k].pred = 0;
      pred_hash[k].pte = 0;
    }
  }
  rewind(nmout);
  mte = mod_table; mtxe = mod_index;
  pte = pred_table;
  for (;;) {
    unsigned int adr, len;
    char typ, *cp;
    char nam[500];
    if (fscanf(nmout, "%x %c %s", &adr, &typ, &nam) == EOF) break;
    if (nam[0] == '_' &&
	nam[1] != '_') {
      if ((cp = rindex(nam, '_')) != &nam[1] &&
	  strcmp(cp+1, "module") == 0 &&
	  strlen(cp+1) == 6) {
	len = compress_ul(nam+1, strlen(nam+1)-7, nam);
	mte->name = malloc(len+1);
	strncpy(mte->name, nam, len);
	mte->name[len] = 0;
	mte->func = (module) adr;
	*mtxe++ = mte;
	mte++;
      } else if (strncmp(nam+1, "predicate_", 10) == 0) {
	char *cpt;
	cp = nam+11;
	do {
	  cp = index(cp, '_');
	} while (*(cp+1) != '_');
	cp += 2;
	cpt = cp + strlen(cp) - 1;
	while (isdigit(*cpt)) cpt--;
	cpt --;
	len = compress_ul(cp, cpt-cp, nam);
	pte->name = malloc(len+1);
	strncpy(pte->name, nam, len);
	pte->name[len] = 0;
	pte->pred = (struct predicate *) adr;
	pte->spied = 0;
	pte->default_trace = 1;
	pte++;
      }
    }
  }
  unlink(tempname);

  qsort(mod_table, n_mod, sizeof(struct mod_table_entry),
	mod_compare_names);
  qsort(mod_index, n_mod, sizeof(struct mod_table_entry *),
	mod_compare_addr);

  for (k = 0; k < n_pred; k++) {
    pred_table[k].mte = get_mte(pred_table[k].pred);
  }

  qsort(pred_table, n_pred, sizeof(struct pred_table_entry),
	pred_compare_names);
  for (k = 0; k < n_pred; k++) {
    enter_pred_hash(pred_table[k].pred, &pred_table[k]);
  }
}

static compress_ul(src, len, dest)
     char *src, *dest;
     int len;
{
  int destlen = 0;
  while (len--) {
    int c = *src++;
    if (c != '_') {
      *dest++ = c;
    } else if (*src == '_') {
      *dest++ = *src++; len--;
    } else {
      char buf[3];
      int n;
      buf[0] = *src++; len--;
      buf[1] = *src++; len--;
      buf[2] = 0;
      sscanf(buf, "%x", &n);
      *dest++ = n;
    }
    destlen++;
  }
  return destlen;
}

/****************************************
  Initiation
****************************************/

static void *trace_trigger_routine();
struct predicate trace_trigger_preds[MAXARGSTRACED+1];

initiate_trace()
{
  int k;
  enabled = leashed = -1;	/* all one */
  leaping = 0;
  make_name_tables();
  for (k=0; k<=MAXARGSTRACED; k++) {
    trace_trigger_preds[k].func = trace_trigger_routine;
    trace_trigger_preds[k].pred = k;
    trace_trigger_preds[k].arity = k+1;
  }
}

/****************************************
  Main Tracing Routines
****************************************/

int trace_flag;
int verbose_print = 0;
static struct goalrec *qp_before;
static q *real_heaplimit;
static struct goalrec *parent;
static char subgoal_trace[100];
static char is_resumed[100];
struct goalrec *trace_fg = 0;

struct goalrec *
alloc_trace_goal_extension()
{
  struct goalrec *ret;
  char *calloc();
  if (trace_fg == 0) {
    int k;
    ret =
      (struct goalrec *)
	calloc(TRACEGOALEXTBULK, sizeof(struct goalrec));
    for (k = TRACEGOALEXTBULK; k > 0; k--) {
      ret->next = trace_fg;
      trace_fg = ret;
      ret++;
    }
  }
  ret = trace_fg;
  trace_fg = trace_fg->next;
  return ret;
}
    
#define free_trace_goal_extension(g)		\
{ (g)->next = trace_fg; trace_fg = (g); }

free_goal_records(g)
     struct goalrec *g;
{
  int k = g->pred->arity + 1;
  struct goalrec *xg;
  do {
    xg = (struct goalrec *)g->args[NUMGOALARGS-1];
    free_trace_goal_extension(g);
    g = xg;
    k -= NUMGOALARGS+1;
  } while (k > 0);
}

#define Traced(g)				\
  ((g)->pred->func == trace_trigger_routine)

trace_goal(g)
     struct goalrec *g;
{
  unsigned int xarity = g->pred->arity;
  struct goalrec *xg = g;
  q a0 = g->args[0];
  while (xarity > NUMGOALARGS) {
    xg = (struct goalrec *)xg->args[NUMGOALARGS-1];
    xarity -= NUMGOALARGS+1;
  }
  if (xarity != NUMGOALARGS) {
    /* save arg 0 as the last arg */
    xg->args[xarity] = a0;
  } else {
    /* on boundary: there's no space in xg to save arg 0 */
    q lastarg = xg->args[NUMGOALARGS-1];
    struct goalrec *xxg = alloc_trace_goal_extension();
    xg->args[NUMGOALARGS-1] = (q) xxg;
    xxg->args[-2] = lastarg;
    xxg->args[-1] = a0;
  }
  g->args[0] = makeatomic(g->pred);
  g->pred = &trace_trigger_preds[g->pred->arity];
}

untrace_goal(g)
     struct goalrec *g;
{
  unsigned int xarity = g->pred->arity - 1;
  struct goalrec *xg = g;
  g->pred = predp(g->args[0]);
  while (xarity > NUMGOALARGS) {
    xg = (struct goalrec *)xg->args[NUMGOALARGS-1];
    xarity -= NUMGOALARGS+1;
  }
  if (xarity != NUMGOALARGS) {
    /* restore arg 0 from the last arg pos */
    g->args[0] = xg->args[xarity];
  } else {
    /* on boundary: another goal record allocated */
    struct goalrec *xxg = (struct goalrec *)xg->args[NUMGOALARGS-1];
    g->args[0] = xxg->args[-1];
    xg->args[NUMGOALARGS-1] = xxg->args[-2];
    free_trace_goal_extension(xxg);
  }
}

struct goalrec *
copy_traced_goal(g)
     struct goalrec *g;
{
  int k = g->pred->arity - NUMGOALARGS;
  struct goalrec *copied = alloc_trace_goal_extension();
  struct goalrec *x = copied;
  *x = *g;
  while (k > 0) {
    x->args[NUMGOALARGS-1] = (q)alloc_trace_goal_extension();
    x = (struct goalrec *)x->args[NUMGOALARGS-1];
    g = (struct goalrec *)g->args[NUMGOALARGS-1];
    *x = *g;
    k -= NUMGOALARGS+1;
  }
  return copied;
}

static void *trace_trigger_routine(glbl, qp, allocp, fg, toppred)
  struct global_variables *glbl;
  struct goalrec *qp;
  struct goalrec *fg;
  q *allocp;
  Const struct predicate *toppred;
{
  unsigned int arity;
  unsigned int k;
  Const struct predicate *traced_pred;
  struct pred_table_entry *pte;
  static call_port_command();

  /* Rewrite the queue top goal record for traced goal */
  untrace_goal(qp);

  traced_pred = qp->pred;
  pte = get_pte(traced_pred);
  spying = pte->spied;
  arity = traced_pred->arity;
  qp_before = qp->next;
  trace_flag = 1;

  if (!leaping || spying) {
    parent = copy_traced_goal(qp);
    if (Enabled(Call)) {
      for (;;) {
	printf("CALL: ");
	print_goal(qp, 0);
	if (Leashed(Call) || spying) {
	  if (!call_port_command()) break;
	} else {
	  printf("\n");
	  break;
	}
      }
    }
  }

  if (trace_flag) {
    real_heaplimit = heaplimit;
    heaplimit = 0;
  }

  return traced_pred->func(glbl, qp, allocp, fg, traced_pred);
}

struct goalrec *trace_after(qp, allocp)
     struct goalrec *qp;
     q *allocp;
{
  struct global_variables *glbl = &globals;
  struct goalrec *qt, *before = qp_before;
  unsigned int subgoals, k;
  struct goalrec **bsearch();
  static commit_port_command();

  for (qt = qp, k = 0;
       qt != before && k < 100;
       qt = qt->next, k++) {
    int j;
    int real_subgoal = 1;
    for (j = 0; j < n_resumed; j++) {
      if (resumed_goals[j] == qt) {
	real_subgoal = 0;
	break;
      }
    }
    if (real_subgoal) {
      subgoal_trace[k] = get_pte(qt->pred)->default_trace;
      is_resumed[k] = 0;
    } else {
      if (subgoal_trace[k] = Traced(qt)) untrace_goal(qt);
      is_resumed[k] = 1;
    }
  }
  subgoals = k;
  n_resumed = 0;

  if (Enabled(Commit) && (!leaping || spying)) {
    for (;;) {
      printf("COMT: ");
      print_goal(parent, 0);
      if (subgoals != 0) {
	qt = qp;
	printf(" :-");
	k = 0;
	do {
	  printf("\n%3d%c%c",
		 k,
		 (k < 100 && is_resumed[k] ? '!' : ':'),
		 (k < 100 && subgoal_trace[k] ? '+' : '-' )),
	  print_goal(qt, parent->pred->func);
	  k++;
	} while ((qt = qt->next) != before);
      }
      if (Leashed(Commit) || spying) {
	if (!commit_port_command(subgoals)) break;
      } else {
	printf("\n");
	break;
      }
    }
  }

  if (trace_flag) {
    for (qt = qp, k = 0;
	 qt != before;
	 qt = qt->next) {
      if (k >= 100 || subgoal_trace[k++]) {
	trace_goal(qt);
      }
    }
    trace_flag = 0;
  }

  if (parent != 0) {
    free_goal_records(parent);
    parent = 0;
  }

  heapp = allocp;
  heaplimit = real_heaplimit;
  return qp;
}

trace_susp(qp, reasonp)
     struct goalrec *qp;
     q *reasonp;
{
  struct global_variables *glbl = &globals;
  static susp_port_command();

  if (Enabled(Susp) && (!leaping || spying)) {
    for (;;) {
      printf("SUSP: ");
      print_goal(qp, 0);
      if (Leashed(Susp) || spying) {
	if (!susp_port_command(qp)) break;
      } else {
	printf("\n");
	break;
      }
    }
  }
  if (trace_flag) {
    trace_goal(qp);
    trace_flag = 0;
  }
  heaplimit = real_heaplimit;
}

trace_resumption(qp)
     struct goalrec *qp;
{
  /*
    This routine only registers the resumed goal in a table
    rather than actually tracing it.
    Real tracing will be done at the commmit port.
    */
  extern char *calloc();
  if (n_resumed == max_resumed_goals) {
    int k;
    struct goalrec **newarea;
    if (max_resumed_goals == 0) {
      max_resumed_goals = 16;
    } else {
      max_resumed_goals *= 2;
    }
    newarea = (struct goalrec **)
      calloc(max_resumed_goals, sizeof(struct goalrec *));
    for (k = 0; k < n_resumed; k++) {
      newarea[k] = resumed_goals[k];
    }
    if (resumed_goals != 0) free(resumed_goals);
    resumed_goals = newarea;
  }
  resumed_goals[n_resumed++] = qp;
}

trace_failure(qp)
     struct goalrec *qp;
{
  static fail_port_command();

  if (Enabled(Fail)) {
    for (;;) {
      printf("FAIL: ");
      print_goal(qp, 0);
      if (Leashed(Fail) || spying) {
	if (!fail_port_command(qp)) break;
      } else {
	printf("\n");
	break;
      }
    }
  }
  trace_flag = 0;
}

/****************************************
  Tracer Command Interpreter
****************************************/

enum tracer_command {
  Continue, Leap,		/* Execution Control */
  Skip, QuasiSkip,
  Abort,
  Unify,			/* PseudoExecution */
  EnablePort, DisablePort, LeashPort, UnLeashPort,
  Trace, NoTrace, Spy, NoSpy, SetDefaultTrace, UnsetDefaultTrace,
  PrintDepth, PrintLength,	/* Print Out Control */
  PrintVerbose,
  ListMod, ListPred,		/* Listing */
  Debugging,			/* Status Query */
  Help, Unknown			/* Miscellaneous */
  };

static char
  command_line[256],
  *clp;

static tracer_error();
static print_help();

#define skip_blanks()  { while (isspace(*clp)) clp++; }

static enum tracer_command trace_parse_command()
{
  Const struct command_table_entry *cte;
  static char command_name[256];
  Const static struct command_table_entry {
    char *name;
    enum tracer_command command;
  } command_table[] = {
    { "", Continue },		/* Execution Control */
    { "c", Continue },
    { "l", Leap },
    { "s", Skip }, { "q", QuasiSkip },
    { "a", Abort },
    { "u", Unify },		/* PseudoExecution */
    { "E", EnablePort }, { "D", DisablePort },
    { "L", LeashPort }, { "U", UnLeashPort },
    { "t", SetDefaultTrace }, { "n", UnsetDefaultTrace },
    { "+", Trace }, { "-", NoTrace },
    { "S", Spy }, { "N", NoSpy },
    { "pd", PrintDepth },	/* Print Out Control */
    { "pl", PrintLength },
    { "pv", PrintVerbose },
    { "lm", ListMod },		/* Listing */
    { "lp", ListPred },
    { "=", Debugging },		/* Status Query */
    { "h", Help }, { "?", Help }, /* Miscellaneous */
    { command_name, Unknown }
  };
  char *cnp = command_name;

  gets(command_line);
  clp = command_line;

  skip_blanks();
  if (isalpha(*clp)) {
    while (isalpha(*clp)) *cnp++ = *clp++;
  } else if (ispunct(*clp)) {
    while (ispunct(*clp)) *cnp++ = *clp++;
  } else {
    while (*clp != '\0' && !isspace(*clp)) *cnp++ = *clp++;
  }
  *cnp = '\0';
  cte = command_table;
  while (strcmp(command_name, cte->name)) cte++;
  return cte->command;
}

static enum trace_port trace_parse_port()
{
  static char port_name[256];
  static struct port_table_entry {
    enum trace_port port;
    char *name;
  } port_table[] = {
    { Call, "call" },
    { Susp, "susp" }, { Susp, "suspend" },
    { Commit, "comt" }, { Commit, "commit" },
    { Fail, "fail"},
    { AllPorts, "all" },
    { UnknownPort, port_name }
  };
  struct port_table_entry *pte;
  char *pnp = port_name;
  skip_blanks();
  if (*clp == '\0') return NoMorePort;
  while (*clp != '\0' && !isspace(*clp)) *pnp++ = tolower(*clp++);
  *pnp = '\0';
  pte = port_table;
  while (strcmp(port_name, pte->name)) pte++;
  return pte->port;
}

static trace_parse_int(addr)
     int *addr;
/*
  Returns:
  0: nothing read
  1: integer value read
  -1: error found while parsing
*/
{
  int c, n;
  if (*clp == '\0') return 0;
  if (!isdigit(*clp)) return -1;
  n = 0;
  do {
    n = 10*n + *clp++ - '0';
  } while (isdigit(*clp));
  *addr = n;
  return 1;
}

static trace_parse_name(np)
     char *np;
{
  if (*clp == '\0') return 0;
  while (isalpha(*clp) || isdigit(*clp) || *clp == '_') {
    *np++ = *clp++;
  }
  *np = '\0';
  return 1;
}

static trace_parse_predicate_name(mod, pred, arity)
     struct mod_table_entry **mod;
     struct pred_table_entry **pred;
     int *arity;
/*
  Returns
  0: nothing read
  1: only module name read ("Module")
  2: module and predicate name read ("Module:Pred")
  3: module and predicate name with arity read ("Module:Pred/Arity")
  -1: error found while parsing
*/
{
  char name[256];
  char  *cp;
  int k;
  struct mod_table_entry *mte;
  struct pred_table_entry *pte;
  *mod = 0;
  *pred = 0;
  skip_blanks();
  if (!trace_parse_name(name)) {
    return 0;
  }
  for (k = 0, mte = mod_table;
       k < n_mod;
       k++, mte++) {
    if (!strcmp(mte->name, name)) {
      *mod = mte;
      goto mod_found;
    }      
  }
  tracer_error("Unknown module: %s", name);
  return -1;
 mod_found:
  if (*clp == '\0') return 1;
  if (*clp != ':') {
    tracer_error("Colon char expected after module name");
    return -1;
  }
  clp++;
  if (!trace_parse_name(name)) {
    tracer_error("Predicate name expected after colon");
    return -1;
  }
  for (k = 0, pte = pred_table;
       k < n_pred;
       k++, pte++) {
    if (pte->pred->func == mte->func &&
	!strcmp(pte->name, name)) {
      *pred = pte;
      goto pred_found;
    }
  }
  tracer_error("Unknown predicate: %s:%s", mte->name, name);
  return -1;
 pred_found:
  if (*clp == '\0') return 2;
  if (*clp++ != '/') {
    tracer_error("Arity should be specified after slash");
    return -1;
  }
  if (trace_parse_int(arity) != 1) {
    tracer_error("Decimal arity expected after slash");
    return -1;
  }
  for (;
       k < n_pred;
       k++, pte++) {
    if (!strcmp(pte->name, name) &&
	pte->pred->arity == *arity) {
      *pred = pte;
      goto arity_found;
    }
  }
  tracer_error("Unknown predicate: %s:%s/%d", mte->name, name, *arity);
  return -1;
 arity_found:
  return 3;
}

static unsigned int
  trace_print_depth = 3,
  trace_print_length = 7;

#define COMMON_COMMANDS\
 Abort: \
 case EnablePort: case DisablePort: \
 case LeashPort: case UnLeashPort: \
 case Spy: case NoSpy: \
 case SetDefaultTrace: case UnsetDefaultTrace: \
 case PrintDepth: case PrintLength: case PrintVerbose: \
 case ListMod: case ListPred: case Debugging: case Unknown

static common_port_command(command, pred)
     enum tracer_command command;
     struct predicate *pred;
{
  enum trace_port port;
  struct mod_table_entry *mte;
  struct pred_table_entry *pte;
  int arity;
  int is_spy, set_flag;
  static
    print_debug_status(), print_port_names(),
    print_modules(), print_predicates();

  switch (command) {
  case Abort:
    exit(1);
  case EnablePort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	enabled = -1;
      } else {
	enabled |= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
  case DisablePort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	enabled = 0;
      } else {
	enabled ^= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
  case LeashPort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	leashed = -1;
      } else {
	leashed |= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
  case UnLeashPort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	leashed= 0;
      } else {
	leashed ^= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
#define SpyOrTrace(x) (is_spy ? (x)->spied : (x)->default_trace)    
  case SetDefaultTrace:
    is_spy = 0;
    set_flag = 1;
    goto set_reset;
  case UnsetDefaultTrace:
    is_spy = 0;
    set_flag = 0;
    goto set_reset;
  case Spy:
    is_spy = 1;
    set_flag = 1;
    goto set_reset;
  case NoSpy:
    is_spy = 1;
    set_flag = 0;
  set_reset:
    {
      Const static char *set_reset[] = { "reset", "set" };
      Const static char *spy_trace[] = { "Default trace", "Spy point" };
      switch (trace_parse_predicate_name(&mte, &pte, &arity)) {
      case -1:
	break;
      case 1:
	for (pte = pred_table;
	     pte < pred_table + n_pred;
	     pte++) {
	  if (pte->pred->func == mte->func) SpyOrTrace(pte) = set_flag;
	}
	printf("\t%s %s on all predicates in module %s\n",
	       spy_trace[is_spy], set_reset[set_flag], mte->name);
	break;
      case 2:
	do {
	  SpyOrTrace(pte) = set_flag;
	  printf("\t%s %s on predicate %s:%s/%d\n",
		 spy_trace[is_spy], set_reset[set_flag],
		 mte->name, pte->name, pte->pred->arity);
	  pte++;
	} while (pte < pred_table + n_pred &&
		 !strcmp((pte-1)->name, pte->name));
	break;
      case 0:
	pte = get_pte(pred);
	mte = pte->mte;
	arity = pred->arity;
	/* fall through */
      case 3:
	SpyOrTrace(pte) = set_flag;
	printf("\t%s %s on predicate %s:%s/%d\n",
	       spy_trace[is_spy], set_reset[set_flag],
	       mte->name, pte->name, pte->pred->arity);
	break;
      }
      break;
    }
  case PrintDepth:
    skip_blanks();
    if (trace_parse_int(&trace_print_depth) != 1) {
      printf("\tCurrent print depth = %d\n", trace_print_depth);
    }
    break;
  case PrintLength:
    skip_blanks();
    if (trace_parse_int(&trace_print_length) != 1) {
      printf("\tCurrent print length = %d\n", trace_print_length);
    }
    break;
  case PrintVerbose:
    verbose_print = !verbose_print;
    printf("\tVerbose printing switched %s\n",
	   (verbose_print ? "ON" : "OFF" ));
    break;
  case ListMod:
    print_modules();
    break;
  case ListPred:
    skip_blanks();
    switch (trace_parse_predicate_name(&mte, &pte, &arity)) {
    case 0:
      print_predicates(0, 0, -1);
      break;
    case 1:
      print_predicates(mte, 0, -1);
      break;
    case 2:
      print_predicates(mte, pte, -1);
      break;
    case 3:
      print_predicates(mte, pte, arity);
      break;
    }
    break;
  case Debugging:
    print_debug_status();
    break;
  case Unknown:
    tracer_error("Unknown command");
    printf("\tTry ? for command list\n");
    break;
  }
}

#define CONTROL_COMMANDS\
 Continue: case Leap: case Skip

static control_command(command)
     enum tracer_command command;
{
  switch(command) {
  case Continue:
    leaping = 0;
    break;
  case Leap:
    leaping = 1;
    break;
  case Skip:
    trace_flag = 0;
    break;
  }
}

static Const char *common_help[] = {
 "E: enable port;   D: disable port;  L: leash port;    U: unleash port;",
 "S <pred>: spy;    N <pred>: nospy;  t <pred>: trace;  n <pred>: notrace;",
 "pd: print depth;  pl: print length; pv: toggle verbose print;",
 "=: debug status   lm: list modules; lp: list predicates;",
 "a: abort execution",
 0 };

static call_port_command()
{
  int again = 0;
  int err = 0;
  int k;
  enum tracer_command command;
  printf("? ");
  switch (command = trace_parse_command()) {
  case COMMON_COMMANDS:
    common_port_command(command, parent->pred);
    again = 1;
    break;
  case CONTROL_COMMANDS:
    control_command(command);
    break;
  default:
    err = 1;
  }    
  if (err) {
    Const static char *call_help[] = {
      "\t*** Commands available at CALL ports ***",
      "<cr>: continue;   c: continue;      s: skip;          l: leap;",
      0
    };
    print_help(call_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static commit_port_command(subgoals)
     unsigned int subgoals;
{
  int k, flag;
  enum tracer_command command;
  int again = 0;
  int err = 0;
  printf("? ");
  switch (command = trace_parse_command()) {
  case Trace:
    flag = 1;
    goto trace_notrace;
  case NoTrace:
    flag = 0;
  trace_notrace:
    if (subgoals == 0) {
      tracer_error("No subgoals to trace");
    } else if (subgoals >= 100) {
      tracer_error("Too many subgoals to control");
    } else {
      skip_blanks();
      if (*clp == '\0') {
	for (k = 0; k < subgoals; k++) {
	  subgoal_trace[k] = flag;
	}
      } else {
	int traced;
	while (1) {
	  switch (trace_parse_int(&traced)) {
	  case 0:
	    goto exit_loop;
	  case 1:
	    if (0 <= traced && traced < subgoals) {
	      subgoal_trace[traced] = flag;
	    } else {
	      tracer_error("Subgoal %d doesn't exist", traced);
	    }
	    skip_blanks();
	    break;
	  default:
	    tracer_error("Subgoal number expected");
	    goto exit_loop;
	  }
	}
      }
    }
  exit_loop:
    again = 1;
    break;
  case CONTROL_COMMANDS:
    control_command(command);
    break;
  case COMMON_COMMANDS:
    common_port_command(command, parent->pred);
    again = 1;
    break;
  default:
    err = 1;
    break;
  }    
  if (err) {
    Const static char *commit_help[] = {
      "\t*** Commands available at COMMIT ports ***",
      "<cr>: continue;   c: continue;      s: skip;          l: leap;",
      "+ <n>: trace;     - <n>: no trace;",
      0
    };
    print_help(commit_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static susp_port_command(qp)
     struct goalrec *qp;
{
  int k;
  enum tracer_command command;
  int again = 0;
  int err = 0;
  printf("? ");
  switch (command = trace_parse_command()) {
  case CONTROL_COMMANDS:
    control_command(command);
    break;
  case COMMON_COMMANDS:
    common_port_command(command, qp->pred);
    again = 1;
    break;
  default:
    err = 1;
    break;
  }    
  if (err) {
    Const static char *susp_help[] = {
      "\t*** Commands available at SUSPENSION ports ***",
      "<cr>: continue;   c: continue;      s: skip;          l: leap;",
      0
    };
    print_help(susp_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static fail_port_command(qp)
     struct goalrec *qp;
{
  int k;
  enum tracer_command command;
  int again = 0;
  int err = 0;
  printf("? ");
  switch (command = trace_parse_command()) {
  case COMMON_COMMANDS:
    common_port_command(command, qp->pred);
    again = 1;
    break;
  default:
    err = 1;
    break;
  }    
  if (err) {
    Const static char *fail_help[] = {
      "\t*** Commands available at FAIL ports ***",
      0
    };
    print_help(fail_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static print_port_status()
{
  static char plusminus[] = { '-', '+' };
  printf("\t   port: Call Susp Comt Fail\n");
  printf("\tenabled:  %c    %c    %c    %c\n",
	 plusminus[Enabled(Call)],
	 plusminus[Enabled(Susp)],
	 plusminus[Enabled(Commit)],
	 plusminus[Enabled(Fail)]);
  printf("\tleashed:  %c    %c    %c    %c\n",
	 plusminus[Leashed(Call)],
	 plusminus[Leashed(Susp)],
	 plusminus[Leashed(Commit)],
	 plusminus[Leashed(Fail)]);
}

static print_debug_status()
{
  print_port_status();
  printf("\tprint %s; depth = %u; length = %u\n",
	 (verbose_print ? "verbose" : "terse"),
	 trace_print_depth, trace_print_length);
}

static print_port_names()
{
  Const char *port_help[] = {
    "Available ports are \"call\", \"susp\", \"comt\" and \"fail\".",
    "You can also specify \"all\".",
    0 };
  print_help(port_help);
}

/****************************************
  Printing Out
****************************************/

static tracer_error(s, a0, a1, a2, a3, a4, a5, a6, a7, a8)
     char *s;
     int a0, a1, a2, a3, a4, a5, a6, a7, a8;
{
  printf("\t!!! ");
  printf(s, a0, a1, a2, a3, a4, a5, a6, a7, a8);
  printf(" !!!\n");
}

static print_help(messages)
     Const char **messages;
{
  while (*messages) {
    printf("    %s\n", *messages++);
  }
}

static trace_print(x)
     q x;
{
  print_partially(x, trace_print_depth, trace_print_length);
}

print_goal(g, mod)
     struct goalrec *g;
     module mod;
{
  Const struct predicate *pred;
  struct pred_table_entry *pte;
  int traced;

  if (traced = Traced(g)) untrace_goal(g);
  pte = get_pte(pred = g->pred);
  if (pred->func != mod) {
    printf("%s:", (pte ? pte->mte->name : "???"));
  }
  printf("%s", (pte ? pte->name : "???"));
  if (pred->arity != 0) {
    int k, pos;
    struct goalrec *xg = g;
    printf("(");
    trace_print(g->args[0]);
    for (k = 1, pos = 1;
	 k < pred->arity;
	 k++, pos++) {
      printf(",");
      if (pos == NUMGOALARGS - 1 && k != pred->arity - 1) {
	xg = (struct goalrec *)xg->args[NUMGOALARGS-1];
	pos -= NUMGOALARGS + 1;
      }
      trace_print(xg->args[pos]);
    }
    printf(")");
  }
  if (traced) trace_goal(g);
}

static print_modules()
{
  int k;

  printf("\t*** Module List ***\n");
  for (k = 0; k < n_mod; k++) {
    struct mod_table_entry *mte = &mod_table[k];
    printf("\t%8X %s\n",
	   (unsigned int)mte->func,
	   mte->name);
  }
  printf("\t*******************\n");
}

static print_predicates(mte, pte, arity)
     struct mod_table_entry *mte;
     struct pred_table_entry *pte;
     int arity;
{
  int k;
  for (k = 0; k < n_pred; k++) {
    struct pred_table_entry *x = &pred_table[k];
    if ((!mte || x->mte == mte) &&
	(!pte || !strcmp(x->name, pte->name)) &&
	(arity < 0 || x->pred->arity == arity)) {
      printf("\t%8X %c%c%s:%s/%d\n",
	     (unsigned int)x->pred,
	     (x->spied ? '*' : ' '),
	     (x->default_trace ? '+' : '-'),
	     x->mte->name,
	     x->name, x->pred->arity);
    }
  }
}
