/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
/*
  struct.h

  Definition of data structures for KL1/C system
*/

#include <klic/param.h>

/****************************************
  UNIVERSAL ONE-WORD DATA

  All the one-word data is represented as type "q".
  This pointer type will always be used after type conversion.
  Using "struct gazonk *" type is for reducing confusion.
****************************************/

typedef struct gazonk *q;

/****************************************
  POINTER TAGS

  [Pointer Tags]
  Lower 2 bits of pointers are used as tags in the following way.

     ...00: Variable Reference
     ...01: Pointer to a Cons Cell
     ...10: Atomic (integer, symbolic atom, etc)
     ...11: Pointer to a Functor Structure

  Variables without suspended goals are represented as
    self-referencing variables.
  Variables _with_ suspended goals are represented as a pointer to
    a cell which contains a special atom.

  [Tags for Atomic Data]
     ..0010: Integer (integers have 28 bits including sign)
     ..0110: Symbolic Atom
     ..1010: Reserved
     ..1110: Reserved
****************************************/

#define VALUEMASK	0xFFFFFFFC
#define ADDRESSMASK	0xFFFFFFFC

#define PTAGBITS	2
#define PTAGMASK	0x3
#define	VARREF		0x0
#define CONS		0x1
#define ATOMIC		0x2
#define FUNCTOR		0x3

#define STRUCT		0x1
#define FUNCTNOTCONS	0x2
#define ATOMNOTREF	0x2

#define ptagof(x)	((unsigned int)(x) & PTAGMASK)
#define addressof(x)	((unsigned int)(x) & ADDRESSMASK)

#define isatomic(x)	(ptagof(x) == ATOMIC)
#define iscons(x)	(ptagof(x) == CONS)
#define isfunctor(x)	(ptagof(x) == FUNCTOR)
#define isref(x)	(ptagof(x) == VARREF)

#define isstruct(x)	((unsigned int)(x) & STRUCT)
#define functnotcons(x)	((unsigned int)(x) & FUNCTNOTCONS)
#define atomicnotref(x)	((unsigned int)(x) & ATOMNOTREF)

#define ATAGBITS	4
#define ATAGMASK	0xF
#define INT		(ATOMIC+0x0)
#define SYM		(ATOMIC+0x4)

#define atagof(x)	((unsigned int)(x) & ATAGMASK)

#define isint(x)	(atagof(x) == INT)
#define issym(x)	(atagof(x) == SYM)

/****************************************
  DATA STRUCTURES

  All the data structures are to be allocated at 4 byte boundaries.
****************************************/

struct functor {
  q functor;			/* principal functor as atomic q object */
  q args[1];			/* arguments */
};

#define functor_of(s)		(functorp(s)->functor)
#define arg(s, k)		(functorp(s)->args[k])

struct cons {
  q cdr, car;
};

#define car_of(x)		(consp(x)->car)
#define cdr_of(x)		(consp(x)->cdr)

#define pcar_of(x)              (makeref(consp(x)+1))
#define pcdr_of(x)              (makeref(consp(x)))

/****************************************
  GOAL MANAGEMENT STRUCTURES
****************************************/

/* Predicate structures are constant structures outside of heap */
/* They are pointed to by atomic tags and treated as atomic in GC */

typedef void * (*module)();

struct predicate {
  module func;			/* pointer to function */
  unsigned short int pred;	/* predicate number */
  unsigned short int arity;	/* number of args */
};

/* Goal records */

/* Suspension records are referenced via pointers with variable tag */
/* They are recognized by its value being a special atom */
/* When a variable with suspended goals is instantiated, */
/*   the first word of this suspension record is used as the cell */

struct goalrec {
  struct goalrec *next;		/* pointer to next suspended goal */
  Const struct predicate *pred;	/* predicate descriptor of the goal */
  q args[NUMGOALARGS];			/* arguments */
};

/*
  Stuffs for suspension
*/
/* Changed */
/*struct susprec {*/
/*  q marker;    /*object tag */
/*  q next;     /* suspension record */
/*};*/

struct susprec {
  q marker;
  q next;
  q back; /* back points the last record */
  q reserved;
};

struct shook {
  struct goalrec *goals;        /* pointer to goal record */
  q next; /* next suspension structure and tag */
};

struct mhook {
  struct goalrec *goals;
  q next;
  q back;         /* back pointer */
  struct mhook *pal; /* pointer to the mhook record which is suspended
		     simultaniously */
};

struct merger{
  struct merger_common *common;
  q next;
};

struct merger_common {
  int counter;
  q variable;
};

/****************************************
  DATA MANIPULATION MACROS
****************************************/

/* Type conversion */

#define refp(x)		((q *)((unsigned int)(x) - VARREF))
#define consp(x)	((struct cons *)((unsigned int)(x) - CONS))
#define functorp(x)	((struct functor *)((unsigned int)(x) - FUNCTOR))
#define goalp(x)	((struct goalrec *)((unsigned int)(x) - ATOMIC))
#define suspp(x)	((struct susprec *)((unsigned int)(x) - VARREF))
#define predp(x)	((struct predicate *)((unsigned int)(x) - ATOMIC))

/* Variables */

#define makeref(x)	(q) ((unsigned int)(x) + VARREF)
#define derefone(x)	(*refp(x))

/* Atomic Values */

#define intval(x)	((unsigned int)(x) >> ATAGBITS)
#define symval(x)	((unsigned int)(x) >> ATAGBITS)

#define makeint(n)	((q)(((n) << ATAGBITS) + INT))
#define makesym(n)	((q)(((n) << ATAGBITS) + SYM))

#define eqatom(x, y)	((x) == (y))

/* for non-heap addr */
#define makeatomic(adr) ((q)((unsigned int)(adr) + ATOMIC))

/* for principal functors */
#define makepf(x)	makesym(x)

/* Conses */
#define makecons(x)	((q)((unsigned int)(x) + CONS))

/* Functors */

#define makefunctor(x) 	((q)((unsigned int)(x) + FUNCTOR))

Extern struct funcdesc {
  q name;			/* atom */
  unsigned int arity;		/* number of args */
} functab[1];

/* Structures for linkage */

struct ext_atom {
  char *name;			/* name string */
  q value;			/* its value with symbol tag */
};

struct ext_funct {
  char *name;			/* name string */
  int arity;			/* arity */
  q value;			/* functor id with atomic tag */
};

struct ext_pred {
  char *module;			/* module name string */
  char *name;			/* predicate name */
  int arity;			/* arity */
  q value;			/* pred id with atomic tag */
};

/* Special Atoms */

#define NILATOM		makesym(0)
/*#define SUSPATOM	makesym(1)*/

/**********************
  Suspension staffs
***********************/
/* Tags for single/multiple suspension */
#define SSUSP 0
#define MSUSP 1
#define MERGER 2
#define SUSPTAG 3
#define SUSPMASK (~SUSPTAG)

struct shook *shookalloc();
struct mhook *mhookalloc();
struct susprec *susprecalloc();

#define makesusprec(rec) { (rec)=susprecalloc(); }
#define initsusprec(susp, ref, hook) { \
  (susp)->marker = makeref(ref); \
  (susp)->next = (q)MSUSP; \
  setnexthook((susp), (hook)); \
  settailhook((susp), (hook)); \
  derefone(ref) = makeref(susp); \
}

#define makeshook(rec, pointer) \
  makehook(rec, shookalloc, pointer, SSUSP)
#define makemhook(rec, pointer) \
  makehook(rec, mhookalloc, pointer, MSUSP)
#define makemerger(rec, pointer) \
  makehook(rec, (struct merger *)shookalloc, pointer, MERGER)
#define collectmerger(s) collectshook((struct shook *)(s))
#define collectmergercommon(s) collectsusprec((struct susprec *)(s))

#define makehook(rec, alloc, pointer, tag) \
  {  \
    (rec) = alloc(); \
    /*printhooks((rec), (tag));*/ \
    (rec)->next = (q)(((int)(((q*)(pointer))+1))|(tag)); \
  }

#ifdef DEBUG
#define printhooks(rec, tag) \
    printf("allocated %s : %x\n", (tag == SSUSP? "shook": "mhook"), rec)
#else
#define printhooks(rec, tag)
#endif


#define setnexthook(rec, nextrec) { \
  (rec)->next = (q)((((int)(&((nextrec)->next))) & SUSPMASK) | \
    (((int)((rec)->next)) & SUSPTAG)); }

#define settailhook(rec, tailrec) { (rec)->back = (q)(((q*)(tailrec))+1); }
#define setbackhook(rec, prevrec) { (rec)->back = (q)(((q*)(prevrec))+1); }

#define makemergercommon(rec) { rec = (struct merger_common *)susprecalloc(); }


#define hooktagword(rec) (*(((q*)(rec)) + 1))

#define hooktag(rec) (((int)hooktagword(rec)) & SUSPTAG)

#define hookpointer(rec) (((int)hooktagword(rec)) & SUSPMASK)

#define getnexthook(rec) ((q*)hookpointer(rec) - 1)
#define getprevhook(rec) ((q*)(((int)((struct mhook *)(rec)->back)) \
			  & SUSPMASK) - 1)

#define hooktagwordp(rec) hooktagword(hookpointer(rec))

#define hooktagp(rec) (((int)hooktagwordp(rec)) & SUSPTAG)

#define hookpoinetrp(rec) (((int)hooktagwordp(rec)) & SUSPMASK)

#define changetagp(rec, newtag) \
  change_next(rec, newtag, SUSPTAG)
#define changepointerp(rec, newpointer) \
  change_next(rec, newpointer, SUSPMASK)

#define change_next(rec, new, mask) \
{ ((int)hooktagwordp(rec)) = ((((int)hooktagwordp(rec)) & (mask)) | \
			      ((int)(new)); }

#define shookp(rec) ((struct shook *)(rec))
#define mhookp(rec) ((struct mhook *)(rec))
#define mergerp(rec) ((struct merger *)(rec))

#define isshook(rec) ((int)hooktagword(rec) == SSUSP)
#define ismhook(rec) ((int)hooktagword(rec) == MSUSP)
#define ismerger(rec) ((int)hooktagword(rec) == MERGER)

/* Global Variables */

#define queue		(glbl->queue0)
#define freegoals	(glbl->freegoals0)
#define secondary_free	(glbl->secondary_free0)
#define heapp		(glbl->heapp0)
#define heaptop		(glbl->heaptop0)
#define heaplimit	(glbl->heaplimit0)
#define heapbottom	(glbl->heapbottom0)
#define heapend		(glbl->heapend0)
#define heapsize	(glbl->heapsize0)
#define totalheapsize	(glbl->totalheapsize0)
#define incrementsize	(glbl->incrementsize0)
#define goalbulk	(glbl->goalbulk0)
#define reasons		(glbl->reasons0)
#ifdef CSUSPS
#define suspensions	(glbl->suspensions0)
#endif

#define calcHeapSize() (totalheapsize)

/* HeapSize and HeapTop are local variables which are expected to be
defined in the context.*/

#define within_heap(p) ((unsigned int)((char *)(p) - (char *)HeapTop) < \
  (unsigned int)HeapSize)

Extern struct global_variables {
  /* First four items are accessed frequently */
  q *heapp0;
  q *heaplimit0;
  struct goalrec *queue0;
  struct goalrec *freegoals0;
  /* The rest are not accessed as often */
  q *heaptop0;
  q *heapbottom0;
  q *heapend0;
  struct goalrec *secondary_free0;
  unsigned int heapsize0, incrementsize0, goalbulk0, totalheapsize0;
  q reasons0[MAXSUSPENSION];	/* suspension reasons */
#ifdef CSUSPS
  unsigned int suspensions0;
#endif
} globals;

/*
  memory block outside of heap.
*/

Extern struct susprec *suspreclist;
Extern struct shook *shookreclist;
Extern struct mhook *mhookreclist;
