/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
write_object(Obj, Info, IF) :-
	write_object(Obj, Info, 0, IF).

write_object([], _, _, _):- !.
write_object([H|T], Info, H0, IF) :-
	write_one(H, Info, H0, Heap),
	write_object(T, Info, Heap, IF).

/* Labels */
write_one(label(Lab), Info, H, H) :-
	format(" ", []), write_lab(Info, Lab), format(":~n", []).
write_one(begin_body(_, Lab, _), Info, H, H) :-
	format(" ", []), write_lab(Info, Lab), format(":~n", []).

/* Register transfer */
write_one(move(To,elem(From,Pos)),_, H, H) :- !,
	format_reg(To,ToX,ToN),
	format("  ~s~d = ", [ToX,ToN]),
	write_elem_access(From,Pos),
	format(";~n", []).
write_one(move(To,From), _, H, H) :-
	format_reg(To,ToX,ToN), format_reg(From,FromX,FromN),
	format("  ~s~d = ~s~d;~n", [ToX, ToN, FromX, FromN]).

/* Element access */
write_one(get_elem(To,From,Pos),_, H, H) :-
	format_reg(To,ToX,ToN), 
	format("  ~s~d = ", [ToX,ToN]),
	write_elem_access(From,Pos),
	format(";~n", []).

/* Indexing */
write_one(deref(X,Lloop,Lsusp), info(Pred,MaxSusp), H, H) :-
	format_reg(X, XX, XN),
	format("  deref_and_jump(~s~d,", [XX,XN]),
	write_lab(info(Pred,MaxSusp),Lloop),
	format(");~n", []),
	format("  *reasonp++ =  ~s~d;~n", [XX,XN]),
	write_goto(info(Pred,MaxSusp),Lsusp).
write_one(push_reason(X,_), _, H, H) :- !,
	format_reg(X, XX, XN),
	format("  *reasonp++ =  ~s~d;~n", [XX,XN]).
write_one(goto(Label), Info, H, H) :-
	write_goto(Info, Label).
write_one(sw_tag(X), _, H, H) :-
	format_reg(X, XX, XN),
	format("  switch (ptagof(~s~d)) {~n", [XX, XN]).
write_one(sw_atom(X), _, H, H) :-
	format_reg(X, XX, XN),
	format("  switch ((unsigned int)~s~d) {~n", [XX,XN]).
write_one(sw_funct(X), _, H, H) :-
	format_reg(X, XX, XN),
	format("  switch ((unsigned int)functor_of(~s~d)) {~n", [XX,XN]).
write_one(sw_vect(X), _, H, H) :-
	format_reg(X, XX, XN),
	format("  switch ((unsigned int)arg(~s~d,0)) {~n", [XX,XN]).
write_one(case_label(S), _, H, H) :-
	format(" case ~s:~n", [S]).
write_one(case_atom(Atom), _, H, H) :-
	format(" case ", []),
	write_atom(Atom),
	format(":~n", []).
write_one(case_funct(Funct), _, H, H) :-
	format(" case ", []),
	write_funct(Funct),
	format(":~n", []).
write_one(case_vect(_/Ari), _, H, H) :-
	format(" case ", []),
	Ari1 is Ari - 1,
	write_atom(Ari1),
	format(":~n", []).
write_one(case_default, _, H, H) :-
	format(" default:~n", []).
write_one(end_sw, _, H, H) :-
	format("  };~n", []).
write_one(eq(X,Y,Lab,FailLab,SuspLab),Info, H, H) :-
	format_reg(X,XX,XN), format_reg(Y,YX,YN),
	format("  if_equal(~s~d, ~s~d, ", [XX,XN,YX,YN]),
	write_lab(Info,Lab),
	format(", ", []),
	write_lab(Info,FailLab),
	format(", ", []),
	write_lab(Info,SuspLab),
	format(");~n", []).

/* Constant loading */
write_one(load_atom(X,A),_,H,H) :-
	format_reg(X,XX,XN),
	format("  ~s~d = ",[XX,XN]),
	write_atom(A), format(";~n",[]).

/* Variable loading */
/* BUG */
write_one(load_newvar(X,V), _, H0, H) :-
	format_reg(X,XX,XN),
	format_reg(V,VX,VN),
	format("  ~s~d = makeref(allocp+~d);~n", [VX,VN,H0]),
	format("  *(allocp+~d) = ~s~d;~n",[H0,VX,VN]),
	format("  ~s~d = ~s~d;~n", [XX,XN,VX,VN]),
	H is H0+1.

/* Allocation */
write_one(make_cons(X),_, H, H) :-
	format_reg(X,XX,XN),
	format("  ~s~d = makecons(allocp+~d);~n", [XX,XN,H]).
write_one(make_functor(X,F,A), _, H, H) :-
	format_reg(X,XX,XN),
	format("  ~s~d = makefunctor(allocp+~d);~n", [XX,XN,H]),
	format("  *(allocp+~d) = ",[H]),
	write_funct(F/A),
	format(";~n", []).
write_one(alloc_atomic(A,K), _, H, H) :-
	format("  *(allocp+~d+~d) = ",[H,K]),
	write_atom(A),
	format(";~n", []).
write_one(alloc_value(elem(X,P),K),_, H, H) :- !,
	format("  *(allocp+~d+~d) = ", [H,K]),
	write_elem_access(X,P),
	format(";~n", []).
write_one(alloc_value(X,K),_, H, H) :-
	format_reg(X,XX,XN),
	format("  *(allocp+~d+~d) = ~s~d;~n",[H,K,XX,XN]).
write_one(alloc_var(X,K),_, H, H) :-
	format_reg(X,XX,XN),
	format("  ~s~d = makeref(allocp+~d+~d);~n", [XX,XN,H,K]),
	format("  *(allocp+~d+~d) = ~s~d;~n",[H,K,XX,XN]).
write_one(alloc_elem(X,K,KK),_, H, H) :-
	format("  *(allocp+~d+~d) = ", [H,KK]),
	write_elem_access(X,K),
	format(";~n",[]).
write_one(make_space(N),_, H0, H) :-
	H is H0+N.

/* Unificaiton */
write_one(unify_atom(X,A), _, H, H) :-
	format_reg(X,XX,XN),
	format("  unify(~s~d, ", [XX,XN]),
	write_atom(A),
	format(");~n", []).
write_one(unify(X,Y),_, H, H) :-
	format_reg(X,XX,XN), format_reg(Y,YX,YN),
	format("  unify(~s~d, ~s~d);~n", [XX,XN,YX,YN]).

/* Execution control */
write_one(execute(Name,Arity,Susp),_, H0, 0) :-
	(
	    H0 =:= 0 -> true
	;
	    format("  allocp += ~d;~n", [H0])
	),
	(
	    Susp = 0 ->
	    format("  execute(~w_~d_loop);~n", [Name,Arity])
	;
	    format("  execute(~w_~d_clear_reason);~n", [Name,Arity])
	),
	format("  goto ~w_~d_ext_interrupt;~n", [Name,Arity]).
write_one(proceed,_, H0, 0) :- !,
	( H0 =:= 0, !; format("  allocp += ~d;~n", [H0]) ),
	format("  proceed();~n", []).
write_one(set_pred(Name,Arity),info(M/_/_,_), H, H) :-
	format("  qp->pred = &", []),
	write_pred_name(M,Name,Arity),
	format(";~n", []).
write_one(push_goal,_,H,H) :-
	format("  push_goal();~n", []).
write_one(extend_goalrec(GR0,GR),_,H,H) :-
	format("  extend_goalrec(~w,~w);~n", [GR0, GR]).
write_one(set_ext_pred(Module,Name,Arity), _, H, H) :-
	format("  qp->pred = &", []),
	write_pred_name(Module,Name,Arity),
	format(";~n", []).
write_one(set_value(P,X,GR),_, H, H) :-
	format_reg(X,XX,XN),
	format("  ~w->args[~d] = ~s~d;~n", [GR,P,XX,XN]).
write_one(set_atomic(P,A,GR), _, H, H) :-
	format("  ~w->args[~d] = ", [GR,P]),
	write_atom(A),
	format(";~n", []).
write_one(set_newvar(P,X,GR),_, H0, H) :-
	format_reg(X,XX,XN),
	format("  ~s~d = makeref(allocp+~d);~n", [XX,XN,H0]),
	format("  *(allocp+~d) = ~s~d;~n", [H0,XX,XN]),
	H is H0+1,
	format("  ~w->args[~d] = ~s~d;~n", [GR,P,XX,XN]).
write_one(set_elem(P,X,K,GR),_, H, H) :-
	format("  ~w->args[~d] = ", [GR,P]),
	write_elem_access(X,K),
	format(";~n",[]).

/* Builtin */
write_one(blt_times(r(X)), _, H, H) :-
	format_reg(X,XX,XN),
	format("  ~s~d = blt_times();~n", [XX,XN]).

/* Arithmetics */
write_one(arith(X,Y,Z,Label,Opr),Info, H, H) :-
	arith_routine_and_operator(Opr, Routine, Operator),
	format("  arith(",[]),
	format_arith(X),
	format_arith(Y),
	format_arith(Z),
	write_lab(Info,Label),
	format(",~s,~s);~n", [Routine,Operator]).

/* Guard builtin */
write_one(gblt(F,_,Args,Lint), Info, H, H) :-
	format("  ~w(",[F]),
	write_gargs(Args),
	write_lab(Info,Lint),
	format(");~n",[]).

write_one(otherwise, Info, H, H) :-
	format("  otherwise(",[]),
	write_lab(Info,interrupt),
	format(");~n",[]).
write_one(alternatively, _, H, H) :-
	format("  alternatively;~n",[]).

write_one(equiv(R1,R0),_,H,H) :- !,
	format_reg(R1,RR1,RR1N),
	format_reg(R0,RR0,RR0N),
	format("  ~s~d = ~s~d;~n",[RR0,RR0N,RR1,RR1N]).

/* others */
write_gargs([]):- !.
write_gargs([atom(Int)|Args]):- integer(Int),!,
	format("makeint(~d),",[Int]),
	write_gargs(Args).
write_gargs([atom(Atom)|Args]):- atom(Atom),!,
	format("makeatom(atom_~w),",[Atom]),
	write_gargs(Args).
write_gargs([r(E)|Args]):-
	format_reg(E,EE,EN),
	format("~s~d,",[EE,EN]),
	write_gargs(Args).
write_gargs([e(Reg,E)|Args]):-
	write_elem_access(Reg,E),
	format(",",[]),
/*	format_reg(E,EE,EN),
	format("~s~d,",[EE,EN]), */
	write_gargs(Args).
write_gargs([E|Args]):-
	format_reg(E,EE,EN),
	format("~s~d,",[EE,EN]),
	write_gargs(Args).

arith_routine_and_operator(add, "do_add", "+").
arith_routine_and_operator(subtract, "do_subtract", "-").
arith_routine_and_operator(lshift, "do_lshift", "<<").
arith_routine_and_operator(rshift, "do_rshift", ">>").

format_arith(a(N)):- !,
	format("makeint(~d),",[N]).
format_arith(r(X)):-
	format_reg(X,XX,XN),
	format("~s~d,", [XX,XN]).

/* Element access */
write_elem_access(X, car) :- !,
	format_reg(X, XX, XN), format("car_of(~s~d)", [XX,XN]).
write_elem_access(X, cdr) :- !,
	format_reg(X, XX, XN), format("cdr_of(~s~d)", [XX,XN]).
write_elem_access(X, Pos) :- !,
	format_reg(X, XX, XN), format("arg(~s~d, ~d)", [XX,XN,Pos]).

write_lab(info(_/N/A,_),other(Lab)):- !,
	format("~a_~d_otherwise_~d",[N,A,Lab]).
write_lab(info(_/N/A,_),alter(Lab)):- !,
	format("~a_~d_alternatively_~d",[N,A,Lab]).
write_lab(info(_/N/A,_),interrupt):- !,
	format("~a_~d_interrupt",[N,A]).
write_lab(info(_/N/A,_),Lab) :- format("~a_~d_~w", [N,A,Lab]).
write_goto(Info,Lab) :-
	format("  goto ",[]), write_lab(Info,Lab), format(";~n",[]).

format_reg(a(N),"a",N).
format_reg(x(N),"x",N).
format_reg(r(R),N,NN):-
   format_reg(R,N,NN).
