/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <klic/basic.h>
#include <klic/struct.h>
#include <klic/primitives.h>
#include <klic/trace.h>
#include <nlist.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

extern void *realloc();
extern void *malloc();
extern void *calloc();
extern void free();

static void dump_queue();
char *program_name;

static struct mod_table_entry {
  module func;
  char *name;
} *mod_table = 0, **mod_index;

static struct pred_table_entry {
  struct predicate *pred;
  struct mod_table_entry *mte;
  char *name;
  char spied;
  char default_trace;
} *pred_table = 0;

enum trace_port {
  Call, Susp, Reduce, Fail, AllPorts, NoMorePort, UnknownPort };
static int enabled, leashed, leaping, spying;

#define Enabled(port)	(enabled >> (port) & 1)
#define Leashed(port)	(leashed >> (port) & 1)

FILE *tracer_input;

/* For remembering goals resumed during reduction */

static int n_resumed;
static struct goalrec **resumed_goals;
static int max_resumed_goals = 0;

/****************************************
  Predicate and Module Tables
****************************************/

static unsigned int n_mod = 0, n_pred = 0;

static mod_compare_names(x, y)
     struct mod_table_entry *x, *y;
{
  return strcmp(x->name, y->name);
}

static mod_compare_addr(x, y)
     struct mod_table_entry **x, **y;
{
  return (long)(*x)->func - (long)(*y)->func;
}

static pred_compare_names(x, y)
     struct pred_table_entry *x, *y;
{
  int result;
  if (!(result = strcmp(x->mte->name, y->mte->name)))
    if (!(result = strcmp(x->name, y->name)))
      result = x->pred->arity - y->pred->arity;
  return result;
}  

static struct mod_table_entry *get_mte(pred)
     struct predicate *pred;
{
  module mod = pred->func;
  unsigned int low = 0, high = n_mod, mid;
  while (low != high) {
    mid = ((high+low) >> 1);
    if (mod_index[mid]->func == mod) {
      return mod_index[mid];
    } else if ((unsigned long)mod_index[mid]->func > (unsigned long)mod) {
      high = mid;
    } else {
      low = mid + 1;
    }
  }
  return 0;
}

static struct pred_hash_entry {
  struct predicate *pred;
  struct pred_table_entry *pte;
} *pred_hash;

static unsigned long pred_hash_mask;

static enter_pred_hash(pred, pte)
     struct predicate *pred;
     struct pred_table_entry *pte;
{
  long h = ((((unsigned long) pred) >> 2) & pred_hash_mask);
  while (pred_hash[h].pred != 0) {
    h = (h + 1) & pred_hash_mask;
  }
  pred_hash[h].pred = pred;
  pred_hash[h].pte = pte;
}  

static struct pred_table_entry *get_pte(pred)
     struct predicate *pred;
{
  long h = ((((unsigned long) pred) >> 2) & pred_hash_mask);
  while (pred_hash[h].pred != pred) {
    h = (h + 1) & pred_hash_mask;
  }
  return pred_hash[h].pte;
}  

static make_name_tables()
{
  char tempname[16];
  char command[100];
  FILE *nmout;
  int k;
  static long compress_ul();
  struct mod_table_entry *mte, **mtxe;
  struct pred_table_entry *pte, **ptxe;

  (void)strcpy(tempname, "/tmp/klicXXXXXX");
  mktemp(tempname);
#ifdef SYSV
  sprintf(command, "nm -hp %s >%s", program_name, tempname);
#else
  sprintf(command, "nm -gp %s >%s", program_name, tempname);
#endif
  system(command);
  nmout = fopen(tempname, "r");
  for (;;) {
    unsigned long adr, len;
    char typ, *cp;
    char nam[500];
#ifdef SYSV
    if (fscanf(nmout, "%d %c", &adr, &typ) == EOF) break;
    if (isupper(typ)) {
      fscanf(nmout, "%s", nam);
      if (nam[0] != '_') {
	if (strncmp(nam, "module_", 7) == 0) {
	  n_mod++;
	} else if (strncmp(nam, "predicate_", 10) == 0) {
	  n_pred++;
	}
      }
    } else {
      while (getc(nmout) != '\n')
	;
    }
#else
    if (fscanf(nmout, "%x %c %s", &adr, &typ, nam) == EOF) break;
    if (nam[0] == '_' && nam[1] != '_') {
      if (strncmp(nam+1, "module_", 7) == 0) {
	n_mod++;
      } else if (strncmp(nam+1, "predicate_", 10) == 0) {
	n_pred++;
      }
    }
#endif
  }
  mod_table =
    (struct mod_table_entry *)
      calloc(n_mod, sizeof(struct mod_table_entry));
  mod_index =
    (struct mod_table_entry **)
      calloc(n_mod, sizeof(struct mod_table_entry *));
  pred_table =
    (struct pred_table_entry *)
      calloc(n_pred, sizeof(struct pred_table_entry));
  {
    unsigned long hash_size = 1;
    while (hash_size <= (3*n_pred/2)) hash_size *= 2;
    pred_hash =
      (struct pred_hash_entry *)
	calloc(hash_size, sizeof(struct pred_hash_entry));
    pred_hash_mask = hash_size - 1;
    for (k = 0; k < hash_size; k++) {
      pred_hash[k].pred = 0;
      pred_hash[k].pte = 0;
    }
  }
  rewind(nmout);
  mte = mod_table; mtxe = mod_index;
  pte = pred_table;
  for (;;) {
    unsigned long adr, len;
    char typ, *cp;
    char nam[500];
#ifdef SYSV
    if (fscanf(nmout, "%d %c", &adr, &typ) == EOF) break;
    if (isupper(typ)) {
      fscanf(nmout, "%s", nam);
      if (nam[0] != '_') {
	if (strncmp(nam, "module_", 7) == 0) {
	  cp = nam+7;
	  len = compress_ul(cp, strlen(cp), nam);
	  mte->name = (char *)malloc(len+1);
	  strncpy(mte->name, nam, len);
	  mte->name[len] = 0;
	  mte->func = (module) adr;
	  *mtxe++ = mte;
	  mte++;
	} else if (strncmp(nam, "predicate_", 10) == 0) {
	  char *cpt;
	  cp = nam+10;
	  while (1) {
	    cp = strchr(cp, '_');
	    if (*(cp+1) == 'x') break;
	    cp += 2;
	  }
	  cp += 2;
	  cpt = cp + strlen(cp) - 1;
	  while (isdigit(*cpt)) cpt--;
	  len = compress_ul(cp, cpt-cp, nam);
	  pte->name = (char *)malloc(len+1);
	  strncpy(pte->name, nam, len);
	  pte->name[len] = 0;
	  pte->pred = (struct predicate *) adr;
	  pte->spied = 0;
	  pte->default_trace = 1;
	  pte++;
	}
      }
    } else {
      while (getc(nmout) != '\n')
	;
    }
#else
    if (fscanf(nmout, "%x %c %s", &adr, &typ, nam) == EOF) break;
    if (nam[0] == '_' && nam[1] != '_') {
      if (strncmp(nam+1, "module_", 7) == 0) {
	cp = nam+8;
	len = compress_ul(cp, strlen(cp), nam);
	mte->name = (char *)malloc(len+1);
	strncpy(mte->name, nam, len);
	mte->name[len] = 0;
	mte->func = (module) adr;
	*mtxe++ = mte;
	mte++;
      } else if (strncmp(nam+1, "predicate_", 10) == 0) {
	char *cpt;
	cp = nam+11;
	while (1) {
	  cp = strchr(cp, '_');
	  if (*(cp+1) == 'x') break;
	  cp += 2;
	}
	cp += 2;
	cpt = cp + strlen(cp) - 1;
	while (isdigit(*cpt)) cpt--;
	len = compress_ul(cp, cpt-cp, nam);
	pte->name = (char *)malloc(len+1);
	strncpy(pte->name, nam, len);
	pte->name[len] = 0;
	pte->pred = (struct predicate *) adr;
	pte->spied = 0;
	pte->default_trace = 1;
	pte++;
      }
    }
#endif
  }
  unlink(tempname);

  qsort(mod_table, n_mod, sizeof(struct mod_table_entry),
	mod_compare_names);
  qsort(mod_index, n_mod, sizeof(struct mod_table_entry *),
	mod_compare_addr);

  for (k = 0; k < n_pred; k++) {
    pred_table[k].mte = get_mte(pred_table[k].pred);
  }

  qsort(pred_table, n_pred, sizeof(struct pred_table_entry),
	pred_compare_names);
  for (k = 0; k < n_pred; k++) {
    enter_pred_hash(pred_table[k].pred, &pred_table[k]);
  }
}

static long compress_ul(src, len, dest)
     char *src, *dest;
     long len;
{
  long destlen = 0;
  while (len--) {
    int c = *src++;
    if (c != '_') {
      *dest++ = c;
    } else if (*src == '_') {
      *dest++ = *src++; len--;
    } else {
      char buf[3];
      int n;
      buf[0] = *src++; len--;
      buf[1] = *src++; len--;
      buf[2] = 0;
      sscanf(buf, "%x", &n);
      *dest++ = n;
    }
    destlen++;
  }
  return destlen;
}

/****************************************
  Initiation
****************************************/

static void *trace_trigger_routine();
static struct predicate trace_trigger_preds[MAXSTDARGS+1];
static unsigned int trace_seq;
struct enqueue_trace_rec *trace_enqueued_goals;
static unsigned long trace_print_depth, trace_print_length;

void initiate_trace()
{
  int k;
  enabled = leashed = -1;	/* all one */
  leaping = 0;
#ifndef SYSV
  setlinebuf(stderr);
#endif
  make_name_tables();
  for (k=0; k<=MAXSTDARGS; k++) {
    trace_trigger_preds[k].func = trace_trigger_routine;
    trace_trigger_preds[k].pred = k;
    trace_trigger_preds[k].arity = k+2;
  }
  trace_seq = 2;
  trace_enqueued_goals = 0;
  trace_print_depth = 3;
  trace_print_length = 7;
  n_resumed = 0;
  max_resumed_goals = 16;
  resumed_goals = (struct goalrec **)
    malloc(max_resumed_goals*sizeof(struct goalrec *));
  if ((tracer_input = fopen("/dev/tty", "r")) == NULL) {
    fatal("Can't open /dev/tty for trace command input");
  }
}

/****************************************
  Main Tracing Routines
****************************************/

int trace_flag;
int spontaneous_susp;
int verbose_print = 0;
static struct goalrec *qp_before;
static struct goalrec *parent;
static unsigned long parent_seq;
static char subgoal_trace[100];
static char is_resumed[100];
static int subgoal_seq[100];

#define Traced(g)				\
  ((g)->pred->func == trace_trigger_routine)

struct goalrec *trace_goal(g, seq)
     struct goalrec *g;
     unsigned int seq;
{
  if (!Traced(g)) {
    struct global_variables *glbl = &globals;
    struct goalrec *ng;
    Const struct predicate *pred = g->pred;
    unsigned int k, arity;
    arity = pred->arity;
    heapalloc(ng, arity+4, (struct goalrec *));
    ng->next = g->next;
    ng->pred = &trace_trigger_preds[arity];
    for (k=0; k!=arity; k++) {
      ng->args[k] = g->args[k];
    }
    ng->args[k] = makeatomic(pred);
    ng->args[k+1] = makeint(seq);
    return ng;
  } else {
    return g;
  }
}

static unsigned int trace_seq_of(g)
     struct goalrec *g;
{
  return intval(g->args[g->pred->arity-1]);
}

static unsigned int untrace_goal(g)
     struct goalrec *g;
{
  if (Traced(g)) {
    unsigned int real_arity = g->pred->arity-2;
    g->pred = predp(g->args[real_arity]);
    return intval(g->args[real_arity+1]);
  } else {
    return 0;
  }
}

/*
static struct goalrec *
copy_traced_goal(g)
     struct goalrec *g;
{
  int k;
  struct goalrec *cg =
    (struct goalrec *)malloc((g->pred->arity+2)*sizeof(q*));
  cg->pred = g->pred;
  for (k=0; k!=g->pred->arity; k++) {
    cg->args[k] = g->args[k];
  }
  return cg;
}
*/

static void *trace_trigger_routine(glbl, qp, allocp, toppred)
  struct global_variables *glbl;
  struct goalrec *qp;
  q *allocp;
  Const struct predicate *toppred;
{
  unsigned int arity, k;
  Const struct predicate *traced_pred;
  struct pred_table_entry *pte;
  static call_port_command();

  /* Rewrite the queue top goal record for traced goal */
  parent_seq = untrace_goal(qp);

  traced_pred = qp->pred;
  pte = get_pte(traced_pred);
  spying = pte->spied;
  arity = traced_pred->arity;
  qp_before = qp->next;
  trace_flag = 1;

  if (!leaping || spying) {
    parent = qp;
    if (Enabled(Call)) {
      for (;;) {
	fprintf(stderr, "%4d CALL:", parent_seq);
	print_goal(qp, 0);
	if (Leashed(Call) || spying) {
	  if (!call_port_command(qp, get_mte(qp->pred))) break;
	} else {
	  fprintf(stderr, "\n");
	  break;
	}
      }
    }
  }

  if (trace_flag) heaplimit = 0;
  return traced_pred->func(glbl, qp, allocp, traced_pred);
}

struct goalrec *trace_after(qp, allocp)
     struct goalrec *qp;
     q *allocp;
{
  struct global_variables *glbl = &globals;
  struct goalrec *qt, *before = qp_before;
  unsigned int subgoals, k;
  struct goalrec **bsearch();
  static reduce_port_command();
  struct enqueue_trace_rec *tr;

  for (qt = qp, k = 0;
       qt != before && k < 100;
       qt = qt->next, k++) {
    int j;
    resumed_goals[n_resumed] = qt;
    for (j = 0; resumed_goals[j] != qt; j++)
      ;
    if (j == n_resumed) {
      subgoal_trace[k] = get_pte(qt->pred)->default_trace;
      is_resumed[k] = 0;
      subgoal_seq[k] = trace_seq++;
    } else {
      if (subgoal_trace[k] = Traced(qt)) {
	subgoal_seq[k] = trace_seq_of(qt);
      } else {
	subgoal_seq[k] = trace_seq++;
      }
      is_resumed[k] = 1;
    }
  }
  for (tr = trace_enqueued_goals;
       tr != 0 && k < 100;
       tr = tr->next, k++) {
    int j;
    resumed_goals[n_resumed] = tr->g;
    for (j = 0; resumed_goals[j] != tr->g; j++)
      ;
    if (j == n_resumed) {
      subgoal_seq[k] = trace_seq++;
      subgoal_trace[k] = 1;
      is_resumed[k] = 0;
    } else {
      if (subgoal_trace[k] = Traced(tr->g)) {
	subgoal_seq[k] = trace_seq_of(tr->g);
      } else {
	subgoal_seq[k] = trace_seq++;
      }
      is_resumed[k] = 1;
    }
  }
       
  subgoals = k;
  n_resumed = 0;

  if (Enabled(Reduce) && (!leaping || spying)) {
    for (;;) {
      fprintf(stderr, "%4d REDU:", parent_seq);
      print_goal(parent, 0);
      if (subgoals != 0) {
	fprintf(stderr, " :-");
	k = 0;
	for (qt = qp;
	     qt != before;
	     qt = qt->next, k++) {
	  fprintf(stderr, "\n%4d%4d%c%c",
		  subgoal_seq[k],
		  k,
		  (k < 100 && is_resumed[k] ? '!' : ':'),
		  (k < 100 && subgoal_trace[k] ? '+' : '-' )),
	  print_goal(qt, parent->pred->func);
	}
	for (tr = trace_enqueued_goals;
	     tr != 0 && k < 100;
	     tr = tr->next, k++) {
	  fprintf(stderr, "\n%4d%4d%c%c",
		  subgoal_seq[k],
		  k,
		  (k < 100 && is_resumed[k] ? '#' : '*'),
		  (k < 100 && subgoal_trace[k] ? '+' : '-' )),
	  print_goal(tr->g, parent->pred->func);
	  fprintf(stderr, "@prio(%d)", tr->prio);
	}
      }
      if (Leashed(Reduce) || spying) {
	if (!reduce_port_command(subgoals, qp, get_mte(parent->pred)))
	  break;
      } else {
	fprintf(stderr, "\n");
	break;
      }
    }
  }

  {
    struct goalrec **qpp;
    for (qpp = &qp, k = 0;
	 *qpp != before;
	 qpp = &(*qpp)->next, k++) {
      if (trace_flag && subgoal_trace[k]) {
	*qpp = trace_goal(*qpp, subgoal_seq[k]);
      } else {
	(void) untrace_goal(*qpp);
      }
    }
  }
  for (tr = trace_enqueued_goals;
       tr != 0 && k < 100;
       k++) {
    int trace_save = trace_flag;
    struct enqueue_trace_rec *next;
    if (trace_flag && subgoal_trace[k]) {
      tr->g = trace_goal(tr->g, subgoal_seq[k]);
    } else {
      (void) untrace_goal(tr->g);
    }
    trace_flag = 0;
    enqueue_goal(qp, tr->prio, tr->g, glbl);
    trace_flag = trace_save;
    next = tr->next;
    free(tr);
    tr = next;
  }
  trace_enqueued_goals = 0;
  trace_flag = 0;

  parent = 0;
  heapp = allocp;
  heaplimit = real_heaplimit;
  return qp;
}

struct goalrec *trace_susp(qp, reasonp)
     struct goalrec *qp;
     q *reasonp;
{
  struct global_variables *glbl = &globals;
  static susp_port_command();
  unsigned int seq = ( spontaneous_susp ? trace_seq++ : parent_seq );

  if (Enabled(Susp) && (!leaping || spying)) {
    for (;;) {
      fprintf(stderr, "%4d SUSP:", seq);
      print_goal(qp, 0);
      if (Leashed(Susp) || spying) {
	if (!susp_port_command(qp, get_mte(qp->pred))) break;
      } else {
	fprintf(stderr, "\n");
	break;
      }
    }
  }
  if (trace_flag) {
    qp = trace_goal(qp, seq);
  }
  trace_flag = spontaneous_susp;
  spontaneous_susp = 0;
  return qp;
}

trace_resumption(qp)
     struct goalrec *qp;
{
  /*
    This routine only registers the resumed goal in a table
    rather than actually tracing it.
    Real tracing will be done at the commmit port.
    */
  if (++n_resumed == max_resumed_goals) {
    int k;
    struct goalrec **newarea;
    max_resumed_goals *= 2;
    resumed_goals = (struct goalrec **)
      realloc(max_resumed_goals*sizeof(struct goalrec *));
  }
  resumed_goals[n_resumed-1] = qp;
}

trace_failure(qp)
     struct goalrec *qp;
{
  static fail_port_command();
  unsigned int seq = (qp==parent ? parent_seq : trace_seq++);

  if (Enabled(Fail)) {
    for (;;) {
      fprintf(stderr, "%4d FAIL:", seq);
      print_goal(qp, 0);
      if (Leashed(Fail) || spying) {
	if (!fail_port_command(qp, get_mte(qp->pred))) break;
      } else {
	fprintf(stderr, "\n");
	break;
      }
    }
  }
  trace_flag = 0;
}

/****************************************
  Tracer Command Interpreter
****************************************/

enum tracer_command {
  Continue, Leap,		/* Execution Control */
  Skip, QuasiSkip,
  Abort,
  ToggleTrace,
  Unify,			/* PseudoExecution */
  EnablePort, DisablePort, LeashPort, UnLeashPort,
  Trace, NoTrace, Spy, NoSpy, SetDefaultTrace, UnsetDefaultTrace,
  PrintDepth, PrintLength,	/* Print Out Control */
  PrintVerbose,
  ListMod, ListPred,		/* Listing */
  Debugging,			/* Status Query */
  DumpQueue,			/* Dump Ready Queue */
  Help, Unknown			/* Miscellaneous */
  };

static char
  command_line[256],
  *clp;

static tracer_error();
static print_help();

#define skip_blanks()  { while (isspace(*clp)) clp++; }

static enum tracer_command trace_parse_command()
{
  Const struct command_table_entry *cte;
  static char command_name[256];
  Const static struct command_table_entry {
    char *name;
    enum tracer_command command;
  } command_table[] = {
    { "", Continue },		/* Execution Control */
    { "c", Continue },
    { "l", Leap },
    { "s", Skip }, { "q", QuasiSkip },
    { "a", Abort },
    { "u", Unify },		/* PseudoExecution */
    { "E", EnablePort }, { "D", DisablePort },
    { "L", LeashPort }, { "U", UnLeashPort },
    { "t", SetDefaultTrace }, { "n", UnsetDefaultTrace },
    { "+", Trace }, { "-", NoTrace },
    { "S", Spy }, { "N", NoSpy },
    { "pd", PrintDepth },	/* Print Out Control */
    { "pl", PrintLength },
    { "pv", PrintVerbose },
    { "lm", ListMod },		/* Listing */
    { "lp", ListPred },
    { "=", Debugging },		/* Status Query */
    { "Q", DumpQueue },		/* Dump Ready Queue */
    { "h", Help }, { "?", Help }, /* Miscellaneous */
    { command_name, Unknown }
  };
  char *cnp = command_name;

  fprintf(stderr, "? ");
  fflush(stderr);
  fgets(command_line, 255, tracer_input);
  clp = command_line;

  skip_blanks();
  if (isdigit(*clp)) {
    return ToggleTrace;
  }
  if (isalpha(*clp)) {
    while (isalpha(*clp)) *cnp++ = *clp++;
  } else if (ispunct(*clp)) {
    while (ispunct(*clp)) *cnp++ = *clp++;
  } else {
    while (*clp != 0 && !isspace(*clp)) *cnp++ = *clp++;
  }
  *cnp = '\0';
  cte = command_table;
  while (strcmp(command_name, cte->name)) cte++;
  return cte->command;
}

static enum trace_port trace_parse_port()
{
  static char port_name[256];
  static struct port_table_entry {
    enum trace_port port;
    char *name;
  } port_table[] = {
    { Call, "c" }, { Call, "call" },
    { Susp, "s" }, { Susp, "susp" }, { Susp, "suspend" },
    { Reduce, "r" }, { Reduce, "redu" }, { Reduce, "reduce" },
    { Fail, "f" }, { Fail, "fail"},
    { AllPorts, "a" }, { AllPorts, "all" },
    { UnknownPort, port_name }
  };
  struct port_table_entry *pte;
  char *pnp = port_name;
  skip_blanks();
  if (*clp == '\0') return NoMorePort;
  while (!isspace(*clp)) *pnp++ = tolower(*clp++);
  *pnp = '\0';
  pte = port_table;
  while (strcmp(port_name, pte->name)) pte++;
  return pte->port;
}

static long trace_parse_int(addr)
     long *addr;
/*
  Returns:
  0: nothing read
  1: integer value read
  -1: error found while parsing
*/
{
  long c, n;
  if (*clp == '\0') return 0;
  if (!isdigit(*clp)) return -1;
  n = 0;
  do {
    n = 10*n + *clp++ - '0';
  } while (isdigit(*clp));
  *addr = n;
  return 1;
}

static trace_parse_name(np)
     char *np;
{
  if (*clp == '\0') return 0;
  while (isalpha(*clp) || isdigit(*clp) || *clp == '_') {
    *np++ = *clp++;
  }
  *np = '\0';
  return 1;
}

static trace_parse_predicate_name(mod, pred, arity, curmod)
     struct mod_table_entry **mod;
     struct pred_table_entry **pred;
     int *arity;
     struct mod_table_entry *curmod;
/*
  Returns
  0: nothing read
  1: only module name read ("Module")
  2: module and predicate name read ("Module:Pred")
  3: module and predicate name with arity read ("Module:Pred/Arity")
  -1: error found while parsing
*/
{
  char name[256];
  char  *cp;
  int k;
  struct mod_table_entry *mte;
  struct pred_table_entry *pte;
  *mod = 0;
  *pred = 0;
  skip_blanks();
  if (!trace_parse_name(name)) {
    return 0;
  }
  if (*clp == ':') {
    for (k = 0, mte = mod_table;
	 k < n_mod;
	 k++, mte++) {
      if (!strcmp(mte->name, name)) {
	*mod = mte;
	goto mod_found;
      }      
    }
    tracer_error("Unknown module: %s", name);
    return -1;
  mod_found:
    clp++;
    if (isspace(*clp)) return 1;
    if (!trace_parse_name(name)) {
      tracer_error("Predicate name expected after colon");
      return -1;
    }
  } else {
    *mod = mte = curmod;
  }
  for (k = 0, pte = pred_table;
       k < n_pred;
       k++, pte++) {
    if (pte->pred->func == mte->func &&
	!strcmp(pte->name, name)) {
      *pred = pte;
      goto pred_found;
    }
  }
  tracer_error("Unknown predicate: %s:%s", mte->name, name);
  return -1;
 pred_found:
  if (isspace(*clp)) return 2;
  if (*clp++ != '/') {
    tracer_error("Arity should be specified after slash");
    return -1;
  }
  if (trace_parse_int(arity) != 1) {
    tracer_error("Decimal arity expected after slash");
    return -1;
  }
  for (;
       k < n_pred;
       k++, pte++) {
    if (!strcmp(pte->name, name) &&
	pte->pred->arity == *arity) {
      *pred = pte;
      goto arity_found;
    }
  }
  tracer_error("Unknown predicate: %s:%s/%d", mte->name, name, *arity);
  return -1;
 arity_found:
  return 3;
}

#define COMMON_COMMANDS\
 Abort: \
 case EnablePort: case DisablePort: \
 case LeashPort: case UnLeashPort: \
 case Spy: case NoSpy: \
 case SetDefaultTrace: case UnsetDefaultTrace: \
 case PrintDepth: case PrintLength: case PrintVerbose: \
 case ListMod: case ListPred: case Debugging: case Unknown

static common_port_command(command, pred, curmod)
     enum tracer_command command;
     struct predicate *pred;
     struct mod_table_entry *curmod;
{
  enum trace_port port;
  struct mod_table_entry *mte;
  struct pred_table_entry *pte;
  int arity;
  int is_spy, set_flag;
  static
    print_debug_status(), print_port_names(),
    print_modules(), print_predicates();

  switch (command) {
  case Abort:
    exit(-1);
  case EnablePort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	enabled = -1;
      } else {
	enabled |= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
  case DisablePort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	enabled = 0;
      } else {
	enabled ^= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
  case LeashPort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	leashed = -1;
      } else {
	leashed |= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
  case UnLeashPort:
    if ((port = trace_parse_port()) == NoMorePort) {
      print_port_names();
      break;
    }
    do {
      if (port == UnknownPort) {
	print_port_names();
	break;
      } else if (port == AllPorts) {
	leashed= 0;
      } else {
	leashed ^= (1 << port);
      }
    } while ((port = trace_parse_port()) != NoMorePort);
    break;
#define SpyOrTrace(x) (is_spy ? (x)->spied : (x)->default_trace)    
#define SetSpyOrTrace(x, y) \
    { if (is_spy) (x)->spied = (y); else (x)->default_trace = (y); }
  case SetDefaultTrace:
    is_spy = 0;
    set_flag = 1;
    goto set_reset;
  case UnsetDefaultTrace:
    is_spy = 0;
    set_flag = 0;
    goto set_reset;
  case Spy:
    is_spy = 1;
    set_flag = 1;
    goto set_reset;
  case NoSpy:
    is_spy = 1;
    set_flag = 0;
  set_reset:
    {
      Const static char *set_reset[] = { "reset", "set" };
      Const static char *spy_trace[] = { "Default trace", "Spy point" };
      switch (trace_parse_predicate_name(&mte, &pte, &arity, curmod)) {
      case -1:
	break;
      case 1:
	for (pte = pred_table;
	     pte < pred_table + n_pred;
	     pte++) {
	  if (pte->pred->func == mte->func) SetSpyOrTrace(pte, set_flag);
	}
	fprintf(stderr, "\t%s %s on all predicates in module %s\n",
		spy_trace[is_spy], set_reset[set_flag], mte->name);
	break;
      case 2:
	do {
	  SetSpyOrTrace(pte, set_flag);
	  fprintf(stderr, "\t%s %s on predicate %s:%s/%d\n",
		  spy_trace[is_spy], set_reset[set_flag],
		  mte->name, pte->name, pte->pred->arity);
	  pte++;
	} while (pte < pred_table + n_pred &&
		 !strcmp((pte-1)->name, pte->name));
	break;
      case 0:
	pte = get_pte(pred);
	mte = pte->mte;
	arity = pred->arity;
	/* fall through */
      case 3:
	SetSpyOrTrace(pte, set_flag);
	fprintf(stderr, "\t%s %s on predicate %s:%s/%d\n",
		spy_trace[is_spy], set_reset[set_flag],
		mte->name, pte->name, pte->pred->arity);
	break;
      }
      break;
    }
  case PrintDepth:
    skip_blanks();
    if (trace_parse_int(&trace_print_depth) != 1) {
      fprintf(stderr, "\tCurrent print depth = %d\n", trace_print_depth);
    }
    break;
  case PrintLength:
    skip_blanks();
    if (trace_parse_int(&trace_print_length) != 1) {
      fprintf(stderr, "\tCurrent print length = %d\n", trace_print_length);
    }
    break;
  case PrintVerbose:
    verbose_print = !verbose_print;
    fprintf(stderr, "\tVerbose printing switched %s\n",
	    (verbose_print ? "ON" : "OFF" ));
    break;
  case ListMod:
    print_modules();
    break;
  case ListPred:
    skip_blanks();
    switch (trace_parse_predicate_name(&mte, &pte, &arity, curmod)) {
    case 0:
      print_predicates(0, 0, -1);
      break;
    case 1:
      print_predicates(mte, 0, -1);
      break;
    case 2:
      print_predicates(mte, pte, -1);
      break;
    case 3:
      print_predicates(mte, pte, arity);
      break;
    }
    break;
  case Debugging:
    print_debug_status();
    break;
  case Unknown:
    tracer_error("Unknown command");
    fprintf(stderr, "\tTry ? for command list\n");
    break;
  }
}

#define CONTROL_COMMANDS\
 Continue: case Leap: case Skip

static control_command(command)
     enum tracer_command command;
{
  switch(command) {
  case Continue:
    leaping = 0;
    break;
  case Leap:
    leaping = 1;
    break;
  case Skip:
    trace_flag = 0;
    break;
  }
}

static Const char *common_help[] = {
 "E: enable port;   D: disable port;  L: leash port;    U: unleash port;",
 "S <pred>: spy;    N <pred>: nospy;  t <pred>: trace;  n <pred>: notrace;",
 "pd: print depth;  pl: print length; pv: toggle verbose print;",
 "=: debug status   lm: list modules; lp: list predicates;",
 "a: abort execution",
 0 };

static call_port_command(qp, curmod)
     struct goalrec *qp;
     struct mod_table_entry *curmod;
{
  int again = 0;
  int err = 0;
  int k;
  enum tracer_command command;
  switch (command = trace_parse_command()) {
  case COMMON_COMMANDS:
    common_port_command(command, parent->pred, curmod);
    again = 1;
    break;
  case CONTROL_COMMANDS:
    control_command(command);
    break;
  case DumpQueue:
    dump_queue(qp);
    again = 1;
    break;
  default:
    err = 1;
  }    
  if (err) {
    Const static char *call_help[] = {
      "\t*** Commands available at CALL ports ***",
      "<cr>: continue;   c: continue;      s: skip;          l: leap;",
      0
    };
    print_help(call_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static reduce_port_command(subgoals, qp, curmod)
     unsigned long subgoals;
     struct goalrec *qp;
     struct mod_table_entry *curmod;
{
  int k, flag;
  enum tracer_command command;
  int again = 0;
  int err = 0;
  switch (command = trace_parse_command()) {
  case ToggleTrace:
    flag = -1; goto trace_notrace;
  case Trace:
    flag = 1; goto trace_notrace;
  case NoTrace:
    flag = 0;
  trace_notrace:
    if (subgoals == 0) {
      tracer_error("No subgoals to trace");
    } else if (subgoals >= 100) {
      tracer_error("Too many subgoals to control");
    } else {
      skip_blanks();
      if (*clp == '\0') {
	for (k = 0; k < subgoals; k++) {
	  subgoal_trace[k] = flag;
	}
      } else {
	int traced;
	while (1) {
	  switch (trace_parse_int(&traced)) {
	  case 0:
	    goto exit_loop;
	  case 1:
	    if (0 <= traced && traced < subgoals) {
	      subgoal_trace[traced] =
		( flag >= 0 ? flag : !subgoal_trace[traced] );
	    } else {
	      tracer_error("Subgoal %d doesn't exist", traced);
	    }
	    skip_blanks();
	    break;
	  default:
	    tracer_error("Subgoal number expected");
	    goto exit_loop;
	  }
	}
      }
    }
  exit_loop:
    again = 1;
    break;
  case CONTROL_COMMANDS:
    control_command(command);
    break;
  case COMMON_COMMANDS:
    common_port_command(command, parent->pred, curmod);
    again = 1;
    break;
  case DumpQueue:
    dump_queue(qp);
    again = 1;
    break;
  default:
    err = 1;
    break;
  }    
  if (err) {
    Const static char *reduce_help[] = {
      "\t*** Commands available at REDUCE ports ***",
      "<cr>: continue;   c: continue;      s: skip;          l: leap;",
      "+ <n>: trace;     - <n>: no trace;  <n>: toggle trace",
      0
    };
    print_help(reduce_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static susp_port_command(qp, curmod)
     struct goalrec *qp;
     struct mod_table_entry *curmod;
{
  int k;
  enum tracer_command command;
  int again = 0;
  int err = 0;
  switch (command = trace_parse_command()) {
  case CONTROL_COMMANDS:
    control_command(command);
    break;
  case COMMON_COMMANDS:
    common_port_command(command, qp->pred, curmod);
    again = 1;
    break;
  default:
    err = 1;
    break;
  }    
  if (err) {
    Const static char *susp_help[] = {
      "\t*** Commands available at SUSPENSION ports ***",
      "<cr>: continue;   c: continue;      s: skip;          l: leap;",
      0
    };
    print_help(susp_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static fail_port_command(qp, curmod)
     struct goalrec *qp;
     struct mod_table_entry *curmod;
{
  int k;
  enum tracer_command command;
  int again = 0;
  int err = 0;
  switch (command = trace_parse_command()) {
  case COMMON_COMMANDS:
    common_port_command(command, qp->pred, curmod);
    again = 1;
    break;
  case DumpQueue:
    dump_queue(qp);
    again = 1;
    break;
  default:
    err = 1;
    break;
  }    
  if (err) {
    Const static char *fail_help[] = {
      "\t*** Commands available at FAIL ports ***",
      0
    };
    print_help(fail_help);
    print_help(common_help);
    again = 1;
  }
  return again;
}

static print_port_status()
{
  static char plusminus[] = { '-', '+' };
  fprintf(stderr, "\t   port: Call Susp Redu Fail\n");
  fprintf(stderr, "\tenabled:  %c    %c    %c    %c\n",
	 plusminus[Enabled(Call)],
	 plusminus[Enabled(Susp)],
	 plusminus[Enabled(Reduce)],
	 plusminus[Enabled(Fail)]);
  fprintf(stderr, "\tleashed:  %c    %c    %c    %c\n",
	 plusminus[Leashed(Call)],
	 plusminus[Leashed(Susp)],
	 plusminus[Leashed(Reduce)],
	 plusminus[Leashed(Fail)]);
}

static print_debug_status()
{
  print_port_status();
  fprintf(stderr, "\tprint %s; depth = %u; length = %u\n",
	 (verbose_print ? "verbose" : "terse"),
	 trace_print_depth, trace_print_length);
}

static print_port_names()
{
  static Const char *port_help[] = {
    "Available ports are \"call\", \"susp\", \"redu\" and \"fail\".",
    "You can also specify \"all\".",
    0 };
  print_help(port_help);
}

/****************************************
  Printing Out
****************************************/

static tracer_error(s, a0, a1, a2, a3, a4, a5, a6, a7, a8)
     char *s;
     long a0, a1, a2, a3, a4, a5, a6, a7, a8;
{
  fprintf(stderr, "\t!!! ");
  fprintf(stderr, s, a0, a1, a2, a3, a4, a5, a6, a7, a8);
  fprintf(stderr, " !!!\n");
}

static print_help(messages)
     Const char **messages;
{
  while (*messages) {
    fprintf(stderr, "    %s\n", *messages++);
  }
}

static trace_print(stream, x)
     FILE *stream;
     q x;
{
  fprint_partially(stream, x, trace_print_depth, trace_print_length);
}

print_goal(g, mod)
{
  fprint_goal(stderr, g, mod);
}

fprint_goal(stream, g, mod)
     FILE *stream;
     struct goalrec *g;
     module mod;
{
  Const struct predicate *pred;
  struct pred_table_entry *pte;
  int traced;

  traced = Traced(g);
  pred = (traced ? predp(g->args[g->pred->arity-2]) : g->pred);
  pte = get_pte(pred);
  if (pred->func != mod) {
    fprintf(stream, "%s:", (pte ? pte->mte->name : "???"));
  }
  fprintf(stream, "%s", (pte ? pte->name : "???"));
  if (pred->arity != 0) {
    unsigned int k;
    fprintf(stream, "(");
    trace_print(stream, g->args[0]);
    for (k = 1;
	 k < pred->arity;
	 k++) {
      fprintf(stream, ",");
      trace_print(stream, g->args[k]);
    }
    fprintf(stream, ")");
  }
}

static print_modules()
{
  int k;

  fprintf(stderr, "\t*** Module List ***\n");
  for (k = 0; k < n_mod; k++) {
    struct mod_table_entry *mte = &mod_table[k];
    fprintf(stderr, "\t%8X %s\n",
	    (unsigned long)mte->func,
	    mte->name);
  }
  fprintf(stderr, "\t*******************\n");
}

static print_predicates(mte, pte, arity)
     struct mod_table_entry *mte;
     struct pred_table_entry *pte;
     int arity;
{
  int k;
  for (k = 0; k < n_pred; k++) {
    struct pred_table_entry *x = &pred_table[k];
    if ((!mte || x->mte == mte) &&
	(!pte || !strcmp(x->name, pte->name)) &&
	(arity < 0 || x->pred->arity == arity)) {
      fprintf(stderr, "\t%8X %c%c%s:%s/%d\n",
	      (unsigned long)x->pred,
	      (x->spied ? '*' : ' '),
	      (x->default_trace ? '+' : '-'),
	      x->mte->name,
	      x->name, x->pred->arity);
    }
  }
}

static void dump_one_queue(prio, qp)
     long prio;
     struct goalrec *qp;
{
  Const extern struct predicate queue_empty_pred;
  Const extern struct predicate topsucceed_pred;
  if (qp->pred != &queue_empty_pred &&
      qp->pred != &topsucceed_pred) {
    fprintf(stderr, "    Priority = %d\n", prio);
    do {
      unsigned int k;
      if (Traced(qp)) {
	fprintf(stderr, "%4d", trace_seq_of(qp));
      }
      fprintf(stderr, "\t");
      fprint_goal(stderr, qp, 0);
      fprintf(stderr, "\n");
      qp = qp->next;
    } while (qp->pred != &queue_empty_pred &&
	     qp->pred != &topsucceed_pred);
  }
}

static void dump_queue(qp)
     struct goalrec *qp;
{
  struct global_variables *glbl = &globals;
  struct prioqrec *pq = prioq.next;
  dump_one_queue(current_prio, qp);
  while (pq->prio >= 0) {
    dump_one_queue(pq->prio, pq->q);
    pq = pq->next;
  }
}
