/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
write_out(module(Name,Preds,Info)) :-
	write_module_header(Name,Preds,MinArity,MaxArity,Info),
	write_predicates(Preds,Name,MinArity,Info),
	write_module_tailer(Name,MaxArity).

write_module_header(Name,Predicates,MinArity,MaxArity,Info) :-
	klicformat("void *module_~q();~n", [Name]),
	write_pred_structs(Predicates, Name),
	declare_externals(Info),
	klicformat("~n", []),
	klicformat("void *module_~q(glbl, qp, allocp, toppred)~n", [Name]),
	klicformat("  struct global_variables *glbl;~n", []),
	klicformat("  struct goalrec *qp;~n", []),
	klicformat("  register q *allocp;~n", []),
	klicformat("  Const struct predicate *toppred;~n", []),
	klicformat("{~n", []),
	(
	    member(Info,const(_,_)) ->
	    klicformat("  static Const q constants[] = {~n", []),
	    declare_constants(Info,0),
	    klicformat("  };~n", [])
	;
	    true
	),
	pred_min_max(Predicates,99999,MinArity,0,MaxArity),
	declare_registers(MaxArity, "a"),
	klicformat("~n", []),
	klicformat("  q *reasonp;~n", []),
	klicformat(" module_top:~n", []),
	write_get_args(0, MinArity),
	klicformat("  switch_on_pred() {~n", []),
	write_pred_dispatch(Predicates),
	klicformat("  }~n", []).

declare_constants([],_) :- !.
declare_constants([const(X,Const)|T],N0) :- !,
	make_const(Const,X0,N0,N),
	( X0=cons(X) -> true; X0=funct(X) ),
	declare_constants(T,N).
declare_constants([_|T],N) :- declare_constants(T,N).

make_const_elems([],[],N,N).
make_const_elems([H|T],[X|L],N0,N) :-
	make_const(H,X,N0,N1),
	make_const_elems(T,L,N1,N).

make_const(A,atom(A),N,N) :- atom(A), !.
make_const(I,int(I),N,N) :- integer(I), !.
make_const([H|T],cons(N1),N0,N) :- !,
	make_const_elems([T,H],Elems,N0,N1),
	write_const_elems(Elems,N1,N).
make_const(S,funct(N1),N0,N) :-
	functor(S,F,A),
	S=..[F|Args],
	make_const_elems(Args,Elems,N0,N1),
	klicformat("    makesym(~f),~n", [F/A]),
	N2 is N1+1,
	write_const_elems(Elems,N2,N).

write_const_elems([],N,N).
write_const_elems([H|T],N0,N) :-
	write_const_elem(H),
	N1 is N0+1,
	write_const_elems(T,N1,N).

write_const_elem(atom(A)) :-
	klicformat("    ~a,~n", [A]).
write_const_elem(int(N)) :-
	klicformat("    makeint(~d),~n", [N]).
write_const_elem(cons(Off)) :-
	klicformat("    (q)((char*)&constants[~d]+CONS),~n", [Off]).
write_const_elem(funct(Off)) :-
	klicformat("    (q)((char*)&constants[~d]+FUNCTOR),~n", [Off]).

% Predicate information:
%	pred(Name, Arity, SeqNum, Works, Object)

write_pred_structs([], _).
write_pred_structs([pred(Name, Arity, SeqNum, _, _)|Rest], Module) :-
	klicformat("Const struct predicate ~p =~n", [Module/Name/Arity]),
	klicformat("   { module_~q, ~d, ~d };~n", [Module, SeqNum, Arity]),
	write_pred_structs(Rest, Module).

declare_externals([]).
declare_externals([ext(P)|T]) :- !,
	( member(T,ext(P)) -> true
	; klicformat("extern Const struct predicate ~p;~n", [P]) ),
	declare_externals(T).
declare_externals([gnew(Class)|T]) :- !,
	( member(T,gnew(Class)) -> true
	; name(Class,Name),
	  make_atom_name_string(Name,String),
	  klicformat("extern q ~s_g_new();~n",[String]) ),
	declare_externals(T).
declare_externals([_|T]) :-
	declare_externals(T).

pred_min_max([], MinA, MinA, MaxA, MaxA).
pred_min_max([pred(_,Arity,_,_,_)|Rest], MinA0, MinA, MaxA0, MaxA) :-
	min(Arity, MinA0, MinA1),
	max(Arity, MaxA0, MaxA1),
	pred_min_max(Rest, MinA1, MinA, MaxA1, MaxA).

declare_registers(0, _) :- !.
declare_registers(N, S) :-
	write_comma_list(N, S, "  q "),
	klicformat(";~n", []).

write_comma_list(0, _, _) :- !.
write_comma_list(N, S, Header) :-
	N1 is N-1,
	klicformat("~s", [Header]),
	write_comma_list1(0, N1, S),
	klicformat("~s~d", [S,N1]).

write_comma_list1(N, N, _) :- !.
write_comma_list1(K, N, S) :-
	klicformat("~s~d, ", [S,K]),
	K1 is K+1,
	write_comma_list1(K1, N, S).

write_pred_dispatch([pred(Name, Arity, SeqNum, _, _)]) :- !,
	klicformat("    last_case_pred(~d, ~q_~d_top);~n",
		[SeqNum, Name, Arity]).
write_pred_dispatch([pred(Name, Arity, SeqNum, _, _)|Rest]) :-
	klicformat("    case_pred(~d, ~q_~d_top);~n",
		[SeqNum, Name, Arity]),
	write_pred_dispatch(Rest).

write_predicates([],_,_,_).
write_predicates([pred(Name,Arity,_,Works,Object)|Rest],
		 Module,Min,Info) :-
	klicformat("~n", []),
	klicformat(" ~q_~d_top: {~n", [Name, Arity]),
	declare_registers(Works, "x"),
	write_get_args(Min, Arity),
	klicformat("  qp = qp->next;~n", []),
	klicformat(" ~q_~d_clear_reason:~n", [Name, Arity]),
	klicformat("  reasonp = reasons;~n", []),
	write_object(Object, info(Module/Name/Arity)),
	klicformat(" ~q_~d_ext_interrupt:~n", [Name,Arity]),
	klicformat("  reasonp = 0l;~n", []),
	klicformat(" ~q_~d_interrupt:~n",[Name,Arity]),
	(
	    member(Info, exec(Name/Arity)) ->
	    klicformat("  toppred = &~p;~n", [Module/Name/Arity])
	;
	    true
	),
	klicformat("  goto interrupt_~d;~n", [Arity]),
	klicformat(" }~n", []),
	write_predicates(Rest,Module,Min,Info).

write_get_args(N,N) :- !.
write_get_args(K,N) :-
	klicformat("  a~d = qp->args[~d];~n", [K,K]),
	K1 is K+1,
	write_get_args(K1, N).

write_module_tailer(Name, MaxArity) :-
	write_interrupt_call(MaxArity, MaxArity),
	klicformat(" proceed_label:~n", []),
	klicformat("  loop_within_module(module_~q);~n", [Name]),
	klicformat("}~n", []).

/*
  Interrupt labels have the following structure.

  interrupt_N_cont:
    Entry for continued from N+1 argument case;
    If at the boundary,
      Store argument #(N-1) to goal record extension;
      Allocate goal record extension if needed;
      Store pointer to extension to qp or newly allocated extension;
      goto interrupt_(N-1)_cont;
    Otherwise,
      goto interrupt_N_common;
  interrupt_N:
    Entry for N argumeng case;
    Allocate goal record extension, if needed;
  interrupt_N_common:
    Store argument #(N-1) to goal record;
*/

write_interrupt_call(0, _) :- !,
	klicformat(" interrupt_0:~n", []),
	klicformat("  allocp[1] = (q)toppred;~n", []),
	klicformat("  allocp[0] = (q)qp;~n", []),
	klicformat(
	"  qp = interrupt_goal((struct goalrec *)allocp, reasonp);~n", []),
	klicformat("  allocp += toppred->arity + 2;~n", []).

write_interrupt_call(N, Max) :-
	N1 is N-1,
	Off is N1+2,
	klicformat(" interrupt_~d:~n", [N]),
	klicformat("  allocp[~d] = a~d;~n", [Off, N1]),
	write_interrupt_call(N1, Max).

% Writing out inter-module info to the header file

write_header_file(module(Mod,Preds,Info)) :-
	make_ext_table(Info,[],A,[],F),
	write_ext_atoms(A),
	write_ext_functs(F),
	klicformat("module_~q~n",[Mod]),
	write_defined_preds(Preds,Mod).

write_ext_atoms([]).
write_ext_atoms([A|T]) :- member(T,A), !, write_ext_atoms(T).
write_ext_atoms([A|T]) :- klicformat("atom_~q~n",[A]), write_ext_atoms(T).

write_ext_functs([]).
write_ext_functs([FA|T]) :- member(T,FA), !, write_ext_functs(T).
write_ext_functs([F/A|T]) :-
	write_funct_name(F,A), nl,
	write_ext_functs(T).

write_defined_preds([],_).
write_defined_preds([pred(Name,Arity,_,_,_)|Rest],Module) :-
	klicformat("~p~n", [Module/Name/Arity]),
	write_defined_preds(Rest,Module).

write_ext_preds([]).
write_ext_preds([P|T]) :- member(T,P), !, write_ext_preds(T).
write_ext_preds([P|T]) :-
	klicformat("~p~n", [P]),
	write_ext_preds(T).

% Subroutines for writing atom and functor names in a canonical manner

write_atom([]) :- !, klicformat("NILATOM", []).
write_atom(.) :- !, klicformat("PERIODATOM", []).
write_atom(N) :- integer(N), !, klicformat("makeint(~dL)", [N]).
write_atom(Atom) :- klicformat("makesym(atom_~q)", [Atom]).

write_funct_name(Funct,Arity) :-
	klicformat("functor_~q_~d", [Funct,Arity]).

write_pred_name(Module,Pred,Arity) :-
	klicformat("predicate_~q_x~q_~d",[Module,Pred,Arity]).

make_atom_name_string([],Str) :- Str = [].
make_atom_name_string([H|T],Str) :-
	(
	    "0" =< H, H =< "9"
	;
	    "A" =< H, H =< "Z"
	;
	    "a" =< H, H =< "z"
	), !,
	Str = [H|StrT],
	make_atom_name_string(T, StrT).
make_atom_name_string([0'_|T],[0'_,0'_|StrT]) :-
	make_atom_name_string(T, StrT).
make_atom_name_string([H|T],[0'_|Str]) :-
	ascii_string(H, Str, StrT),
	make_atom_name_string(T, StrT).

ascii_string(C, [Upper, Lower|T], T) :-
	U is C>>4, hexchar(U, Upper),
	L is C/\15, hexchar(L, Lower).

hexchar(C, N) :- C < 10, !, N is "0" + C.
hexchar(C, N) :- N is "A" + C - 10.
