// imframe.cpp : implementation file for main Imogene window
// Copyright 1993 by Harley Davis

#include <math.h>
#include "stdafx.h"
#include "iminl.h"
#include "imobj.h"
#include "imogene.h"
#include "imframe.h"
#include "imdoc.h"
#include "imview.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CImFrame

IMPLEMENT_DYNCREATE(CImFrame, CMDIChildWnd)

CImFrame::CImFrame()
{
	m_bCycling = FALSE;
	m_nCycleSpeed = 80;	// ms between color cycles
	m_bCycleIn = TRUE;
	m_pPalEntries = NULL;
	m_nColors = 256; // should base this on available palette size.
	m_pPalette = new CPalette;
	m_bNewPalette = TRUE;
	InitPalette();
}

CImFrame::~CImFrame()
{
	delete m_pPalette;
}

// Palette stuff

void Hsi2Rgb(double h, double s, double i, int index, LPLOGPALETTE pLogPalette, int nColors)
// Convert hue, saturation, intensity to red, green, blue.
// This code is based on a similar fn found in Ezzell's "Windows 3.1 Graphics Programming".
{
	double t = 2.0*pi*h;
	double rv = 1.0+s*sin(t-2.0*pi/3.0);
	double gv = 1.0+s*sin(t);
	double bv = 1.0+s*sin(t+2.0*pi/3.0);
	t = ((double)(nColors)-.01)*i/2.0;
	BYTE RVal = (BYTE)(rv*t);
	BYTE GVal = (BYTE)(gv*t);
	BYTE BVal = (BYTE)(bv*t);
	// TRACE("Palette %d: %d, %d, %d\n", index, (int)RVal, (int)GVal, (int)BVal);
	pLogPalette->palPalEntry[index].peRed = RVal;
	pLogPalette->palPalEntry[index].peGreen = GVal;
	pLogPalette->palPalEntry[index].peBlue = BVal;
	pLogPalette->palPalEntry[index].peFlags = PC_RESERVED;
}

void CImFrame::InitPalette()
// Initialize the palette with m_ncolors equally spaced colors.
{
	static const int nPalSize = sizeof(LOGPALETTE)+(m_nColors*sizeof(PALETTEENTRY));
	
	LPLOGPALETTE pLogPalette = (LPLOGPALETTE)malloc(nPalSize);
	ASSERT(pLogPalette!=NULL);
	pLogPalette->palVersion = 0x300;
	pLogPalette->palNumEntries = m_nColors;
	// set up colors
	// use hsi with s & i as constants.
	double s = 1.0;
	double i = 1.0;
	double h = 0.0;
	double dh = 1.0/256.0;
	for (int index = 0; index<m_nColors; index++)
	{
		Hsi2Rgb(h, s, i, index, pLogPalette, m_nColors);
		h += dh;
	}
	if(m_bNewPalette) 
	{
		m_pPalette->CreatePalette(pLogPalette);
		m_bNewPalette = FALSE;
	}
	m_pPalette->AnimatePalette(0, m_nColors, pLogPalette->palPalEntry);
} 


// Sizing & Splitter window management

const int IM_NCOLS = 3;			// # of columns \\ should be modifiable
const int IM_NROWS = 3;			// # of rows 	\\ should be modifiable

const int cyCaption = GetSystemMetrics(SM_CYCAPTION);
const int cxBorder = GetSystemMetrics(SM_CXBORDER);
const int cyBorder = GetSystemMetrics(SM_CYBORDER);
const int cxFrame = GetSystemMetrics(SM_CXFRAME);
const int cyFrame = GetSystemMetrics(SM_CYFRAME);

// Splitter width & height should be 4 if I've understood WINSPLIT.CPP.
#ifdef _DEBUG
// make windows smaller in debug for faster initialization.
const int nMaxViewHeight = 100;
const int nMaxViewWidth = 100;
#else
const int nMaxViewHeight = 200;
const int nMaxViewWidth = 200;
#endif

const int cxOverhead = 2*cxBorder+2*cxFrame+(IM_NCOLS-1)*4;	
const int cyOverhead = cyCaption+2*cyBorder+2*cyFrame+(IM_NROWS-1)*4;	
const int nMaxHeight = (nMaxViewHeight*IM_NROWS)+cyOverhead;
const int nMaxWidth = (nMaxViewWidth*IM_NCOLS)+cxOverhead;

void CImFrame::GetFrameDims(int & viewWidth, int & viewHeight)
{
	// compute the initial size of a frame.  This is either 200x200 if it fits,
	// or (clientHeight/3)x(clientHeight/3).
	CRect* pRect = new CRect;
	(pTheApp->m_pMainWnd)->GetClientRect(pRect);
	int bottom = pRect->bottom;
	viewHeight = min(nMaxHeight,(bottom-cyOverhead));
	// int right = pRect->right;
	// int viewWidth =  min(nMaxWidth, (right-cxOverhead));
	// \\ I think it's better to have squares, and height is the limit on most screens.
	viewWidth = viewHeight;
	TRACE("GetViewDims: w = %d, h = %d\n", viewWidth, viewHeight);
	delete pRect;
	return;
}

BOOL CImFrame::OnCreateClient(LPCREATESTRUCT lpcs, CCreateContext* pContext)
{
	// get view height and width
	// Check out the splitter window documentation to understand this cruft. 
    int viewHeight, viewWidth;
    GetFrameDims(viewHeight, viewWidth);
    viewHeight = viewHeight/IM_NROWS;
    viewWidth = viewWidth/IM_NCOLS;
	
	TRACE("Creating views: height %d, width %d\n", viewHeight, viewWidth);

	if (!m_wndSplitter.CreateStatic(this, IM_NROWS, IM_NCOLS, WS_CHILD | WS_VISIBLE | WS_BORDER))
	{
		TRACE("** Failed to create the static splitter\n");
		return FALSE;
	}
	for (int i = 0; i < IM_NROWS; i++)
		{
			m_wndSplitter.SetRowInfo(i, viewHeight, 0);
			m_wndSplitter.SetColumnInfo(i, viewWidth, 0);
			for (int j = 0; j < IM_NCOLS; j++)
				{
					TRACE("Setting up pane @ [%d, %d]\n", i, j);
					if (!m_wndSplitter.CreateView(i, j, pContext->m_pNewViewClass,
													CSize(viewWidth,viewHeight), pContext))
					{
						TRACE("** Failed to create pane at (%d, %d)\n", i, j);
						return FALSE;
					}
					CImView* viewCur = (CImView*)m_wndSplitter.GetPane(i,j);
					viewCur->SetActive(i==0 && j==0);
					viewCur->SetRow(i);
					viewCur->SetColumn(j);
					viewCur->SetHeight(viewHeight);
					viewCur->SetWidth(viewWidth);
				}
		}
	ASSERT(((m_wndSplitter.GetPane(0,0))->GetDC())!=((m_wndSplitter.GetPane(2,2))->GetDC()));
	SetActiveView((CView*)m_wndSplitter.GetPane(0,0));
	return TRUE;
}

BOOL CImFrame::PreCreateWindow(CREATESTRUCT& cs)
// MFC kindly gives us a lousy default window size which is a 
// rectangle instead of a square, so we fix it up.
{
	TRACE("Original dimensions: h = %d, w = %d\n", cs.cy, cs.cx);   
	GetFrameDims(cs.cy, cs.cx);
	TRACE("New dimensions:      h = %d, w = %d\n", cs.cy, cs.cx);
	return CMDIChildWnd::PreCreateWindow(cs);
}

BEGIN_MESSAGE_MAP(CImFrame, CMDIChildWnd)
	//{{AFX_MSG_MAP(CImFrame)
	ON_WM_SIZE()
	ON_COMMAND(ID_TOGGLE_CYCLING, OnToggleCycling)
	ON_COMMAND(ID_LOAD_COLOR_MAP, OnLoadColorMap)
	ON_COMMAND(ID_DEFAULT_COLOR_MAP, OnDefaultColorMap)
	ON_COMMAND(ID_ACCEL_CYCLING, OnAccelCycling)
	ON_COMMAND(ID_DECEL_CYCLING, OnDecelCycling)
	ON_COMMAND(ID_CYCLE_IN, OnCycleIn)
	ON_COMMAND(ID_CYCLE_OUT, OnCycleOut)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CImFrame message handlers


void CImFrame::OnSize(UINT nType, int cx, int cy)
// this should probably just stretch the bitmap instead of 
// recalculating it.
{
	CMDIChildWnd::OnSize(nType, cx, cy);
	
	// Change splitter window sizes & recalc.
	int nTotalWidth = 0;
	int nTotalHeight = 0;
	for(int i = 0; i<IM_NROWS; i++)
	{
		int nColWidth, nColMin;
		int nRowHeight, nRowMin;
		m_wndSplitter.GetRowInfo(i, nRowHeight, nRowMin);
		m_wndSplitter.GetColumnInfo(i, nColWidth, nColMin);
		nTotalWidth+=nColWidth;
		nTotalHeight+=nRowHeight;
	}
	for(i = 0; i<IM_NROWS; i++)
	{
		m_wndSplitter.SetRowInfo(i, nTotalHeight/IM_NROWS, 0);
		m_wndSplitter.SetColumnInfo(i, nTotalWidth/IM_NCOLS, 0);
	}
	m_wndSplitter.RecalcLayout();
}

const UINT TimerID = 0xC1C; 

void CALLBACK EXPORT CycleCallback(HWND hWnd, UINT, UINT nIDEvent, DWORD)
{
	// called by timer when it's time to cycle.
	ASSERT(nIDEvent == TimerID);
	CImFrame* pFrame = (CImFrame*)CWnd::FromHandlePermanent(hWnd);
	ASSERT(pFrame);
	pFrame->CycleColors();
}

void CImFrame::OnToggleCycling()
// turn cycling on or off.
{
	if (m_bCycling)
	// turn off cycling.
	{
		KillTimer(TimerID);	// will do nothing if not started.
		if (m_pPalEntries)
		{
			free(m_pPalEntries);
			m_pPalEntries = NULL;
		}
		m_bCycling = FALSE;
	}
	else
	// turn on cycling.
	{
		if(SetTimer(TimerID, (int)m_nCycleSpeed, &CycleCallback))
		{
			m_pPalEntries = (LPPALETTEENTRY)malloc(m_nColors*sizeof(PALETTEENTRY));
			m_bCycling = TRUE;
		}
		else
		{
			AfxMessageBox("Couldn't create cycling timer; sorry.");
		}
	}
}

void CImFrame::CycleColors()
{
	// Do one cycle.
	CPalette* pPal = GetPalette();
	int nEntries = pPal->GetPaletteEntries(0, m_nColors, m_pPalEntries);
	ASSERT(nEntries==m_nColors);
	if (m_bCycleIn)
	// move it in, move it out
	// move it in, move it around
	// disco baby.
	// imogene is a sort of 70's revival program, right?
	{
		PALETTEENTRY entry1 = m_pPalEntries[0];
		for (int i = 0; i<m_nColors-1; i++)
		{
			m_pPalEntries[i] = m_pPalEntries[i+1];
		}
		m_pPalEntries[m_nColors-1] = entry1;
	}
	else
	{
		PALETTEENTRY entry1 = m_pPalEntries[m_nColors-1];
		for (int i = m_nColors-1; i>0; i--)
		{
			m_pPalEntries[i] = m_pPalEntries[i-1];
		}
		m_pPalEntries[0] = entry1;
	}
	pPal->AnimatePalette(0, m_nColors, m_pPalEntries);
}

void CImFrame::OnLoadColorMap()
// pop a file asker and then load the selected WinFract color map.
// Thanks to Bert Tyler for permission to redistribute the WinFract .MAP files.
{   
	BOOL bWasCycling = m_bCycling;
	if (bWasCycling) OnToggleCycling(); // turn off cycling.
	
	CFileDialog dlg(TRUE, "MAP", NULL, NULL, 
	"Winfract maps (*.map) | *.map | All files (*.*) | *.* ||");
	
	if (dlg.DoModal()==IDOK)
	{
		// pull up a color map.
		CString sPath = dlg.GetPathName();
		
		static const int nPalSize = sizeof(LOGPALETTE)+(256*sizeof(PALETTEENTRY));
	
		LPLOGPALETTE pLogPalette = (LPLOGPALETTE)malloc(nPalSize);
		ASSERT(pLogPalette!=NULL);
		pLogPalette->palVersion = 0x300;
		pLogPalette->palNumEntries = 256;
		BYTE r, g, b;
		
		char* line = new char[101];
		FILE* f = fopen((const char*)sPath, "r");
		if (f == NULL)
		{
			AfxMessageBox("File not found");
			return;
		}
		TRACE("Loading map file %s\n", sPath);
		for (int index = 0; index < 256; index++)
		{
			if (fgets(line,100,f) == NULL)
				break;
			sscanf(line, "%d %d %d", &r, &g, &b);
			pLogPalette->palPalEntry[index].peRed = r;
			pLogPalette->palPalEntry[index].peGreen = g;
			pLogPalette->palPalEntry[index].peBlue = b;
			pLogPalette->palPalEntry[index].peFlags = PC_RESERVED;
		}
		TRACE("Loaded %d colors.\n", index);
		fclose(f);
		m_nColors = index;
		if(!(m_pPalette->ResizePalette(index)))
		{
			AfxMessageBox("ResizePalette failed!");
			return;
		}
		m_pPalette->AnimatePalette(0, index, pLogPalette->palPalEntry);
		free(pLogPalette);
		delete line;
		return;
	}
	
	if (bWasCycling) OnToggleCycling(); // turn cycling back on if it was on.
	
}

void CImFrame::OnDefaultColorMap()
// in case of weirdness, we can regenerate the initial color map
// from scratch.
{
	m_nColors = 256;
	InitPalette();		
}

const long MaxCycleSpeed = 2000;
const long MinCycleSpeed = 10;

void CImFrame::OnAccelCycling()
{
	// to accelerate, decrease time between cycles.
	if (m_nCycleSpeed<=MinCycleSpeed)
		m_nCycleSpeed = MinCycleSpeed;
	else
		m_nCycleSpeed = (90*m_nCycleSpeed)/100;
		
	TRACE("New cycle speed is %d\n", m_nCycleSpeed);
	if (m_bCycling)
	{
		OnToggleCycling();
		OnToggleCycling();
	}	
}

void CImFrame::OnDecelCycling()
{
	// to decelerate, increase time between cycles.
	if (m_nCycleSpeed>=MaxCycleSpeed)
		m_nCycleSpeed = MaxCycleSpeed;
	else
		m_nCycleSpeed = (110*m_nCycleSpeed)/100;
		
	TRACE("New cycle speed is %d\n", m_nCycleSpeed);
	if (m_bCycling)
	{
		OnToggleCycling();
		OnToggleCycling();
	}		
}

void CImFrame::OnCycleIn()
// this is what happens when you press the left arrow.
{
	m_bCycleIn = TRUE;	
}

void CImFrame::OnCycleOut()
// and this happens when you press the right arrow.
{
	m_bCycleIn = FALSE;
}
