Random subclass: #GeneticRandom
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticRandom methodsFor: 'enumerating'!

next: anInteger
	"This returns a random integer in range [1, anInteger]"
	^(self next * anInteger) truncated + 1.!

nextBoolean
	"This returns a random Boolean"
	^self nextBoolean: 0.5.!

nextBoolean: aFloat
	"This returns a random Boolean with a biased range"
	^self next < aFloat.!

nextZeroOne
	"This returns a random of either zero or one"
	^self next < 0.5 ifTrue: [0] ifFalse: [1].! !

!GeneticRandom methodsFor: 'accessing'!

seed

	"This returns the instance variable"

	^seed.! !

Object subclass: #GeneticObject
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticObject methodsFor: 'evaluation'!

eval: anEnvironment
	"This method is for the evaluation of a GeneticObject
	during fitness computation"
	^self subclassResponsibility.!

postEval: anEnvironment
	"This method is invoked after to the evaluation of a GeneticObject
	during fitness computation. The method is designed to allow some
	initialization of the program to occur after evaluation. For example,

	we might want to gather statistics on aspects of the evaluation."

	^anEnvironment!

preEval: anEnvironment
	"This method is invoked prior to the evaluation of a GeneticObject
	during fitness computation. The method is designed to allow some
	initialization of the program to occur prior to evaluation."

	^anEnvironment! !

!GeneticObject methodsFor: 'accessing'!

size
	"This computes the size of the GeneticObject"
	^self subclassResponsibility.! !

!GeneticObject methodsFor: 'evolution'!

crossover: aGeneticObject Using: aGeneticAlgorithm
	"This generates a new pair of genetic objects"
	^self subclassResponsibility!

mutateUsing: aGeneticAlgorithm
	"This generates a mutated genetic object"
	^self subclassResponsibility! !

!GeneticObject methodsFor: 'initialization'!

initialize
	"This initializes a newly created instance. This is
	provided in order that the class new method will 
	find the behaviour"! !

!GeneticObject methodsFor: 'printing'!

postStoreOn: aStream

	"This stores subclass-specific information"!

storeOn: aStream

	"This stores the parameter components of the object on the stream. The postStore: method 

	in the subclass stakes care of the subclass-specific information."

	aStream nextPutAll: '(('.

	aStream nextPutAll: self class name.

	aStream nextPutAll: ' new) '.

	self postStoreOn: aStream.

	aStream nextPutAll: '; yourself)'.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticObject class
	instanceVariableNames: ''!


!GeneticObject class methodsFor: 'instance creation'!

generateUsing: anObject
	"This allows a random object to be generated"
	self subclassResponsibility.!

new
	"This creates a new instance"
	^super new initialize.! !

GeneticObject subclass: #GeneticProgramObject
	instanceVariableNames: 'function '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticProgramObject methodsFor: 'accessing'!

function
	"This returns the instance variable"
	^function.!

function: aSymbol
	"This sets the instance variable"
	^function := aSymbol.!

functor
	"This returns the instance variable"
	^function.! !

!GeneticProgramObject methodsFor: 'evolution'!

crossover: aGeneticProgram Using: aGeneticAlgorithm
	"This generates a pair of GeneticPrograms which have swapped subtrees"
	| aCopy1 aCopy2 aNode1 aNode2 |
	aCopy1 := self copy.
	aCopy2 := aGeneticProgram copy.
	aNode1 := aCopy1 selectNodeUsing: aGeneticAlgorithm.
	aNode2 := aCopy2 selectNodeUsing: aGeneticAlgorithm.
	aNode1 become: aNode2.
	aCopy1 depth > aGeneticAlgorithm maxDepth ifTrue: [aCopy1 := self copy].
	aCopy2 depth > aGeneticAlgorithm maxDepth ifTrue: [aCopy2 := aGeneticProgram copy].
	^Array with: aCopy1 with: aCopy2.!

mutateUsing: aGeneticAlgorithm
	"This generates a mutated GeneticProgram. Note, it is possible for 
	a GeneticProgramTerminal to become a GeneticProgram via this
	mutation process."
	| aCopy aNode oldNode |
	aCopy := self copy.
	oldNode := aNode := aCopy selectNodeUsing: aGeneticAlgorithm.
	aNode become: (self class generateUsing: aGeneticAlgorithm).
	aCopy depth > aGeneticAlgorithm maxDepth ifTrue: [aNode become: oldNode].
	^aCopy.! !

!GeneticProgramObject methodsFor: 'copying'!

copy
	"This generates a copy of the GeneticProgram. It's rather in the nature 
	of a deepCopy as the whole subtree is copied. We access the instance
	variables directly here for speed reasons."
	| aCopy |
	aCopy := self class new.
	aCopy function: (function copy).

	^aCopy.! !

!GeneticProgramObject methodsFor: 'printing'!

postStoreOn: aStream

	"This stores subclass-specific information"

	aStream nextPutAll: ' function: '.

	self function storeOn: aStream.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticProgramObject class
	instanceVariableNames: ''!


!GeneticProgramObject class methodsFor: 'arity'!

arity

	"This returns the function arity for the functions in the behaviour protocol"

	^#().! !

GeneticProgramObject subclass: #GeneticProgramTerminal
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticProgramTerminal methodsFor: 'evaluation'!

eval: anEnvironment
	"This computes the results of the evaluation. Although it is intended
	that terminals are atomic in nature - integers, floats, strings and such -
	we could envisage using blocks or even methods with no arguments."
	^function value.! !

!GeneticProgramTerminal methodsFor: 'accessing'!

depth
	"This returns the depth of a GeneticProgramTerminal"
	^1.!

objectsAtDepth: anInteger
	"This returns the object if it's at the right level, nil otherwise"
	^anInteger = 1 ifTrue: [self] ifFalse: [nil].!

size
	"This returns the size of a GeneticProgramTerminal"
	^1.! !

!GeneticProgramTerminal methodsFor: 'printing'!

printOn: aStream
	"This appends a string representation of itself to the stream"
	self class printOn: aStream.
	aStream nextPutAll: ' value ', self function printString.! !

!GeneticProgramTerminal methodsFor: 'private'!

findAllNodesOfClass: aClass
	"This returns an OrderedCollection with all nodes of a given class in it"
	| allNodes |
	allNodes := OrderedCollection new.
	self class == aClass ifTrue: [allNodes add: self].
	^allNodes.!

selectNodeUsing: aGeneticAlgorithm
	"Only one node exists, so it better be this one!!"
	^self.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticProgramTerminal class
	instanceVariableNames: ''!


!GeneticProgramTerminal class methodsFor: 'instance creation'!

generateUsing: aGeneticProgramming
	"This allows a random object to be generated"
	^self new: (aGeneticProgramming terminals at: (aGeneticProgramming random next: aGeneticProgramming terminals size)).!

new: anObject
	"This allows a random object to be generated."
	| aGPObject |

	aGPObject := self new.

	aGPObject function: anObject.
	^aGPObject.! !

GeneticProgramTerminal subclass: #GeneticAntTerminal
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticAntTerminal methodsFor: 'behaviour'!

advance: anAnt

	"This causes the ant to advance"

	anAnt advance.!

turnLeft: anAnt

	"This causes the ant to turn left"

	anAnt turnLeft.!

turnRight: anAnt

	"This causes the ant to turn right"

	anAnt turnRight.! !

!GeneticAntTerminal methodsFor: 'evaluation'!

eval: anEnv
	"This just evaluates the function"
	| foodLeft |
	self perform: function with: anEnv.
	foodLeft := anEnv foodTotal - anEnv foodFound.
	^foodLeft.!

preEval: anEnv
	"This just creates a new environment"
	^(Smalltalk at: anEnv) new.! !

GeneticProgramObject subclass: #GeneticProgram
	instanceVariableNames: 'arguments '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!
GeneticProgram comment:
'I am a program represented as a tree structure. While I inherit from GeneticObject, it means
that I can have my fitness evaluated at any node in the tree. This is useful for storing the
"value" of a program fragment, for example.'!


!GeneticProgram methodsFor: 'evaluation'!

eval: anEnvironment
	"This evaluates the GeneticProgram. We don't need to check the arguments as all
	leaf nodes are GeneticProgramTerminal objects."
	^self perform: function with: anEnvironment.! !

!GeneticProgram methodsFor: 'generation'!

random: aRandom functions: anOrderedCollection terminals: anotherOrderedCollection depth: anInteger
	"This method generates a random GeneticProgram. Note that object mutation is used for the generation
	of terminal nodes."
	|  fnSize maxSize index functionArity functionDef  |
	anInteger isZero ifTrue: [
		^self become: (self objectTerminalClass new: (anotherOrderedCollection at: (aRandom next: anotherOrderedCollection size))).
	]
	ifFalse: [
		fnSize := anOrderedCollection size.
		maxSize :=  fnSize + (anotherOrderedCollection size).
		index := aRandom next: maxSize.
		index > fnSize ifTrue: [
			^self become: (self objectTerminalClass new: (anotherOrderedCollection at: (index - fnSize))).
		]
		ifFalse: [		"The function set in anOrderedCollection are Arrays"
			functionDef := anOrderedCollection at: index.
			functionArity := functionDef at: 2.
			function := functionDef at: 1.
			arguments := Array new: functionArity.
			1 to: functionArity do: [:i |
				arguments at: i 
					put: (self class 
						random: aRandom 
						functions: anOrderedCollection 
						terminals: anotherOrderedCollection 
						maxDepth: (anInteger - 1)).
			]
		]
	].! !

!GeneticProgram methodsFor: 'accessing'!

arguments
	"This returns the instance variable"
	^arguments.!

arguments: anArray
	"This sets the instance variable"
	^arguments := anArray.!

depth
	"This returns the depth of a GeneticProgram in terms of the number of levels
	in the tree"
	^1 + (self arguments inject: 0 into: [:sum :arg | sum max: arg depth]).!

objectsAtDepth: anInteger
	"This returns the objects at a given depth in the tree"
	| objects argObj |
	anInteger = 1 
		ifTrue: [^self] 
		ifFalse: [
			objects := OrderedCollection new.
			self arguments do: [ :arg | 
				argObj := arg objectsAtDepth: (anInteger - 1).
				argObj isNil ifFalse: [
					(argObj isMemberOf: OrderedCollection)
						ifTrue: [objects addAll: argObj]
						ifFalse: [objects add: argObj].
				].
			].
			objects isEmpty ifTrue: [^nil] ifFalse: [^objects].
		].!

size
	"This returns the size of a GeneticProgram in terms of the number of nodes
	in the tree"
	^self arguments inject: 1 into: [:sum :arg | sum + arg size].! !

!GeneticProgram methodsFor: 'private'!

findAllNodesOfClass: aClass
	"This returns an OrderedCollection with all nodes of a given class in it"
	| allNodes |
	allNodes := OrderedCollection new.
	self arguments do: [ :arg| allNodes addAll: (arg findAllNodesOfClass: aClass)].
	self class == aClass ifTrue: [allNodes add: self].
	^allNodes.!

objectTerminalClass

	"This returns the name of the class used for terminals"

	^Smalltalk at: (self class name asString, 'Terminal') asSymbol.!

selectNodeUsing: aGeneticAlgorithm
	"This method randomly selects a node biased 90-10 towards internal nodes"
	| random objects |
	random := aGeneticAlgorithm random.
	objects := (random nextBoolean: 0.9) 
		ifTrue: [self findAllNodesOfClass: self class]
		ifFalse: [self findAllNodesOfClass: self objectTerminalClass].

	^objects at: (random next: objects size).!

selectNodeUsingOld: aGeneticAlgorithm
	"This method randomly selects a node from the tree. Firstly, it selects a level.
	Then it collects all of the objects from that level. Finally, it randomly selects an
	object from that level."
	| random level objects |
	random := aGeneticAlgorithm random.
	level := random next: self depth.
	objects := self objectsAtDepth: level.
	^(objects isKindOf: OrderedCollection) ifTrue: [objects at: (random next: objects size)] ifFalse: [objects].! !

!GeneticProgram methodsFor: 'copying'!

copy
	"This generates a copy of the GeneticProgram. It's rather in the nature 
	of a deepCopy as the whole subtree is copied. We access the instance
	variables directly here for speed reasons."
	| aCopy size |
	aCopy := super copy.
	size := arguments size.
	aCopy arguments: (Array new: size).
	1 to: size do: [ :i |
		aCopy arguments at: i put: ((arguments at: i) copy).
	].
	^aCopy.! !

!GeneticProgram methodsFor: 'printing'!

postStoreOn: aStream

	"This stores subclass-specific information"

	super postStoreOn: aStream.

	aStream nextPutAll: '; arguments: '.

	self arguments storeOn: aStream.!

printOn: aStream
	"This appends a string representation of itself to the stream"
	self class printOn: aStream.
	aStream nextPutAll: ' with ', self size printString, ' nodes and of depth ', self depth printString.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticProgram class
	instanceVariableNames: ''!


!GeneticProgram class methodsFor: 'instance creation'!

generateUsing: aGeneticProgramming
	"This allows a random object to be generated"
	^self
		random: (aGeneticProgramming random) 
		functions: (aGeneticProgramming functions) 
		terminals: (aGeneticProgramming terminals) 
		maxDepth: (aGeneticProgramming maxInitialDepth).!

random: aRandom functions: anOrderedCollection terminals: anotherOrderedCollection maxDepth: anInteger
	"This method generates a new random GeneticProgram"
	| aGeneticProgram |
	aGeneticProgram := self new.
	aGeneticProgram random: aRandom functions: anOrderedCollection terminals: anotherOrderedCollection depth: anInteger.
	^aGeneticProgram! !

GeneticProgram subclass: #GeneticAnt
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticAnt methodsFor: 'behaviour'!

ifSensor: anAnt
	"This looks at the food at the next location (as determined by advance)
	and if food is present executes the 'true' code, otherwise the 'false'
	code is executed."

	anAnt senseFood 
		ifTrue: [(arguments at: 1) eval: anAnt]
		ifFalse: [(arguments at: 2) eval: anAnt].!

progn: anAnt
	"This evaluates the function in the current environment"

	arguments do: [ :arg | arg perform: arg function with: anAnt].! !

!GeneticAnt methodsFor: 'evaluation'!

eval: anEnv
	"This evaluates the Ant program fitness"
	| foodLeft |

	foodLeft := anEnv foodTotal - anEnv foodFound.
	[foodLeft > 0 and: [anEnv steps > 0]] whileTrue: [
		super eval: anEnv.
		foodLeft := anEnv foodTotal - anEnv foodFound.
	].
	^foodLeft.!

preEval: anEnv
	"This just creates a new environment"
	^(Smalltalk at: anEnv) new.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticAnt class
	instanceVariableNames: ''!


!GeneticAnt class methodsFor: 'arity'!

arity

	"This returns the function arity for the functions in the behaviour protocol"

	^#(2 3).! !

!GeneticAnt class methodsFor: 'examples'!

example
	"This evaluates a given Ant program"
	"GeneticAnt example"
	| antProgram anEnv aUI aGASTE |
	antProgram := (Filename named: 'gpeist1.gpf') fileIn.
	anEnv := JohnMuirTrailEnv new.
	aGASTE := GASTE new.
	aGASTE experiment population fittestMember: antProgram.
	aUI := GeneticAntAnimationViewer openWith: aGASTE.
	anEnv addDependent: aUI.! !

GeneticObject subclass: #GeneticBitString
	instanceVariableNames: 'string '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticBitString methodsFor: 'generation'!

random: aGeneticRandom
	"This initializes the string to random bits"
	| size |
	size := self string size.
	1 to: size do: [:i |
		self string at: i put: aGeneticRandom nextZeroOne.
	].! !

!GeneticBitString methodsFor: 'copying'!

copy
	"This copies the bitstring portion of the GeneticObject"
	| aGeneticObject |
	aGeneticObject := self class new.
	aGeneticObject string: (self string copy).
	^aGeneticObject.! !

!GeneticBitString methodsFor: 'accessing'!

size
	"This returns the size of the bit string"
	^self string size.!

string
	"This returns the instance variable"
	^string.!

string: anArray
	"This sets the instance variable"
	^string := anArray.! !

!GeneticBitString methodsFor: 'evolution'!

crossover: aGeneticBitString Using: aGeneticAlgorithm
	"This causes bits to be exchanged between solutions"
	| bit cut bitString1 bitString2 |
	cut := aGeneticAlgorithm random next: aGeneticBitString size.
	bitString1 := self copy.
	bitString2 := aGeneticBitString copy.
	1 to: cut do: [ :i |
		bit := bitString1 string at: i.
		bitString1 string at: i put: (bitString2 string at: i).
		bitString2 string at: i put: bit.
	].
	^Array with: bitString1 with: bitString2.!

mutateUsing: aGeneticAlgorithm
	"This causes bits to be mutated for this solution"
	| size |
	size := self size.
	1 to: size do: [ :i |
		(aGeneticAlgorithm random nextBoolean: aGeneticAlgorithm pMutation) ifTrue: [
			self string at: i put: aGeneticAlgorithm random nextZeroOne.
		]
	].! !

!GeneticBitString methodsFor: 'printing'!

postStoreOn: aStream

	"This stores subclass-specific information"

	aStream nextPutAll: '; string: '.

	self string storeOn: aStream.!

printOn: aStream
	"This prints a string representation of itself on the stream"
	self string do: [ :bit |
		bit printOn: aStream.
	].
	aStream cr.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticBitString class
	instanceVariableNames: ''!


!GeneticBitString class methodsFor: 'instance creation'!

generateUsing: aGeneticBitStringAlgorithm
	"This allows a random object to be generated"
	| aGeneticBitString |
	aGeneticBitString := self new: aGeneticBitStringAlgorithm stringSize.
	aGeneticBitString random: aGeneticBitStringAlgorithm random.
	^aGeneticBitString!

new: anInteger
	"This creates a new instance"
	| aGeneticBitString |
	aGeneticBitString := self new initialize.
	aGeneticBitString string: (Array new: anInteger).
	^aGeneticBitString.! !

GeneticBitString subclass: #GeneticAllOnesString
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GAObj'!


!GeneticAllOnesString methodsFor: 'evaluation'!

eval: anEnvironment
	"This method is for the evaluation of a GeneticObject
	during fitness computation"
	^self string inject: self string size into: [:sum :i | sum - i].! !
