Model subclass: #GeneticAlgorithm
	instanceVariableNames: 'pMutation pCrossover sMechanism trials converged population random maxEvaluations objectClass env evaluations fitnessThreshold outputAtEvaluations outputFrequency isChanged semaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GA'!
GeneticAlgorithm comment:
'I represent a class of algorithms called Genetic Algorithms. I am based upon ideas
taken from Genetics and Natural Selection. In a Genetic Algorithm a population of 
candidate solutions to a given problem are each assessed as to how well they
solve the problem. Evolution occurs by preferentially reproducing solutions which
best solve the problem and then mating them in order to generate new combinations
of genetic material.'!


!GeneticAlgorithm methodsFor: 'initialization'!

initialize
	"This initializes the instance"
	self objectClass: #GeneticObject.
	self env: GeneticEnvironment.
	self fitnessThreshold: 0.0.
	self converged: false.
	self outputAtEvaluations: 0.
	self isChanged: false.
	self outputFrequency: 100.
	self maxEvaluations: 5000.
	self evaluations: 0.
	self sMechanism: #tournament.
	self trials: 2.
	self random: GeneticRandom new.
	self pMutation: 0.001.
	self pCrossover: 0.8.
	self population: GeneticPopulation new.
	self initializeQueue.!

initializePopulation
	"This creates a population of genetic objects"
	self population initializePopulationUsing: self.!

initializeQueue
	"This initializes the instance' queue"
	self semaphore: (SharedQueue new).! !

!GeneticAlgorithm methodsFor: 'evaluation'!

populationFitness
	"This evaluates the fitness of each object in the population and
	computes the total fitness of the population."
	self population populationFitness: self.
	self population computeFittestMember.! !

!GeneticAlgorithm methodsFor: 'evolution'!

evolvePopulation
	"This causes changes to be made in the population by applying
	crossover and mutation operations"
	self population evolvePopulationUsing: self.! !

!GeneticAlgorithm methodsFor: 'testing'!

convergenceCheck
	"This method checks to see if we've converged on a solution"
	self converged: (self evaluations > self maxEvaluations or: [self population fittestMember rawFitness <= self fitnessThreshold])!

outputCheck
	"This method determines whether it is time for some output. If it
	is, notify the view to update itself"

	((self evaluations - self outputAtEvaluations) >= self outputFrequency) ifTrue: [
		self outputAtEvaluations: self evaluations.
		self populationFitness.
		self updateDependents.
	].!

shouldCrossover

	"This returns true if we should crossover, false otherwise"

	^self random nextBoolean: self pCrossover.!

shouldMutate

	"This returns true if we should mutate, false otherwise"

	^self random nextBoolean: self pMutation.! !

!GeneticAlgorithm methodsFor: 'algorithm'!

algorithm
	"This method performs the genetic experiment"

	self isChanged: true.
	[self converged] whileFalse: [
		self evolvePopulation.
		self populationFitness.
		self convergenceCheck.
		"self outputCheck."
	].

	self sortPopulation.
	"Don't output the same measures twice"
	self outputAtEvaluations = self evaluations ifFalse: [self updateDependents].
	self informOfEndOfRun.!

continueExperiment
	"This method performs the genetic experiment"

	self maxEvaluations: (self evaluations + self maxEvaluations).
	self initializeQueue.
	self converged: false.
	self algorithm.!

runExperiment
	"This method performs the genetic experiment"
	self converged: false.
	self initializeQueue.
	self initializePopulation.

	self evaluations: 0.
	self  outputAtEvaluations: (self sizeOfPopulation - self outputFrequency).
	self populationFitness.
	self algorithm.! !

!GeneticAlgorithm methodsFor: 'accessing'!

converged
	"This returns the instance variable"
	^converged.!

converged: aBoolean
	"This sets the instance variable"
	^converged := aBoolean.!

env
	"This returns the instance variable"
	^env.!

env: anObject
	"This sets the instance variable"
	anObject = env ifFalse: [self isChanged: true].
	^env := anObject.!

evaluations
	"This returns the instance variable"
	^evaluations.!

evaluations: anInteger
	"This sets the instance variable"
	anInteger = evaluations ifFalse: [self isChanged: true].
	^evaluations := anInteger.!

fitnessThreshold
	"This returns the instance variable"
	^fitnessThreshold.!

fitnessThreshold: aFloat
	"This returns the instance variable"
	aFloat = fitnessThreshold ifFalse: [self isChanged: true].
	(aFloat positive and: [aFloat <= 1.0]) ifTrue: [fitnessThreshold := aFloat].
	^fitnessThreshold.!

incrementEvaluations
	"This adds one to the number of evaluations"
	self evaluations: (self evaluations + 1).
	self outputCheck.
	^self evaluations.!

isChanged
	"This returns the instance variable"
	^isChanged.!

isChanged: aBoolean
	"This sets the instance variable"
	^isChanged := aBoolean.!

maxEvaluations
	"This returns the instance variable"
	^maxEvaluations.!

maxEvaluations: anInteger
	"This sets the instance variable"
	anInteger = maxEvaluations ifFalse: [self isChanged: true].
	^maxEvaluations := anInteger.!

next
	"This returns the next random number in a sequence"
	^random next.!

objectClass
	"This returns the instance variable"
	^objectClass.!

objectClass: aSymbol
	"This sets the instance variable"
	aSymbol = objectClass ifFalse: [self isChanged: true].
	^objectClass := aSymbol.!

outputAtEvaluations
	"This returns the instance variable"
	^outputAtEvaluations.!

outputAtEvaluations: anInteger
	"This returns the instance variable"
	^outputAtEvaluations := anInteger.!

outputFrequency
	"This returns the instance variable"
	^outputFrequency.!

outputFrequency: anInteger
	"This sets the instance variable"
	anInteger = outputFrequency ifFalse: [self isChanged: true].
	^outputFrequency := anInteger.!

pCrossover
	"This returns the probability of crossover"
	^pCrossover.!

pCrossover: aFloat
	"This sets the probability of crossover"
	aFloat = pCrossover ifFalse: [self isChanged: true].
	(aFloat between: 0.0 and: 1.0) ifTrue: [pCrossover := aFloat].
	^pCrossover.!

pMutation
	"This returns the probability of mutation"
	^pMutation.!

pMutation: aFloat
	"This sets the probability of mutation"
	aFloat = pMutation ifFalse: [self isChanged: true].
	(aFloat between: 0.0 and: 1.0) ifTrue: [pMutation := aFloat].
	^pMutation.!

population
	"This returns the instance variable"
	^population.!

population: aGeneticPopulation
	"This sets the instance variable"
	^population := aGeneticPopulation.!

random
	"This returns the instance variable"
	^random.!

random: aRandom
	"This sets the instance variable"
	^random := aRandom.!

seed
	"This sets the seed for the random number generator"
	^self random seed.!

seed: aSmallInteger
	"This sets the seed for the random number generator"
	(aSmallInteger = self random seed) ifFalse: [self isChanged: true].
	^self random seed: aSmallInteger.!

semaphore
	"This returns the instance variable"
	^semaphore.!

semaphore: aSemaphore
	"This sets the instance variable"
	^semaphore := aSemaphore.!

sizeOfPopulation
	"This returns the instance variable"
	^self population sizeOfPopulation!

sizeOfPopulation: anInteger
	"This sets the instance variable"
	anInteger = self population sizeOfPopulation ifFalse: [self isChanged: true].
	^self population sizeOfPopulation: anInteger!

sMechanism
	"This returns the instance variable"
	^sMechanism.!

sMechanism: aSymbol
	"This sets the instance variable"
	aSymbol == sMechanism ifFalse: [self isChanged: true].
	(aSymbol == #roulette or: [aSymbol == #tournament]) ifTrue: [sMechanism := aSymbol].
	^sMechanism.!

trials
	"This returns the instance variable"
	^trials.!

trials: anInteger
	"This sets the instance variable"
	anInteger = trials ifFalse: [self isChanged: true].
	^trials := anInteger.! !

!GeneticAlgorithm methodsFor: 'sorting'!

sortPopulation

	"This sorts the population using fitness values after it's initial population

	has been evaluated"

	self population sortPopulation.! !

!GeneticAlgorithm methodsFor: 'printing'!

postStoreOn: aStream

	"This stores subclass-specific information"!

storeOn: aStream

	"This stores the parameter components of the algorithm on the stream. The postStore: method 

	in the subclass stakes care of the subclass-specific information."

	aStream nextPutAll: '(('.
	aStream nextPutAll: self class name.
	aStream nextPutAll: ' new) '.
	self storeParametersOn: aStream.
	self postStoreOn: aStream.
	aStream nextPutAll: '; yourself)'.!

storeParametersOn: aStream

	"This stores the parameter components of the algorithm on the stream"

	aStream nextPutAll: 'pMutation: '.
	self pMutation storeOn: aStream.
	aStream nextPutAll: '; pCrossover: '.
	self pCrossover storeOn: aStream.
	aStream nextPutAll: '; sMechanism: '.
	self sMechanism storeOn: aStream.
	aStream nextPutAll: '; trials: '.
	self trials storeOn: aStream.
	aStream nextPutAll: '; maxEvaluations: '.
	self maxEvaluations storeOn: aStream.
	aStream nextPutAll: '; objectClass: '.
	self objectClass storeOn: aStream.
	self env isNil ifFalse: [
		aStream nextPutAll: '; env: '.
		self env storeOn: aStream.
	].
	aStream nextPutAll: '; fitnessThreshold: '.
	self fitnessThreshold storeOn: aStream.
	aStream nextPutAll: '; outputFrequency: '.
	self outputFrequency storeOn: aStream.

	aStream nextPutAll: '; sizeOfPopulation: '.
	self sizeOfPopulation storeOn: aStream.
	aStream nextPutAll: '; seed: '.
	self seed storeOn: aStream.! !

!GeneticAlgorithm methodsFor: 'private'!

informOfEndOfRun
	"This method informs all dependents that the run has ended"
	self changed: #endOfRun.!

updateDependents
	"This method ensures that all dependent objects are updated"

	self outputAtEvaluations: self evaluations.
	self changed: #output with: self population computeFittestMember.!

updateStatus: anObject
	"This method causes updates to occur in the status field of the
	GASTE window"

	self changed: #status with: anObject.! !

!GeneticAlgorithm methodsFor: 'updating'!

changed: aSymbol with: anObject
	"This allows a changed message to be sent to my dependents"

	super changed: aSymbol with: anObject.
	self dependents do: [:obj | self semaphore next].! !

!GeneticAlgorithm methodsFor: 'protocol'!

continue
	"This allows the GA to continue."

	self semaphore nextPut: self.
	Processor yield.! !

!GeneticAlgorithm methodsFor: 'statistics'!

statisticalDepthMeasures
	"This computes mean and std. dev. of population depth. An array is returned
	with the mean as the first element and the variance as the second element"

	| total totalSquared depth mean |
	total := totalSquared := 0.0.
	self population members do: [ :member |
		depth := member object depth.
		total := total + depth.
		totalSquared := totalSquared + (depth squared).
	].
	mean := total / self population sizeOfPopulation.
	^Array with: mean with: ((totalSquared / self population sizeOfPopulation) - (mean * mean)).!

statisticalFitnessMeasures
	"This computes mean and std. dev. of population fitness. An array is returned
	with the mean as the first element and the variance as the second element"

	| total totalSquared fitness mean |
	total := totalSquared := 0.0.
	self population members do: [ :member |
		fitness := member  fitness.
		total := total + fitness.
		totalSquared := totalSquared + (fitness squared).
	].
	mean := total / self population sizeOfPopulation.
	^Array with: mean with: ((totalSquared / self population sizeOfPopulation) - (mean * mean)).!

statisticalRawFitnessMeasures
	"This computes mean and std. dev. of population fitness. An array is returned
	with the mean as the first element and the variance as the second element"

	| total totalSquared fitness mean |
	total := totalSquared := 0.0.
	self population members do: [ :member |
		fitness := member  rawFitness.
		total := total + fitness.
		totalSquared := totalSquared + (fitness squared).
	].
	mean := total / self population sizeOfPopulation.
	^Array with: mean with: ((totalSquared / self population sizeOfPopulation) - (mean * mean)).!

statisticalRawMinMaxFitnessMeasures
	"This computes the minimum and maximum of population fitness. An array is returned
	with the minimum as the first element and the maximum as the second element"

	|  max min fitness |
	max := 0.0. min := 1.0e30.
	self population members do: [ :member |
		fitness := member  rawFitness.
		fitness < min ifTrue: [min := fitness].
		fitness > max ifTrue: [max := fitness].
	].
	^Array with: min with: max.!

statisticalSizeMeasures
	"This computes mean and std. dev. of population size. An array is returned
	with the mean as the first element and the variance as the second element"

	| total totalSquared size mean |
	total := totalSquared := 0.0.
	self population members do: [ :member |
		size := member object size.
		total := total + size.
		totalSquared := totalSquared + (size squared).
	].
	mean := total / self population sizeOfPopulation.
	^Array with: mean with: ((totalSquared / self population sizeOfPopulation) - (mean * mean)).! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticAlgorithm class
	instanceVariableNames: ''!


!GeneticAlgorithm class methodsFor: 'instance creation'!

new
	"This creates and initializes an instance"
	^super new initialize.! !

GeneticAlgorithm subclass: #GeneticBitStringAlgorithm
	instanceVariableNames: 'stringSize '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GA'!


!GeneticBitStringAlgorithm methodsFor: 'accessing'!

stringSize
	"This returns the instance variable"
	^stringSize.!

stringSize: anInteger
	"This returns the instance variable"
	^stringSize := anInteger.! !

!GeneticBitStringAlgorithm methodsFor: 'initialization'!

initialize
	"This initializes an instance of GeneticBitStringAlgorithm"
	super initialize.
	self objectClass: #GeneticBitString.
	self stringSize: 50.
	self population sizeOfPopulation: 100.! !

!GeneticBitStringAlgorithm methodsFor: 'printing'!

postStoreOn: aStream

	"This is used to store an instance of an experiment on disk for further experimentation"

	aStream nextPutAll: '; stringSize: '.

	self stringSize storeOn: aStream.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticBitStringAlgorithm class
	instanceVariableNames: ''!


!GeneticBitStringAlgorithm class methodsFor: 'examples'!

example1
	"GeneticBitStringAlgorithm example1"
	| aGeneticBitStringAlgorithm |
	aGeneticBitStringAlgorithm := self new.
	aGeneticBitStringAlgorithm initializePopulation.
	aGeneticBitStringAlgorithm inspect.!

example2
	"GeneticBitStringAlgorithm example2"
	| aGeneticAlgorithm |
	aGeneticAlgorithm := self new.
	aGeneticAlgorithm objectClass: #GeneticAllOnesString.
	aGeneticAlgorithm runExperiment.
	aGeneticAlgorithm inspect.! !

GeneticAlgorithm subclass: #GeneticProgramming
	instanceVariableNames: 'functions terminals maxDepth maxInitialDepth '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GA'!


!GeneticProgramming methodsFor: 'accessing'!

functions
	"This returns the instance variable"
	^functions.!

functions: anArray
	"This sets the instance variable"
	^functions := anArray.!

maxDepth
	"This returns the instance variable"
	^maxDepth.!

maxDepth: anInteger
	"This returns the instance variable"
	^maxDepth := anInteger.!

maxInitialDepth
	"This returns the instance variable"
	^maxInitialDepth.!

maxInitialDepth: anInteger
	"This sets the instance variable"
	anInteger <= self maxDepth ifTrue: [ maxInitialDepth := anInteger].
	^maxInitialDepth.!

objectClass: aSymbol

	"This adds behaviour to the super class in order to assign functions and

	terminals for the GeneticProgramming run"

	super objectClass: aSymbol.

	"Now we have to find the functions and terminals and set them up correctly"

	self setupFunctionsBasedOn: self objectClass.

	self setupTerminalsBasedOn: self objectClass.!

terminals
	"This returns the instance variable"
	^terminals.!

terminals: anArray
	"This sets the instance variable"
	^terminals := anArray.! !

!GeneticProgramming methodsFor: 'initialization'!

initialize
	"This method initializes a new GeneticProgramming instance."
	super initialize.
	self pMutation: 0.0.
	self maxDepth: 15.
	self maxInitialDepth: 6.
	self functions: Array new.
	self terminals: Array new.
	self objectClass: #GeneticAnt.
	self env: #JohnMuirTrailEnv.
	self population sizeOfPopulation: 300.! !

!GeneticProgramming methodsFor: 'private'!

setupFunctionsBasedOn: aSymbol

	"This sets up the Genetic Programming experiment functions

	and terminals based upon the object class. We do nothing

	if the class of interest does not exist."

	| classOfInterest behaviourFunctions behaviourArity |

	classOfInterest := Smalltalk at: aSymbol ifAbsent: [^self].

	behaviourFunctions := classOfInterest organization listAtCategoryNamed: self class functionsProtocol.

	(classOfInterest respondsTo: #arity) ifTrue: [

		behaviourArity := classOfInterest arity.

		self functions: (OrderedCollection new: behaviourFunctions size).

		behaviourFunctions with: behaviourArity do: [ :fn :arity |

			self functions add: (Array with: fn with: arity)

		]

	].!

setupTerminalsBasedOn: aSymbol

	"This sets up the Genetic Programming experiment terminals

	and terminals based upon the object class. We do nothing

	if the class of interest does not exist."

	| classOfInterest |

	classOfInterest := Smalltalk at: (self  terminalsClass: aSymbol) ifAbsent: [^self].

	self terminals: (classOfInterest organization listAtCategoryNamed: self class terminalsProtocol).! !

!GeneticProgramming methodsFor: 'naming'!

terminalsClass: aSymbol

	"This is the leaf node class in which all terminals are stored"

	^(aSymbol asString, self class terminalsClass) asSymbol.! !

!GeneticProgramming methodsFor: 'printing'!

postStoreOn: aStream

	"This is used to store an instance of an experiment on disk for further experimentation"

	aStream nextPutAll: '; maxDepth: '.

	self maxDepth storeOn: aStream.

	aStream nextPutAll: '; maxInitialDepth: '.

	self maxInitialDepth storeOn: aStream.

	aStream nextPutAll: '; functions: '.

	self functions storeOn: aStream.

	aStream nextPutAll: '; terminals: '.

	self terminals storeOn: aStream.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticProgramming class
	instanceVariableNames: ''!


!GeneticProgramming class methodsFor: 'examples'!

example1
	"This is my test program"
	"GeneticProgramming example1"
	| aGeneticProgramming |
	aGeneticProgramming := GeneticProgramming new.
	aGeneticProgramming 
			functions: #(#(#f1 2) #(#f2 2) #(f3 3));
			terminals: #(1 2 3).
	aGeneticProgramming initializePopulation.
	^aGeneticProgramming.!

example2
	"This is a test of the crossover operator"
	"GeneticProgramming example2"
	|  aGPrun aGP1 aGP2 array1 array2 |
	aGPrun  := GeneticProgramming example3.
	aGP1 := aGPrun  population members at: 1.
	aGP2 := aGPrun  population members at: 2.
	array1 := Array with: aGP1 with: aGP2.
	array2 := aGP1 crossover: aGP2 Using: aGPrun.
	Transcript show: 'Before: '.
	1 to: 2 do: [ :i | Transcript show: (array1 at: i) object printString, ' '].
	Transcript cr; show: 'After: '.
	1 to: 2 do: [ :i | Transcript show: (array2 at: i) object printString, ' '].
	Transcript cr.
	TreeBrowser openOn: (array1 at: 1) object.
	^Array with: array1 with: array2.!

example3
	"This is an implementation of the Ant Farm problem. In the Ant Farm problem,
	ants learn to follow a trail of food. The trail consists of straights, turns, and has gaps"
	"GeneticProgramming example3"
	|  aGPrun |
	aGPrun  := GeneticProgramming new.
	aGPrun functions: #( #(#ifSensor: 2) #(#progn: 3)); terminals: #(#advance: #turnLeft: #turnRight:); objectClass: #GeneticAnt.

	aGPrun initializePopulation.
	aGPrun storeOn: Transcript.

	Transcript cr; show: 'Yo'; cr.

	^aGPrun.!

example4
	"This is an implementation of the Ant Farm problem. In the Ant Farm problem,
	ants learn to follow a trail of food. The trail consists of straights, turns, and has gaps"
	"GeneticProgramming example4"
	|  aGPrun |
	aGPrun  := GeneticProgramming new.
	aGPrun 

		functions: #( #(#ifSensor: 2) #(#progn: 3)); 

		terminals: #(#advance: #turnLeft: #turnRight:); 

		env: #AntEnvironment;

		objectClass: #GeneticAnt;

		maxEvaluations: 4000.

	aGPrun runExperiment.
	^aGPrun.! !

!GeneticProgramming class methodsFor: 'class constants'!

animationViewerClass
	"This is the name of the class where the animation viewer is stored"

	^ 'AnimationViewer'.!

arityMethod
	"This is the actual name of the arity method"

	^#arity.!

arityProtocol
	"This is the protocol in which the arity method is stored"

	^#arity.!

evalMethod
	"This is the method name for the eval method"

	^#eval:.!

evalProtocol
	"This is the protocol name for the eval method"

	^#evaluation.!

functionsProtocol

	"This is the protocol in which all functions are stored"

	^#behaviour.!

postEvalMethod
	"This is the method name for the postEval method"

	^#postEval:.!

preEvalMethod
	"This is the method name for the preEval method"

	^#preEval:.!

terminalsClass

	"This is the leaf node class in which all terminals are stored"

	^ 'Terminal'.!

terminalsProtocol

	"This is the protocol in which all terminals are stored"

	^#behaviour.! !

!GeneticProgramming class methodsFor: 'naming'!

terminalsClass: aSymbol
	"This is the leaf node class in which all terminals are stored"

	^(aSymbol asString, self terminalsClass) asSymbol.! !

Model subclass: #GeneticStatistics
	instanceVariableNames: 'statistics '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GA'!
GeneticStatistics comment:
'I am an object that stores statistics as they are computed for a particular Genetic Algorithm run.
The variables that I store are defined in the statisticsStored class method.

Name			Meaning
====			=======
statistics		A Dictionary containing all of the measures computed for the run.
				The value corresponding to each key will be an OrderedCollection with
				each entry representing a single output point.'!


!GeneticStatistics methodsFor: 'searching'!

at: aSymbol
	"This retrieves the information stored for the requested statistic"
	^self statistics at: aSymbol ifAbsent: [OrderedCollection new].!

at: aSymbol entry: anInteger
	"This retrieves the information stored for the requested statistic at 
	a given data point"
	| aStatistic |
	aStatistic := self at: aSymbol.
	aStatistic isNil ifFalse: [
		aStatistic size > anInteger ifFalse: [
			^aStatistic at: anInteger.
		]
	].
	^nil.! !

!GeneticStatistics methodsFor: 'updating'!

addTo: aSymbol value: anObject
	"This adds an entry to the data base for the recorded statistic. All
	dependents are told of the update to the stored values"
	| storedValues |
	storedValues := self statistics at: aSymbol ifAbsent: [OrderedCollection new].
	storedValues add: anObject.
	self statistics at: aSymbol put: storedValues.
	self changed: aSymbol with: anObject.!

addToStoredStatistics: aGeneticAlgorithm
	"This updates all information from the GeneticAlgorithm that is to be
	stored in the data base. The class method statisticsStored returns
	an Array of method names that are used to compute values to be
	stored in the statistics data base."

	self class statisticsStored do: [:aSymbol |
		self addTo: aSymbol value: (aGeneticAlgorithm perform: aSymbol)
	]!

update: aSymbol with: anObject from: aSender
	"This is invoked whenever output occurs from the GeneticAlgorithm"
	aSymbol == #output ifTrue: [
		self addToStoredStatistics: aSender.
		aSender continue.
	].! !

!GeneticStatistics methodsFor: 'accessing'!

statistics
	"This returns the instance variable"
	^statistics.!

statistics: aDictionary
	"This sets the instance variable"
	^statistics := aDictionary.! !

!GeneticStatistics methodsFor: 'initialization'!

initialize
	"This initializes a newly-created instance"
	self statistics: (Dictionary new).! !

!GeneticStatistics methodsFor: 'protocol'!

continue
	"This does nothing. It's just here in case somebody tries to send the GeneticStatistics
	object a continue message -- thinking that it's similiar to a GeneticAlgorithm object
	in terms of protocol."! !

!GeneticStatistics methodsFor: 'inspecting'!

inspect
	"This allows the GeneticStatistics object to be inspected"
	self statistics inspect.! !

!GeneticStatistics methodsFor: 'printing'!

representBinaryOn: binWriter
	"Save only the dictionary, nothing else. If we try and save the
	whole thing it means that we'll include the dependencies of the
	object which includes GraphicsHandle instances."
	^self statistics representBinaryOn: binWriter.!

storeOn: aStream
	"This stores an ASCII representation of itself on a stream"
	aStream nextPutAll: '(('.
	aStream nextPutAll: self class name.
	aStream nextPutAll: ' new) statistics: '.
	self statistics storeOn: aStream.
	aStream nextPutAll: '; yourself)'.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticStatistics class
	instanceVariableNames: ''!


!GeneticStatistics class methodsFor: 'instance creation'!

new
	"This creates and initializes an instance"
	^super new initialize.! !

!GeneticStatistics class methodsFor: 'class constants'!

statisticsStored
	"This is the set of GeneticAlgorithm method names used for calculation of
	stored statistical values"
	^#(
		#evaluations
		#statisticalFitnessMeasures
		#statisticalRawFitnessMeasures
		#statisticalRawMinMaxFitnessMeasures
		#statisticalDepthMeasures
		#statisticalSizeMeasures
	).! !

Object subclass: #GeneticPopulation
	instanceVariableNames: 'totalFitness sizeOfPopulation members fittestMember fittestMemberIndex statisticalMeasures '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GA'!
GeneticPopulation comment:
'I am a population of genetic objects. Each member of the population represents a 
potential solution to the problem being studied.'!


!GeneticPopulation methodsFor: 'sorting'!

sortPopulation
	"This sorts the population by fitness values, and returns them in an OrderedCollection"
	self members: (SortedCollection withAll: self members sortBlock: [:x :y | x fitness >= y fitness]).
	self members: (self members collect: [:obj | obj]).! !

!GeneticPopulation methodsFor: 'statistics'!

computeFittestMember
	"This method determines the fittest member of the population. This
	method assumes that the fitness of each population member has been
	pre-computed"
	| member |
	self fittestMember: (self members at: 1).
	self fittestMemberIndex: 1.
	2 to: self sizeOfPopulation do: [ :i |
		member := self members at: i.
		member isNil ifFalse: [
			member fitness > self fittestMember fitness ifTrue: [
				self fittestMember: member.
				self fittestMemberIndex: i.
			]
		]
		ifTrue: [^nil].
	].
	^self fittestMember.!

computeTotalFitness
	"This method determines the total population fitness"
	^self members inject: 0.0 into: [:sum :member | sum + member fitness].!

populationFitness: aGeneticAlgorithm
	"This method determines the total fitness of the population. This
	method assumes that the fitness of each population member has been
	pre-computed"

	| member |
	1 to: self sizeOfPopulation do: [ :i |
		member := self members at: i.
		member fitnessEvaluation: aGeneticAlgorithm.
	].!

resetStatistics: aGeneticAlgorithm
	"This resets the statistics associated with the experiment"
	| newStatistics |
	newStatistics := GeneticStatistics new.
	self statisticalMeasures isNil ifFalse: [
		self statisticalMeasures changed: #newStatistics with: newStatistics
	].
	aGeneticAlgorithm isNil ifFalse: [
		self statisticalMeasures isNil ifFalse: [
			aGeneticAlgorithm removeDependent: self statisticalMeasures
		]
	].
	self statisticalMeasures: newStatistics.
	aGeneticAlgorithm isNil ifFalse: [
		aGeneticAlgorithm addDependent: self statisticalMeasures
	].! !

!GeneticPopulation methodsFor: 'accessing'!

fittestMember
	"This returns the instance variable"
	^fittestMember isNil ifTrue: [self computeFittestMember] ifFalse: [fittestMember].!

fittestMember: aGeneticObject
	"This sets the instance variable"
	^fittestMember := aGeneticObject.!

fittestMemberIndex
	"This returns the instance variable"
	^fittestMemberIndex.!

fittestMemberIndex: anInteger
	"This sets the instance variable"
	^fittestMemberIndex := anInteger!

members
	"This returns the instance variable"
	^members.!

members: anArrayOfGeneticObjects
	"This sets the instance variable"
	sizeOfPopulation := anArrayOfGeneticObjects size.
	^members := anArrayOfGeneticObjects.!

sizeOfPopulation
	"This returns the instance variable"
	^sizeOfPopulation.!

sizeOfPopulation: anInteger
	"This sets the instance variable"
	anInteger = sizeOfPopulation ifFalse: [members := Array new: anInteger].
	^sizeOfPopulation := anInteger.!

statisticalMeasures
	"This returns the instance variable"
	^statisticalMeasures.!

statisticalMeasures: aGeneticStatistics
	"This sets the instance variable"
	^statisticalMeasures := aGeneticStatistics.!

totalFitness
	"This returns the instance variable"
	^totalFitness.!

totalFitness: anInteger
	"This sets the instance variable"
	^totalFitness := anInteger.! !

!GeneticPopulation methodsFor: 'initialization'!

initialize
	"This method initializes new instances of a GeneticPopulation.
	Note sizeOfPopulation is initialized in the members: method."

	self totalFitness: 0.
	self resetStatistics: nil.
	self members: (Array new).!

initializePopulationUsing: aGeneticAlgorithm
	"This initializes the starting population of genetic objects"
	self resetStatistics: aGeneticAlgorithm.
	1 to: self sizeOfPopulation do: [ :i | 
		self members 
			at: i 
			put: (GeneticPopulationMember new: ((Smalltalk at: aGeneticAlgorithm objectClass) generateUsing: aGeneticAlgorithm))
	].! !

!GeneticPopulation methodsFor: 'selection'!

selectPopulationMembers: aGeneticAlgorithm
	"This method randomly selects two population members for genetic manipulation"
	| population random size |
	population := self members.
	random := aGeneticAlgorithm random.
	size := self sizeOfPopulation.
	^Array with: (population at: (random next: size)) with: (population at: (random next: size)).! !

!GeneticPopulation methodsFor: 'private'!

crossoverMembers: anArray Using: aGeneticAlgorithm
	"This causes crossover to occur between population members"
	^aGeneticAlgorithm shouldCrossover 
		ifTrue: [(anArray at: 1) crossover: (anArray at: 2) Using: aGeneticAlgorithm]
		ifFalse: [anArray].!

mutateMembers: anArray Using: aGeneticAlgorithm
	"This causes mutation to occur between population members."

	| obj1 obj2 |

	aGeneticAlgorithm shouldMutate 
		ifTrue: [obj1 := (anArray at: 1) mutateUsing: aGeneticAlgorithm]
		ifFalse: [obj1 := anArray at: 1].
	aGeneticAlgorithm shouldMutate 
		ifTrue: [obj2 := (anArray at: 2) mutateUsing: aGeneticAlgorithm]
		ifFalse: [obj2 := anArray at: 2].
	^Array with: obj1 with: obj2.!

rouletteWheel
	"This computes the roulette wheel for this generation"
	| size rouletteWheel last current total |
	size := self sizeOfPopulation.
	rouletteWheel := Array new: size.
	total := self computeTotalFitness.
	last := 0.0.
	1 to: size do: [ :i |
		current := last + ((members at: i) fitness / total).
		rouletteWheel at: i put: current.
		last := current.
	].
	^rouletteWheel.!

spin: anArray with: aGeneticAlgorithm size: anInteger
	"This returns the member index"
	| last rand f |
	last := 0.0.
	rand := aGeneticAlgorithm next.
	1 to: anInteger do: [:i | 
		f := (anArray at: i).
		(rand < f and: [f >= last]) ifTrue: [^i]	.
		last := f.
	].
	^anInteger.! !

!GeneticPopulation methodsFor: 'evolution'!

evolvePopulationByRouletteWheelUsing: aGeneticAlgorithm
	"This causes updates to occur in the population by using roulette wheel selection"
	| size rouletteWheel newMembers newObjectArray index |
	size := self sizeOfPopulation.
	rouletteWheel := self rouletteWheel.
	newMembers := Array new: size.
	"Create intermediate population, derived using the computed roulette wheel"
	1 to: size do: [:i |
		index := self spin: rouletteWheel with: aGeneticAlgorithm size: size.
		newMembers at: i put: (members at: index) copy.
	].
	self members: newMembers.
	"Create final population by crossover and mutation"
	1 to: size by: 2 do: [:i |
		newObjectArray := self mutateMembers: (
								self crossoverMembers: (
									self selectPopulationMembers: aGeneticAlgorithm) Using: aGeneticAlgorithm) Using: aGeneticAlgorithm.
		newMembers at: i  put: (newObjectArray at: 1).
		newMembers at: (i+1) put: (newObjectArray at: 2).
	].
	"Now swap the old and new populations"
	self members: newMembers.!

evolvePopulationByTournamentUsing: aGeneticAlgorithm
	"This causes updates to occur in the population by using tournament selection"
	|  newCandidate size newObjectArray index |

	newObjectArray := self mutateMembers: (
						self crossoverMembers: (
							self selectPopulationMembers: aGeneticAlgorithm) Using: aGeneticAlgorithm) Using: aGeneticAlgorithm.
	size := self sizeOfPopulation.
	newCandidate := Array with: 1 with: 1.
	"Perform the tournament selection on the population"
	1 to: newObjectArray size do: [ :i |
		1 to: aGeneticAlgorithm trials do: [ :j |
			index := aGeneticAlgorithm random next: size.
			(members at: index) fitness < (members at: (newCandidate at: i)) fitness ifTrue: [newCandidate at: i put: index].
		].
	].
	"Update the actual members now. Just replace the objects in the population"
	1 to: newObjectArray size do: [ :i |
		members at: (newCandidate at: i) put: (newObjectArray at: i).
	].!

evolvePopulationUsing: aGeneticAlgorithm
	"This causes updates to occur in the population by application of
	various GeneticOperators"
	aGeneticAlgorithm sMechanism == #tournament ifTrue: [^self evolvePopulationByTournamentUsing: aGeneticAlgorithm].
	aGeneticAlgorithm sMechanism == #roulette ifTrue: [^self evolvePopulationByRouletteWheelUsing: aGeneticAlgorithm].

	"An illegal value crept into the system, so just give an error"
	self error: 'Illegal selection mechanism'.
	aGeneticAlgorithm converged: true.! !

!GeneticPopulation methodsFor: 'printing'!

storeOn: aStream

	"This stores the parameter components of the algorithm on the stream."

	aStream nextPutAll: '(('.
	aStream nextPutAll: self class name.
	aStream nextPutAll: ' new) members: '.
	self members storeOn: aStream.
	aStream nextPutAll: '; sizeOfPopulation: '.
	self sizeOfPopulation storeOn: aStream.
	aStream nextPutAll: '; statisticalMeasures: '.
	self statisticalMeasures storeOn: aStream.
	aStream nextPutAll: '; yourself)'.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticPopulation class
	instanceVariableNames: ''!


!GeneticPopulation class methodsFor: 'instance creation'!

new
	"This creates and initializes an instance"
	^super new initialize.!

new: anInteger
	"This creates and initializes an instance"
	| aGeneticPopulation |
	aGeneticPopulation := self new.
	aGeneticPopulation sizeOfPopulation: anInteger.! !

Model subclass: #GeneticPopulationMember
	instanceVariableNames: 'fitness rawFitness object fitnessEvaluated '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'GA'!
GeneticPopulationMember comment:
'I am a GeneticObject -- I represent a solution to the particular problem
being studied.'!


!GeneticPopulationMember methodsFor: 'accessing'!

fitness
	"This returns the instance variable"
	^fitness.!

fitness: aFloat
	"This sets the instance variable"
	^(aFloat > 0.0 and: [aFloat <= 1.0]) ifTrue: [fitness := aFloat] ifFalse: [fitness].!

fitnessEvaluated
	"This returns the instance variable"
	^fitnessEvaluated.!

fitnessEvaluated: aBoolean
	"This sets the instance variable"
	^fitnessEvaluated := aBoolean.!

object
	"This returns the instance variable"
	^object.!

object: anObject
	"This returns the instance variable"
	^object := anObject.!

rawFitness
	"This returns the instance variable"
	^rawFitness.!

rawFitness: aFloat
	"This sets the instance variable"
	self fitness: (1.0 / (1.0 + aFloat)).
	^rawFitness := aFloat!

size
	"This is used to compute the size of the object being genetically manipulated."
	^self object size.! !

!GeneticPopulationMember methodsFor: 'initialization'!

initialize
	"This initializes a GeneticPopulationMember instance"
	self fitnessEvaluated: false.
	self fitness: 0.0.! !

!GeneticPopulationMember methodsFor: 'evaluation'!

fitnessEvaluation: aGeneticAlgorithm
	"This computes the fitness of a GeneticPopulationMember if necessary and
	otherwise returns the previously computed value. This (potentially) avoids
	a good deal of unnecessary calculation." 
	| env |
	self fitnessEvaluated ifFalse: [
		env := self object preEval: aGeneticAlgorithm env.
		self rawFitness: (self object eval: env).
		self object postEval: env.
		self fitnessEvaluated: true.
		aGeneticAlgorithm incrementEvaluations.
	].
	^self fitness.! !

!GeneticPopulationMember methodsFor: 'copying'!

copy
	"This generates a new copy of the instance"
	| aGeneticObject |
	aGeneticObject := self class new.
	aGeneticObject rawFitness: self rawFitness.
	aGeneticObject fitnessEvaluated: self fitnessEvaluated.
	aGeneticObject object: self object copy.
	^aGeneticObject.! !

!GeneticPopulationMember methodsFor: 'evolution'!

crossover: aGeneticPopulationMember Using: aGeneticAlgorithm
	"This performs the crossover of two GeneticPopulationMembers"
	| newObjects |
	newObjects := self object crossover: aGeneticPopulationMember object Using: aGeneticAlgorithm.
	^Array with: (GeneticPopulationMember new: (newObjects at: 1)) with: (GeneticPopulationMember new: (newObjects at: 2)).!

mutateUsing: aGeneticAlgorithm
	"This performs the mutation of a GeneticPopulationMember"
	^GeneticPopulationMember new: (self object mutateUsing: aGeneticAlgorithm).! !

!GeneticPopulationMember methodsFor: 'printing'!

printOn: aStream
	"Create a printable representation of the object and print it on
	the stream"

	self object class printOn: aStream.
	self fitnessEvaluated ifTrue: [
		aStream nextPutAll: ' of fitness: '.
		self rawFitness printOn: aStream.
	]
	ifFalse: [
		aStream nextPutAll: ' of unknown fitness.'.
	].!

storeOn: aStream
	"This stores the parameter components of the algorithm on the stream. The postStore: method 
	in the subclass stakes care of the subclass-specific information."

	aStream nextPutAll: '(('.
	aStream nextPutAll: self class name.
	aStream nextPutAll: ' new) object: '.
	self object storeOn: aStream.
	aStream nextPutAll: '; fitnessEvaluated: '.
	self fitnessEvaluated storeOn: aStream.
	self fitnessEvaluated ifTrue: [
		aStream nextPutAll: '; fitness: '.
		self fitness storeOn: aStream.
		aStream nextPutAll: '; rawFitness: '.
		self rawFitness storeOn: aStream.
	].

	aStream nextPutAll: '; yourself)'.! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

GeneticPopulationMember class
	instanceVariableNames: ''!


!GeneticPopulationMember class methodsFor: 'instance creation'!

new
	"This creates and initializes an instance"
	^super new initialize.!

new: anObject
	"This creates and initializes an instance"
	| aGeneticObject |
	aGeneticObject := self new.
	aGeneticObject object: anObject.
	^aGeneticObject.! !
