#include "geppetto.h"
#include "proto.h"

#define NUMBER_OF_FITNESS_CASES	16384

static int var_j;
static int randNum[NUMBER_OF_FITNESS_CASES];

#define ErrorDivideByZero	ErrorUserDefined+0
const char *MsgDivideByZero =	"Divide by Zero";
#define ErrorIllegalShift	ErrorUserDefined+1
const char *MsgIllegalShift =	"Illegal Shift";

static objectList *randTerminals P((NOARGS));
static result *opAdd P((int, const object **, void *));
static result *opSubtract P((int, const object **, void *));
static result *opMultiply P((int, const object **, void *));
static result *opDivide P((int, const object **, void *));
static result *opModulus P((int, const object **, void *));
static result *opShiftLeft P((int, const object **, void *));
static result *opShiftRight P((int, const object **, void *));
static objectList *randFunctions P((NOARGS));
static void *randCaseInitialize P((int, int));
static void randCaseFitness P((result *, int, int *, double *, double *,
			       void *));
static int randTerminateRun P((int, int, double, double));

static objectList *
randTerminals()
{
  objectList *list;
  variable *vp;
  constantSrc *csp;

  list = objectListCreate(2);
  if (list) {

    vp = variableCreate(dtInteger, "J", &var_j);
    if (!vp || objectListAdd(list, vp)) {
      variableFree(vp);
      objectListFree(list);
      return(0);
    }

    csp = integerSrcCreate(0, 0);
    if (!csp || objectListAdd(list, csp)) {
      constantSrcFree(csp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static result *
opAdd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* return the result */
  resultSetInteger(r0, resultInteger(r0) + resultInteger(r1));
  resultFree(r1);
  return(r0);
}

static result *
opSubtract(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* return the result */
  resultSetInteger(r0, resultInteger(r0) - resultInteger(r1));
  resultFree(r1);
  return(r0);
}

static result *
opMultiply(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* return the result */
  resultSetInteger(r0, resultInteger(r0) * resultInteger(r1));
  resultFree(r1);
  return(r0);
}

static result *
opDivide(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* don't allow division by zero */
  if (resultInteger(r1) == 0)
    resultSetError(r0, ErrorDivideByZero);
  else
    resultSetInteger(r0, resultInteger(r0) / resultInteger(r1));

  resultFree(r1);
  return(r0);
}

static result *
opModulus(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* don't allow division by zero */
  if (resultInteger(r1) == 0)
    resultSetError(r0, ErrorDivideByZero);
  else
    resultSetInteger(r0, resultInteger(r0) % resultInteger(r1));

  resultFree(r1);
  return(r0);
}

static result *
opShiftLeft(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* don't allow division by zero */
  if (resultInteger(r1) < 0)
    resultSetError(r0, ErrorIllegalShift);
  else
    resultSetInteger(r0, resultInteger(r0) << resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opShiftRight(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* don't allow division by zero */
  if (resultInteger(r1) < 0)
    resultSetError(r0, ErrorIllegalShift);
  else
    resultSetInteger(r0, resultInteger(r0) >> resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static objectList *
randFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(7);
  if (list) {

    osp = complexOperatorSrcCreate("+", opAdd, dtInteger, 2, 2, dtInteger,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("-", opSubtract, dtInteger, 2, 2, dtInteger,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("*", opMultiply, dtInteger, 2, 2, dtInteger,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("/", opDivide, dtInteger, 2, 2, dtInteger,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("%", opModulus, dtInteger, 2, 2, dtInteger,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("<<", opShiftLeft, dtInteger, 2, 2,
				   dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate(">>", opShiftRight, dtInteger, 2, 2,
				   dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

  }

  return(list);
}

static void *
randCaseInitialize(popNum, fc)
int popNum;
int fc;
{
  var_j = fc;
  return(0);
}

static void
randCaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  int diff;

#ifdef DEBUG_RAND_OPS
  {
    charString *cstr;

    cstr = charStringCreate();
    charStringSet(cstr, "Result is ");
    resultToString(rp, cstr);
    charStringPrint(cstr);
    charStringFree(cstr);
  }
#endif /* DEBUG_RAND_OPS */

  /* if there's an error... */
  if (resultIsError(rp)) {

    *rawp += NUMBER_OF_FITNESS_CASES;
  } else {

    /* get absolute value of difference */
    diff = resultInteger(rp) - randNum[fc];
    if (diff < 0)
      diff = -(diff);

    /* got a hit they're equal */
    if (diff == 0)
      (*hitp)++;

    /* add difference to raw fitness */
    *rawp += diff;
  }

  /* set standardized  fitness after final case */
  *stdp = NUMBER_OF_FITNESS_CASES - *rawp;
}

static int
randTerminateRun(popNum, hits, raw, std)
int popNum;
int hits;
double raw;
double std;
{
  return(hits == NUMBER_OF_FITNESS_CASES);
}

void
appInitialize(gp, pop, popNum)
void *gp;
population *pop;
int popNum;
{
  int i;

  /* load the answers */
  for (i = 0; i < NUMBER_OF_FITNESS_CASES; i++) {
    srnd(i);
    randNum[i] = irnd();
  }

  errorCodeSetMsgPtr(ErrorDivideByZero, &MsgDivideByZero);
  errorCodeSetMsgPtr(ErrorIllegalShift, &MsgIllegalShift);

  populationSetFitnessCases(pop, NUMBER_OF_FITNESS_CASES);
  populationSetTerminalList(pop, randTerminals());
  populationSetFunctionList(pop, randFunctions());
  populationSetReturnTypes(pop, dtInteger);

  populationSetCaseInitializeFunc(pop, randCaseInitialize);
  populationSetCaseFitnessFunc(pop, randCaseFitness);
  populationSetTerminateRunFunc(pop, randTerminateRun);
}

#ifdef DEBUG_RAND

const char *proglist[] = {
  "(+ (* J 7) (J >> 1)",
  0
};

#endif /* DEBUG_NEURAL */
