#include "geppetto.h"
#include "proto.h"

#if MULTIPLEX == 11
#define ADDR_BITS	3
#define ADDR_MASK	0x7
#define DATA_BITS	8
#define DATA_MASK	0xff
#define FITNESS_CASES	2048
#else /* MULTIPLEX != 11 */
#if MULTIPLEX == 6
#define ADDR_BITS	2
#define ADDR_MASK	0x3
#define DATA_BITS	4
#define DATA_MASK	0xf
#define FITNESS_CASES	64
#else /* MULTIPLEX != 6 */
#if MULTIPLEX == 3
#define ADDR_BITS	1
#define ADDR_MASK	0x1
#define DATA_BITS	2
#define DATA_MASK	0x3
#define FITNESS_CASES	8
#else /* MULTIPLEX != 3 */
#define ADDR_BITS	0
#define ADDR_MASK	0x0
#define DATA_BITS	0
#define DATA_MASK	0x0
#define FITNESS_CASES	0
#endif /* MULTIPLEX == 3 */
#endif /* MULTIPLEX == 6 */
#endif /* MULTIPLEX == 11 */

const int fitnessCases = FITNESS_CASES;

static bool addr[ADDR_BITS], data[DATA_BITS];

static objectList *multiplexTerminals P((NOARGS));
static result *opNot P((int, const object **, void *));
static result *opAnd P((int, const object **, void *));
static result *opOr P((int, const object **, void *));
static result *opIf P((int, const object **, void *));
static objectList *multiplexFunctions P((NOARGS));
static void *multiplexCaseInitialize P((int, int));
static void multiplexCaseFitness P((result *, int, int *, double *, double *,
				  void *));
static int multiplexTerminateRun P((int, int, double, double));

static objectList *
multiplexTerminals()
{
  objectList *list;
  int i;
  char name[3];
  variable *vp;

  list = objectListCreate(11);
  if (list) {

    /* set up address bits */
    name[0] = 'A';
    name[2] = 0;
    for (i = 0; i < ADDR_BITS; i++) {
      name[1] = '0' + i;

      vp = variableCreate(dtBoolean, name, &addr[i]);
      if (!vp || objectListAdd(list, vp)) {
	variableFree(vp);
	objectListFree(list);
	return(0);
      }
    }

    /* set up data bits */
    name[0] = 'D';
    name[2] = 0;
    for (i = 0; i < DATA_BITS; i++) {
      name[1] = '0' + i;

      vp = variableCreate(dtBoolean, name, &data[i]);
      if (!vp || objectListAdd(list, vp)) {
	variableFree(vp);
	objectListFree(list);
	return(0);
      }
    }
  }

  return(list);

}

static result *
opNot(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp;

  /* evaluate argument */
  rp = objectEval(argv[0], envp);
  if (!resultIsError(rp)) {

    /* make sure we got a boolean result */
    if (!resultIsBoolean(rp))
      resultSetError(rp, ErrorBadDataType);
    else {

      /* negate the result */
      resultSetBoolean(rp, !resultBoolean(rp));
    }
  }

  /* return result */
  return(rp);
}

static result *
opAnd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r1, *r2;

  /* evaluate arguments */
  r1 = objectEval(argv[0], envp);
  if (resultIsError(r1))
    return(r1);
  r2 = objectEval(argv[1], envp);
  if (resultIsError(r2)) {
    resultFree(r1);
    return(r2);
  }

  /* make sure both results are valid */
  if (!resultIsBoolean(r1) || !resultIsBoolean(r2))
    resultSetError(r1, ErrorBadDataType);
  else
    resultSetBoolean(r1, resultBoolean(r1) & resultBoolean(r2));

  /* return the result */
  resultFree(r2);
  return(r1);
}

static result *
opOr(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r1, *r2;

  /* evaluate arguments */
  r1 = objectEval(argv[0], envp);
  if (resultIsError(r1))
    return(r1);
  r2 = objectEval(argv[1], envp);
  if (resultIsError(r2)) {
    resultFree(r1);
    return(r2);
  }

  /* make sure both results are valid */
  if (!resultIsBoolean(r1) || !resultIsBoolean(r2))
    resultSetError(r1, ErrorBadDataType);
  else
    resultSetBoolean(r1, resultBoolean(r1) | resultBoolean(r2));

  /* return the result */
  resultFree(r2);
  return(r1);
}

static result *
opIf(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp;
  bool bval;

  /* evaluate condition */
  rp = objectEval(argv[0], envp);
  if (!resultIsError(rp)) {

    /* make sure we got a boolean result */
    if (!resultIsBoolean(rp))
      resultSetError(rp, ErrorBadDataType);
    else {

      /* find out which branch we should take */
      bval = resultBoolean(rp);
      resultFree(rp);

      /* evaluate branch */
      rp = objectEval(argv[(bval ? 1 : 2)], envp);
    }
  }

  /* return result */
  return(rp);
}

static objectList *
multiplexFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(4);
  if (list) {

    osp = complexOperatorSrcCreate("NOT", opNot, dtBoolean, 1, 1, dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("AND", opAnd, dtBoolean, 2, 2, dtBoolean,
				   dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("OR", opOr, dtBoolean, 2, 2, dtBoolean,
				   dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("IF", opIf, dtBoolean, 3, 3, dtBoolean,
				   dtBoolean, dtBoolean);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static void *
multiplexCaseInitialize(popNum, fc)
int popNum;
int fc;
{
  int i, n;

  /* set address bits */
  for (i = 0, n = (fc >> DATA_BITS) & ADDR_MASK; i < ADDR_BITS; i++, n >>= 1)
    addr[i] = (n & 0x1) == 1;

  /* set data bits */
  for (i = 0, n = fc & DATA_MASK; i < DATA_BITS; i++, n >>= 1)
    data[i] = (n & 0x1) == 1;

  return(0);
}

static void
multiplexCaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  int a;

#ifdef NOISY
  {
    charString *cstr;

    cstr = charStringCreate();
    charStringSet(cstr, "Result is");
    resultToString(rp, cstr);
    charStringPrint(cstr);
    charStringFree(cstr);
  }
#endif /* NOISY */

  /* the result should be a boolean */
  if (!resultIsBoolean(rp))
    *stdp += 1.0;
  else {

    a = (fc >> DATA_BITS) & ADDR_MASK;

#ifdef DEBUG_MULTIPLEX
    {
      int i;

      printf("Addr: %x, Data: ", a);
      for (i = DATA_BITS-1; i >= 0; i--)
	putc(data[i] ? '1' : '0', stdout);
      printf(", Computed: %d\n", resultBoolean(rp));
    }
#endif /* DEBUG_MULTIPLEX */

    if ((data[a] != 0) == (resultBoolean(rp) != 0))
      *hitp += 1;
    else
      *stdp += 1.0;
  }
}

static int
multiplexTerminateRun(popNum, hits, raw, std)
int popNum;
int hits;
double raw;
double std;
{
  return(hits == fitnessCases);
}

void
appInitialize(gp, pop, popNum)
void *gp;
population *pop;
int popNum;
{
  populationSetFitnessCases(pop, fitnessCases);
  populationSetTerminalList(pop, multiplexTerminals());
  populationSetFunctionList(pop, multiplexFunctions());
  populationSetReturnTypes(pop, dtBoolean);
  populationSetCaseInitializeFunc(pop, multiplexCaseInitialize);
  populationSetCaseFitnessFunc(pop, multiplexCaseFitness);
  populationSetTerminateRunFunc(pop, multiplexTerminateRun);
}

#ifdef DEBUG_MULTIPLEX

const char *proglist[] = {
#if MULTIPLEX == 11
  "(IF A2 (IF A1 (IF A0 D7 D6) (IF A0 D5 D4)) (IF A1 (IF A0 D3 D2) (IF A0 D1 D0)))",
#else /* MULTIPLEX != 11 */
#if MULTIPLEX == 3
  "(IF A0 D1 D0)",
#else /* MULTIPLEX != 3 */
#if MULTIPLEX == 6
  "(IF A1 (IF A0 D3 D2) (IF A0 D1 D0))",
#else /* MULTIPLEX != 6 */
  "(Invalid MULTIPLEX value)",
#endif /* MULTIPLEX == 6 */
#endif /* MULTIPLEX == 3 */
#endif /* MULTIPLEX == 11 */
  0
};

#endif /* DEBUG_MULTIPLEX */
