/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include <math.h>
#include "pmrand.h"
#include "program.h"
#include "ckptio.h"
#include "population.h"
#include "proto.h"

#ifdef DEBUG_POPULATION
#define DEBUG_POP_CODE
#endif /* DEBUG_POPULATION */

#ifndef HUGE_VAL
#define HUGE_VAL	HUGE
#endif /* !HUGE_VAL */

static program *findCumPct P((program **, int, double));
static program *selectFitProp P((const population *));
static program *selectGreedy P((const population *));
static program *selectTourn P((const population *));

#define MAX_ATTEMPTS	20	/* # of attempts before depth is increased */
#define MAX_CYCLES	5	/* # of depth increases before aborting */

population *
populationCreate(num)
int num;
{
  population *pop;
  int i;

  /* create a new population */
  pop = (population *)malloc(sizeof(population));
  if (pop) {

    /* initialize function pointers */
    pop->caseInit = 0;
    pop->caseTerminate = 0;
    pop->caseFitness = 0;
    pop->evalCleanup = 0;
    pop->terminateRun = 0;
    pop->destructor = 0;

    /* initialize manditory variables */
    pop->cases = 0;
    pop->tList = 0;
    pop->fList = 0;

    /* initialize optional variables */
    pop->alwaysEval = 0;
    pop->initialDepth = 6;
    pop->maxLoops = 1000;
    pop->maxMutateDepth = 4;
    pop->maxTotalDepth = 17;
    pop->overselectionPct = -1.0;
    pop->selectionMethod = psmFitnessProportionate;
    pop->parsimony = 0.0;
    pop->size = 500;
    pop->creationMethod = pcmRamped;
    pop->returnTypes = dtAll;
    pop->tournamentRounds = 3;

    /* initialize internal variables */
    pop->environment = 0;

    /* initialize breed percentages */
    for (i = 0; i < pbtMaxProgramBreedFunctions; i++)
      pop->percentage[i] = 0.0;
    pop->percentage[pbtReproduction] = 0.095;
    pop->percentage[pbtXoverInternal] = 0.695;
    pop->percentage[pbtXoverAny] = 0.195;

    /* initialize program info */
    pop->programs = pop->workspace = 0;
    pop->bestOfRun = 0;
    pop->bestGeneration = 0;

  }

  return(pop);
}

population *
populationCopy(pop)
const population *pop;
{
  population *npop;

  npop = (population *)malloc(sizeof(population));
  if (npop) {
    memcpy(npop, pop, sizeof(population));
    npop->programs = npop->workspace = 0;
    npop->bestOfRun = 0;
  }

  return(npop);
}

void
populationGenesis(pop)
population *pop;
{
  unsigned progSize;
  program **pSpace, **wSpace;
  program *pp;
  int i, j, attempts, cycles;
  int minDepth, depth;
  double pctTotal;

  /* make sure percentages add up to 1.0 */
  for (i = 0, pctTotal = 0.0; i < pbtMaxProgramBreedFunctions; i++)
    pctTotal += pop->percentage[i];
  if (pctTotal > 1.0) {
    fprintf(stderr, "Breeding method percentages exceed 100%%!\n");
    exit(1);
  } else if (pctTotal < 1.0) {

    /* try to assign remainder to last empty function */
    for (i = pbtMaxProgramBreedFunctions - 1; i >= 0; i--)
      if (pop->percentage[i] == 0.0) {
	pop->percentage[i] = 1.0 - pctTotal;
	pctTotal = 0.0;
	break;
      }

    /* assign remainder to last function */
    if (pctTotal > 0.0)
      pop->percentage[pbtMaxProgramBreedFunctions-1] = pctTotal;
  }

  /* set up greedy overselection percentage if it's uninitialized */
  if ((pop->selectionMethod == psmGreedyOverselection) &&
      (pop->overselectionPct < 0)) {
    pop->overselectionPct = .32;
    i = 1000;
    while (pop->size > i) {
      i <<= 1;
      pop->overselectionPct /= 2;
    }
  }

  /* cruise if we've already got some programs */
  if (pop->programs)
    return;

  /* allocate space for programs */
  progSize = sizeof(program *)*pop->size;
  pSpace = (program **)malloc(progSize);
  if (pSpace)
    wSpace = (program **)malloc(progSize);
  else
    wSpace = 0;

  if (wSpace) {

    /* initialize new program set */
    pop->programs = pSpace;
    pop->workspace = wSpace;
    pop->bestGeneration = 0;
    pop->bestOfRun = NULL;

    /* create all the programs */
    minDepth = 2;
    for (i = 0; i < pop->size; i++) {
      cycles = attempts = 0;

      do {

	/* create next program */
	switch (pop->creationMethod) {
	case pcmFull:
	case pcmGrow:
	  pp = programCreate(pop->tList, pop->fList, pop->returnTypes,
			     pop->initialDepth, pop->creationMethod);
	  break;
	case pcmRamped:
	  if (minDepth >= pop->initialDepth)
	    depth = minDepth;
	  else
	    depth = irndrng(minDepth, pop->initialDepth);
	  pp = programCreate(pop->tList, pop->fList, pop->returnTypes, depth,
			     (irnd() & 1 ? pcmFull : pcmGrow));
	  break;
	default:
	  /* bad creation method ... abort */
	  return;
	}

	/* if we created a program */
	if (pp) {

	  /* look for a duplicate */
	  for (j = 0; j < i; j++)
	    if (programCompare(pp, pop->programs[j])) {
	      programFree(pp);
	      pp = 0;
	      break;
	    }
	}

	/* see if we should increase the minimum depth */
	if (pp == 0) {
	  attempts++;
	  if (attempts >= MAX_ATTEMPTS) {
	    cycles++;
	    minDepth++;
	    if (minDepth > pop->initialDepth)
	      pop->initialDepth = minDepth;
	    attempts = 0;
	  }
	}
      } while ((pp == 0) && (cycles < MAX_CYCLES));

      /* quit if there's a problem */
      if (!pp) {
/*
Couldn't create program NNNNN after NNN attempts!  There's probably a
problem with argument types.  Please check the objects in the function
and terminal list.
*/
	fprintf(stderr, "Couldn't create program %d after %d attempts!",
		i, MAX_ATTEMPTS * MAX_CYCLES);
	fprintf(stderr, "  There's probably a\nproblem with argument types.");
	fprintf(stderr, "  Please check the objects in the function\n");
	fprintf(stderr, "and terminal list.\n");
	exit(1);
      }

      /* add new program */
      pop->programs[i] = pp;
    }
  }
}

static program *
findCumPct(ppp, plen, val)
program **ppp;
int plen;
double val;
{
  int len, base, mid;
  double pval;

  /* do a binary search to find a particular cumulative percentage */
  base = 0;
  for (len = plen; len != 0; len >>= 1) {
    mid = base + (len >> 1);

    /* if we found a match, we're done */
    pval = programCumulativePercentage(ppp[mid]);
    if (val == pval)
      return(ppp[mid]);

    /* move base to the right */
    if (val > pval) {
      base = mid + 1;
      len--;
    }
  }

  /* return the correct program */
  if (base >= 0 && base < plen)
    return(ppp[base]);

  /* didn't find it */
  return(NULL);
}

static program *
selectFitProp(pop)
const population *pop;
{
  return(findCumPct(pop->programs, pop->size, drnd()));
}

static program *
selectGreedy(pop)
const population *pop;
{
  double r;

  /* choose group to select from */
  if (drnd() < 0.8)
    r = drnd();
  else
    r = 1.0 - drndrng(pop->overselectionPct, 1.0);

  /* return program */
  return(findCumPct(pop->programs, pop->size, r));
}

static program *
selectTourn(pop)
const population *pop;
{
  int i, r, n;

  r = irndrng(0, pop->size-1);
  for (i = 1; i < pop->tournamentRounds; i++) {
    n = irndrng(0, pop->size-1);
    if (n < r)
      r = n;
  }

#ifdef DEBUG_POP_CODE
  printf("selectTourn: chose %d\n", r);
#endif /* DEBUG_POP_CODE */
  return(pop->programs[r]);
}

void
populationBreed(pop)
population *pop;
{
  parentSelectionFunction *selectFunc = NULL;
  double r;
  int i, n;
  double pctTotal;
  program *p1, *p2;
  program **tmp;
  programCrossoverType pxo;

  /* get routine to use for parent selection */
  switch (pop->selectionMethod) {
  case psmFitnessProportionate:
    selectFunc = selectFitProp;
    break;
  case psmGreedyOverselection:
    selectFunc = selectGreedy;
    break;
  case psmTournament:
    selectFunc = selectTourn;
    break;
  default:
    fprintf(stderr, "Invalid selection method %d\n", pop->selectionMethod);
    return;
  }

  /* make sure we don't exceed the percentage */
  for (i = 0, pctTotal = 0.0; i < pbtMaxProgramBreedFunctions; i++)
    pctTotal += pop->percentage[i];

  /* generate new programs */
  n = pop->size;
  while (n > 0) {

    /* choose method for breeding next program(s) */
    r = drnd();
    for (i = 0; i < pbtMaxProgramBreedFunctions; i++) {

      r -= pop->percentage[i];
      if (r <= 0)
	switch (i) {
	case pbtReproduction:
	  pop->workspace[--n] = programCopy(selectFunc(pop));
#ifdef DEBUG_POP_CODE
	  printf("Program %d copied\n", n);
#endif /* DEBUG_POP_CODE */
	  i = pbtMaxProgramBreedFunctions;
	  break;
	case pbtXoverInternal:
	case pbtXoverTerminal:
	case pbtXoverAny:
	  if (n > 1) {
	    p1 = programCopy(selectFunc(pop));
	    p2 = programCopy(selectFunc(pop));

	    pxo = (i == pbtXoverInternal ? pxoInternal :
		   (i == pbtXoverTerminal ? pxoTerminal : pxoAny));

	    programCrossover(p1, p2, pxo, pop->maxTotalDepth, pop->returnTypes);

	    programResetFitness(p1);
	    programResetFitness(p2);
#ifdef DEBUG_POP_CODE
	    printf("Programs %d and %d crossed over (%s)\n", n-1, n-2,
		   (i == pbtXoverInternal ? "internal" :
		    (i == pbtXoverTerminal ? "terminal" : "any")));
#endif /* DEBUG_POP_CODE */

	    pop->workspace[--n] = p1;
	    pop->workspace[--n] = p2;
	    i = pbtMaxProgramBreedFunctions;
	  }
	  break;
	case pbtCreation:
	  p1 = 0;
	  switch (pop->creationMethod) {
	  case pcmFull:
	  case pcmGrow:
	    p1 = programCreate(pop->tList, pop->fList, pop->returnTypes,
			       pop->initialDepth, pop->creationMethod);
	    break;
	  case pcmRamped:
	    p1 = programCreate(pop->tList, pop->fList, pop->returnTypes,
			     irndrng(2, pop->initialDepth),
			     (irnd() & 1 ? pcmFull : pcmGrow));
	    break;
	  default:
	    break;
	  }

	  /* if we created a program */
	  if (p1) {
	    programResetFitness(p1);
#ifdef DEBUG_POP_CODE
	    printf("Program %d created\n", n);
#endif /* DEBUG_POP_CODE */
	    pop->workspace[--n] = p1;
	    i = pbtMaxProgramBreedFunctions;
	  }
	  break;
	case pbtHoist:
	  p1 = programHoist(programCopy(selectFunc(pop)), pop->returnTypes);
	  if (p1) {
	    programResetFitness(p1);
#ifdef DEBUG_POP_CODE
	    printf("Program %d hoisted\n", n);
#endif /* DEBUG_POP_CODE */
	    pop->workspace[--n] = p1;
	    i = pbtMaxProgramBreedFunctions;
	  }
	  break;
	case pbtMutation:
	  p1 = programMutate(programCopy(selectFunc(pop)), pop->tList, pop->fList,
			     pop->returnTypes, pop->maxMutateDepth, pop->maxTotalDepth);
	  if (p1) {
	    programResetFitness(p1);
#ifdef DEBUG_POP_CODE
	    printf("Program %d mutated\n", n);
#endif /* DEBUG_POP_CODE */
	    pop->workspace[--n] = p1;
	    i = pbtMaxProgramBreedFunctions;
	  }
	  break;
	default:
	  fprintf(stderr, "Unknown program breed function %s\n",
		  programBreedTypeName(i));
	  break;
	}
    }
  }

  /* switch workspace and program list */
  tmp = pop->workspace;
  pop->workspace = pop->programs;
  pop->programs = tmp;

  /* don't free first program if it's the best of run */
  if (pop->workspace[0] != pop->bestOfRun)
    programFree(pop->workspace[0]);

  /* free old programs */
  for (n = 1; n < pop->size; n++)
    programFree(pop->workspace[n]);
}

int
populationCompare(pop1, pop2, cmpProgs)
const population *pop1, *pop2;
int cmpProgs;
{
  int p;

  /* NULL pointers can match */
  if (!pop1 && !pop2)
    return(1);

  /* fail on cases where one pointer is NULL but the other isn't */
  if (!pop1 || !pop2)
    return(0);

  /* fail if functions differ */
  if ((pop1->caseInit != pop2->caseInit) ||
      (pop1->caseTerminate != pop2->caseTerminate) ||
      (pop1->caseFitness != pop2->caseFitness) ||
      (pop1->evalCleanup != pop2->evalCleanup) ||
      (pop1->terminateRun != pop2->terminateRun) ||
      (pop1->destructor != pop2->destructor))
    return(0);

  /* fail if manditory variables differ */
  if ((pop1->size != pop2->size) || (pop1->cases != pop2->cases) ||
      !objectListCompare(pop1->tList, pop2->tList) ||
      !objectListCompare(pop1->fList, pop2->fList))
    return(0);

  /* fail if optional variables differ */
  if ((pop1->alwaysEval != pop2->alwaysEval) ||
      (pop1->initialDepth != pop2->initialDepth) ||
      (pop1->maxLoops != pop2->maxLoops) ||
      (pop1->maxMutateDepth != pop2->maxMutateDepth) ||
      (pop1->maxTotalDepth != pop2->maxTotalDepth) ||
      (pop1->overselectionPct != pop2->overselectionPct) ||
      (pop1->selectionMethod != pop2->selectionMethod) ||
      (pop1->parsimony != pop2->parsimony) ||
      (pop1->creationMethod != pop2->creationMethod) ||
      (pop1->returnTypes != pop2->returnTypes) ||
      (pop1->tournamentRounds != pop2->tournamentRounds))
    return(0);

  /* fail if percentages differ */
  for (p = 0; p < pbtMaxProgramBreedFunctions; p++)
    if ((pop1->percentage[p] < (pop2->percentage[p] - 0.0001)) ||
	(pop1->percentage[p] > (pop2->percentage[p] + 0.0001)))
      return(0);

  /* if they want to compare programs... */
  if (cmpProgs) {

    /* fail if the best-of-run programs differ */
    if (pop1->bestGeneration != pop2->bestGeneration)
      return(0);
    if (programCompare(pop1->bestOfRun, pop2->bestOfRun) == 0)
      return(0);

    /* compare programs */
    for (p = 0; p < pop1->size; p++)
      if (!programCompare(pop1->programs[p], pop2->programs[p]))
	return(0);
  }

  /* must be identical */
  return(1);
}

void
populationDump(pop, verbose, countUnique)
const population *pop;
int verbose, countUnique;
{
  int i, j;
  const program *pp;
  int hits, hitsTotal = 0, hitsBest = 0, hitsWorst = 0;
  double std, stdTotal = 0.0, stdBest = 0, stdWorst = 0;
  int stdSamples = 0;
  double norm, normTotal = 0.0, normBest = 0, normWorst = 0;
  int depth, depthTotal = 0, deepest = 0, shallowest = 0;
  int complexity = 0;
  int unique = 0;
  const char *bestName;
#ifdef TRACK_BREED_TYPE
  int breedTotal[pbtMaxProgramBreedTypes];
#endif /* TRACK_BREED_TYPE */

#ifdef TRACK_BREED_TYPE
  /* clear array of breed types */
  for (i = 0; i < pbtMaxProgramBreedTypes; i++)
    breedTotal[i] = 0;
#endif /* TRACK_BREED_TYPE */

  for (i = 0; i < pop->size; i++) {
    pp = pop->programs[i];

    /* print program (if desired) */
    if (verbose) {
      printf("%d ", i);
      programDump(pp, 1);
    }

    /* gather hits statistics */
    hits = programHits(pp);
    hitsTotal += hits;
    if (i == 0) {
      hitsBest = hitsWorst = hits;
    } else {
      if (hits > hitsBest)
	hitsBest = hits;
      if (hits < hitsWorst)
	hitsWorst = hits;
    }

    /* gather standardized statistics */
    std = programStandardizedFitness(pp);
    if (stdSamples == 0) {
      stdBest = stdWorst = std;
    } else if (std != HUGE_VAL) {
      if (std < stdBest)
	stdBest = std;
      if (std > stdWorst)
	stdWorst = std;
    }
    if (std != HUGE_VAL) {
      stdTotal += std;
      stdSamples++;
    }

    /* gather normalized statistics */
    norm = programNormalizedFitness(pp);
    normTotal += norm;
    if (i == 0) {
      normBest = normWorst = norm;
    } else {
      if (norm < normBest)
	normBest = norm;
      if (norm > normWorst)
	normWorst = norm;
    }

    /* gather depth statistics */
    depth = programDepth(pp);
    depthTotal += depth;
    if (i == 0) {
      deepest = shallowest = depth;
    } else {
      if (depth > deepest)
	deepest = depth;
      if (depth < shallowest)
	shallowest = depth;
    }

    /* total number of nodes for all programs */
    complexity += programInternalNodes(pp) + programTerminalNodes(pp);

    if (countUnique) {
      /* see if this program is unique */
      for (j = 0; j < i; j++)
	if (programCompare(pp, pop->programs[j]))
	  break;
      if (j >= i)
	unique++;
    }

#ifdef TRACK_BREED_TYPE
    /* track this program's source */
    breedTotal[programBreedSource(pp)]++;
#endif /* TRACK_BREED_TYPE */
  }

  hitsTotal /= pop->size;
  if (stdSamples > 0)
    stdTotal /= stdSamples;
  normTotal /= pop->size;
  depthTotal /= pop->size;
  complexity /= pop->size;

  bestName = pop->bestOfRun == pop->programs[0] ? "Run" : "Generation";

  printf("Hits: Best=%d, Average=%d, Worst=%d\n", hitsBest, hitsTotal,
	 hitsWorst);
  printf("Standardized Fitness: Best=%f, Average=%f, Worst=%f\n", stdBest,
	 stdTotal, stdWorst);
  printf("Normalized Fitness: Best=%f, Average=%f, Worst=%f\n", normBest,
	 normTotal, normWorst);
  printf("Depth: Deepest=%d, Average=%d, Most Shallow=%d\n", deepest,
	 depthTotal, shallowest);
  printf("Structural Complexity: ");
  printf("Best Of %s=%d", bestName, programInternalNodes(pop->programs[0]) +
	 programTerminalNodes(pop->programs[0]));
  printf(", Average=%d\n", complexity);
  if (countUnique)
    printf("%% of Unique Programs: %f\n",
	   (unique * 100.0)/(double )pop->size);
#ifdef TRACK_BREED_TYPE
  printf("Percentage of programs generated by:\n");
  for (i = 0; i < pbtMaxProgramBreedTypes; i++)
    if (breedTotal[i] > 0)
      printf("  %25s %5.2f%% (%d total)\n", programBreedTypeName(i),
	     ((double )breedTotal[i]*100)/pop->size, breedTotal[i]);
#endif /* TRACK_BREED_TYPE */
  printf("Best Of %s Program:\n", bestName);
  programDump(pop->programs[0], 1);
  printf("\n");
}

population *
populationReadCheckpoint(gp, num, popInitFunc, infile)
void *gp;
int num;
initializeFunction *popInitFunc;
FILE *infile;
{
  population *pop;
  charString *cstr;
  const char *cp;
  int rval, i, notEOF;
  unsigned progSize;
  double stdBest;

  /* create an input buffer */
  cstr = charStringCreate();
  if (!cstr)
    return(NULL);

  /* create population at last possible moment */
  pop = populationCreate(num);
  if (!pop) {
    charStringFree(cstr);
    return(NULL);
  }

  /* initialize new population */
  if (popInitFunc)
    (*popInitFunc)(gp, pop, num);

  /* read checkpoint data */
  rval = (readCreationMethod(infile, cstr, &(pop->creationMethod)) ||
	  readInteger(infile, cstr, "initDepth", &(pop->initialDepth)) ||
	  readInteger(infile, cstr, "maxTotDepth", &(pop->maxTotalDepth)) ||
	  readInteger(infile, cstr, "maxMutDepth", &(pop->maxMutateDepth)) ||
	  readSelectionMethod(infile, cstr, &(pop->selectionMethod)));
  if (!rval)
    for (i = 0; !rval && (i < pbtMaxProgramBreedFunctions); i++) {
      rval = readDouble(infile, cstr, programBreedTypeName(i),
			&(pop->percentage[i]));
    }
  if (!rval)
    rval = (readDouble(infile, cstr, "parsimony", &(pop->parsimony)) ||
	    readBoolean(infile, cstr, "alwaysEval", &(pop->alwaysEval)) ||
	    readInteger(infile, cstr, "popSize", &(pop->size)));

  /* get next string */
  if (!rval)
    rval = charStringRead(cstr, infile) || (charStringLength(cstr) == 0);

  /* see if it's the best of run program */
  if (!rval) {

    cp = charStringBuffer(cstr);
    if (strncmp(cp, "bestGeneration ", 15) == 0) {

      /* get best generation number */
      sscanf(cp+15, "%d %lf", &(pop->bestGeneration), &stdBest);

      /* find start of program */
      for (cp += 15; *cp && !isspace(*cp); cp++)
	;
      for ( ; *cp && isspace(*cp); cp++)
	;
      for ( ; *cp && !isspace(*cp); cp++)
	;

      /* parse best-of-run program */
      charStringChop(cstr);
      pop->bestOfRun = programParse(pop->tList, pop->fList, cp);
      if (!pop->bestOfRun)
	rval = 1;
      else {

	/* set standardized fitness for best-of-run program */
	programSetStandardizedFitness(pop->bestOfRun, stdBest);

	/* get next string */
	rval = charStringRead(cstr, infile) || (charStringLength(cstr) == 0);
      }
    }
  }

  /* make sure we're ready to read the programs */
  if (!rval) {
    cp = charStringBuffer(cstr);
    if (strncmp(cp, "programs", 8) != 0)
      rval = 1;
    else {

      /* allocate space for programs */
      progSize = sizeof(program *)*pop->size;
      pop->programs = (program **)malloc(progSize);
      if (!pop->programs) {
	pop->workspace = 0;
	rval = 1;
      } else
	pop->workspace = (program **)malloc(progSize);
    }
  }

  if (!rval) {

    /* parse all the programs */
    notEOF = 1;
    for (i = 0; i < pop->size; i++) {

      /* read next program */
      if (notEOF)
	notEOF = !(charStringRead(cstr, infile) ||
		   charStringLength(cstr) == 0);

      /* parse program */
      if (notEOF) {
	charStringChop(cstr);
	pop->programs[i] = programParse(pop->tList, pop->fList,
					charStringBuffer(cstr));
	if (pop->programs[i] == 0)
	  fprintf(stderr, "Failed to parse \"%s\"\n", charStringBuffer(cstr));
      } else
	pop->programs[i] = 0;
    }
  }

  /* clean up */
  charStringFree(cstr);
  return(pop);
}

int
populationWriteCheckpoint(pop, outfile)
const population *pop;
FILE *outfile;
{
  charString *cstr;
  int rval, i;

  /* create an output buffer */
  cstr = charStringCreate();
  if (!cstr)
    return(-1);

  /* write checkpoint data */
  rval = (writeCreationMethod(outfile, pop->creationMethod) ||
	  writeInteger(outfile, "initDepth", pop->initialDepth) ||
	  writeInteger(outfile, "maxTotDepth", pop->maxTotalDepth) ||
	  writeInteger(outfile, "maxMutDepth", pop->maxMutateDepth) ||
	  writeSelectionMethod(outfile, pop->selectionMethod));
  if (!rval)
    for (i = 0; !rval && (i < pbtMaxProgramBreedFunctions); i++)
      rval = writeDouble(outfile, programBreedTypeName(i), pop->percentage[i]);
  if (!rval)
    rval = (writeDouble(outfile, "parsimony", pop->parsimony) ||
	    writeBoolean(outfile, "alwaysEval", pop->alwaysEval) ||
	    writeInteger(outfile, "popSize", pop->size));

  /* write best of run program (if it exists) */
  if (pop->bestOfRun) {

    fprintf(outfile, "bestGeneration %d %lf ", pop->bestGeneration,
	    programStandardizedFitness(pop->bestOfRun));
    programCheckpoint(pop->bestOfRun, cstr);
    charStringCatenate(cstr, "\n");
    charStringWrite(cstr, outfile);
  }

  /* write programs */
  fprintf(outfile, "programs\n");
  for (i = 0; i < pop->size; i++) {
    charStringClear(cstr);
    programCheckpoint(pop->programs[i], cstr);
    charStringCatenate(cstr, "\n");
    charStringWrite(cstr, outfile);
  } 

  /* clean up */
  charStringFree(cstr);
  return(0);
}

void
populationFree(pop)
population *pop;
{
  int i;

  /* free best-of-run if it's not from this generation */
  if (pop->bestOfRun && (pop->bestOfRun != pop->programs[0]))
    programFree(pop->bestOfRun);

  /* free all programs */
  if (pop->programs) {
    for (i = 0; i < pop->size; i++)
      programFree(pop->programs[i]);
    free(pop->programs);
  }
  if (pop->workspace)
    free(pop->workspace);
  free(pop);
}

#ifdef DEBUG_POPULATION

#include <string.h>
#include "constntsrc.h"
#include "operatrsrc.h"
#include "optrivial.h"

static objectList *tstTerminals P((NOARGS));
static objectList *tstFunctions P((NOARGS));
static void tstInitialize P((void *, population *, int));
static void evalProgram P((program *, double));
static int byStandard P((const void *, const void *));
static int eval P((population *, int));
static void chkGenesis P((population *, programCreationMethod));
static void randomizeBreedPct P((population *));
static void checkCopy P((const population *pop));
static void checkCkpt P((const population *));
int main P((int, char *[]));

/* global values */
int uniqueToggle = 1;

char *progname;

static objectList *
tstTerminals()
{
  objectList *list;
  constantSrc *csp;

  list = objectListCreate(1);
  csp = integerSrcCreate(0, 100);
  if (!csp || objectListAdd(list, csp)) {
    constantSrcFree(csp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static objectList *
tstFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(2);

  osp = simpleOperatorSrcCreate("+", opAdd, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  osp = simpleOperatorSrcCreate("-", opSubtract, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static void
tstInitialize(gp, pop, num)
void *gp;
population *pop;
int num;
{
  populationSetFitnessCases(pop, 1);
  populationSetTerminalList(pop, tstTerminals());
  populationSetFunctionList(pop, tstFunctions());

  populationClearAlwaysEvaluateMode(pop);
  populationSetBreedPercentage(pop, pbtReproduction, 0.095);
  populationSetBreedPercentage(pop, pbtXoverInternal, 0.695);
  populationSetBreedPercentage(pop, pbtXoverTerminal, 0.0);
  populationSetBreedPercentage(pop, pbtXoverAny, 0.195);
  populationSetBreedPercentage(pop, pbtCreation, 0.0);
  populationSetBreedPercentage(pop, pbtHoist, 0.0);
  populationSetBreedPercentage(pop, pbtMutation, 0.0);
  populationSetInitialTreeDepth(pop, 3);
  populationSetMaximumLoops(pop, 1000);
  populationSetMaximumMutationDepth(pop, 6);
  populationSetMaximumTreeDepth(pop, 6);
  populationSetOverselectionPercentage(pop, -1.0);
  populationSetParentSelectionMethod(pop, psmFitnessProportionate);
  populationSetParsimony(pop, 0.0);
  populationSetProgramCreationMethod(pop, pcmRamped);
  populationSetReturnTypes(pop, dtAll);
  populationSetSize(pop, 16);
  populationSetTournamentRounds(pop, 3);
}

static void
evalProgram(pp, parsimony)
program *pp;
double parsimony;
{
  void *envp = 0;
  int hits = 0;
  double raw = 0;
  double std = 0;
  int val;
  result *rp;

  /* make sure we didn't get a bogus program */
  if (pp == 0)
    return;

  /* evaluate program */
  rp = programEval(pp, envp);

  /* evaluate the fitness of the result */
  if (rp) {
    val = resultInteger(rp);
    if (val < 0)
      val = 0;
    hits += val;
    std += 100 - val/10;
    resultFree(rp);
  }

  /* set number of hits for this program */
  programSetHits(pp, hits);

  /* modify standardized fitness by parsimony factor if desired */
  if (parsimony > 0.0)
    std += (programInternalNodes(pp) + programTerminalNodes(pp)) * parsimony;

  /* set final raw and standardized fitness */
  programSetRawFitness(pp, raw);
  programSetStandardizedFitness(pp, std);
}

static int
byStandard(a, b)
const void *a, *b;
{
  const program * const *p1 = (const program * const *)a;
  const program * const *p2 = (const program * const *)b;

  if (programStandardizedFitness(*p1) < programStandardizedFitness(*p2))
    return(-1);
  if (programStandardizedFitness(*p1) > programStandardizedFitness(*p2))
    return(1);
  return(0);
}

static int
eval(pop, genNum)
population *pop;
int genNum;
{
  int i;
  program *pp;
  double adjtot = 0, pct = 0;

  for (i = 0; i < populationSize(pop); i++) {
    pp = populationProgram(pop, i);

    /* if this was reproduced, we don't have to reevaluate it */
    if ((programHits(pp) == 0) && (programStandardizedFitness(pp) == 0.0)) {
      evalProgram(pp, populationParsimony(pop));
    }

    /* compute adjusted fitness */
    programSetAdjustedFitness(pp, 1.0/(1.0 + programStandardizedFitness(pp)));

    /* total all adjusted fitnesses */
    adjtot += programAdjustedFitness(pp);
  }


  /* now go through and compute normalized fitness */
  for (i = 0; i < populationSize(pop); i++) {
    pp = populationProgram(pop, i);
    programSetNormalizedFitness(pp, programAdjustedFitness(pp) / adjtot);
  }

  /* sort by standardized fitness */
  populationSort(pop, byStandard);

  /* set cumulative percentage */
  pct = 0;
  for (i = 0; i < populationSize(pop); i++) {
    pp = populationProgram(pop, i);
    pct += programNormalizedFitness(pp);
    programSetCumulativePercentage(pp, pct);
  }

  /* see if there's a new best-of-run program */
  pp = populationProgram(pop, 0);
  if ((populationBestOfRun(pop) == NULL) ||
      (programStandardizedFitness(pp) <
       programStandardizedFitness(populationBestOfRun(pop)))) {
    populationSetBestOfRun(pop, pp);
    populationSetBestGeneration(pop, genNum);
  }

  /* see if this program is good enough to terminate the run */
  return(programHits(populationBestOfRun(pop)) >= 1000);
}

static void
chkGenesis(pop, method)
population *pop;
programCreationMethod method;
{
  if (pop->programs)
    free(pop->programs);
  if (pop->workspace)
    free(pop->workspace);

  populationSetProgramCreationMethod(pop, method);
  populationGenesis(pop);

  uniqueToggle = !uniqueToggle;
  printf("\nchkGenesis Dump:\n");
  populationDump(pop, 1, uniqueToggle);
}

static void
randomizeBreedPct(pop)
population *pop;
{
  int i;
  double pct[pbtMaxProgramBreedFunctions];
  double pctTotal = 1.0;
  double tmp;

  /* clear out values */
  for (i = 0; i < pbtMaxProgramBreedFunctions; i++)
    pct[i] = 0.0;

  /* generate random values */
  while (pctTotal > 0.0) {

    /* get next percentage */
    tmp = drndrng(0.0, 0.5);
    if (pctTotal > tmp)
      pctTotal -= tmp;
    else {
      tmp = pctTotal;
      pctTotal = 0;
    }

    /* assign to a slot */
    pct[irndrng(0, pbtMaxProgramBreedFunctions-1)] += tmp;
  }

  /* make sure percentages add up to 1.0 */
  pctTotal = 0.0;
  for (i = 0; i < pbtMaxProgramBreedFunctions; i++)
    pctTotal += pct[i];

  /* add in extra bit */
  if (pctTotal != 1.0) {
    i = pbtMaxProgramBreedFunctions;
    while (--i >= 0)
      if (pct[i] != 0) {
	pct[i] += 1.0 - pctTotal;
	break;
      }
  }

  /* assign percentages */
  for (i = 0; i < pbtMaxProgramBreedFunctions; i++)
    populationSetBreedPercentage(pop, i, pct[i]);

  /* print percentages */
  printf("\nPercentages:\n");
  pctTotal = 0.0;
  for (i = 0; i < pbtMaxProgramBreedFunctions; i++) {
    printf("%23s: %f\n", programBreedTypeName(i),
	   populationBreedPercentage(pop, i));
    pctTotal += populationBreedPercentage(pop, i);
  }
  if (pctTotal != 1.0)
    fprintf(stderr, "Error = %f\n", 1.0 - pctTotal);
}

static void
checkCopy(pop)
const population *pop;
{
  population *npop;

  npop = populationCopy(pop);

  /* compare the populations */
  if (!populationCompare(pop, npop, 0)) {
    fprintf(stderr, "%s: Populations didn't match!\n", progname);
    printf("***************** OLD POPULATION *****************\n");
    populationDump(pop, 1, 1);
    printf("***************** NEW POPULATION *****************\n");
    populationDump(npop, 1, 1);
  } else
    printf("Population copy works\n");

  /* get rid of new population */
  populationFree(npop);
}

static void
checkCkpt(pop)
const population *pop;
{
  char filename[L_tmpnam];
  FILE *ckpt;
  int rtnval;
  population *new;

  /* create a temporary file */
  tmpnam(filename);
  ckpt = fopen(filename, "w");
  if (!ckpt) {
    fprintf(stderr, "%s: Couldn't open temporary file!\n", progname);
    return;
  }

  /* write checkpoint info, close file */
  rtnval = populationWriteCheckpoint(pop, ckpt);
  fclose(ckpt);
  if (rtnval) {
    fprintf(stderr, "%s: Problems encountered while writing ckpt file\n",
	    progname);
  } else {

    /* reopen the checkpoint file for reading */
    ckpt = fopen(filename, "r");
    if (!ckpt) {
      fprintf(stderr, "%s: Couldn't reopen checkpoint file!\n", progname);
    } else {

      /* read checkpoint file we just created */
      new = populationReadCheckpoint(NULL, 0, tstInitialize, ckpt);
      if (!new) {
	fprintf(stderr, "%s: Problems encountered while reading ckpt file!\n",
		progname);
	return;
      } else {

	/* compare the populations */
	if (!populationCompare(pop, new, 1)) {
	  fprintf(stderr, "%s: Populations didn't match!\n", progname);
	  printf("***************** OLD POPULATION *****************\n");
	  populationDump(pop, 1, 1);
	  printf("***************** NEW POPULATION *****************\n");
	  populationDump(new, 1, 1);
	} else
	  printf("Population checkpoint works\n");

	/* get rid of new population */
	populationFree(new);
      }

      /* close checkpoint file */
      fclose(ckpt);
    }
  }

  /* get rid of temporary file */
  unlink(filename);
}

int
main(argc, argv)
int argc;
char *argv[];
{
  int randomSeed = 123456789;
  int generations = 6;
  population *pop = 0;
  int g, done;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
/*    dbmallopt(MALLOC_CKCHAIN, &moa); */
  }
#endif

  /* save program name */
  if ((progname = strrchr(argv[0], '/')) == 0)
    progname = argv[0];
  else
    progname++;

  printf("Seed = %d\n", randomSeed);
  srnd(randomSeed);

  /* create population base */
  pop = populationCreate(0);
  tstInitialize(NULL, pop, 0);

  /* check program genesis */
  chkGenesis(pop, pcmFull);
  chkGenesis(pop, pcmGrow);
  chkGenesis(pop, pcmRamped);

  /* evolve population */
  for (done = g = 0; !done && g < generations; g++) {
    if (g) {
      randomizeBreedPct(pop);
      populationBreed(pop);
    }
    done = eval(pop, g);
    printf("\nGeneration %d Summary:\n", g);
    populationDump(pop, 1, 1);
  }

  checkCopy(pop);
  checkCkpt(pop);

  populationFree(pop);

#ifdef KEEP_ALLOCATED_MEMORY
  resultFreeStack();
  constantFreeStack();
  constantSrcFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_POPULATION */
