/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include "operator.h"
#include "operatrsrc.h"
#include "proto.h"

operatorSrc *
simpleOperatorSrcCreate(name, proc, args)
const char *name;
opSimpProc *proc;
int args;
{
  operatorSrc *osp;
  int i;

  /* create a new instance of operatorSrc */
  osp = (operatorSrc *)malloc(sizeof(operatorSrc) + sizeof(int)*args);
  if (osp) {
    objectSetType(osp, otSimpleOperatorSrc);
    objectSetDataType(osp, dtAll);
    osp->name = name;
    osp->proc.simp = proc;
    osp->minArgs = args;
    osp->maxArgs = args;
    osp->print = 0;

    osp->argtypes = (datatype *)((char *)osp + sizeof(operatorSrc));
    for (i = 0; i < args; i++)
      osp->argtypes[i] = dtAll;
  }

  return(osp);
}

operatorSrc *
#ifdef __STDC__
complexOperatorSrcCreate(const char *name, opCompProc *proc,
			 datatype returnTypes, int minArgs, int maxArgs, ...)
#else
complexOperatorSrcCreate(va_alist)
va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  const char *name;
  opCompProc *proc;
  datatype returnTypes;
  int minArgs, maxArgs;
#endif
  operatorSrc *osp;
  int i;

#ifdef __STDC__
  va_start(ap, maxArgs);
#else /* !__STDC__ */
  va_start(ap);
  name = va_arg(ap, const char *);
  proc = va_arg(ap, const opCompProc *);
  returnTypes = va_arg(ap, datatype);
  minArgs = va_arg(ap, int);
  maxArgs = va_arg(ap, int);
#endif /* __STDC__ */

  /* create a new instance of operatorSrc */
  osp = (operatorSrc *)malloc(sizeof(operatorSrc) + sizeof(int)*maxArgs);
  if (osp) {
    objectSetType(osp, otComplexOperatorSrc);
    objectSetDataType(osp, returnTypes);
    osp->name = name;
    osp->proc.comp = proc;
    osp->minArgs = minArgs;
    osp->maxArgs = maxArgs;
    osp->print = 0;

    osp->argtypes = (datatype *)((char *)osp + sizeof(operatorSrc));
    for (i = 0; i < maxArgs; i++)
      osp->argtypes[i] = va_arg(ap, datatype);
  }

  va_end(ap);
  return(osp);
}

operator *
operatorSrcCopy(osp)
const operatorSrc *osp;
{
  return(operatorCreate(osp, osp->maxArgs));
}

result *
operatorSrcEval(osp)
const operatorSrc *osp;
{
  return(resultCreate(dtError, ErrorBadDataType));
}

int
operatorSrcCompare(osp1, osp2)
const operatorSrc *osp1, *osp2;
{
  int i;

  /* try to mismatch on easy stuff */
  if ((objectType(osp1) != objectType(osp2)) ||
      (objectDataType(osp1) != objectDataType(osp2)) ||
      (objectType(osp1) == otSimpleOperator &&
       (osp1->proc.simp != osp2->proc.simp)) ||
      (objectType(osp1) == otComplexOperator &&
       (osp1->proc.comp != osp2->proc.comp)) ||
      (osp1->minArgs != osp2->minArgs) || (osp1->maxArgs != osp2->maxArgs) ||
      (strcmp(osp1->name, osp2->name) != 0))
    return(0);

  /* check argument types */
  for (i = 0; i < osp1->maxArgs; i++)
    if (osp1->argtypes[i] != osp2->argtypes[i])
      return(0);

  /* they're the same */
  return(1);
}

const operatorSrc **
operatorSrcNodePtr(ospp, count, nodeNum, internal, typeptr)
const operatorSrc **ospp;
int *count;
int nodeNum, internal;
datatype *typeptr;
{
  /* operatorSrcs are terminal points */
  if (internal)
    return(0);

  /* see if we're the node they want */
  if (((*count)++ == nodeNum) &&
      ((*typeptr == 0) || ((*typeptr & objectDataType(*ospp)) != 0)))
    return(ospp);

  /* we aren't the right node */
  return(0);
}

int
operatorSrcToString(osp, cstr)
const operatorSrc *osp;
charString *cstr;
{
  int rval, i;

  rval = charStringCatenate(cstr, "(");
  if (!rval)
    rval = charStringCatenate(cstr, osp->name);
  if (!rval)
    for (i = 0; !rval && i < osp->maxArgs; i++)
      rval = datatypeToString(osp->argtypes[i], cstr);
  if (!rval)
    rval = charStringCatenate(cstr, " ->");
  if (!rval)
    rval = datatypeToString(objectDataType(osp), cstr);
  if (!rval)
    rval = charStringCatenate(cstr, ")");
  return(rval);
}

void
operatorSrcFree(osp)
operatorSrc *osp;
{
  free(osp);
}

#ifdef DEBUG_OPERATORSRC

int main P((NOARGS));

int
main()
{
  charString *cstr;
  operatorSrc *osp;
  operator *op;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  cstr = charStringCreate();

  osp = simpleOperatorSrcCreate("sample", 0, 2);
  charStringSet(cstr, "operatorSrc is ");
  operatorSrcToString(osp, cstr);
  charStringPrint(cstr);
  operatorSrcFree(osp);

  osp = simpleOperatorSrcCreate("sample", 0, 2);
  op = operatorSrcCopy(osp);
  charStringSet(cstr, "Original is ");
  operatorSrcToString(osp, cstr);
  charStringCatenate(cstr, ", copy is ");
  operatorToString(op, cstr);
  charStringPrint(cstr);
  operatorSrcFree(osp); operatorFree(op);

  osp = complexOperatorSrcCreate("sample", 0, dtMath, 2, 2, dtMath, dtMath);
  charStringSet(cstr, "operatorSrc is ");
  operatorSrcToString(osp, cstr);
  charStringPrint(cstr);
  operatorSrcFree(osp);

  osp = complexOperatorSrcCreate("sample", 0, dtMath, 2, 2, dtMath, dtMath);
  op = operatorSrcCopy(osp);
  charStringSet(cstr, "Original is ");
  operatorSrcToString(osp, cstr);
  charStringCatenate(cstr, ", copy is ");
  operatorToString(op, cstr);
  charStringPrint(cstr);
  operatorSrcFree(osp); operatorFree(op);

  charStringFree(cstr);

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_OPERATORSRC */
