/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include "object.h"
#include "operatrsrc.h"
#include "operator.h"
#include "proto.h"

operator *
operatorCreate(osp, numArgs)
const operatorSrc *osp;
int numArgs;
{
  operator *op;
  int isSimple;
  int alsize, rlsize;

  /* make sure there are a valid number of arguments */
  if ((numArgs < operatorSrcMinArgs(osp)) ||
      (numArgs > operatorSrcMaxArgs(osp)))
    return(0);

  /* if this is a complex operator, comute length of result list */
  isSimple = objectType(osp) == otSimpleOperatorSrc;
  if (isSimple)
    rlsize = sizeof(result *)*numArgs;
  else
    rlsize = 0;

  /* build a new operator */
  alsize = sizeof(object *)*numArgs;
  op = (operator *)malloc(sizeof(operator) + alsize + rlsize);
  if (op) {
    objectSetType(op, (isSimple ? otSimpleOperator : otComplexOperator));
    objectSetDataType(op, objectDataType(osp));
    op->src = osp;
    op->numArgs = 0;
    op->maxArgs = numArgs;
    op->arglist = (object **)((char *)op + sizeof(operator));
    if (isSimple)
      op->reslist = (result **)((char *)op + sizeof(operator) + alsize);
    else
      op->reslist = 0;
  }
  return(op);
}

int
operatorAddArg(op, arg)
operator *op;
object *arg;
{
  int i;

  /* cruise if it's full already */
  if (op->numArgs >= op->maxArgs)
    return(1);

  /* don't add empty arguments */
  if (arg == NULL)
    return(1);

  /* add argument */
  i = op->numArgs++;
  op->arglist[i] = arg;
  return(0);
}

operator *
operatorCopy(op, everything)
const operator *op;
int everything;
{
  operator *nop;
  int i;

  nop = operatorCreate(op->src, operatorSrcMaxArgs(op->src));
  if (everything)
    for (i = 0; i < op->numArgs; i++)
      if (operatorAddArg(nop, objectCopy(op->arglist[i], everything)))
	break;
  return(nop);
}

result *
simpleOperatorEval(op, envp)
const operator *op;
void *envp;
{
  const operatorSrc *osp;
  result **rpp;
  int i, j;
  opSimpProc *proc;
  result *rp;

  /* evaluate all the arguments */
  osp = op->src;
  rpp = op->reslist;
  for (i = 0; i < op->numArgs; i++) {
    rpp[i] = objectEval(op->arglist[i], envp);

    /* check for problems */
    if (!rpp[i] || (objectDataType(rpp[i]) & osp->argtypes[i]) == 0) {

      /* free previous results */
      for (j = 0; j < i; j++)
	resultFree(rpp[j]);

      /* return bad result */
      return(rpp[i]);
    }
  }

  /* evaluate this operator */
  proc = operatorSrcSimpleProc(osp);
  rp = (*proc)((const result **)rpp, envp);

  /* free prior results, return new result */
  for (i = 0; i < op->numArgs; i++)
    resultFree(rpp[i]);
  return(rp);
}

result *
complexOperatorEval(op, envp)
const operator *op;
void *envp;
{
  opCompProc *proc;

  proc = operatorSrcComplexProc(op->src);
  return((*proc)(op->numArgs, (const object **)op->arglist, envp));
}

int
operatorCount(op, internal)
const operator *op;
int internal;
{
  int i, t;

  /* count number of submembers of this operator */
  for (i = t = 0; i < op->numArgs; i++)
    t += objectCount(op->arglist[i], internal);

  /* remember to count operator */
  return(((internal && op->numArgs > 0) || (!internal && op->numArgs == 0)) ?
	 t+1 : t);
}

int
operatorDepth(op, currentDepth)
const operator *op;
int currentDepth;
{
  int i, d, deepest;

  /* count the current node */
  currentDepth++;

  /* take soundings to find deepest argument tree */
  deepest = currentDepth;
  for (i = d = 0; i < op->numArgs; i++) {
    d = objectDepth(op->arglist[i], currentDepth);
    if (d > deepest)
      deepest = d;
  }

  /* return max depth */
  return(deepest);
}

int
operatorCompare(op1, op2)
const operator *op1, *op2;
{
  int i;

  /* try to mismatch on easy stuff */
  if ((objectType(op1) != objectType(op2)) ||
      (objectDataType(op1) != objectDataType(op2)) ||
      (objectCompare((const object *)(op1->src), (const object *)(op2->src)) == 0) ||
      (op1->numArgs != op2->numArgs) ||
      (op1->maxArgs != op2->maxArgs))
    return(0);

  /* check argument list */
  for (i = 0; i < op1->numArgs; i++)
    if (objectCompare(op1->arglist[i], op2->arglist[i]) == 0)
      return(0);

  /* they're the same */
  return(1);
}

object **
operatorNodePtr(opp, count, nodeNum, internal, typeptr)
const operator **opp;
int *count;
int nodeNum, internal;
datatype *typeptr;
{
  int i;
  object **obp;

  /* see if this is the internal node they want */
  if ((internal && (*opp)->numArgs > 0) ||
      (!internal && (*opp)->numArgs == 0)) {
    if (((*count)++ >= nodeNum) &&
	((*typeptr == 0) || ((*typeptr & objectDataType(*opp)) != 0)))
      return((object **)opp);
  }

  /* check arguments */
  for (i = 0; i < (*opp)->numArgs; i++) {
    obp = objectNodePtr((const object **)&((*opp)->arglist[i]), count,
			nodeNum, internal, typeptr);

    /* if we found it... */
    if (obp && *obp) {

      /* return mask of all valid types for this argument */
      if (operatorSource(*opp) && (*obp == (*opp)->arglist[i]))
	*typeptr = operatorSrcArgType(operatorSource(*opp), i);

      /* return the argument */
      return(obp);
    }
  }

  /* didn't find the right node */
  return(0);
}

int
operatorToLispString(op, checkpoint, cstr)
const operator *op;
int checkpoint;
charString *cstr;
{
  int rval = 1;
  int i;

  rval = charStringCatenate(cstr, "(");
  if (!rval) {
    rval = charStringCatenate(cstr, operatorSrcName(op->src));
    if (!rval) {
      for (i = 0; !rval && i < op->numArgs; i++) {
	rval = charStringCatenate(cstr, " ");
	if (!rval) {
	  if (checkpoint)
	    rval = objectCheckpoint(op->arglist[i], cstr);
	  else
	    rval = objectToString(op->arglist[i], cstr);
	}
      }
    }
  }
  if (!rval)
    rval = charStringCatenate(cstr, ")");
  return(rval);
}

void
operatorFree(op)
operator *op;
{
  int i;

  for (i = 0; i < op->numArgs; i++)
    objectFree(op->arglist[i]);
  free(op);
}

#ifdef DEBUG_OPERATOR

#include "constant.h"
#include "optrivial.h"

static operator *complexOperator P((operatorSrc *, int));
static void checkNodeCode P((operatorSrc *));
int main P((NOARGS));

static int constValue = 0;

static operator *
complexOperator(osp, depth)
operatorSrc *osp;
int depth;
{
  operator *op;
  int i;

  op = operatorCreate(osp, operatorSrcMinArgs(osp));
  depth--;
  for (i = 0; i < operatorSrcMinArgs(osp); i++)
    if (depth <= 1)
      operatorAddArg(op, (object *)integerCreate(constValue++));
    else
      operatorAddArg(op, (object *)complexOperator(osp, depth));
  return(op);
}

static void
checkNodeCode(osp)
operatorSrc *osp;
{
  charString *cstr;
  operator *op;
  int nI, nT, nMax, i;
  char buffer[128];
  object **node;
  int count;
  datatype rtntype = 0;

  /* build a 3-deep tree */
  op = complexOperator(osp, 3);

  /* get totals for node types */
  nI = operatorCount(op, 1);
  nT = operatorCount(op, 0);
  nMax = nI > nT ? nI : nT;

  /* dump initial results */
  cstr = charStringCreate();
  charStringSet(cstr, "Operator is ");
  operatorToString(op, cstr);
  sprintf(buffer, " (%d nodes deep, %d internal, %d terminal nodes)",
	  operatorDepth(op, 0), nI, nT);
  charStringCatenate(cstr, buffer);
  charStringPrint(cstr);

  /* print all nodes */
  for (i = 0; i < nMax; i++) {
    count = 0;
    node = objectNodePtr((const object **)&op, &count, i, 0, &rtntype);
    sprintf(buffer, "Terminal node %d is ", i);
    charStringSet(cstr, buffer);
    if (node && *node)
      objectToString(*node, cstr);
    charStringCatenate(cstr, ", Valid type(s):");
    datatypeToString(rtntype, cstr);
    charStringPrint(cstr);

    count = 0;
    node = objectNodePtr((const object **)&op, &count, i, 1, &rtntype);
    sprintf(buffer, "Internal node %d is ", i);
    charStringSet(cstr, buffer);
    if (node && *node)
      objectToString(*node, cstr);
    charStringCatenate(cstr, ", Valid type(s):");
    datatypeToString(rtntype, cstr);
    charStringPrint(cstr);
  }
    
  operatorFree(op);
  charStringFree(cstr);
}

int
main()
{
  charString *cstr;
  operatorSrc *osp;
  operator *op, *nop;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  cstr = charStringCreate();

  /* create a complex operator template */
  osp = complexOperatorSrcCreate("+", opComplexAdd, dtInteger, 2, 2,
				 dtInteger, dtInteger);

  /* create an operator */
  op = operatorCreate(osp, operatorSrcMinArgs(osp));
  charStringSet(cstr, "Operator is ");
  operatorToString(op, cstr);
  charStringPrint(cstr);
  operatorFree(op);

  /* create a more complex operator */
  op = complexOperator(osp, 2);
  charStringSet(cstr, "Operator is ");
  operatorToString(op, cstr);
  charStringPrint(cstr);
  operatorFree(op);

  /* copy a complex operator */
  op = complexOperator(osp, 3);
  nop = operatorCopy(op, 1);
  charStringSet(cstr, "Original is ");
  operatorToString(op, cstr);
  charStringCatenate(cstr, ", copy is ");
  operatorToString(nop, cstr);
  if (operatorCompare(op, nop) == 0)
    charStringCatenate(cstr, " (but compare failed)");
  charStringPrint(cstr);
  operatorFree(op);
  operatorFree(nop);

  /* check node counting/returning functions */
  checkNodeCode(osp);

  operatorSrcFree(osp);
  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_OPERATOR */
