/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include "objtypes.h"
#include "genlist.h"
#include "proto.h"

#define NUM_ALIASES	10
static datatype dtalias[NUM_ALIASES+1] = {
  dtError, dtError, dtError, dtError, dtError,
  dtError, dtError, dtError, dtError, dtError
};

static genericList *errorMessage = 0;

int
datatypeMakeAlias(udt, adt)
datatype udt, adt;
{
  switch (udt) {
  case dtUserDef0:
    dtalias[0] = adt;
    return(1);
  case dtUserDef1:
    dtalias[1] = adt;
    return(1);
  case dtUserDef2:
    dtalias[2] = adt;
    return(1);
  case dtUserDef3:
    dtalias[3] = adt;
    return(1);
  case dtUserDef4:
    dtalias[4] = adt;
    return(1);
  case dtUserDef5:
    dtalias[5] = adt;
    return(1);
  case dtUserDef6:
    dtalias[6] = adt;
    return(1);
  case dtUserDef7:
    dtalias[7] = adt;
    return(1);
  case dtUserDef8:
    dtalias[8] = adt;
    return(1);
  case dtUserDef9:
    dtalias[9] = adt;
    return(1);
  case dtVoid:
  case dtBoolean:
  case dtShort:
  case dtInteger:
  case dtLong:
  case dtFloat:
  case dtDouble:
  case dtList:
  case dtBlob:
  case dtError:
    break;
  }
  return(0);
}

datatype
datatypeLookupAlias(dt)
datatype dt;
{
  switch (dt) {
  case dtUserDef0:
    return(dtalias[0]);
  case dtUserDef1:
    return(dtalias[1]);
  case dtUserDef2:
    return(dtalias[2]);
  case dtUserDef3:
    return(dtalias[3]);
  case dtUserDef4:
    return(dtalias[4]);
  case dtUserDef5:
    return(dtalias[5]);
  case dtUserDef6:
    return(dtalias[6]);
  case dtUserDef7:
    return(dtalias[7]);
  case dtUserDef8:
    return(dtalias[8]);
  case dtUserDef9:
    return(dtalias[9]);
  case dtVoid:
  case dtBoolean:
  case dtShort:
  case dtInteger:
  case dtLong:
  case dtFloat:
  case dtDouble:
  case dtList:
  case dtBlob:
  case dtError:
    break;
  }
  return(dtError);
}

int
datatypeToString(dt, cstr)
datatype dt;
charString *cstr;
{
  int rval = 0;
  unsigned bit = 0x0001;
  const char *name = 0;
  int num = 0;
  const char *sepchar;

  while (!rval && (dt & dtAll)) {
    if (dt & bit) {
      switch(bit) {
      case dtVoid:
	name = "Void";
	break;
      case dtBoolean:
	name = "Boolean";
	break;
      case dtShort:
	name = "Short";
	break;
      case dtInteger:
	name = "Integer";
	break;
      case dtLong:
	name = "Long";
	break;
      case dtFloat:
	name = "Float";
	break;
      case dtDouble:
	name = "Double";
	break;
      case dtList:
	name = "List";
	break;
      case dtBlob:
	name = "Blob";
	break;
      case dtUserDef0:
	name = "UserDef0";
	break;
      case dtUserDef1:
	name = "UserDef1";
	break;
      case dtUserDef2:
	name = "UserDef2";
	break;
      case dtUserDef3:
	name = "UserDef3";
	break;
      case dtUserDef4:
	name = "UserDef4";
	break;
      case dtUserDef5:
	name = "UserDef5";
	break;
      case dtUserDef6:
	name = "UserDef6";
	break;
      case dtUserDef7:
	name = "UserDef7";
	break;
      case dtUserDef8:
	name = "UserDef8";
	break;
      case dtUserDef9:
	name = "UserDef9";
	break;
      case dtError:
	name = "Error";
	break;
      }

      /* if we found another type, add it to the string */
      if (name) {
	sepchar = (num++ == 0 ? " " : "|");
	rval = charStringCatenate(cstr, sepchar);
	if (!rval)
	  rval = charStringCatenate(cstr, name);
	name = 0;
      }

      /* turn off this bit */
      dt &= ~bit;
    }

    /* move on to next bit */
    bit <<= 1;
  }

  return(rval);
}

errorCode
errorCodeSetMsgPtr(code, message)
errorCode code;
const char **message;
{
  /* make sure we've got a valid errorCode */
  code -= ErrorUserDefined;
  if (code < 0)
    return(-1);

  /* try to create the list if it doesn't exist */
  if (!errorMessage) {
    errorMessage = genericListCreate(sizeof(const char *), 16);
    if (!errorMessage)
      return(-2);
  }

  /* try to add a new message */
  if (genericListSet(errorMessage, code, message))
    return(-3);

  /* return index of new message */
  return(0);
}

const char *
errorCodeMessage(code)
errorCode code;
{
  /* check static errors first */
  if (code == ErrorBadDataType)
    return("Bad DataType");
  if (code == ErrorNoError)
    return("No Error");

  /* check user-defined errors */
  if (code >= ErrorUserDefined) {
    code -= ErrorUserDefined;
    if (errorMessage && code <= genericListLength(errorMessage))
      return(*(char **)genericListEntry(errorMessage, code));
  }

  /* don't know what this error is */
  return("Unknown Error");
}

void
errorCodeFreeStack()
{
  if (errorMessage)
    genericListFree(errorMessage);
}

#ifdef DEBUG_OBJTYPES

#include <stdio.h>

static void checkObjTypes P((NOARGS));
static void checkErrors P((NOARGS));
int main P((NOARGS));

static void
checkObjTypes()
{
  int errflag = 0;
  charString *str;

  if (!datatypeMakeAlias(dtUserDef0, dtFloat)) {
    fprintf(stderr, "MakeAlias(UserDef0, Float) failed!\n");
    errflag++;
  }
  if (!datatypeMakeAlias(dtUserDef1, dtVoid)) {
    fprintf(stderr, "MakeAlias(UserDef1, Void) failed!\n");
    errflag++;
  }
  if (!datatypeMakeAlias(dtUserDef2, dtInteger)) {
    fprintf(stderr, "MakeAlias(UserDef2, Integer) failed!\n");
    errflag++;
  }
  if (!datatypeMakeAlias(dtUserDef3, dtList)) {
    fprintf(stderr, "MakeAlias(UserDef3, List) failed!\n");
    errflag++;
  }

  if (!datatypeMakeAlias(dtUserDef0, dtBoolean)) {
    fprintf(stderr, "MakeAlias(UserDef0, Boolean) failed!\n");
    errflag++;
  }

  if (datatypeLookupAlias(dtUserDef0) != dtBoolean) {
    fprintf(stderr, "LookupAlias(UserDef0) failed!\n");
    errflag++;
  }
  if (datatypeLookupAlias(dtUserDef1) != dtVoid) {
    fprintf(stderr, "LookupAlias(UserDef1) failed!\n");
    errflag++;
  }
  if (datatypeLookupAlias(dtUserDef2) != dtInteger) {
    fprintf(stderr, "LookupAlias(UserDef2) failed!\n");
    errflag++;
  }
  if (datatypeLookupAlias(dtUserDef3) != dtList) {
    fprintf(stderr, "LookupAlias(UserDef3) failed!\n");
    errflag++;
  }

  str = charStringCreate();
  datatypeToString(dtAll, str);
  charStringPrint(str);
  charStringFree(str);

  if (!errflag)
    printf("datatype aliases work\n");
}

static const char *msg[] = {
	"Error 0", "Error 1", "Error 2", "Error 3", "Error 4", "Error 5",
	"Error 6", "Error 7", "Error 8", "Error 9", "Error 10", "Error 11",
	"Error 12", "Error 13", "Error 14", "Error 15", "Error 16",
};

static void
checkErrors()
{
  int i, j;
  const char *numsg;
  int errflag = 0;

  for (i = 16; i >= 0; i--) {
    errorCodeSetMsgPtr(ErrorUserDefined+i, &msg[i]);
    for (j = 16; j >= i; j--) {
      numsg = errorCodeMessage(ErrorUserDefined+j);
      if (!numsg || strcmp(numsg, msg[j])) {
	fprintf(stderr, "User defined error %d: \"%s\" != \"%s\"\n", j,
		(numsg ? numsg : "<EMPTY>"), msg[i]);
	errflag = 1;
      }
    }
  }

  for (i = 0; i < 17; i++) {
    numsg = errorCodeMessage(ErrorUserDefined+i);
    if (!numsg || strcmp(numsg, msg[i])) {
      fprintf(stderr, "User defined error %d: \"%s\" != \"%s\"\n", i,
	      (numsg ? numsg : "<EMPTY>"), msg[i]);
      errflag = 1;
    } else {
      printf("User defined error %d: \"%s\" == \"%s\"\n", i, numsg, msg[i]);
    }
  }

  if (!errflag)
    printf("errorCode messages work\n");
}

int
main()
{
  checkObjTypes();
  checkErrors();

  exit(0);
}
#endif /* DEBUG_OBJTYPES */
