/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdlib.h>
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include "variable.h"
#include "operatrsrc.h"
#include "operator.h"
#include "objectlist.h"
#include "proto.h"

typedef struct sortkey {
  datatype dtype;
  int num;
} sortkey;

static int byFrequency P((const void *, const void *));
static int byPopularity P((const void *, const void *));

static sortkey *keyList = 0;
static int keyListLen;

int
objectListAdd(olp, op)
objectList *olp;
void *op;
{
  return(genericListAdd(olp, (object **)(&op)));
}

static int
byFrequency(a, b)
const void *a, *b;
{
  const sortkey *p0 = (const sortkey *)a;
  const sortkey *p1 = (const sortkey *)b;

  return(p1->num - p0->num);
}

static int
byPopularity(a, b)
const void *a, *b;
{
  const object * const *p0 = (const object * const *)a;
  const object * const *p1 = (const object * const *)b;
  int i, n0, n1;

  for (i = n0 = 0; i < keyListLen; i++)
    if (keyList[i].dtype == objectDataType(*p0)) {
      n0 = keyList[i].num;
      break;
    }

  for (i = n1 = 0; i < keyListLen; i++)
    if (keyList[i].dtype == objectDataType(*p1)) {
      n1 = keyList[i].num;
      break;
    }

  return(n1 - n0);
}

int
objectListOrganize(olp)
objectList *olp;
{
  int i, j;
  object *ob, **nobl;
  float current, step;
  int first, num;
  int offset, entry;
  int objListLen = objectListLength(olp);
  unsigned objListBytes = objListLen * sizeof(object *);

  /* create a long enough list */
  keyList = (sortkey *)malloc(sizeof(sortkey) * objListLen);
  if (!keyList)
    return(1);

  /* build the list */
  for (i = keyListLen = 0; i < objListLen; i++) {
    ob = objectListEntry(olp, i);

    /* see if we've already got this type */
    for (j = 0; j < keyListLen; j++)
      if (objectDataType(ob) == keyList[j].dtype) {
	keyList[j].num++;
	break;
      }

    /* if we didn't find it, add a new type */
    if (j >= keyListLen) {
      keyList[keyListLen].dtype = objectDataType(ob);
      keyList[keyListLen].num = 1;
      keyListLen++;
    }
  }

  /* if there's only one type or there are no duplicate types, we're done */
  if ((keyListLen == 1) || (keyListLen == objListLen)) {
    free(keyList);
    return(0);
  }

  /* sort the keys */
  qsort((char *)keyList, (unsigned )keyListLen, sizeof(sortkey), byFrequency);

#ifdef DEBUG_OBJECTLIST
  /* print the popularity list */
  for (i = 0; i < keyListLen; i++)
    printf("%04x: %d\n", keyList[i].dtype, keyList[i].num);
#endif /* DEBUG_OBJECTLIST */

  /* sort the objects by popularity */
  qsort((char *)olp->list, (unsigned )objListLen, sizeof(object *),
	byPopularity);

  /* create a new object list and clear it */
  nobl = (object **)malloc(objListBytes);
  if (!nobl) {
    free(keyList);
    return(1);
  }
  bzero(nobl, objListBytes);

  /* reorganize the list */
  first = 0;
  i = j = 0;
  while ((i < objListLen) && (j < keyListLen)) {

    /* find first empty entry */
    while (nobl[first])
      first++;

    /* get current entry and step */
    current = first++;
    step = (float )objListLen / (float )keyList[j].num;

    /* assign all pointers of this type */
    num = keyList[j].num;
#ifdef DEBUG_OBJECTLIST
    printf("Assigning %d elements of type %d\n", keyList[j].num,
	   keyList[j].dtype);
#endif /* DEBUG_OBJECTLIST */
    while (num > 0) {
#ifdef DEBUG_OBJECTLIST
      printf("CURRENT=%f, STEP=%f\n", current, step);
#endif /* DEBUG_OBJECTLIST */

      /* find nearest empty slot */
      offset = 0;
      while (nobl[(int )current]) {
	offset++;

	/* check ahead */
	entry = current + offset;
	if (entry >= objListLen)
	  entry -= objListLen;
	if (nobl[entry] == 0) {
	  current += offset;
	  if (current >= objListLen)
	    current -= objListLen;
	} else {

	  /* check behind */
	  entry = current - offset;
	  if (entry < 0)
	    entry += objListLen;
	  if (nobl[entry] == 0) {
	    current -= offset;
	    if (current < 0)
	      current += objListLen;
	  }
	}
      }

      /* assign object, update counters */
      nobl[(int )current] = objectListEntry(olp, i);
#ifdef DEBUG_OBJECTLIST
      printf("Element %d is datatype %04x\n", (int )current,
	     objectDataType(nobl[(int )current]));
#endif /* DEBUG_OBJECTLIST */
      num--;
      i++;
      current += step;
      if (current > objListLen)
	current -= objListLen;
    }
    j++;
#ifdef DEBUG_OBJECTLIST
    printf("\n");
#endif /* DEBUG_OBJECTLIST */
  }

  /* copy reorganized list */
  bcopy(nobl, olp->list, objListBytes);

  /* free memory and return */
  free(nobl);
  free(keyList);
  return(0);
}

int
objectListCompare(olp1, olp2)
const objectList *olp1, *olp2;
{
  int i;

  /* make sure they're the same length */
  if (objectListLength(olp1) != objectListLength(olp2))
    return(0);

  /* compare objects in each list */
  for (i = 0; i < objectListLength(olp1); i++)
    if (objectCompare(objectListEntry(olp1, i),
		      objectListEntry(olp2, i)) == 0)
      return(0);

  /* must be identical */
  return(1);
}

object *
objectListLookup(olp, s, len)
const objectList *olp;
const char *s;
unsigned len;
{
  int i;
  object *ob;
  const char *name = 0;

  for (i = 0; i < objectListLength(olp); i++) {

    /* get next object */
    ob = objectListEntry(olp, i);

    /* find object name */
    switch (objectType(ob)) {
    case otVariable:
      name = variableName((const variable *)ob);
      break;
    case otSimpleOperatorSrc:
    case otComplexOperatorSrc:
      name = operatorSrcName((const operatorSrc *)ob);
      break;
    case otSimpleOperator:
    case otComplexOperator:
      name = operatorName((const operator *)ob);
      break;
    default:
      break;
    }

    if (name) {
      if ((strncmp(s, name, len) == 0) && (strlen(name) == len))
	return(ob);
    }
  }

  return(0);
}

int
objectListToString(olp, cstr)
const objectList *olp;
charString *cstr;
{
  int rval = 0;
  char buffer[16];
  int i;

  rval = charStringCatenate(cstr, "<");
  if (objectListLength(olp) <= 0) {
    if (!rval)
      rval = charStringCatenate(cstr, "**empty list**");
  } else {
    for (i = 0; !rval && i < objectListLength(olp); i++) {
      sprintf(buffer, "%s%d: ", (i == 0 ? "" : " "), i);
      rval = charStringCatenate(cstr, buffer);
      if (!rval)
	objectToString(objectListEntry(olp, i), cstr);
    }
  }

  if (!rval)
    rval = charStringCatenate(cstr, ">");
  return(rval);
}

void
objectListFree(olp)
objectList *olp;
{
  int i;

  for (i = 0; i < objectListLength(olp); i++)
    objectFree(objectListEntry(olp, i));
  genericListFree(olp);
}

#ifdef DEBUG_OBJECTLIST

#include "pmrand.h"
#include "constant.h"

static void checkOrganization P((charString *));
int main P((NOARGS));

static void
checkOrganization(cstr)
charString *cstr;
{
  objectList *olp;
  int i, count;

  olp = objectListCreate(10);

  count = 0;

  for (i = 0; i < irndrng(0, 20); i++)
    objectListAdd(olp, booleanCreate(count++ & 0x1));

  for (i = 0; i < irndrng(0, 20); i++)
    objectListAdd(olp, shortCreate((short )count++));

  for (i = 0; i < irndrng(0, 20); i++)
    objectListAdd(olp, integerCreate(count++));

  for (i = 0; i < irndrng(0, 20); i++)
    objectListAdd(olp, longCreate((long )count++));

  printf("Before organization (%d elements):\n", objectListLength(olp));
  fflush(stdout);

  charStringClear(cstr);
  objectListToString(olp, cstr);
  charStringPrint(cstr);

  for (i = count = 0; i < objectListLength(olp); i++)
    if (objectListEntry(olp, i) == NULL)
      count++;
  if (count > 0)
    printf("%d NULL entries\n", count);

  objectListOrganize(olp);

  printf("After organization (%d elements):\n", objectListLength(olp));
  fflush(stdout);

  charStringClear(cstr);
  objectListToString(olp, cstr);
  charStringPrint(cstr);

  for (i = count = 0; i < objectListLength(olp); i++)
    if (objectListEntry(olp, i) == NULL) {
      printf("NULL entry at %d\n", i);
      count++;
    }
  if (count > 0)
    printf("%d NULL entries\n", count);

  objectListFree(olp);
}

int
main()
{
  objectList *olp, *nolp;
  charString *cstr;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  olp = objectListCreate(5);
  objectListAdd(olp, integerCreate(34));
  objectListAdd(olp, floatCreate(7.15));

  printf("%d elements in list\n", objectListLength(olp));
  fflush(stdout);
  cstr = charStringCreate();
  objectListToString(olp, cstr);
  charStringPrint(cstr);

  nolp = objectListCopy(olp);
  printf("%d elements in copied list\n", objectListLength(nolp));
  fflush(stdout);

  charStringClear(cstr);
  objectListToString(nolp, cstr);
  charStringPrint(cstr);

  if (objectListCompare(olp, nolp) == 0) {
    printf("objectLists are *NOT* identical!\n");
    fflush(stdout);
  }

  objectListFree(olp);

/*  srnd(time(0) ^ getpid()); */
  checkOrganization(cstr);

  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_OBJECTLIST */
