/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdio.h>
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include <sys/time.h>
#include "../geppetto.h"
#include "program.h"
#include "population.h"
#include "proto.h"

static void argPrint P((const char *));
static global *processArgs P((int, char *[], global *));
int main P((int, char *[]));

char *progname;

/*
 * This is set when a checkpoint file is read
 */
static int firstGen = 0;

static void
argPrint(str)
const char *str;
{
  static int lineLen = 0;
  static int current = 0;
  int len;

  /* find line length */
  if (lineLen == 0) {
    lineLen = 80;
    current = lineLen;
  }

  len = strlen(str);
  current -= len;
  if (current > 0) {
    fputs(str, stderr);
  } else {
    fprintf(stderr, "\n\t%s", str);
    current = lineLen - (8 + len);
  }
}

static global *
processArgs(argc, argv, gp)
int argc;
char *argv[];
global *gp;
{
  int c, errflag = 0;
  extern char *optarg;
  int intval;
  char *cp;
  double dblval;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

#ifdef MASSIVE_CHECKING
    /* check malloc chain every time malloc() is called */
    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
#else
    /* don't validate pointers during calls to mem*,b*,str* routines */
    moa.i = 0;
    dbmallopt(MALLOC_CKDATA, &moa);
#endif
  }
#endif

  /* process all arguments */
  while ((c = getopt(argc, argv, "aAc:d:f:F:g:i:m:o:p:r:s:t:uUvVx:z:")) != EOF)
    switch (c) {
    case 'a':
      globalClearAlwaysEvaluateMode(gp);
      break;
    case 'A':
      globalSetAlwaysEvaluateMode(gp);
      break;
    case 'c':
      switch (*optarg) {
      case 'f':
      case 'F':
	globalSetProgramCreationMethod(gp, pcmFull);
	break;
      case 'g':
      case 'G':
	globalSetProgramCreationMethod(gp, pcmGrow);
	break;
      case 'r':
      case 'R':
	globalSetProgramCreationMethod(gp, pcmRamped);
	break;
      default:
	fprintf(stderr, "%s: Invalid creation method '%s'\n", progname, optarg);
	fprintf(stderr, "\tCreation Method must be one of:\n");
	fprintf(stderr, "\t\tF(ull)\n");
	fprintf(stderr, "\t\tG(row)\n");
	fprintf(stderr, "\t\tR(amped half and half)\n");
	errflag = 1;
	break;
      }
      break;
    case 'd':
      intval = strtol(optarg, 0, 0);
      if (intval <= 0) {
	fprintf(stderr,
		"%s: Maximum total depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetMaximumTreeDepth(gp, intval);
      break;
    case 'f':
      cp = (char *)malloc(strlen(optarg)+1);
      if (!cp) {
	fprintf(stderr, "%s: Couldn't get memory to save file name!\n",
		progname);
	exit(1);
      }
      strcpy(cp, optarg);
      globalSetCheckpointName(gp, cp);
      break;
    case 'F':
      cp = (char *)malloc(strlen(optarg)+1);
      if (!cp) {
	fprintf(stderr, "%s: Couldn't get memory to save file name!\n",
		progname);
	exit(1);
      }
      strcpy(cp, optarg);
      globalFree(gp);
      gp = globalReadCheckpointFile(cp, appInitialize, &firstGen);
      if (!gp) {
	fprintf(stderr,	"%s: Error reading checkpoint file '%s'!\n",
		progname, cp);
	exit(1);
      }
      break;
    case 'g':
      intval = strtol(optarg, 0, 0);
      if (intval <= 0) {
	fprintf(stderr, "%s: Generation '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetGenerations(gp, intval);
      break;
    case 'i':
      intval = strtol(optarg, 0, 0);
      if (intval <= 0) {
	fprintf(stderr, "%s: Initial depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetInitialTreeDepth(gp, intval);
      break;
    case 'm':
      intval = strtol(optarg, 0, 0);
      if (intval <= 0) {
	fprintf(stderr, "%s: Maximum mutation depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetMaximumMutationDepth(gp, intval);
      break;
    case 'o':
      dblval = strtod(optarg, 0);
      if (dblval < 0.0 || dblval > 1.0) {
	fprintf(stderr, "%s: Invalid overselection percentage '%s'\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetOverselectionPercentage(gp, dblval);
      break;
    case 'p':
      intval = strtol(optarg, 0, 0);
      if (intval <= 0) {
	fprintf(stderr, "%s: Population size '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetPopulationSize(gp, intval);
      break;
    case 'r':
      intval = strtol(optarg, 0, 0);
      if (intval <= 0) {
	fprintf(stderr, "%s: Invalid random seed '%s'\n", progname, optarg);
	errflag = 1;
      } else
	globalSetRandomNumberSeed(gp, intval);
      break;
    case 's':
      switch (*optarg) {
      case 'f':
      case 'F':
	globalSetParentSelectionMethod(gp, psmFitnessProportionate);
	break;
      case 'g':
      case 'G':
	globalSetParentSelectionMethod(gp, psmGreedyOverselection);
	break;
      case 't':
      case 'T':
	globalSetParentSelectionMethod(gp, psmTournament);
	break;
      default:
	fprintf(stderr, "%s: Invalid selection type '%s'\n", progname, optarg);
	fprintf(stderr, "\tSelection Type must be one of:\n");
	fprintf(stderr, "\t\tF(itness proportionate)\n");
	fprintf(stderr, "\t\tG(reedy overselection)\n");
	fprintf(stderr, "\t\tT(ournament)\n");
	errflag = 1;
	break;
      }
      break;
    case 't':
      intval = strtol(optarg, 0, 0);
      if (intval < 1 || intval > 10) {
	fprintf(stderr, "%s: Invalid number of tournament rounds '%s'\n", progname, optarg);
	errflag = 1;
      } else
	globalSetTournamentRounds(gp, intval);
      break;
    case 'u':
      globalSetCountUniqueMode(gp);
      break;
    case 'U':
      globalClearCountUniqueMode(gp);
      break;
    case 'v':
      globalSetVerboseMode(gp);
      break;
    case 'V':
      globalClearVerboseMode(gp);
      break;
    case 'x':
      intval = strtol(optarg, 0, 0);
      if (intval < 1) {
	fprintf(stderr,
		"%s: Checkpoint frequency '%s' must be greater than 0\n",
		progname, optarg);
	errflag = 1;
      } else
	globalSetCheckpointFrequency(gp, intval);
      break;
    case 'z':
      dblval = strtod(optarg, 0);
      globalSetParsimony(gp, dblval);
      break;
    case '?':
      fprintf(stderr, "%s: Bad argument '%c'\n", progname, c);
      errflag = 1;
      break;
    }

  /* make sure they specified a file name if they're checkpointing */
  if ((globalCheckpointFrequency(gp) > 0) &&
      (globalCheckpointName(gp) == 0)) {
    fprintf(stderr, "%s: Please specify a checkpoint file name with -f\n",
	    progname);
    errflag = 1;
  }

  /* die if there were problems */
  if (errflag) {
    argPrint("Usage: ");
    argPrint(progname);
    argPrint(" [-a(lwaysEvaluateProgram)]");
    argPrint(" [-A <opposite of -a>]");
    argPrint(" [-c creationMethod)]");
    argPrint(" [-d maximumTotalDepth]");
    argPrint(" [-f checkpointFilename]");
    argPrint(" [-F checkpointInputFilename]");
    argPrint(" [-g #generations]");
    argPrint(" [-i initialDepth]");
    argPrint(" [-m maximumMutationDepth]");
    argPrint(" [-o overselectionPercentage]");
    argPrint(" [-p populationSize]");
    argPrint(" [-r randomseed]");
    argPrint(" [-s selectionMethod]");
    argPrint(" [-t tournamentRounds]");
    argPrint(" [-u(niqueProgramStatistics)]");
    argPrint(" [-U <opposite of -u>]");
    argPrint(" [-v(erbose)]");
    argPrint(" [-V <opposite of -v>]");
    argPrint(" [-x checkpointFreq]");
    argPrint(" [-z parsimonyFactor]");
    argPrint("\n");
    exit(1);
  }

  return(gp);
}

int
main(argc, argv)
int argc;
char *argv[];
{
  global *gp;
  struct timeval first, second;
  struct timezone tzp;
  double elapsed;

  /* save program name */
  if ((progname = strrchr(argv[0], '/')) == 0)
    progname = argv[0];
  else
    progname++;

  /* create application global object */
  gp = globalCreate(appInitialize);
  if (!gp) {
    fprintf(stderr, "%s: Couldn't create application global!\n", progname);
    exit(1);
  }

#ifdef DEBUG_CHECKPOINT
  /* initialize checkpoint file name */
  checkptName = "ckpt.debug";
#endif

  /* process the argument list */
  gp = processArgs(argc, argv, gp);

  /* print parameters for this run */
  globalDumpParams(gp);

  /* start timing */
  gettimeofday(&first, &tzp);

  /* do stuff */
  globalGenesis(gp);
  globalLoop(gp, firstGen);

  /* stop timing, print elapsed time */
  gettimeofday(&second, &tzp);
  if (first.tv_usec > second.tv_usec) {
    second.tv_sec--;
    second.tv_usec += 1000000;
  }
  elapsed = (double )(second.tv_sec - first.tv_sec) +
	(double )(second.tv_usec - first.tv_usec)/1000000.0;
  printf("Elapsed time = %f seconds\n", elapsed);

  /* clean up allocated memory */
  globalFree(gp);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
  constantSrcFreeStack();
  resultFreeStack();
  variableFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */
  errorCodeFreeStack();

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
