/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#ifdef CHECK_RSRC_USAGE
#if defined(hpux) || defined(sun)
#include <sys/resource.h>
#ifdef hpux
#include <sys/syscall.h>
#define getrusage(a, b)	syscall(SYS_GETRUSAGE, a, b)
#endif /* hpux */
#else /* !(hpux || sun) */
#undef CHECK_RSRC_USAGE
#endif /* hpux || sun */
#endif /* CHECK_RSRC_USAGE */
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "pmrand.h"
#include "ckptio.h"
#include "global.h"
#include "proto.h"

#ifndef NULL
#define NULL	0
#endif

extern const char *progname;
static int byStandard P((const void *, const void *));
static int safeRename P((const char *, const char *));

global *
globalCreate(popInitFunc)
initializeFunction *popInitFunc;
{
  global *gp;
  population **popp;
  int num;

  /* allocate memory */
  gp = (global *)malloc(sizeof(global));
  if (gp) {

    /* initialize everything */
    gp->popInitFunc = popInitFunc;
    gp->checkptFreq = 0;
    gp->checkptName = 0;
    gp->countUnique = 0;
    gp->generations = 50;
    gp->verbose = 0;
    gp->evalFunc = globalDiscreteEval;
    gp->useSingleEnvironment = 0;
    gp->useFinalResult = 1;

    /* initialize random number seed */
    gp->randomSeed = time(0) ^ getpid();
    srnd(gp->randomSeed);

    /* create first population */
    gp->popp = (population **)malloc(sizeof(population *)*2);
    if (gp->popp) {
      gp->maxPops = 2;
      gp->numPops = 1;
      gp->popp[1] = 0;
      gp->popp[0] = populationCreate(0);
      if (popInitFunc)
	(*popInitFunc)(gp, gp->popp[0], 0);

      /* if they asked for more populations during the initialization... */
      if (gp->numPops > 1) {

	/* if we need more space, try to extend the population list */
	if (gp->numPops > gp->maxPops) {
	  popp = (population **)realloc(gp->popp,
					sizeof(population *) * gp->numPops);
	  if (!popp) {
	    populationFree(gp->popp[0]);
	    gp->popp[0] = 0;
	    gp->numPops = 0;
	  }

	  gp->popp = popp;
	  gp->maxPops = gp->numPops;
	}

	/* add new populations */
	num = 1;
	while (gp->numPops > num) {
	  gp->popp[num] = populationCopy(gp->popp[0]);
	  if (gp->popInitFunc)
	    (*(gp->popInitFunc))(gp, gp->popp[num], num);
	  num++;
	}
      }
    }

    /* get rid of failed global object */
    if (!gp->popp || !gp->popp[0]) {
      if (gp->popp)
	free(gp->popp);
      free(gp);
      gp = 0;
    }
  }
      
  return(gp);
}

void
#ifdef __STDC__
globalSetPopulationOption(const global *gp, populationOption opt, ...)
#else
globalSetPopulationOption(va_alist)
va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  global *gp;
  populationOption opt;
#endif
  int p;
  int intValue = 0;
  double dblValue = 0.0;
  parentSelectionMethod psmValue = psmFitnessProportionate;
  programCreationMethod pcmValue = pcmRamped;

#ifdef __STDC__
  va_start(ap, opt);
#else /* !__STDC__ */
  va_start(ap);
  gp = va_arg(ap, global *);
  opt = va_arg(ap, populationOption);
#endif /* __STDC__ */

  /* get option value */
  switch (opt) {
  case poClearAlwaysEvaluateMode:
  case poSetAlwaysEvaluateMode:
    break;
  case poInitialTreeDepth:
  case poMaximumMutationDepth:
  case poMaximumTreeDepth:
  case poSize:
  case poTournamentRounds:
    intValue = va_arg(ap, int);
    break;
  case poOverselectionPercentage:
    dblValue = va_arg(ap, double);
    break;
  case poParentSelectionMethod:
    psmValue = va_arg(ap, parentSelectionMethod);
    break;
  case poProgramCreationMethod:
    pcmValue = va_arg(ap, programCreationMethod);
    break;
  case poCaseInitializeFunc:
  case poCaseTerminateFunc:
  case poCaseFitnessFunc:
  case poEvalCleanupFunc:
  case poTerminateRunFunc:
  case poDestructorFunc:
  case poFitnessCases:
  case poTerminalList:
  case poFunctionList:
  case poBreedPercentage:
  case poMaximumLoops:
  case poParsimony:
  case poReturnTypes:
    break;
  }

  for (p = 0; p < gp->numPops; p++) {

    switch (opt) {
    case poClearAlwaysEvaluateMode:
      populationClearAlwaysEvaluateMode(gp->popp[p]);
      break;
    case poSetAlwaysEvaluateMode:
      populationClearAlwaysEvaluateMode(gp->popp[p]);
      break;
    case poInitialTreeDepth:
      populationSetInitialTreeDepth(gp->popp[p], intValue);
      break;
    case poMaximumMutationDepth:
      populationSetMaximumMutationDepth(gp->popp[p], intValue);
      break;
    case poMaximumTreeDepth:
      populationSetMaximumTreeDepth(gp->popp[p], intValue);
      break;
    case poOverselectionPercentage:
      populationSetOverselectionPercentage(gp->popp[p], dblValue);
      break;
    case poParentSelectionMethod:
      populationSetParentSelectionMethod(gp->popp[p], psmValue);
      break;
    case poProgramCreationMethod:
      populationSetProgramCreationMethod(gp->popp[p], pcmValue);
      break;
    case poSize:
      populationSetSize(gp->popp[p], intValue);
      break;
    case poTournamentRounds:
      populationSetTournamentRounds(gp->popp[p], intValue);
      break;
    case poCaseInitializeFunc:
    case poCaseTerminateFunc:
    case poCaseFitnessFunc:
    case poEvalCleanupFunc:
    case poTerminateRunFunc:
    case poDestructorFunc:
    case poFitnessCases:
    case poTerminalList:
    case poFunctionList:
    case poBreedPercentage:
    case poMaximumLoops:
    case poParsimony:
    case poReturnTypes:
      fprintf(stderr, "Not handling populationOption '%d'\n", opt);
      break;
    }
  }
  va_end(ap);
}

void
globalDumpParams(gp)
const global *gp;
{
  int i, j;
  const objectList *list;

  printf("Global:\n");
  if (globalCheckpointFrequency(gp) > 0) {
    printf("\tCheckpoint File Name: %s\n", globalCheckpointName(gp));
    printf("\tCheckpoint Frequency: %d\n", globalCheckpointFrequency(gp));
  }
  printf("\t%sPrint %% of Unique Programs\n",
	 (globalCountUniqueMode(gp) ? "" : "Don't "));
  printf("\tNumber of Generations: %d\n", globalGenerations(gp));
  printf("\tRandom Seed: %d\n", globalRandomNumberSeed(gp));
  printf("\t%sDump All Programs\n",
	 (globalVerboseMode(gp) ? "" : "Don't "));
  printf("\n");

  for (i = 0; i < gp->numPops; i++) {

    if (gp->numPops > 1)
      printf("\t\t\tPopulation %d\n", i);

    printf("Creation:\n");
    printf("\tInitial Depth: %d\n", populationInitialTreeDepth(gp->popp[i]));
    printf("\tPopulation Size: %d\n", populationSize(gp->popp[i]));
    printf("\tCreationMethod: %s\n\n",
	   (populationProgramCreationMethod(gp->popp[i]) == pcmFull ? "Full" :
	    (populationProgramCreationMethod(gp->popp[i]) == pcmGrow ? "Grow" :
	     (populationProgramCreationMethod(gp->popp[i]) == pcmRamped ?
	      "Ramped" :
	      "Unknown"))));

    printf("Evaluation:\n");
    if (populationAlwaysEvaluateMode(gp->popp[i]))
      printf("\tAlways Run Programs\n");
    else
      printf("\tDon't Run Copied Programs\n");
    if (populationParsimony(gp->popp[i]) > 0.0)
      printf("\tParsimony Factor: %f\n", populationParsimony(gp->popp[i]));
    if (populationHasCaseTerminateFunc(gp->popp[i]))
      printf("\tMaximum Number of Loops: %d\n",
	     populationMaximumLoops(gp->popp[i]));
    printf("\n");

    printf("Breeding:\n");
    printf("\tMaximum Mutation Depth: %d\n",
	   populationMaximumMutationDepth(gp->popp[i]));
    printf("\tMaximum Total Depth: %d\n",
	   populationMaximumTreeDepth(gp->popp[i]));
    printf("\tSelection Method:\n\t\t%s\n\n",
	   (populationParentSelectionMethod(gp->popp[i]) ==
	    psmFitnessProportionate ? "Fitness Proportionate" :
	    (populationParentSelectionMethod(gp->popp[i]) ==
	     psmGreedyOverselection ? "Greedy Overselection" :
	     (populationParentSelectionMethod(gp->popp[i]) == psmTournament ?
	      "Tournament" : "Unknown"))));

    if ((populationParentSelectionMethod(gp->popp[i]) ==
	 psmFitnessProportionate) ||
	(populationParentSelectionMethod(gp->popp[i]) ==
	 psmGreedyOverselection)) {
      printf("Breeding Method Percentage:\n");
      for (j = 0; j < pbtMaxProgramBreedFunctions; j++)
	if (populationBreedPercentage(gp->popp[i], j) > 0.0)
	  printf("\t%% %s: %5.2f\n", programBreedTypeName(j),
		 populationBreedPercentage(gp->popp[i], j)*100);
    }

    if (populationParentSelectionMethod(gp->popp[i]) == psmGreedyOverselection)
      printf("Greedy Overselection Point: %f\n",
	     populationOverselectionPercentage(gp->popp[i]));
    else if (populationParentSelectionMethod(gp->popp[i]) == psmTournament)
      printf("Tournament Rounds: %d\n",
	     populationTournamentRounds(gp->popp[i]));

#ifdef DUMP_LISTS
    list = populationTerminalList(gp->popp[i]);
    if (list) {
      charString *cstr;

      cstr = charStringCreate();
      charStringSet(cstr, "Terminal List: ");
      objectListToString(list, cstr);
      charStringCatenate(cstr, "\n");
      charStringWrite(cstr, stdout);
    }

    list = populationFunctionList(gp->popp[i]);
    if (list) {
      charString *cstr;

      cstr = charStringCreate();
      charStringSet(cstr, "Function List: ");
      objectListToString(list, cstr);
      charStringCatenate(cstr, "\n");
      charStringWrite(cstr, stdout);
    }
#endif /* DUMP_LISTS */

    printf("\n");
  }
}

void
globalGenesis(gp)
global *gp;
{
  int i;

  for (i = 0; i < gp->numPops; i++)
    populationGenesis(gp->popp[i]);
}

void
globalBreed(gp)
global *gp;
{
  int i;

  for (i = 0; i < gp->numPops; i++)
    populationBreed(gp->popp[i]);
}

static int
byStandard(a, b)
const void *a, *b;
{
  const program * const *p1 = (const program * const *)a;
  const program * const *p2 = (const program * const *)b;

  if (programStandardizedFitness(*p1) < programStandardizedFitness(*p2))
    return(-1);
  if (programStandardizedFitness(*p1) > programStandardizedFitness(*p2))
    return(1);
  return(0);
}

void
globalEvalSingle(gp, popNum, pp)
global *gp;
int popNum;
program *pp;
{
  population *pop;
  int i, l;
  void *envp = 0;
  int maxLoops;
  int hits = 0;
  double raw = 0;
  double std = 0;
  result *rp;

  /* make sure we didn't get a bogus program */
  if (pp == 0)
    return;

  /* get this population */
  if (popNum >= gp->numPops)
    return;
  pop = gp->popp[popNum];

  /* run through all the fitness cases */
  for (i = 0; i < populationFitnessCases(pop); i++) {

    /* set up environment for this case */
    if (populationHasCaseInitializeFunc(pop))
      envp = populationCaseInitializeFunc(pop, 0, i);

    /* evaluate this case */
    if (!populationHasCaseTerminateFunc(pop)) {
      rp = programEval(pp, envp);	/* only need to run program once */
    } else {

      /* repeatedly evaluate the program */
      l = 0;
      rp = 0;
      maxLoops = populationMaximumLoops(pop);
      do {
	if (rp)
	  resultFree(rp);
	rp = programEval(pp, envp);
      } while (!populationCaseTerminateFunc(pop, rp, envp, i) &&
	       (objectDataType(rp) != dtError) &&
	       (l++ < maxLoops));
    }

    /* evaluate the fitness of the result */
    if (populationHasCaseFitnessFunc(pop))
      populationCaseFitnessFunc(pop, rp, i, &hits, &raw, &std, envp);
    else {
      hits++;
      std++;
    }

    /* lose final result */
    if (rp)
      resultFree(rp);
  }

  /* clean up */
  if (populationHasEvalCleanupFunc(pop))
    populationEvalCleanupFunc(pop, envp);

  /* set number of hits for this program */
  programSetHits(pp, hits);

  /* modify standardized fitness by parsimony factor if desired */
  if (populationParsimony(pop) > 0.0)
    std += (programInternalNodes(pp) + programTerminalNodes(pp)) *
      populationParsimony(pop);

  /* set final raw and standardized fitness */
  programSetRawFitness(pp, raw);
  programSetStandardizedFitness(pp, std);
}

int
globalDiscreteEval(gp, genNum)
global *gp;
int genNum;
{
  int i, p;
  population *pop;
  program *pp;
  double adjtot = 0, pct = 0;
#ifdef DETECT_RESULT_LEAKS
  int leaked = 0;
#endif /* DETECT_RESULT_LEAKS */
  const program *bor;

  for (i = 0; i < gp->numPops; i++) {
    pop = gp->popp[i];

    for (p = 0; p < populationSize(pop); p++) {
      pp = populationProgram(pop, p);

      /* if this was reproduced, we don't have to reevaluate it */
      if (populationAlwaysEvaluateMode(pop) ||
	  ((programHits(pp) == 0) && (programStandardizedFitness(pp) == 0.0))) {
	globalEvalSingle(gp, i, pp);
#ifdef DETECT_RESULT_LEAKS
	if (resultFoundLeak())
	  leaked++;
#endif /* DETECT_RESULT_LEAKS */
      }

      /* compute adjusted fitness */
      programSetAdjustedFitness(pp, 1.0/(1.0 + programStandardizedFitness(pp)));

      /* total all adjusted fitnesses */
      adjtot += programAdjustedFitness(pp);
    }

    /* now go through and compute normalized fitness */
    for (p = 0; p < populationSize(pop); p++) {
      pp = populationProgram(pop, p);
      programSetNormalizedFitness(pp, programAdjustedFitness(pp) / adjtot);
    }

    /* sort by standardized fitness */
    populationSort(pop, byStandard);

    /* set cumulative percentage */
    pct = 0;
    for (p = 0; p < populationSize(pop); p++) {
      pp = populationProgram(pop, p);
      pct += programNormalizedFitness(pp);
      programSetCumulativePercentage(pp, pct);
    }

    /* see if there's a new best-of-run program */
    pp = populationProgram(pop, 0);
    if ((populationBestOfRun(pop) == NULL) ||
	(programStandardizedFitness(pp) <
	 programStandardizedFitness(populationBestOfRun(pop)))) {
      populationSetBestOfRun(pop, pp);
      populationSetBestGeneration(pop, genNum);
    }

#ifdef DETECT_RESULT_LEAKS
    /* report number of leaks found */
    if (leaked > 0) {
      fprintf(stderr, "WARNING\nWARNING: Some result objects");
      fprintf(stderr, " were not freed!\nWARNING\n\n");
    }
#endif /* DETECT_RESULT_LEAKS */

    /* see if this program is good enough to terminate the run */
    if (populationHasTerminateRunFunc(pop)) {
      bor = populationBestOfRun(pop);
      return(populationTerminateRunFunc(pop, 0, bor));
    }
  }

  return(0);
}

typedef struct codata {
  program *pp;
  int progNum;
  population *pop;
  void *envp;
  result *rp;
  int hits;
  double raw, std;
} codata;

void
globalEvalMultiple(gp, info, numInfo)
global *gp;
progInfo info[];
int numInfo;
{
  int i, j, l;
  int maxLoops;
  result *rp = 0;

  /* set up data array */
  for (j = 0; j < numInfo; j++) {
    info[j].hits = programHits(info[j].pp);
    info[j].raw = programRawFitness(info[j].pp);
    info[j].std = programStandardizedFitness(info[j].pp);
  }

  /* run through all the fitness cases */
  for (i = 0; i < populationFitnessCases(info[0].pop); i++) {

    /* set up environment for this case */
    for (j = 0; j < numInfo; j++)
      if (gp->useSingleEnvironment && j) {
	info[j].envp = info[0].envp;
      } else if (populationHasCaseInitializeFunc(info[j].pop))
	info[j].envp = populationCaseInitializeFunc(info[j].pop, j, i);
      else
	info[j].envp = 0;

    /* evaluate this case */
    if (!populationHasCaseTerminateFunc(info[0].pop)) {
      for (j = 0; j < numInfo; j++)
	rp = info[j].rp = programEval(info[j].pp, info[j].envp);
    } else {

      /* set up variables */
      maxLoops = populationMaximumLoops(info[0].pop);
      for (j = 0; j < numInfo; j++) {
	info[j].rp = 0;
	if (maxLoops < populationMaximumLoops(info[j].pop))
	  maxLoops = populationMaximumLoops(info[j].pop);
      }

      /* repeatedly evaluate the program */
      j = numInfo - 1;
      l = 0;
      do {
	j = (j + 1) % numInfo;
	if (info[j].rp)
	  resultFree(info[j].rp);
	info[j].rp = programEval(info[j].pp, info[j].envp);
      } while (!populationCaseTerminateFunc(info[j].pop,
					    info[j].rp, info[j].envp, i) &&
	       (objectDataType(info[j].rp) != dtError) &&
	       (l++ < maxLoops));
      rp = info[j].rp;
    }

    /* evaluate the fitness of the result */
    for (j = 0; j < numInfo; j++) {
      if (populationHasCaseFitnessFunc(info[j].pop))
	populationCaseFitnessFunc(info[j].pop,
				  (gp->useFinalResult ? rp : info[j].rp), i,
				  &(info[j].hits), &(info[j].raw),
				  &(info[j].std), info[j].envp);
      else {
	info[j].hits++;
	info[j].std++;
      }

      /* lose this result (unless it's the final result) */
      if (info[j].rp && (info[j].rp != rp))
	resultFree(info[j].rp);
    }

    /* lose final result */
    if (rp)
      resultFree(rp);
  }

  /* clean up */
  for (j = 0; j < numInfo; j++) {
    if (!(gp->useSingleEnvironment) || (j == 0))
      if (populationHasEvalCleanupFunc(info[j].pop))
	populationEvalCleanupFunc(info[j].pop, info[j].envp);

    /* set number of hits for this program */
    programSetHits(info[j].pp, info[j].hits);

    /* modify standardized fitness by parsimony factor if desired */
    if (populationParsimony(info[j].pop) > 0.0)
      info[j].std += (programInternalNodes(info[j].pp) +
		      programTerminalNodes(info[j].pp)) *
			populationParsimony(info[j].pop);

    /* set final raw and standardized fitness */
    programSetRawFitness(info[j].pp, info[j].raw);
    programSetStandardizedFitness(info[j].pp, info[j].std);
  }
}

int
globalCoEval(gp, genNum)
global *gp;
int genNum;
{
  int i, j, p1, p2;
  population *pop;
  program *pp;
  double adjtot, pct;
#ifdef DETECT_RESULT_LEAKS
  int leaked = 0;
#endif /* DETECT_RESULT_LEAKS */
  const program *bor;
  int terminated = 1;
  progInfo piList[2];

  /* clear out hits, raw, std values */
  for (i = 0; i < (gp->numPops - 1); i++) {
    pop = gp->popp[i];

    for (p1 = 0; p1 < populationSize(pop); p1++) {
      pp = populationProgram(pop, p1);

      programSetHits(pp, 0);
      programSetRawFitness(pp, 0.0);
      programSetStandardizedFitness(pp, 0.0);
    }
  }

  /* coevaluate programs */
  for (i = 0; i < (gp->numPops - 1); i++) {
    piList[0].pop = gp->popp[i];

    for (p1 = 0; p1 < populationSize(gp->popp[i]); p1++) {
      piList[0].pp = populationProgram(gp->popp[i], p1);

      for (j = i + 1; j < gp->numPops; j++) {
	piList[1].pop = gp->popp[j];

	for (p2 = 0; p2 < populationSize(gp->popp[j]); p2++) {
	  piList[1].pp = populationProgram(gp->popp[j], p2);

	  globalEvalMultiple(gp, piList, 2);

#ifdef DETECT_RESULT_LEAKS
	  if (resultFoundLeak())
	    leaked++;
#endif /* DETECT_RESULT_LEAKS */
	}

      }
    }
  }

  for (i = 0; i < gp->numPops; i++) {
    pop = gp->popp[i];

    adjtot = 0;
    for (j = 0; j < populationSize(pop); j++) {
      pp = populationProgram(pop, j);

      /* compute adjusted fitness */
      programSetAdjustedFitness(pp,
				1.0/(1.0 + programStandardizedFitness(pp)));

      /* total all adjusted fitnesses */
      adjtot += programAdjustedFitness(pp);
    }

    /* now go through and compute normalized fitness */
    for (j = 0; j < populationSize(pop); j++) {
      pp = populationProgram(pop, j);
      programSetNormalizedFitness(pp, programAdjustedFitness(pp) / adjtot);
    }

    /* sort by standardized fitness */
    populationSort(pop, byStandard);

    /* set cumulative percentage */
    pct = 0;
    for (j = 0; j < populationSize(pop); j++) {
      pp = populationProgram(pop, j);
      pct += programNormalizedFitness(pp);
      programSetCumulativePercentage(pp, pct);
    }

    /* see if there's a new best-of-run program */
    pp = populationProgram(pop, 0);
    if ((populationBestOfRun(pop) == NULL) ||
	(programStandardizedFitness(pp) <
	 programStandardizedFitness(populationBestOfRun(pop)))) {
      populationSetBestOfRun(pop, pp);
      populationSetBestGeneration(pop, genNum);
    }

#ifdef DETECT_RESULT_LEAKS
    /* report number of leaks found */
    if (leaked > 0) {
      fprintf(stderr, "WARNING\nWARNING: Some result objects");
      fprintf(stderr, " were not freed!\nWARNING\n\n");
    }
#endif /* DETECT_RESULT_LEAKS */

    /* see if this program is good enough to terminate the run */
    if (populationHasTerminateRunFunc(pop)) {
      bor = populationBestOfRun(pop);
      terminated &= populationTerminateRunFunc(pop, i, bor);
    }
  }

  return(terminated);
}

static int
safeRename(oldname, newname)
const char *oldname, *newname;
{
  struct stat statbuf;

  /* remove new file if it exists */
  if (stat(newname, &statbuf) == 0)
    unlink(newname);

  /* move old file to new name */
  if (rename(oldname, newname))
    return(1);

  /* success */
  return(0);
}

global *
globalReadCheckpointFile(name, initFunc, genPtr)
const char *name;
initializeFunction *initFunc;
int *genPtr;
{
  FILE *in;
  charString *cstr;
  global *gp;
  int i, rval;
  int numPops;

  /* try to open the checkpoint file */
  in = fopen(name, "r");
  if (!in)
    return(0);

  /* create an input buffer */
  cstr = charStringCreate();
  if (!cstr) {
    fclose(in);
    return(NULL);
  }

  /* create global structure */
  gp = globalCreate(initFunc);
  if (!gp) {
    charStringFree(cstr);
    fclose(in);
    return(NULL);
  }

  /* read checkpoint data */
  rval = (readInteger(in, cstr, "currentGen", genPtr) ||
	  readInteger(in, cstr, "maxGen", &(gp->generations)) ||
	  readInteger(in, cstr, "checkptFreq", &(gp->checkptFreq)) ||
	  readBoolean(in, cstr, "countUnique", &(gp->countUnique)) ||
	  readBoolean(in, cstr, "verbose", &(gp->verbose)) ||
	  readInteger(in, cstr, "numPops", &numPops));

  /* make sure population array is big enough */
  if (numPops > gp->maxPops) {
    free(gp->popp);
    gp->popp = (population **)malloc(sizeof(population *)*numPops);
    if (gp->popp)
      gp->maxPops = numPops;
    else {
      gp->maxPops = 0;
      rval = 1;
    }
  }

  /* handle population checkpoint info */
  if (!rval) {

    /* clear out old pointers */
    for (i = 0; i < gp->maxPops; i++)
      gp->popp[i] = 0;

    /* read in population data */
    gp->numPops = 0;
    for (i = 0; !rval && (i < numPops); ) {
      gp->popp[i] = populationReadCheckpoint(gp, i, initFunc, in);
      if (gp->popp[i])
	i++;
    }
    gp->numPops = i;
  }

  fclose(in);

  if (rval) {
    globalFree(gp);
    gp = 0;
  }

  return(gp);
}

int
globalWriteCheckpointFile(gp, gen)
const global *gp;
int gen;
{
  const char *ext = "bak";
  char *backupName;
  FILE *out;
  int i, rval;

  /* get enough space for new name */
  backupName = (char *)malloc(strlen(gp->checkptName) + 1 + strlen(ext) + 1);
  if (!backupName)
    return(0);

  /* create new name */
  strcpy(backupName, gp->checkptName);
  strcat(backupName, ".");
  strcat(backupName, ext);

  /* backup old checkpoint file if it exists */
  safeRename(gp->checkptName, backupName);

  /* try to open the checkpoint file */
  out = fopen(gp->checkptName, "w");
  if (!out)
    return(-1);

  /* write checkpoint data */
  rval = (writeInteger(out, "currentGen", gen) ||
	  writeInteger(out, "maxGen", gp->generations) ||
	  writeInteger(out, "checkptFreq", gp->checkptFreq) ||
	  writeBoolean(out, "countUnique", gp->countUnique) ||
	  writeBoolean(out, "verbose", gp->verbose) ||
	  writeInteger(out, "numPops", gp->numPops));
  for (i = 0; !rval && (i < gp->numPops); i++)
    rval = populationWriteCheckpoint(gp->popp[i], out);

  fclose(out);

  /* if we failed, lose new file; otherwise, lose old file */
  if (rval) {
    unlink(gp->checkptName);
    safeRename(backupName, gp->checkptName);
  } else
    unlink(backupName);

  /* free allocated memory, return */
  free(backupName);
  return(rval);
}

int
globalCompare(gp1, gp2)
const global *gp1, *gp2;
{
  int i;

  /* NULL pointers can match */
  if (!gp1 && !gp2)
    return(1);

  /* fail on cases where one pointer is NULL but the other isn't */
  if (!gp1 || !gp2)
    return(0);

  /* fail if variables differ */
  if ((gp1->numPops != gp2->numPops) ||
      (gp1->generations != gp2->generations) ||
      (gp1->checkptFreq != gp2->checkptFreq) ||
#ifdef SAVE_CHECKPT_NAME
      !(gp1->checkptName && gp2->checkptName &&
       (strcmp(gp1->checkptName, gp2->checkptName) == 0)) ||
#endif /* SAVE_CHECKPT_NAME */
      (gp1->countUnique != gp2->countUnique) ||
      (gp1->verbose != gp2->verbose))
    return(0);

  /* fail it populations differ */
  for (i = 0; i < gp1->numPops; i++)
    if (!populationCompare(gp1->popp[i], gp2->popp[i], 1))
      return(0);

  /* they're identical */
  return(1);
}

void
globalDump(gp, gen)
const global *gp;
int gen;
{
  int i;

  printf("\t\t\tGeneration %d\n", gen);

  for (i = 0; i < gp->numPops; i++) {

    if (gp->numPops > 1)
      printf("\t\t\tPopulation %d\n", i);
    populationDump(gp->popp[i], gp->verbose, gp->countUnique);
    printf("\n");
  }
}

void
globalLoop(gp, firstGen)
global *gp;
int firstGen;
{
  int g, i, done;
  int ckpt = 0;

  for (g = firstGen; g <= gp->generations; g++) {
    if (g != firstGen)
      globalBreed(gp);

    if (++ckpt == gp->checkptFreq) {
      if (globalWriteCheckpointFile(gp, g))
	fprintf(stderr, "%s: Couldn't write checkpoint file \"%s\"!\n",
		progname, gp->checkptName);
      else
	printf("\t<Checkpoint written>\n");
      ckpt = 0;
    }

    done = globalEval(gp, g);
    globalDump(gp, g);

#ifdef CHECK_RSRC_USAGE
    {
      struct rusage rusage;

      if (getrusage(RUSAGE_SELF, &rusage))
	perror(progname);
      else
	printf("RsrcUsg: UTime=%ds%du STime=%ds%du MaxRSS=%d, IntRSS=%d, Swaps=%d\n",
	       rusage.ru_utime.tv_sec, rusage.ru_utime.tv_usec,
	       rusage.ru_stime.tv_sec, rusage.ru_stime.tv_usec,
	       rusage.ru_maxrss, rusage.ru_idrss, rusage.ru_nswap);
    }
#endif /* CHECK_RSRC_USAGE */

    /* outta here if we met the termination criteria */
    if (done)
      break;
  }

  /* print final summary */
  for (i = 0; i < gp->numPops; i++) {
    printf("Best program");
    if (gp->numPops > 1)
      printf(" for population %d", i);
    printf(" found on generation %d\n", populationBestGeneration(gp->popp[i]));
    if (gp->verbose)
      programDump(populationBestOfRun(gp->popp[i]), 1);
  }
}

void
globalFree(gp)
global *gp;
{
  int i;

  for (i = 0; i < gp->numPops; i++)
    populationFree(gp->popp[i]);
  free(gp->popp);
  free(gp);
}

#ifdef DEBUG_GLOBAL

#include "constant.h"
#include "objectlist.h"
#include "operatrsrc.h"
#include "optrivial.h"

static objectList *appTerminals P((NOARGS));
static objectList *appFunctions P((NOARGS));
static void appCaseFitness P((result *, int, int *, double *, double *,
			      void *));
static void appInitialize P((void *, population *, int));
static void checkCkpt P((global *gp));
int main P((int, char *[]));

const char *progname;

static objectList *
appTerminals()
{
  objectList *list;
  int i;

  /* create list of terminals */
  list = objectListCreate(6);
  for (i = 1; i < 6; i++)
    if (objectListAdd(list, integerCreate(i))) {
      objectListFree(list);
      return(0);
    }
  return(list);
}

static objectList *
appFunctions()
{
  objectList *list;
  operatorSrc *osp;

  /* create list of functions */
  list = objectListCreate(2);

  osp = simpleOperatorSrcCreate("+", opAdd, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  osp = simpleOperatorSrcCreate("-", opSubtract, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static void appCaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  *hitp = resultInteger(rp);
  *stdp = 32767 - *hitp;
}

static void
appInitialize(gp, pop, num)
void *gp;
population *pop;
int num;
{
  globalSetRandomNumberSeed(gp, 123456789);

  populationSetSize(pop, 16);
  populationSetFitnessCases(pop, 1);

  populationSetTerminalList(pop, appTerminals());
  populationSetFunctionList(pop, appFunctions());

  populationSetCaseFitnessFunc(pop, appCaseFitness);
}

static void
checkCkpt(gp)
global *gp;
{
  char filename[L_tmpnam];
  int rtnval;
  global *ngp;
  int gen;

  /* set up a temporary checkpoint file name */
  tmpnam(filename);
  globalSetCheckpointName(gp, filename);

  /* write checkpoint info */
  rtnval = globalWriteCheckpointFile(gp, 66);
  if (rtnval) {
    fprintf(stderr, "%s: Problems encountered while writing ckpt file\n",
	    progname);
  } else {

    /* read checkpoint file we just created */
    ngp = globalReadCheckpointFile(filename, appInitialize, &gen);
    if (!ngp) {
      fprintf(stderr, "%s: Problems encountered while reading ckpt file!\n",
	      progname);
      return;
    } else {

      /* compare everything */
      if ((gen != 66) || !globalCompare(gp, ngp)) {
	fprintf(stderr, "%s: Globals didn't match!\n", progname);
	printf("***************** OLD GLOBAL *****************\n");
	globalDump(gp, 66);
	printf("***************** NEW GLOBAL *****************\n");
	globalDump(ngp, gen);
      } else
	printf("Global checkpoint works\n");

      /* get rid of new global */
      globalFree(ngp);
    }
  }

  /* get rid of temporary file */
  unlink(filename);
}

int
main(argc, argv)
int argc;
char *argv[];
{
  global *gp;
  int done, g, i;

  /* save program name */
  if ((progname = strrchr(argv[0], '/')) == 0)
    progname = argv[0];
  else
    progname++;

  /* create global */
  gp = globalCreate(appInitialize);
  globalSetGenerations(gp, 3);
  globalSetVerboseMode(gp);

  globalDumpParams(gp);

  /* evolve population(s) */
  globalGenesis(gp);
  for (done = g = 0; !done && g < gp->generations; g++) {
    if (g)
      globalBreed(gp);
    done = globalEval(gp, g);
    printf("\nGeneration %d Summary:\n", g);
    for (i = 0; i < gp->numPops; i++) {
      if (i > 1)
	printf("\t\t\tPopulation %d\n", i);
      populationDump(gp->popp[i], gp->verbose, gp->countUnique);
    }
  }

  checkCkpt(gp);

  /* clean up */
  globalFree(gp);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
  resultFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

  printf("GLOBAL works\n");
  exit(0);
}
#endif /* DEBUG_GLOBAL */
