/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "pmrand.h"
#include "object.h"
#include "constntsrc.h"
#include "proto.h"

#ifdef KEEP_ALLOCATED_MEMORY
static constantSrc *freeStack = 0;
#endif /* KEEP_ALLOCATED_MEMORY */

constantSrc *
#ifdef __STDC__
constantSrcCreate(datatype dt, ...)
#else
constantSrcCreate(va_alist)
va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  datatype dt;
#endif
  constantSrc *csp;
  datatype rdt;
  int imin, imax;
  double dmin, dmax;

#ifdef __STDC__
  va_start(ap, dt);
#else /* !__STDC__ */
  va_start(ap);
  dt = va_arg(ap, datatype);
#endif /* __STDC__ */

  /* create a new instance of constantSrc */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    csp = freeStack;
    freeStack = *(constantSrc **)csp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    csp = (constantSrc *)malloc(sizeof(constantSrc));

  /* fill the new constantSrc instance */
  if (csp) {
    objectSetType(csp, otConstantSrc);
    objectSetDataType(csp, dt);

    rdt = datatypeResolve(dt);
    switch (rdt) {
    case dtBoolean:
      break;
    case dtShort:
    case dtInteger:
      imin = va_arg(ap, int);
      imax = va_arg(ap, int);
      if (imin || imax) {
	csp->min.i = imin;
	csp->max.i = imax;
      } else
	csp->min.i = csp->max.i = 0;
      break;
    case dtFloat:
    case dtDouble:
      dmin = va_arg(ap, double);
      dmax = va_arg(ap, double);
      if (dmin || dmax) {
	csp->min.d = dmin;
	csp->max.d = dmax;
      } else
	csp->min.d = csp->max.d = 0;
      break;
    case dtBlob:
      break;
    case dtVoid:
    case dtLong:
    case dtList:
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
    case dtError:
      constantSrcFree(csp);
      csp = 0;
      break;
    }
  }

  /* clean up and cruise */
  va_end(ap);
  return(csp);
}

constant *
constantSrcCopy(csp)
const constantSrc *csp;
{
  datatype rdt;
  int ival;
  double dval;
  blob *bbp;

  rdt = datatypeResolve(objectDataType(csp));
  switch (rdt) {
  case dtBoolean:
    ival = irndrng(0, 1);
    return(booleanCreate(ival));
  case dtShort:
    if (csp->min.i || csp->max.i)
      ival = irndrng(csp->min.i, csp->max.i);
    else
      ival = irnd();
    return(shortCreate(ival));
  case dtInteger:
    if (csp->min.i || csp->max.i)
      ival = irndrng(csp->min.i, csp->max.i);
    else
      ival = irnd();
    return(integerCreate(ival));
  case dtFloat:
    if (csp->min.d || csp->max.d)
      dval = drndrng(csp->min.d, csp->max.d);
    else
      dval = drnd();
    return(floatCreate(dval));
  case dtDouble:
    if (csp->min.d || csp->max.d)
      dval = drndrng(csp->min.d, csp->max.d);
    else
      dval = drnd();
    return(doubleCreate(dval));
  case dtBlob:
    bbp = blobCreate(objectDataType(csp));
    return(blobPtrCreate(bbp, objectDataType(csp)));
  case dtVoid:
  case dtLong:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4: 
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9: 
  case dtError:
    break;
  }
  return(0);
}

result *
constantSrcEval(csp)
const constantSrc *csp;
{
  datatype rdt;
  result *rp = 0;
  bool bval;
  short sval;
  int ival;
  float fval;
  double dval;
  blob *bbp;

  rdt = datatypeResolve(objectDataType(csp));
  switch (rdt) {
  case dtBoolean:
    bval = irndrng(0, 1);
    rp = resultCreate(dtBoolean, (int )bval);
    break;
  case dtShort:
    if (csp->min.i || csp->max.i)
      sval = irndrng(csp->min.i, csp->max.i);
    else
      sval = irnd();
    rp = resultCreate(dtShort, (int )sval);
    break;
  case dtInteger:
    if (csp->min.i || csp->max.i)
      ival = irndrng(csp->min.i, csp->max.i);
    else
      ival = irnd();
    rp = resultCreate(dtInteger, ival);
    break;
  case dtFloat:
    if (csp->min.d || csp->max.d)
      fval = drndrng(csp->min.d, csp->max.d);
    else
      fval = drnd();
    rp = resultCreate(dtFloat, (double )fval);
  case dtDouble:
    if (csp->min.d || csp->max.d)
      dval = drndrng(csp->min.d, csp->max.d);
    else
      dval = drnd();
    rp = resultCreate(dtDouble, dval);
    break;
  case dtBlob:
    bbp = blobCreate(objectDataType(csp));
    rp = resultCreate(objectDataType(csp), bbp);
    break;
  case dtVoid:
  case dtLong:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
  case dtError:
    rp = resultCreate(dtError, ErrorBadDataType);
    break;
  }
  return(rp);
}

int
constantSrcCompare(csp1, csp2)
const constantSrc *csp1, *csp2;
{
  datatype rdt;

  if (objectDataType(csp1) != objectDataType(csp2))
    return(0);

  rdt = datatypeResolve(objectDataType(csp1));
  switch (rdt) {
    case dtBoolean:
      break;
    case dtShort:
    case dtInteger:
      if ((csp1->min.i != csp2->min.i) || (csp1->max.i != csp2->max.i))
	return(0);
      break;
    case dtFloat:
    case dtDouble:
      if ((csp1->min.d != csp2->min.d) || (csp1->max.d != csp2->max.d))
	return(0);
      break;
    case dtBlob:
      break;
    case dtVoid:
    case dtLong:
    case dtList:
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
    case dtError:
      return(0);
    }

  return(1);
}

const constantSrc **
constantSrcNodePtr(cspp, count, nodeNum, internal, typeptr)
const constantSrc **cspp;
int *count;
int nodeNum, internal;
datatype *typeptr;
{
  /* constantSrcs are terminal points */
  if (internal)
    return(0);

  /* see if we're the node they want */
  if (((*count)++ == nodeNum) &&
      ((*typeptr == 0) || ((*typeptr & objectDataType(*cspp)) != 0)))
    return(cspp);

  /* we aren't the right node */
  return(0);
}

int
constantSrcToString(csp, cstr)
const constantSrc *csp;
charString *cstr;
{
  datatype rdt;
  const char *tstr = "??";
  char buffer[128];

  rdt = datatypeResolve(objectDataType(csp));
  switch (rdt) {
  case dtBoolean:
    tstr = "Boolean";
    break;
  case dtShort:
    tstr = "Short";
    break;
  case dtInteger:
    tstr = "Integer";
    break;
  case dtFloat:
    tstr = "Float";
    break;
  case dtDouble:
    tstr = "Double";
    break;
  case dtBlob:
    tstr = "Blob";
    break;
  case dtVoid:
  case dtLong:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
  case dtError:
    tstr = "ILLEGAL";
    break;
  }
  sprintf(buffer, "ConstSource<%s>", tstr);
  return(charStringCatenate(cstr, buffer));
}

void
constantSrcFree(csp)
constantSrc *csp;
{
#ifdef KEEP_ALLOCATED_MEMORY
  if (csp) {
    *(constantSrc **)csp = freeStack;
    freeStack = csp;
  }
#else /* !KEEP_ALLOCATED_MEMORY */
  free(csp);
#endif /* KEEP_ALLOCATED_MEMORY */
}
    
#ifdef KEEP_ALLOCATED_MEMORY
void
constantSrcFreeStack()
{
  constantSrc *csp;

  while (freeStack) {
    csp = freeStack;
    freeStack = *(constantSrc **)csp;
    free(csp);
  }
}
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef DEBUG_CONSTANTSRC

static void checkBoolean P((NOARGS));
static void checkInteger P((NOARGS));
static void checkDouble P((NOARGS));
int main P((NOARGS));

static void
checkBoolean()
{
  constantSrc *csp;
  int i;
  constant *cp;

  /* generate a whole bunch of random booleans */
  csp = booleanSrcCreate();

  for (i = 0; i < 8; i++) {
    cp = constantSrcCopy(csp);
    printf("%s\n", constantBoolean(cp) ? "TRUE" : "FALSE");
    constantFree(cp);
  }
  constantSrcFree(csp);
}

static void
checkInteger()
{
  constantSrc *csp;
  int i, sum[12];
  constant *cp;
  int ival;

  /* create source, zero sum array */
  csp = integerSrcCreate(1, 10);
  for (i = 0; i < 12; i++)
    sum[i] = 0;

  /* generate a whole bunch of random numbers */
  for (i = 0; i < 10000; i++) {
    cp = constantSrcCopy(csp);
    ival = constantInteger(cp);
    if ((ival < 1) || (ival > 10))
      fprintf(stderr, "Bad value %d\n", ival);
    else
      sum[ival]++;
    constantFree(cp);
  }

  /* print final totals */
  for (i = 0; i < 12; i++)
    printf("%2d: %d\n", i, sum[i]);
  constantSrcFree(csp);
}

static void
checkDouble()
{
  constantSrc *csp;
  constant *cp;
  int i;

  /* create source */
  csp = doubleSrcCreate(0.0, 0.0);

  /* generate a whole bunch of random numbers */
  for (i = 0; i < 100; i++) {
    cp = constantSrcCopy(csp);
    printf("%f\n", constantDouble(cp));
    constantFree(cp);
  }
  constantSrcFree(csp);
}

int
main()
{
  charString *cstr;
  constantSrc *csp;
  constant *cp;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  cstr = charStringCreate();

  csp = booleanSrcCreate();
  charStringSet(cstr, "ConstantSrc is ");
  constantSrcToString(csp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp);

  csp = shortSrcCreate(0, 10);
  charStringSet(cstr, "ConstantSrc is ");
  constantSrcToString(csp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp);

  csp = integerSrcCreate(0, 10);
  charStringSet(cstr, "ConstantSrc is ");
  constantSrcToString(csp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp);

  csp = floatSrcCreate(0.0, 0.0);
  charStringSet(cstr, "ConstantSrc is ");
  constantSrcToString(csp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp);

  csp = doubleSrcCreate(0.0, 0.0);
  charStringSet(cstr, "ConstantSrc is ");
  constantSrcToString(csp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp);

  csp = booleanSrcCreate(); cp = constantSrcCopy(csp);
  charStringSet(cstr, "Original is ");
  constantSrcToString(csp, cstr);
  charStringCatenate(cstr, ", copy is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp); constantFree(cp);

  csp = shortSrcCreate(-5, 5); cp = constantSrcCopy(csp);
  charStringSet(cstr, "Original is ");
  constantSrcToString(csp, cstr);
  charStringCatenate(cstr, ", copy is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp); constantFree(cp);

  csp = integerSrcCreate(-5, 5); cp = constantSrcCopy(csp);
  charStringSet(cstr, "Original is ");
  constantSrcToString(csp, cstr);
  charStringCatenate(cstr, ", copy is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp); constantFree(cp);

  csp = floatSrcCreate(0.0, 0.0); cp = constantSrcCopy(csp);
  charStringSet(cstr, "Original is ");
  constantSrcToString(csp, cstr);
  charStringCatenate(cstr, ", copy is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp); constantFree(cp);

  csp = doubleSrcCreate(0, 0); cp = constantSrcCopy(csp);
  charStringSet(cstr, "Original is ");
  constantSrcToString(csp, cstr);
  charStringCatenate(cstr, ", copy is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantSrcFree(csp); constantFree(cp);

  checkBoolean();
  checkInteger();
  checkDouble();

  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  constantSrcFreeStack();
  constantFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_CONSTANTSRC */
