#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "ckptio.h"
#include "proto.h"

extern const char *progname;

static const char *skipName P((FILE *, charString *, const char *));

static const char *
skipName(in, cstr, name)
FILE *in;
charString *cstr;
const char *name;
{
  const char *cp;
  unsigned len;

  /* read string */
  if (charStringRead(cstr, in) || charStringLength(cstr) == 0)
    return(0);

  /* make sure it's the string we're looking for */
  cp = charStringBuffer(cstr);
  len = strlen(name);
  if (strncmp(cp, name, len) != 0 || !isspace(cp[len]))
    return(0);

  /* skip trailing whitespace */
  for (cp += len; *cp && isspace(*cp); cp++)
    ;

  return(cp);
}

int
readCreationMethod(in, cstr, cmp)
FILE *in;
charString *cstr;
programCreationMethod *cmp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, "createMethod");
  if (!cp)
    return(1);

  /* find creation method */
  switch (*cp) {
  case 'f':
  case 'F':
    if (strncasecmp(cp, "Full", 4) == 0 && (cp[4] == 0 || isspace(cp[4]))) {
      *cmp = pcmFull;
      return(0);
    }
    break;
  case 'g':
  case 'G':
    if (strncasecmp(cp, "Grow", 4) == 0 && (cp[4] == 0 || isspace(cp[4]))) {
      *cmp = pcmGrow;
      return(0);
    }
    break;
  case 'r':
  case 'R':
    if (strncasecmp(cp, "Ramped", 6) == 0 && (cp[6] == 0 || isspace(cp[6]))) {
      *cmp = pcmRamped;
      return(0);
    }
    break;
  }

  /* didn't find it */
  fprintf(stderr, "%s: Bad creation method in checkpoint file!\n", progname);
  return(1);
}

int
readSelectionMethod(in, cstr, smp)
FILE *in;
charString *cstr;
parentSelectionMethod *smp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, "selectMethod");
  if (!cp)
    return(1);

  /* find creation method */
  switch (*cp) {
  case 'f':
  case 'F':
    if (strncasecmp(cp, "FitProp", 7) == 0 && (cp[7] == 0 || isspace(cp[7]))) {
      *smp = psmFitnessProportionate;
      return(0);
    }
    break;
  case 'g':
  case 'G':
    if (strncasecmp(cp, "Greedy", 6) == 0 && (cp[6] == 0 || isspace(cp[6]))) {
      *smp = psmGreedyOverselection;
      return(0);
    }
    break;
  case 't':
  case 'T':
    if (strncasecmp(cp, "Tourn", 5) == 0 && (cp[5] == 0 || isspace(cp[5]))) {
      *smp = psmTournament;
      return(0);
    }
    break;
  }

  /* didn't find it */
  fprintf(stderr, "%s: Bad selection method in checkpoint file!\n", progname);
  return(1);
}

int
readBoolean(in, cstr, name, blnp)
FILE *in;
charString *cstr;
const char *name;
int *blnp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, name);
  if (!cp) {
    fprintf(stderr, "%s: Looking for field \"%s\"", progname, name);
    fprintf(stderr, " in checkpoint file, got \"%s\"\n", cp);
    return(1);
  }

  /* grab the boolean value */
  if (*cp == 'T' || *cp == 't')
    *blnp = 1;
  else if (*cp == 'F' || *cp == 'f')
    *blnp = 0;
  else {
    fprintf(stderr, "%s: Bad boolean value '%s'", progname, cp);
    fprintf(stderr, " for field \"%s\" in checkpoint file\n", name);
    return(2);
  }

  return(0);
}

int
readInteger(in, cstr, name, intp)
FILE *in;
charString *cstr;
const char *name;
int *intp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, name);
  if (!cp) {
    fprintf(stderr, "%s: Looking for field \"%s\"", progname, name);
    fprintf(stderr, " in checkpoint file, got \"%s\"\n", cp);
    return(1);
  }

  /* grab the integer */
  sscanf(cp, "%d", intp);

  return(0);
}

int
readDouble(in, cstr, name, dblp)
FILE *in;
charString *cstr;
const char *name;
double *dblp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, name);
  if (!cp) {
    fprintf(stderr, "%s: Looking for field \"%s\"", progname, name);
    fprintf(stderr, " in checkpoint file, got \"%s\"\n", cp);
    return(1);
  }

  /* grab the double value */
  sscanf(cp, "%lf", dblp);

  return(0);
}

int
writeCreationMethod(out, cm)
FILE *out;
programCreationMethod cm;
{
  const char *name = "Unknown";

  switch (cm) {
  case pcmFull:
    name = "Full";
    break;
  case pcmGrow:
    name = "Grow";
    break;
  case pcmRamped:
    name = "Ramped";
    break;
  }
  return(fprintf(out, "createMethod %s\n", name) == EOF);
}

int
writeSelectionMethod(out, sm)
FILE *out;
parentSelectionMethod sm;
{
  const char *name = "Unknown";

  switch (sm) {
  case psmFitnessProportionate:
    name = "FitProp";
    break;
  case psmGreedyOverselection:
    name = "Greedy";
    break;
  case psmTournament:
    name = "Tourn";
    break;
  }
  return(fprintf(out, "selectMethod %s\n", name) == EOF);
}
