#ifndef PROGRAM_H
#define PROGRAM_H

#include "compiler.h"
#include "object.h"
#include "objectlist.h"

typedef enum programCreationMethod {
  pcmFull, pcmGrow, pcmRamped
} programCreationMethod;

typedef enum programCrossoverType {
  pxoInternal, pxoTerminal, pxoAny
} programCrossoverType;

typedef enum programBreedType {
  pbtReproduction, pbtXoverInternal, pbtXoverTerminal, pbtXoverAny,
  pbtCreation, pbtHoist,
  pbtMutation,
  pbtMaxProgramBreedFunctions,
  pbtParse = pbtMaxProgramBreedFunctions,
  pbtCopyViaMutate, pbtCopyViaXover, pbtCopyViaHoist,
  pbtMaxProgramBreedTypes
} programBreedType;

typedef struct program {
  object *tree;
  int internals, terminals, depth;
  int hits;
  double raw, standardized, normalized, cumPercentage;
#ifdef TRACK_BREED_TYPE
  programBreedType breedType;
#endif /* TRACK_BREED_TYPE */
} program;

#define programInternalNodes(pp)		((const int )(pp)->internals)
#define programTerminalNodes(pp)		((const int )(pp)->terminals)
#define programDepth(pp)			((const int )(pp)->depth)

#define programHits(pp)				((const int )(pp)->hits)
#define programRawFitness(pp)			((const double )(pp)->raw)
#define programStandardizedFitness(pp)		\
  ((const double )((pp)->standardized))
#define programAdjustedFitness(pp)		\
  ((const double )((pp)->normalized))
#define programNormalizedFitness(pp)		\
  ((const double )((pp)->normalized))
#define programCumulativePercentage(pp)		\
  ((const double )((pp)->cumPercentage))

#ifdef TRACK_BREED_TYPE
#define programBreedSource(pp)			\
  ((const programBreedType )(pp)->breedType)
#endif /* TRACK_BREED_TYPE */
#define programBreedTypeName(pbt)		\
  ((pbt) == pbtReproduction ? "Reproduction" : \
   ((pbt) == pbtXoverInternal ? "Internal Crossover" : \
    ((pbt) == pbtXoverTerminal ? "Terminal Crossover" : \
     ((pbt) == pbtXoverAny ? "General Crossover" : \
      ((pbt) == pbtCreation ? "Creation" : \
       ((pbt) == pbtHoist ? "Hoist" : \
	((pbt) == pbtMutation ? "Mutation" : \
	 ((pbt) == pbtParse ? "Parse" : \
	  ((pbt) == pbtCopyViaMutate ? "Copy (Failed Mutation)" : \
	   ((pbt) == pbtCopyViaXover ? "Copy (Failed Crossover)" : \
	    "Unknown BreedType"))))))))))

#define programSetHits(pp, h)			(pp)->hits = h
#define programSetRawFitness(pp, d)		(pp)->raw = d
#define programSetStandardizedFitness(pp, d)	(pp)->standardized = d
#define programSetAdjustedFitness(pp, d)	(pp)->normalized = d
#define programSetNormalizedFitness(pp, d)	(pp)->normalized = d
#define programSetCumulativePercentage(pp, d)	(pp)->cumPercentage = d

#define programEval(pp, t)			objectEval((pp)->tree, t)
#define programResetFitness(pp)			(pp)->hits = (pp)->raw = \
  (pp)->standardized = (pp)->normalized = 0

#define programToString(pp, s)			objectToString((pp)->tree, s)
#define programCheckpoint(pp, s)		objectCheckpoint((pp)->tree, s)

program *programCreate P((const objectList *termlist,
			  const objectList *funclist,
			  datatype rtntypes, int depth,
			  programCreationMethod method));
program *programParse P((const objectList *termlist,
			 const objectList *funclist,
			 const char *string));
program *programCopy P((const program *pp));
program *programMutate P((program *pp, const objectList *termlist,
			  const objectList *funclist, datatype rtntypes,
			  int subtreeDepth, int totalDepth));
int programCrossover P((program *pp1, program *pp2, programCrossoverType type,
			int maxDepth, datatype topTypes));
program *programHoist P((program *pp, datatype topTypes));
int programCompare P((const program *pp1, const program *pp2));
void programDump P((const program *pp, int dumpProgram));
void programFree P((program *pp));

#endif /* PROGRAM_H */
