#ifndef POPULATION_H
#define POPULATION_H

#include "compiler.h"
#include "object.h"
#include "objectlist.h"
#include "program.h"

typedef enum parentSelectionMethod {
  psmFitnessProportionate, psmGreedyOverselection, psmTournament
} parentSelectionMethod;

typedef void *(caseInitFunction) P((int, int));
typedef int (caseTerminateFunction) P((result *, void *, int));
typedef void (caseFitnessFunction) P((result *, int, int *, double *, double *,
				      void *));
typedef void (evalCleanupFunction) P((void *));
typedef int (terminateRunFunction) P((int, int, double, double));
typedef void (destroyFunction) P((NOARGS));

typedef enum populationOption {
  poCaseInitializeFunc, poCaseTerminateFunc, poCaseFitnessFunc,
  poEvalCleanupFunc, poTerminateRunFunc, poDestructorFunc, poFitnessCases,
  poTerminalList, poFunctionList, poClearAlwaysEvaluateMode,
  poSetAlwaysEvaluateMode, poBreedPercentage,
  poInitialTreeDepth, poMaximumLoops, poMaximumMutationDepth,
  poMaximumTreeDepth, poOverselectionPercentage, poParentSelectionMethod,
  poParsimony, poProgramCreationMethod, poReturnTypes, poSize,
  poTournamentRounds
} populationOption;

typedef struct population {
  /* functions */
  caseInitFunction *caseInit;
  caseTerminateFunction *caseTerminate;
  caseFitnessFunction *caseFitness;
  evalCleanupFunction *evalCleanup;
  terminateRunFunction *terminateRun;
  destroyFunction *destructor;
  /* manditory variables */
  int size;
  int cases;
  objectList *tList, *fList;
  /* optional variables */
  int alwaysEval;
  double percentage[pbtMaxProgramBreedFunctions];
  int initialDepth;
  int maxLoops;
  int maxMutateDepth;
  int maxTotalDepth;
  double overselectionPct;
  parentSelectionMethod selectionMethod;
  double parsimony;
  programCreationMethod creationMethod;
  unsigned returnTypes;
  int tournamentRounds;
  /* internal variables */
  void *environment;
  /* programs */
  program **programs, **workspace;
  program *bestOfRun;
  int bestGeneration;
} population;

typedef program *(parentSelectionFunction) P((const population *));
typedef void (initializeFunction) P((void *, population *, int));

#define populationProgram(pop, i)	((pop) && ((i) >= 0) &&		\
					 ((i) < (pop)->size) ?	\
					 (program *)((pop)->programs[i]) : 0)
#define populationSort(pop, func)	qsort((char *)(pop)->programs,	\
					      (unsigned )(pop)->size,\
					      sizeof(program *), (func))

#define populationSetCaseInitializeFunc(pop, func) \
     (pop)->caseInit = (func)
#define populationSetCaseTerminateFunc(pop, func) \
     (pop)->caseTerminate = (func)
#define populationSetCaseFitnessFunc(pop, func)	(pop)->caseFitness = (func)
#define populationSetEvalCleanupFunc(pop, func)	(pop)->evalCleanup = (func)
#define populationSetTerminateRunFunc(pop, func) \
     (pop)->terminateRun = (func)
#define populationSetDestructorFunc(pop, func)	(pop)->destructor = (func)

#define populationHasCaseInitializeFunc(pop)	((pop)->caseInit != 0)
#define populationHasCaseTerminateFunc(pop)	((pop)->caseTerminate != 0)
#define populationHasCaseFitnessFunc(pop)	((pop)->caseFitness != 0)
#define populationHasEvalCleanupFunc(pop)	((pop)->evalCleanup != 0)
#define populationHasTerminateRunFunc(pop)	((pop)->terminateRun != 0)
#define populationHasDestructorFunc(pop)	((pop)->destructor != 0)

#define populationCaseInitializeFunc(pop, num, i)			\
	(((pop)->caseInit)(num, i))
#define populationCaseTerminateFunc(pop, rp, envp, i)			\
       (((pop)->caseTerminate)(rp, envp, i))
#define populationCaseFitnessFunc(pop, rp, i, hitp, rawp, stdp, envp)	\
       (((pop)->caseFitness)(rp, i, hitp, rawp, stdp, envp))
#define populationEvalCleanupFunc(pop, envp)	(((pop)->evalCleanup)(envp))
#define populationTerminateRunFunc(pop, popNum, pp)			\
	(((pop)->terminateRun)(popNum, programHits(pp), \
			       programRawFitness(pp), \
			       programStandardizedFitness(pp)))

#define populationSetFitnessCases(pop, fc)	(pop)->cases = (fc)
#define populationSetTerminalList(pop, tl)	\
       objectListOrganize((pop)->tList = (tl))
#define populationSetFunctionList(pop, fl)	\
       objectListOrganize((pop)->fList = (fl))

#define populationTerminalList(pop)	((const objectList *)(pop)->tList)
#define populationFunctionList(pop)	((const objectList *)(pop)->fList)
#define populationFitnessCases(pop)	((const int )(pop)->cases)

#define populationClearAlwaysEvaluateMode(pop)	(pop)->alwaysEval = 0
#define populationSetAlwaysEvaluateMode(pop)	(pop)->alwaysEval = 1
#define populationSetBreedPercentage(pop, pbt, p) \
     (pop)->percentage[pbt] = (p)
#define populationSetInitialTreeDepth(pop, d)	(pop)->initialDepth = (d)
#define populationSetMaximumLoops(pop, ml)	(pop)->maxLoops = (ml)
#define populationSetMaximumMutationDepth(pop, d) \
     (pop)->maxMutateDepth = (d)
#define populationSetMaximumTreeDepth(pop, d)	(pop)->maxTotalDepth = (d)
#define populationSetOverselectionPercentage(pop, o) \
     (pop)->overselectionPct = (o)
#define populationSetParentSelectionMethod(pop, psm) \
     (pop)->selectionMethod = (psm)
#define populationSetParsimony(pop, p)		(pop)->parsimony = (p)
#define populationSetProgramCreationMethod(pop, pcm) \
     (pop)->creationMethod = (pcm)
#define populationSetReturnTypes(pop, dt)	(pop)->returnTypes = (dt)
#define populationSetSize(pop, s)		\
     ((pop)->size = ((pop)->programs == NULL ? (s) : (pop)->size))
#define populationSetTournamentRounds(pop, r)	(pop)->tournamentRounds = (r)

#define populationSetBestOfRun(pop, bor) \
					{ \
					  if ((pop)->bestOfRun) \
					    programFree((pop)->bestOfRun); \
					  (pop)->bestOfRun = (bor); \
					}
#define populationSetBestGeneration(pop, g)	(pop)->bestGeneration = (g)

#define populationAlwaysEvaluateMode(pop)	((pop)->alwaysEval != 0)
#define populationBreedPercentage(pop, pbt) \
     ((const double )(pop)->percentage[pbt])
#define populationBreedPercentageArray(pop) \
     ((const double *)(pop)->percentage)
#define populationInitialTreeDepth(pop)	((const int )(pop)->initialDepth)
#define populationMaximumLoops(pop)	((const int )(pop)->maxLoops)
#define populationMaximumMutationDepth(pop) \
     ((const int )(pop)->maxMutateDepth)
#define populationMaximumTreeDepth(pop)	((const int )(pop)->maxTotalDepth)
#define populationOverselectionPercentage(pop) \
     ((const double )(pop)->overselectionPct)
#define populationParentSelectionMethod(pop) \
     ((const parentSelectionMethod )(pop)->selectionMethod)
#define populationParsimony(pop)	((const double )(pop)->parsimony)
#define populationSize(pop)		((const int )(pop)->size)
#define populationProgramCreationMethod(pop) \
     ((const programCreationMethod )(pop)->creationMethod)
#define populationReturnTypes(pop)	((const unsigned )(pop)->returnTypes)
#define populationTournamentRounds(pop)	((const int )(pop)->tournamentRounds)

#define populationBestOfRun(pop)	(program *)((pop)->bestOfRun)
#define populationBestGeneration(pop)	(const int )(pop)->bestGeneration

population *populationCreate P((int num));
population *populationCopy P((const population *pop));
void populationGenesis P((population *pop));
void populationBreed P((population *pop));
int populationCompare P((const population *pop1, const population *pop2,
			 int cmpProgs));
void populationDump P((const population *pop, int verbose, int countUnique));
population *populationReadCheckpoint P((void *gp, int num,
					initializeFunction *initFunc,
					FILE *infile));
int populationWriteCheckpoint P((const population *pop, FILE *outfile));
void populationFree P((population *pop));

#endif /* POPULATION_H */
