#include "geppetto.h"
#include "proto.h"

#define TREE_LENGTH	63
int tree[TREE_LENGTH] = {
  32, 32, 31, 16, 15, 16, 16, 16, 7, 8, 8, 8, 24, 24, 23, 12,
  3, 4, 4, 4, 20, 20, 19, 12, 28, 28, 27, 12, 11, 12, 12, 12,
  1, 2, 2, 2, 18, 18, 17, 10, 26, 26, 25, 10, 9, 10, 10, 10,
  30, 30, 29, 14, 13, 14, 14, 14, 5, 6, 6, 6, 22, 22, 21
};

typedef struct environment {
  int fitnessCases;
  char playerName;
  population *pop;
} environment;

#ifdef PLAYER_X
const int fitnessCases = 4;
#else /* !PLAYER_X */
#ifdef PLAYER_O
const int fitnessCases = 8;
#else /* !PLAYER_O */
const int fitnessCases = 1;
#endif /* PLAYER_O */
#endif /* PLAYER_X */

static int current, step;
static char xm1, om1, xm2, om2, xm3;
static environment envarray[2];
static int maxHits;

static result *opLeft P((int, const object **, void *));
static result *opRight P((int, const object **, void *));
static objectList *discreteGameTerminals P((NOARGS));
static result *opCaseXMove1 P((int, const object **, void *));
static result *opCaseOMove1 P((int, const object **, void *));
static result *opCaseXMove2 P((int, const object **, void *));
static result *opCaseOMove2 P((int, const object **, void *));
static objectList *discreteGameFunctions P((NOARGS));
static void *discreteGameCaseInitialize P((int, int));
static int discreteGameCaseTerminate P((result *, void *, int));
static void discreteGameCaseFitness P((result *, int, int *, double *,
				       double *, void *));
static int discreteGameTerminateRun P((int, int, double, double));

static result *
opLeft(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  /* move left */
  current -= step;

  /* assign move to proper variable */
  if (step == 16)
    xm1 = 'l';
  else if (step == 8)
    om1 = 'l';
  else if (step == 4)
    xm2 = 'l';
  else if (step == 2)
    om2 = 'l';
  else if (step == 1)
    xm3 = 'l';
  else
    fprintf(stderr, "Bad step value %d in opLeft!\n", step);

  /* remember that we moved */
  step >>= 1;

  return(resultCreate(dtVoid));
}

static result *
opRight(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  /* move right */
  current += step;

  /* assign move to proper variable */
  if (step == 16)
    xm1 = 'r';
  else if (step == 8)
    om1 = 'r';
  else if (step == 4)
    xm2 = 'r';
  else if (step == 2)
    om2 = 'r';
  else if (step == 1)
    xm3 = 'r';
  else
    fprintf(stderr, "Bad step value %d in opRight!\n", step);

  /* remember that we moved */
  step >>= 1;

  return(resultCreate(dtVoid));
}

static objectList *
discreteGameTerminals()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(2);
  if (list) {

    osp = complexOperatorSrcCreate("L", opLeft, dtVoid, 0, 0);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("R", opRight, dtVoid, 0, 0);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);

}

static result *
opCaseXMove1(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  switch (xm1) {
  case 'u':
    return(objectEval(argv[0], 0));
  case 'l':
    return(objectEval(argv[1], 0));
  case 'r':
    return(objectEval(argv[2], 0));
  }

  fprintf(stderr, "Bad value '%c' for xm1 in opCaseXMove1!\n", xm1);
  return(resultCreate(dtVoid));
}

static result *
opCaseOMove1(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  switch (om1) {
  case 'u':
    return(objectEval(argv[0], 0));
  case 'l':
    return(objectEval(argv[1], 0));
  case 'r':
    return(objectEval(argv[2], 0));
  }

  fprintf(stderr, "Bad value '%c' for om1 in opCaseOMove1!\n", om1);
  return(resultCreate(dtVoid));
}

static result *
opCaseXMove2(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  switch (xm2) {
  case 'u':
    return(objectEval(argv[0], 0));
  case 'l':
    return(objectEval(argv[1], 0));
  case 'r':
    return(objectEval(argv[2], 0));
  }

  fprintf(stderr, "Bad value '%c' for xm2 in opCaseXMove2!\n", xm2);
  return(resultCreate(dtVoid));
}

static result *
opCaseOMove2(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  switch (om2) {
  case 'u':
    return(objectEval(argv[0], 0));
  case 'l':
    return(objectEval(argv[1], 0));
  case 'r':
    return(objectEval(argv[2], 0));
  }

  fprintf(stderr, "Bad value '%c' for om2 in opCaseOMove2!\n", om2);
  return(resultCreate(dtVoid));
}

static objectList *
discreteGameFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(4);
  if (list) {

    osp = complexOperatorSrcCreate("CXM1", opCaseXMove1, dtVoid, 3, 3,
				   dtVoid, dtVoid, dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("COM1", opCaseOMove1, dtVoid, 3, 3,
				   dtVoid, dtVoid, dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("CXM2", opCaseXMove2, dtVoid, 3, 3,
				   dtVoid, dtVoid, dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("COM2", opCaseOMove2, dtVoid, 3, 3,
				   dtVoid, dtVoid, dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static void *
discreteGameCaseInitialize(popNum, fc)
int popNum;
int fc;
{
  environment *envp;

  xm1 = om1 = xm2 = om2 = xm3 = 'u';
  current = 31;
  step = 16;

#ifdef PLAYER_O
  if (!(fc & 0x01)) {
    xm1 = 'l';
    current -= step;
  } else {
    xm1 = 'r';
    current += step;
  }

  /* remember that we moved */
  step >>= 1;
#endif /* PLAYER_O */

  envp = &envarray[popNum];
#ifdef PLAYER_X
  envp->playerName = 'X';
  envp->fitnessCases = fitnessCases;
#else /* !PLAYER_X */
#ifdef PLAYER_O
  envp->playerName = 'O';
  envp->fitnessCases = fitnessCases;
#else /* !PLAYER_O */
  if (popNum == 0)
    envp->playerName = 'X';
  else
    envp->playerName = 'O';
  envp->fitnessCases = maxHits;
#endif /* PLAYER_O */
#endif /* PLAYER_X */

  return(envp);
}

static int
discreteGameCaseTerminate(rp, envp, fc)
result *rp;
void *envp;
int fc;
{
  switch (step) {
  case 0x8:
    if (!(fc & 0x01)) {
      om1 = 'l';
      current -= step;
    } else {
      om1 = 'r';
      current += step;
    }
    break;
  case 0x4:
    if (!(fc & 0x02)) {
      xm2 = 'l';
      current -= step;
    } else {
      xm2 = 'r';
      current += step;
    }
    break;
  case 0x2:
    if (!(fc & 0x02)) {
      om2 = 'l';
      current -= step;
    } else {
      om2 = 'r';
      current += step;
    }
    break;
  case 0x1:

    if (!(fc & 0x04)) {
      xm3 = 'l';
      current -= step;
    } else {
      xm3 = 'r';
      current += step;
    }
    break;
  }

  /* remember that we moved */
  step >>= 1;

  /* quit if there aren't any moves left */
  return(step == 0);
}

static void
discreteGameCaseFitness(rp, fc, hitp, rawp, stdp, ep)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *ep;
{
  environment *envp = (environment *)ep;
  int winner;

  /* did this program win? */
  if (envp->playerName == 'X')
    winner = (tree[current] >= 12);
  else
    winner = (tree[current] <= 12);

#define DEBUG_DISCRETEGAME_XXX
#ifdef DEBUG_DISCRETEGAME_XXX
  printf("Game %d: %c %c %c %c %c, %c %s(%d)\n", fc, xm1, om1, xm2, om2, xm3,
	 envp->playerName, winner ? "won" : "lost", tree[current]);
#endif /* DEBUG_DISCRETEGAME */

  /* the result should be a void */
  if (!resultIsVoid(rp))
    *stdp += envp->fitnessCases;
  else if (winner)
    *hitp += 1;
  else
    *stdp += 1.0;
}

static int
discreteGameTerminateRun(popNum, hits, raw, std)
int popNum;
int hits;
double raw;
double std;
{
#ifdef BOTH_PLAYERS
  return(hits == populationSize(envarray[popNum].pop));
#else /* !BOTH_PLAYERS */
  return(hits == maxHits);
#endif /* BOTH_PLAYERS */
}

void
appInitialize(gp, pop, popNum)
void *gp;
population *pop;
int popNum;
{
#ifdef BOTH_PLAYERS
  int popSize;

  popSize = populationSize(pop);
  if (popSize > maxHits)
    maxHits = popSize;
  globalSetNumberOfPopulations(gp, 2);
  globalSetEvalFunc(gp, globalCoEval);

  /* stash population ptr for future reference */
  envarray[1 - popNum].pop = pop;
#else /* !BOTH_PLAYERS */
  maxHits = fitnessCases;
#endif /* BOTH_PLAYERS */

  populationSetFitnessCases(pop, fitnessCases);
  populationSetTerminalList(pop, discreteGameTerminals());
  populationSetFunctionList(pop, discreteGameFunctions());
  populationSetCaseInitializeFunc(pop, discreteGameCaseInitialize);
  populationSetCaseTerminateFunc(pop, discreteGameCaseTerminate);
  populationSetCaseFitnessFunc(pop, discreteGameCaseFitness);
  populationSetTerminateRunFunc(pop, discreteGameTerminateRun);
}

#ifdef DEBUG_DISCRETEGAME

#ifdef DUMP_TREE
static void dumpTree P((NOARGS));

static void
dumpTree()
{
  int i;

  for (i = 0; i < TREE_LENGTH; i++)
    if (!(i & 0x01))
      printf("\t%d\n", tree[i]);
    else if (!(i & 0x02))
      printf("\t\t%d\n", tree[i]);
    else if (!(i & 0x04))
      printf("\t\t\t%d\n", tree[i]);
    else if (!(i & 0x08))
      printf("\t\t\t\t%d\n", tree[i]);
    else if (!(i & 0x10))
      printf("\t\t\t\t\t%d\n", tree[i]);
    else if (!(i & 0x20))
      printf("\t\t\t\t\t\t%d\n", tree[i]);
    else if (!(i & 0x40))
      printf("\t\t\t\t\t\t\t%d\n", tree[i]);
    else if (!(i & 0x80))
      printf("\t\t\t\t\t\t\t\t%d\n", tree[i]);
    else
      printf("??? %d\n", tree[i]);
}
#endif /* DUMP_TREE */

const char *proglist[] = {
#ifdef PLAYER_O
  "(CXM2 (CXM1 L R L) L R)",
#endif /* PLAYER_O */
#ifdef PLAYER_X
  "(COM2 (COM1 L L R) L R)",
  "(COM1 L L R)",
#endif /* PLAYER_X */
	0
};

#endif /* DEBUG_DISCRETEGAME */
