#include "geppetto.h"
#include "opdata.h"
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif

#ifdef DEBUG_COREWAR
#define NOISY
#endif

#define NUM_WARRIORS	2

/* user-defined data types */

#define dtImmediate	dtUserDef0
#define dtDirect	dtUserDef1
#define dtIndirect	dtUserDef2
#define dtDecrement	dtUserDef3
#ifdef HANDLE_INCREMENT
#define dtIncrement	dtUserDef4
#endif /* HANDLE_INCREMENT */
#define dtInstruction	dtUserDef5
#define dtMultiInstr	dtUserDef6
#define dtWarrior	dtUserDef7

typedef enum envStatus {
  uninitialized, gotProgram, gotError, gotResults
} envStatus;

typedef struct coreWarEnv {
  envStatus status;
  int num;
  program_t *prog;
  int wins;
} coreWarEnv;

typedef struct progData {
  coreWarEnv *envp;
  program_t *prog;
  int wins;
} progData;

/* KotH data */

static void initKotH P((void));
static void fight P((progData *, unsigned));
#ifdef NOISY
static void disasm P((core_el_t *, aval_t, aval_t));
#endif /* NOISY */
static void progFree P((program_t *));

char  cwerrstr[STR_ERRLEN] = "";
const char  *opnames[] = {
	"DAT", "MOV", "ADD", "SUB", "JMP", "JMZ", "JMN",
	"DJN", "CMP", "SPL", "SLT", "END", "EQU", NULL};
const char  *amodenames[] = {
	"immediate", "direct", "indirect", "predecrement", "postincrement"};
core_el_t  *core = NULL;

aval_t  coresize = 8000;
aval_t  maxprogsize = 100;
unsigned  maxthreads = 8000;
unsigned  amode_restrict = TRUE;  /* TRUE is ICWS '88 */
unsigned  postinc = FALSE;  /* FALSE is ICWS '88 */
unsigned  interactive = FALSE;
unsigned  split_id = FALSE;

unsigned  max_cycles;
unsigned  w_radius; /* Maximum distance to do a write. */
unsigned  rounds = 3;

static progData progList[NUM_WARRIORS];
static int numProgs = 0;
static int KotHInitialized = 0;

static void
initKotH()
{
  w_radius = coresize / 2;
  max_cycles = coresize * 10;

  KotHInitialized = 1;
}

static void
fight(list, nprogs)
progData *list;
unsigned nprogs;
{
  int simlen, i, j;

  ld_init(nprogs);
  new_core();
  for (i = 0;  i < nprogs;  ++i)
    load_core(list[i].prog, ld_addr(i), i);
    simlen = simulate(max_cycles, nprogs);
    for (i = 0, j = 0;  i < nprogs;  ++i)
      if (proc_count(i) == 0)
	++j;
    for (i = 0;  i < nprogs;  ++i)
      if (proc_count(i))
	list[i].wins += j + 1;
}

#ifdef NOISY
static void
disasm(listing, len, start)
core_el_t *listing;
aval_t len;
aval_t start;
{
  static char amname[] = "# @<>";
  aval_t  addr;

  for (addr = 0;  addr < len;  ++addr)  {
    if (addr == start)
      printf("START");
    printf("\t%s\t%c", opnames[JTOP(listing->index)],
	   amname[JTAM1(listing->index)]);
    if (listing->val1 == AVAL_PROGID)
      printf("ID");
    else
      printf("%4d", listing->val1);
    printf(",%c", amname[JTAM2(listing->index)]);
    if (listing->val2 == AVAL_PROGID)
      printf("ID\n");
    else
      printf("%4d\n", listing->val2);
    ++listing;
  }
}
#endif /* NOISY */

static void
progFree(prog)
program_t *prog;
{
  if (prog->listing)
    free(prog->listing);
  free(prog);
}

/* CoreWar BLOB code */

blob *
blobCreate(dtype)
datatype dtype;
{
  /* don't allow blobs to be created */
  return((blob *)0);
}

blob *
blobCopy(bbp, dtype)
const blob *bbp;
datatype dtype;
{
  opData *odp;

  switch (dtype) {
  case dtInstruction:
  case dtMultiInstr:
    odp = opDataCopy((const opData *)bbp);
    return((blob *)odp);
  default:
    break;
  }

  return(NULL);
}

int
blobCompare(bbp0, bbp1, dtype)
const blob *bbp0, *bbp1;
datatype dtype;
{
  return(0);
}

blob *
blobParse(sp, dtp)
const char **sp;
datatype *dtp;
{
  fprintf(stderr, "blobParse() called\n");
  return(0);
}

int
blobToString(bbp, dtype, cstr)
const blob *bbp;
datatype dtype;
charString *cstr;
{
  return(charStringCatenate(cstr, "??UNKNOWN??"));
}

void
blobFree(bbp, dtype)
blob *bbp;
datatype dtype;
{
  switch (dtype) {
  case dtInstruction:
  case dtMultiInstr:
    opDataFree((opData *)bbp);
    break;
  default:
    break;
  }
}

/* GP CoreWar code */

static coreWarEnv env[NUM_WARRIORS];

#ifdef HANDLE_INCREMENT
#define dtAnyMode	(dtImmediate|dtDirect|dtIndirect|dtDecrement|dtIncrement)
#else /* !HANDLE_INCREMENT */
#define dtAnyMode	(dtImmediate|dtDirect|dtIndirect|dtDecrement)
#endif /* HANDLE_INCREMENT */

#define ErrorDivideByZero	ErrorUserDefined+0
const char *MsgDivideByZero =	"Divide by Zero";
#define ErrorOutOfMemory	ErrorUserDefined+1
const char *MsgOutOfMemory =	"Out Of Memory";
#define ErrorProgramTooLong	ErrorUserDefined+2
const char *MsgProgramTooLong =	"Program Too Long";

static objectList *corewarTerminals P((void));
static result *opAdd P((int, const object **, void *));
static result *opSubtract P((int, const object **, void *));
static result *opMultiply P((int, const object **, void *));
static result *opDivide P((int, const object **, void *));
static result *opModeGeneric P((datatype, const object *, void *));
static result *opModeImmediate P((int, const object **, void *));
static result *opModeDirect P((int, const object **, void *));
static result *opModeIndirect P((int, const object **, void *));
static result *opModeDecrement P((int, const object **, void *));
#ifdef HANDLE_INCREMENT
static result *opModeIncrement P((int, const object **, void *));
#endif /* HANDLE_INCREMENT */
amode_t resultMode P((result *));
static result *opOpGeneric P((const object *, const object *, op_t, void *));
static result *opOpData P((int, const object **, void *));
static result *opOpMove P((int, const object **, void *));
static result *opOpAdd P((int, const object **, void *));
static result *opOpSubtract P((int, const object **, void *));
static result *opOpJump P((int, const object **, void *));
static result *opOpJumpZero P((int, const object **, void *));
static result *opOpJumpNonZero P((int, const object **, void *));
static result *opOpDecAndJumpNonZero P((int, const object **, void *));
static result *opOpCompare P((int, const object **, void *));
static result *opOpSplit P((int, const object **, void *));
static result *opOpSkipLessThan P((int, const object **, void *));
static result *opOpJumpBack P((int, const object **, void *));
static result *opOpJumpFrwd P((int, const object **, void *));
static result *opOpJumpZeroBack P((int, const object **, void *));
static result *opOpJumpZeroFrwd P((int, const object **, void *));
static result *opOpJumpNonZeroBack P((int, const object **, void *));
static result *opOpJumpNonZeroFrwd P((int, const object **, void *));
static result *opBlock P((int, const object **, void *));
static result *opAssemble P((int, const object **, void *));
static objectList *corewarFunctions P((NOARGS));
static void *corewarCaseInitialize P((int, int));
static int corewarCaseTerminate P((result *, void *, int));
static void corewarCaseFitness P((result *, int, int *, double *, double *,
				  void *));
static int corewarTerminateRun P((int, int, double, double));

objectList *
corewarTerminals()
{
  objectList *list;
  constantSrc *csp;

  list = objectListCreate(1);
  if (list) {

    csp = integerSrcCreate(-5, 5);
    if (objectListAdd(list, csp)) {
      constantSrcFree(csp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static result *
opAdd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else
    resultSetInteger(r0, resultInteger(r0) + resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opSubtract(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else
    resultSetInteger(r0, resultInteger(r0) - resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opMultiply(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else
    resultSetInteger(r0, resultInteger(r0) * resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opDivide(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else if (resultInteger(r1) == 0)
    resultSetError(r0, ErrorDivideByZero);
  else
    resultSetInteger(r0, resultInteger(r0) / resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opModeGeneric(dtype, obp, envp)
datatype dtype;
const object *obp;
void *envp;
{
  result *rp;
  int ival;

  /* evaluate argument */
  rp = objectEval(obp, envp);
  if (resultIsError(rp))
    return(rp);

  /* make sure result is valid */
  if (!resultIsInteger(rp))
    resultSetError(rp, ErrorBadDataType);

  /* adjust value so it's in the range 0..coresize */
  ival = resultInteger(rp);
  if (ival < 0) {
    ival = -(ival) % coresize;
    if (ival == 0)
      return(0);
    else
      ival = coresize - ival;
  } else
    ival = ival % coresize;

  /* return the new result */
  resultFree(rp);
  return(resultCreate(dtype, ival));
}

static result *
opModeImmediate(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opModeGeneric(dtImmediate, argv[0], envp));
}

static result *
opModeDirect(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opModeGeneric(dtDirect, argv[0], envp));
}

static result *
opModeIndirect(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opModeGeneric(dtIndirect, argv[0], envp));
}

static result *
opModeDecrement(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opModeGeneric(dtDecrement, argv[0], envp));
}

#ifdef HANDLE_INCREMENT
static result *
opModeIncrement(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opModeGeneric(dtIncrement, argv[0], envp));
}
#endif /* HANDLE_INCREMENT */

amode_t
resultMode(rp)
result *rp;
{
  switch (objectDataType((object *)rp)) {
  case dtImmediate:
    return(am_imm);
  case dtDirect:
    return(am_dir);
  case dtIndirect:
    return(am_ind);
  case dtDecrement:
    return(am_dec);
#ifdef HANDLE_INCREMENT
  case dtIncrement:
    return(am_inc);
#endif /* HANDLE_INCREMENT */
  default:
    break;
  }
  return(-1);
}

static result *
opOpGeneric(arg0, arg1, optype, envp)
const object *arg0, *arg1;
op_t optype;
void *envp;
{
  result *r0, *r1;
  opData od;

  /* evaluate arguments */
  r0 = objectEval(arg0, envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(arg1, envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1)) {
    resultFree(r0);
    r0 = resultCreate(dtError, ErrorBadDataType);
  } else {

    /* set up new operator */
    od.op = optype;
    od.aMode = resultMode(r0);
    od.aVal = resultInteger(r0);
    od.bMode = resultMode(r1);
    od.bVal = resultInteger(r1);
    od.next = NULL;
    resultFree(r0);
    r0 = resultCreate(dtInstruction, &od);
  }

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opOpData(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_dat, envp));
}

static result *
opOpMove(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_mov, envp));
}

static result *
opOpAdd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_add, envp));
}

static result *
opOpSubtract(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_sub, envp));
}

static result *
opOpJump(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_jmp, envp));
}

static result *
opOpJumpZero(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_jmz, envp));
}

static result *
opOpJumpNonZero(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_jmn, envp));
}

static result *
opOpDecAndJumpNonZero(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_djn, envp));
}

static result *
opOpCompare(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_cmp, envp));
}

static result *
opOpSplit(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_spl, envp));
}

static result *
opOpSkipLessThan(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  return(opOpGeneric(argv[0], argv[1], op_slt, envp));
}

static result *
opOpJumpBack(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp = 0;
  opData *top = 0, *bottom;
  int jumpAmt = 7999;
  opData od;

  /* evaluate argument */
  rp = objectEval(argv[0], envp);
  if (!resultIsBlobPtr(rp)) {
    resultFree(rp);
    rp = resultCreate(dtError, ErrorBadDataType);
  } else {
    top = bottom = (opData *)resultBlobPtr(rp);

    /* set pointer to last statement */
    while (bottom->next) {
      bottom = bottom->next;
      jumpAmt--;
    }

    /* set up new operator */
    od.op = op_jmp;
    od.aMode = am_dir;
    od.aVal = jumpAmt;
    od.bMode = am_imm;
    od.bVal = 0;
    od.next = NULL;
    bottom->next = &od;

    /* set result type to void so blob doesn't get freed */
    resultSetType(rp, dtVoid);
    resultFree(rp);

    /* copy new data */
    rp = resultCreate(dtInstruction, top);
    bottom->next = NULL;
    opDataFree(top);
  }

  /* return the result */
  return(rp);
}

static result *
opOpJumpFrwd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp = 0;
  opData *top = 0, *bottom;
  int jumpAmt = 1;
  opData od;

  /* evaluate argument */
  rp = objectEval(argv[0], envp);
  if (!resultIsBlobPtr(rp)) {
    resultFree(rp);
    rp = resultCreate(dtError, ErrorBadDataType);
  } else {
    top = bottom = (opData *)resultBlobPtr(rp);

    /* set pointer to last statement */
    while (bottom->next) {
      bottom = bottom->next;
      jumpAmt++;
    }

    /* set up new operator */
    od.op = op_jmp;
    od.aMode = am_dir;
    od.aVal = jumpAmt;
    od.bMode = am_imm;
    od.bVal = 0;
    od.next = top;

    /* set result type to void so blob doesn't get freed */
    resultSetType(rp, dtVoid);
    resultFree(rp);

    /* set final result */
    rp = resultCreate(dtInstruction, &od);
    opDataFree(top);
  }

  /* return the result */
  return(rp);
}

static result *
opOpJumpZeroBack(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp = 0;
  amode_t mode;
  aval_t val;
  opData *top = 0, *bottom;
  int jumpAmt = 7999;
  opData od;

  /* evaluate final argument (because it should be simpler) */
  rp = objectEval(argv[1], envp);
  if (resultIsError(rp))
    return(rp);

  /* make sure final result is valid */
  if (!resultIsInteger(rp)) {
    resultFree(rp);
    rp = resultCreate(dtError, ErrorBadDataType);
  } else {

    /* save mode & val */
    mode = resultMode(rp);
    val = resultInteger(rp);

    /* evaluate first argument */
    rp = objectEval(argv[0], envp);
    if (!resultIsBlobPtr(rp)) {
      resultFree(rp);
      rp = resultCreate(dtError, ErrorBadDataType);
    } else {
      top = bottom = (opData *)resultBlobPtr(rp);

      /* set pointer to last statement */
      while (bottom->next) {
	bottom = bottom->next;
	jumpAmt--;
      }

      /* set result type to void so blob doesn't get freed */
      resultSetType(rp, dtVoid);
      resultFree(rp);

      /* set up new operator */
      od.op = op_jmz;
      od.aMode = am_dir;
      od.aVal = jumpAmt;
      od.bMode = mode;
      od.bVal = val;
      od.next = NULL;
      bottom->next = &od;

      rp = resultCreate(dtInstruction, top);
      bottom->next = NULL;
      opDataFree(top);
    }
  }

  /* return the result */
  return(rp);
}

static result *
opOpJumpZeroFrwd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp = 0;
  amode_t mode;
  aval_t val;
  opData *top = 0, *bottom;
  int jumpAmt = 1;
  opData od;

  /* evaluate final argument (because it should be simpler) */
  rp = objectEval(argv[1], envp);
  if (resultIsError(rp))
    return(rp);

  /* make sure final result is valid */
  if (!resultIsInteger(rp)) {
    resultFree(rp);
    rp = resultCreate(dtError, ErrorBadDataType);
  } else {

    /* save mode & val */
    mode = resultMode(rp);
    val = resultInteger(rp);

    /* evaluate argument */
    rp = objectEval(argv[0], envp);
    if (!resultIsBlobPtr(rp)) {
      resultFree(rp);
      rp = resultCreate(dtError, ErrorBadDataType);
    } else {
      top = bottom = (opData *)resultBlobPtr(rp);

      /* set pointer to last statement */
      while (bottom->next) {
	bottom = bottom->next;
	jumpAmt++;
      }

      /* set result type to void so blob doesn't get freed */
      resultSetType(rp, dtVoid);
      resultFree(rp);

      /* set up new operator */
      od.op = op_jmz;
      od.aMode = am_dir;
      od.aVal = jumpAmt;
      od.bMode = mode;
      od.bVal = val;
      od.next = top;

      /* set final result */
      rp = resultCreate(dtInstruction, &od);
      opDataFree(top);
    }
  }

  /* return the result */
  return(rp);
}

static result *
opOpJumpNonZeroBack(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp = 0;
  amode_t mode;
  aval_t val;
  opData *top = 0, *bottom;
  int jumpAmt = 7999;
  opData od;

  /* evaluate final argument (because it should be simpler) */
  rp = objectEval(argv[1], envp);
  if (resultIsError(rp))
    return(rp);

  /* make sure final result is valid */
  if (!resultIsInteger(rp)) {
    resultFree(rp);
    rp = resultCreate(dtError, ErrorBadDataType);
  } else {

    /* save mode & val */
    mode = resultMode(rp);
    val = resultInteger(rp);

    /* evaluate first argument */
    rp = objectEval(argv[0], envp);
    if (!resultIsBlobPtr(rp)) {
      resultFree(rp);
      rp = resultCreate(dtError, ErrorBadDataType);
    } else {
      top = bottom = (opData *)resultBlobPtr(rp);

      /* set pointer to last statement */
      while (bottom->next) {
	bottom = bottom->next;
	jumpAmt--;
      }

      /* set result type to void so blob doesn't get freed */
      resultSetType(rp, dtVoid);
      resultFree(rp);

      /* set up new operator */
      od.op = op_jmn;
      od.aMode = am_dir;
      od.aVal = jumpAmt;
      od.bMode = mode;
      od.bVal = val;
      od.next = NULL;
      bottom->next = &od;

      /* set final result */
      rp = resultCreate(dtInstruction, top);
      bottom->next = NULL;
      opDataFree(top);
    }
  }

  /* return the result */
  return(rp);
}

static result *
opOpJumpNonZeroFrwd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *rp = 0;
  amode_t mode;
  aval_t val;
  opData *top = 0, *bottom;
  int jumpAmt = 1;
  opData od;

  /* evaluate final argument (because it should be simpler) */
  rp = objectEval(argv[1], envp);
  if (resultIsError(rp))
    return(rp);

  /* make sure final result is valid */
  if (!resultIsInteger(rp)) {
    resultFree(rp);
    rp = resultCreate(dtError, ErrorBadDataType);
  } else {

    /* save mode & val */
    mode = resultMode(rp);
    val = resultInteger(rp);

    /* evaluate argument */
    rp = objectEval(argv[0], envp);
    if (!resultIsBlobPtr(rp)) {
      resultFree(rp);
      rp = resultCreate(dtError, ErrorBadDataType);
    } else {

      /* save new result */
      top = bottom = (opData *)resultBlobPtr(rp);

      /* set pointer to last statement */
      while (bottom->next) {
	bottom = bottom->next;
	jumpAmt++;
      }

      /* set result type to void so blob doesn't get freed */
      resultSetType(rp, dtVoid);
      resultFree(rp);

      /* set up new operator */
      od.op = op_jmn;
      od.aMode = am_dir;
      od.aVal = jumpAmt;
      od.bMode = mode;
      od.bVal = val;
      od.next = top;

      /* set final result */
      rp = resultCreate(dtInstruction, &od);
      opDataFree(top);
    }
  }

  /* return the result */
  return(rp);
}

static result *
opBlock(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  int i = 0;
  result *rp = 0;
  opData *top = 0, *bottom;

  /* evaluate arguments */
  do {
    if (rp)
      resultFree(rp);
    rp = objectEval(argv[i++], envp);
    if (resultIsBlobPtr(rp)) {

      /* save new result */
      if (!top)
        top = bottom = (opData *)resultBlobPtr(rp);
      else
        bottom->next = (opData *)resultBlobPtr(rp);

      /* set pointer to last statement */
      while (bottom->next)
        bottom = bottom->next;

      /* set result type to void so blob doesn't get freed */
      resultSetType(rp, dtVoid);

    } else if (!resultIsError(rp))
      printf("Block found result which is neither BlobPtr nor Error!\n");

  } while (i < argc);

  /* set final result */
  if (resultIsBlobPtr(rp))
    resultSetBlobPtr(rp, (blob *)top, dtInstruction);
  else if (top) {
    resultFree(rp);
    rp = resultCreate(dtInstruction, top);
    opDataFree(top);
  }

  /* return the result */
  return(rp);
}

static result *
opAssemble(argc, argv, xenvp)
int argc;
const object **argv;
void *xenvp;
{
  coreWarEnv *envp = (coreWarEnv *)xenvp;
  result *rp;
#ifdef NOISY
  charString *cstr;
#endif /* NOISY */
  opData *odp;
  program_t *prog;
  aval_t  cur_addr = 0, start = 0;

  if (envp->status == uninitialized) {

#ifdef NOISY
    cstr = charStringCreate();
    charStringSet(cstr, "ASSEMBLE Argument is ");
    objectToString(argv[0], cstr);
    charStringPrint(cstr);
    charStringFree(cstr);
#endif /* NOISY */

    prog = (program_t *)malloc(sizeof(program_t));
    if (!prog)
      rp = resultCreate(dtError, ErrorOutOfMemory);
    else {
      strcpy(prog->pname, "x Unknown");
      strcpy(prog->retaddr, "x Unknown");
      strcpy(prog->author, "x Unknown");
      strcpy(prog->pname, "Unknown");
      prog->fname[0] = 0;
      prog->listing = NULL;

      /* evaluate argument */
      rp = objectEval(argv[0], envp);
      if (resultIsBlobPtr(rp)) {

        /* find program starting address and length */
        odp = (opData *)resultBlobPtr(rp);
	while (odp != NULL) {
	  if (cur_addr++ == start && odp->op == op_dat)
	    start++;
	  odp = odp->next;
	}

	/* set up new program */
	odp = (opData *)resultBlobPtr(rp);
	prog->proglen = cur_addr;
	prog->startaddr = start;
	prog->listing = (core_el_t *)malloc(cur_addr * sizeof(core_el_t));
	if (!prog->listing) {
	  resultFree(rp);
	  rp = resultCreate(dtError, ErrorOutOfMemory);
	} else {
	  cur_addr = 0;
	  while (odp != NULL)  {
	    prog->listing[cur_addr].val1 = odp->aVal;
	    prog->listing[cur_addr].val2 = odp->bVal;
	    prog->listing[cur_addr].index = JTINDEX(odp->op, odp->aMode,
						    odp->bMode);
	    prog->listing[cur_addr].instr =
	      jmptable[prog->listing[cur_addr].index];
	    cur_addr++;
	    odp = odp->next;
	  }

	  /* make sure we didn't exceed the max program length */
	  if (cur_addr > maxprogsize)  {
	    resultFree(rp);
	    rp = resultCreate(dtError, ErrorProgramTooLong);
#ifdef NOISY
	  } else {
	    disasm(prog->listing, prog->proglen, prog->startaddr);
#endif /* NOISY */
	  }
	}
      }
    }

#ifdef NOISY
    if (resultIsError(rp)) {
      cstr = charStringCreate();
      charStringSet(cstr, "ASSEMBLE Result is ");
      resultToString(rp, cstr);
      charStringPrint(cstr);
      charStringFree(cstr);
    }
#endif /* NOISY */

    /* if we didn't hit an error, save new program */
    if (resultIsError(rp)) {
      envp->status = gotError;
      if (prog)
	progFree(prog);
    } else {
      envp->prog = prog;
      envp->status = gotProgram;
      resultFree(rp);
      rp = resultCreate(dtWarrior);
    }
  }

  /* return the result */
  return(rp);
}

static objectList *
corewarFunctions()
{
  objectList *list;
  operatorSrc *osp;
  int i;

#ifdef HANDLE_INCREMENT
  list = objectListCreate(35);
#else
  list = objectListCreate(34);
#endif /* HANDLE_INCREMENT */
  if (list) {

#ifdef USE_MATH_OPERATORS
    osp = complexOperatorSrcCreate("+", opAdd, dtInteger, 2, 2,
				   dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("-", opSubtract, dtInteger, 2, 2,
				   dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("*", opMultiply, dtInteger, 2, 2,
				   dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("/", opDivide, dtInteger, 2, 2,
				   dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
#endif /* USE_MATH_OPERATORS */

    osp = complexOperatorSrcCreate("#", opModeImmediate, dtImmediate, 1, 1,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("$", opModeDirect, dtDirect, 1, 1,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("@", opModeIndirect, dtIndirect, 1, 1,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("<", opModeDecrement, dtDecrement, 1, 1,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

#ifdef HANDLE_INCREMENT
    osp = complexOperatorSrcCreate(">", opModeIncrement, dtIncrement, 1, 1,
				   dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
#endif /* HANDLE_INCREMENT */

    osp = complexOperatorSrcCreate("MOV", opOpMove, dtInstruction, 2, 2,
				   dtAnyMode, dtDirect|dtIndirect|dtDecrement);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("ADD", opOpAdd, dtInstruction, 2, 2,
				   dtAnyMode, dtDirect|dtIndirect|dtDecrement);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("SUB", opOpSubtract, dtInstruction, 2, 2,
				   dtAnyMode, dtDirect|dtIndirect|dtDecrement);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("CMP", opOpCompare, dtInstruction, 2, 2,
				   dtAnyMode, dtDirect|dtIndirect|dtDecrement);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("SLT", opOpSkipLessThan, dtInstruction, 2, 2,
				   dtAnyMode, dtDirect|dtIndirect|dtDecrement);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMP", opOpJump, dtInstruction, 2, 2,
				   dtDirect|dtIndirect|dtDecrement, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMZ", opOpJumpZero, dtInstruction, 2, 2,
				   dtDirect|dtIndirect|dtDecrement, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMN", opOpJumpNonZero, dtInstruction, 2, 2,
				   dtDirect|dtIndirect|dtDecrement, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("DJN", opOpDecAndJumpNonZero,
				   dtInstruction, 2, 2,
				   dtDirect|dtIndirect|dtDecrement, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("SPL", opOpSplit, dtInstruction, 2, 2,
				   dtDirect|dtIndirect|dtDecrement, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("DAT", opOpData, dtInstruction, 2, 2,
				   dtImmediate|dtDecrement,
				   dtImmediate|dtDecrement);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMPBack", opOpJumpBack,
				   dtInstruction, 1, 1,
				   dtInstruction|dtMultiInstr);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMPFrwd", opOpJumpFrwd,
				   dtInstruction, 1, 1,
				   dtInstruction|dtMultiInstr);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMZBack", opOpJumpZeroBack,
				   dtInstruction, 2, 2,
				   dtInstruction|dtMultiInstr, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMZFrwd", opOpJumpZeroFrwd,
				   dtInstruction, 2, 2,
				   dtInstruction|dtMultiInstr, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMNBack", opOpJumpNonZeroBack,
				   dtInstruction, 2, 2,
				   dtInstruction|dtMultiInstr, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("JMNFrwd", opOpJumpNonZeroFrwd,
				   dtInstruction, 2, 2,
				   dtInstruction|dtMultiInstr, dtAnyMode);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    for (i = 0; i < 4; i++) {
      osp = complexOperatorSrcCreate("progn2", opBlock, dtMultiInstr, 2, 2,
				     dtInstruction|dtMultiInstr,
				     dtInstruction|dtMultiInstr);
      if (objectListAdd(list, osp)) {
        operatorSrcFree(osp);
        objectListFree(list);
        return(0);
      }
    }

    for (i = 0; i < 4; i++) {
      osp = complexOperatorSrcCreate("progn3", opBlock, dtMultiInstr, 3, 3,
				     dtInstruction|dtMultiInstr,
				     dtInstruction|dtMultiInstr,
				     dtInstruction|dtMultiInstr);
      if (objectListAdd(list, osp)) {
        operatorSrcFree(osp);
        objectListFree(list);
        return(0);
      }
    }

    osp = complexOperatorSrcCreate("assemble", opAssemble, dtWarrior, 1, 1,
				   dtMultiInstr);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

void *
corewarCaseInitialize(progNum, fc)
int progNum;
int fc;
{
  env[progNum].num = progNum;
  env[progNum].status = uninitialized;
  env[progNum].prog = NULL;

  return((void *)&env[progNum]);
}

static int
corewarCaseTerminate(rp, xenvp, fc)
result *rp;
void *xenvp;
int fc;
{
#ifndef DEBUG_COREWAR
  coreWarEnv *envp = (coreWarEnv *)xenvp;
  int i;

  /* if we've assembled all the programs... */
  if (envp->num == (NUM_WARRIORS-1)) {

    /* make sure we got more that one valid program */
    numProgs = 0;
    for (i = 0; i < NUM_WARRIORS; i++)
      if (env[i].status == gotProgram) {
	progList[numProgs].envp = &env[i];
	progList[numProgs].prog = env[i].prog;
	progList[numProgs].wins = 0;
	numProgs++;
      } else
	env[i].wins = 0;

    /* run it if we've got enough for a battle */
    if (numProgs > 1) {
      for (i = 0;  i < rounds;  ++i)
	fight(progList, numProgs);
    } else {
      for (i = 0; i < numProgs; i++)
        progList[i].wins += (NUM_WARRIORS - numProgs) * rounds;
    }

    /* save number of wins */
    for (i = 0; i < numProgs; i++) {
      progList[i].envp->wins = progList[i].wins;
      progList[i].envp->status = gotResults;
    }

    /* all done */
    return(1);
  }

  /* not yet */
  return(0);
#else /* DEBUG_COREWAR */
  return(1);
#endif /* !DEBUG_COREWAR */
}

void
corewarCaseFitness(rp, fc, hitp, rawp, stdp, xenvp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *xenvp;
{
  coreWarEnv *envp = (coreWarEnv *)xenvp;

  if (envp->status != gotProgram) {
    if (envp->status != gotResults)
      *stdp += NUM_WARRIORS * 2 * rounds;
    else {
      *hitp += envp->wins;
      *stdp += (NUM_WARRIORS * rounds) - envp->wins;
#ifdef NOISY
      printf("HITS=%d, STD=%6.1f\n", *hitp, *stdp);
/*
      disasm(envp->prog->listing, envp->prog->proglen, envp->prog->startaddr);
*/
#endif
    }
  }

  /* free allocated memory */
  if (envp->prog)
    progFree(envp->prog);
}

int
corewarTerminateRun(progNum, hits, raw, std)
int progNum;
int hits;
double raw, std;
{
  return(0);
}

void
appInitialize(gp, pop, progNum)
void *gp;
population *pop;
int progNum;
{
  objectList *tList, *fList;

  datatypeMakeAlias(dtImmediate, dtInteger);
  datatypeMakeAlias(dtDirect, dtInteger);
  datatypeMakeAlias(dtIndirect, dtInteger);
  datatypeMakeAlias(dtDecrement, dtInteger);
#ifdef HANDLE_INCREMENT
  datatypeMakeAlias(dtIncrement, dtInteger);
#endif /* HANDLE_INCREMENT */
  datatypeMakeAlias(dtInstruction, dtBlob);
  datatypeMakeAlias(dtMultiInstr, dtBlob);
  datatypeMakeAlias(dtWarrior, dtVoid);

  errorCodeSetMsgPtr(ErrorDivideByZero, &MsgDivideByZero);
  errorCodeSetMsgPtr(ErrorOutOfMemory, &MsgOutOfMemory);
  errorCodeSetMsgPtr(ErrorProgramTooLong, &MsgProgramTooLong);

  /* build terminal and function lists */
  tList = corewarTerminals();
  fList = corewarFunctions();

  /* set global variables */
/*  globalSetRandomNumberSeed(gp, 12345678); */
  globalClearSingleEnvironmentFlag(gp);
  globalClearFinalResultFlag(gp);
  globalSetNumberOfPopulations(gp, 2);

  /* set app-specific variables */
  populationSetTerminalList(pop, tList);
  populationSetFunctionList(pop, fList);
  populationSetReturnTypes(pop, dtWarrior);
  populationSetFitnessCases(pop, 1);
  populationSetInitialTreeDepth(pop, 16);

  /* set global functions */
  globalSetEvalFunc(gp, globalCoEval);

  /* set app-specific functions */
  populationSetCaseInitializeFunc(pop, corewarCaseInitialize);
  populationSetCaseTerminateFunc(pop, corewarCaseTerminate);
  populationSetCaseFitnessFunc(pop, corewarCaseFitness);
  populationSetTerminateRunFunc(pop, corewarTerminateRun);

  /* initialize everything else */
  if (!KotHInitialized)
    initKotH();
}

#ifdef DEBUG_COREWAR

#include "program.h"

int main P((int, char *[]));

int
main(argc, argv)
int argc;
char *argv[];
{
  global *gp;
  population *pop;
  void *envp;
  program *pp;
  const char *src = "(assemble (ADD (< (+ 1 2)) (< (- 7 4))))";
  int i = 1;

  gp = globalCreate(appInitialize);
  pop = globalPopulation(gp, 0);

  do {
    if (argc != 1)
      src = argv[i];
    pp = programParse(populationTerminalList(pop), populationFunctionList(pop),
		      src);
    if (pp == 0) {
      printf("Bad program '%s'!\n", src);
    } else {
      envp = populationCaseInitializeFunc(pop, 0, 0);
      globalEvalProgram(gp, 0, pp);
      programDump(pp, 1);
      programFree(pp);
    }
  } while (++i < argc);

  globalFree(gp);
  exit(0);
}
#endif /* DEBUG_COREWAR */
