#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include <math.h>
#include "geppetto.h"
#include "proto.h"

#ifndef HUGE_VAL
#define HUGE_VAL	HUGE
#endif /* !HUGE_VAL */

#define dtTwoVal	dtUserDef0
#define dtThreeVal	dtUserDef1

typedef struct blob2 {
  int val[2];
} blob2;

typedef struct blob3 {
  int val[3];
} blob3;

blob *
blobCreate(dtype)
datatype dtype;
{
  blob2 *b2p;
  blob3 *b3p;

  if (dtype == dtTwoVal) {
    b2p = (blob2 *)malloc(sizeof(blob2));
    if (b2p) {
      b2p->val[0] = irndrng(0, 5);
      b2p->val[1] = irndrng(2, 6);
      return((blob *)b2p);
    }
  } else if (dtype == dtThreeVal) {
    b3p = (blob3 *)malloc(sizeof(blob3));
    if (b3p) {
      b3p->val[0] = irndrng(0, 2);
      b3p->val[1] = irndrng(3, 5);
      b3p->val[2] = irndrng(6, 8);
      return((blob *)b3p);
    }
  }

  return(0);
}

blob *
blobCopy(bbp, dtype)
const blob *bbp;
datatype dtype;
{
  const blob2 *b2p;
  blob2 *nb2p;
  const blob3 *b3p;
  blob3 *nb3p;

  if (dtype == dtTwoVal) {
    b2p = (const blob2 *)bbp;
    nb2p = (blob2 *)malloc(sizeof(blob2));
    if (nb2p) {
      nb2p->val[0] = b2p->val[0];
      nb2p->val[1] = b2p->val[1];
      return((blob *)nb2p);
    }
  } else if (dtype == dtThreeVal) {
    b3p = (const blob3 *)bbp;
    nb3p = (blob3 *)malloc(sizeof(blob3));
    if (nb3p) {
      nb3p->val[0] = b3p->val[0];
      nb3p->val[1] = b3p->val[1];
      nb3p->val[2] = b3p->val[2];
      return((blob *)nb3p);
    }
  }

  return(0);
}

int
blobCompare(bbp0, bbp1, dtype)
const blob *bbp0, *bbp1;
datatype dtype;
{
  int cmp;

  if (dtype == dtTwoVal)
    cmp = ((((const blob2 *)bbp0)->val[0] == ((const blob2 *)bbp1)->val[0]) &&
	   (((const blob2 *)bbp0)->val[1] == ((const blob2 *)bbp1)->val[1]));
  else if (dtype == dtThreeVal) {
    cmp = ((((const blob3 *)bbp0)->val[0] == ((const blob3 *)bbp1)->val[0]) &&
	   (((const blob3 *)bbp0)->val[1] == ((const blob3 *)bbp1)->val[1]) &&
	   (((const blob3 *)bbp0)->val[2] == ((const blob3 *)bbp1)->val[2]));
  } else
    cmp = 0;

  return(cmp);
}

blob *
blobParse(sp, dtp)
const char **sp;
datatype *dtp;
{
  fprintf(stderr, "blobParse() called\n");
  return(0);
}

int
blobToString(bbp, dtype, cstr)
const blob *bbp;
datatype dtype;
charString *cstr;
{
  const blob2 *b2p;
  const blob3 *b3p;
  char buffer[32];

  if (dtype == dtTwoVal) {
    b2p = (const blob2 *)bbp;
    sprintf(buffer, "2Val %d,%d", b2p->val[0], b2p->val[1]);
  } else if (dtype == dtThreeVal) {
    b3p = (const blob3 *)bbp;
    sprintf(buffer, "3Val %d,%d,%d", b3p->val[0], b3p->val[1], b3p->val[2]);
  } else
    strcpy(buffer, "UNKNOWN");

  return(charStringCatenate(cstr, buffer));
}

void
blobFree(bbp, dtype)
blob *bbp;
datatype dtype;
{
  free(bbp);
}

static objectList *symregTerminals P((NOARGS));
static result *opAdd P((int argc, const object **, void *));
static result *opSubtract P((int argc, const object **, void *));
static result *opMultiply P((int argc, const object **, void *));
static result *opDivide P((int argc, const object **, void *));
static result *opBlob2x3 P((int argc, const object **, void *));
static result *opBlob3x2 P((int argc, const object **, void *));
static objectList *symregFunctions P((NOARGS));
static void *symregCaseInitialize P((int, int));
static void symregCaseFitness P((result *, int, int *, double *, double *,
				 void *));
static int symregTerminateRun P((int, int, double, double));

const int fitnessCases = 20;

#define ErrorDivideByZero	ErrorUserDefined+0
const char *MsgDivideByZero =	"Divide by Zero";

static struct srvalues {
  int x;
  int answer;
} fitnessCase[20];

static int x;

static objectList *
symregTerminals()
{
  objectList *list;
  variable *vp;
  constantSrc *csp;

  list = objectListCreate(3);
  if (list) {

    vp = variableCreate(dtInteger, "X", &x);
    if (objectListAdd(list, vp)) {
      variableFree(vp);
      objectListFree(list);
      return(0);
    }

    csp = blobSrcCreate(dtTwoVal);
    if (objectListAdd(list, csp)) {
      constantSrcFree(csp);
      objectListFree(list);
      return(0);
    }

    csp = blobSrcCreate(dtThreeVal);
    if (objectListAdd(list, csp)) {
      constantSrcFree(csp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static result *
opAdd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else
    resultSetFloat(r0, resultInteger(r0) + resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opSubtract(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else
    resultSetInteger(r0, resultInteger(r0) - resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opMultiply(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else
    resultSetInteger(r0, resultInteger(r0) * resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opDivide(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* make sure both results are valid */
  if (!resultIsInteger(r0) || !resultIsInteger(r1))
    resultSetError(r0, ErrorBadDataType);
  else if (resultInteger(r1) == 0)
    resultSetError(r0, ErrorDivideByZero);
  else
    resultSetInteger(r0, resultInteger(r0) / resultInteger(r1));

  /* return the result */
  resultFree(r1);
  return(r0);
}

static result *
opBlob2x3(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;
  blob2 *b2p;
  blob3 *b3p;
  int val;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* do the operation */
  if (objectDataType(r0) != dtTwoVal || objectDataType(r1) != dtThreeVal)
   resultSetError(r0, ErrorBadDataType);
  else {
    b2p = (blob2 *)resultBlobPtr(r0);
    b3p = (blob3 *)resultBlobPtr(r1);
    val = (b2p->val[0] * b2p->val[1]) -
      (b3p->val[0] + b3p->val[1] + b3p->val[2]);
  }

  /* clean up and return */
  resultFree(r0);
  resultFree(r1);
  return(resultCreate(dtInteger, val));
}

static result *
opBlob3x2(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  result *r0, *r1;
  blob3 *b3p;
  blob2 *b2p;
  int val;

  /* evaluate arguments */
  r0 = objectEval(argv[0], envp);
  if (resultIsError(r0))
    return(r0);
  r1 = objectEval(argv[1], envp);
  if (resultIsError(r1)) {
    resultFree(r0);
    return(r1);
  }

  /* do the operation */
  if (objectDataType(r0) != dtThreeVal || objectDataType(r1) != dtTwoVal)
   resultSetError(r0, ErrorBadDataType);
  else {
    b3p = (blob3 *)resultBlobPtr(r0);
    b2p = (blob2 *)resultBlobPtr(r1);
    val = (b3p->val[0] - b3p->val[1] - b3p->val[2]) *
      (b2p->val[0] + b2p->val[1]);
  }

  /* clean up and return */
  resultFree(r0);
  resultFree(r1);
  return(resultCreate(dtInteger, val));
}

static objectList *
symregFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(6);
  if (list) {

    osp = complexOperatorSrcCreate("+", opAdd, dtInteger,
				   2, 2, dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("-", opSubtract, dtInteger,
				   2, 2, dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("*", opMultiply, dtInteger,
				   2, 2, dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("/", opDivide, dtInteger,
				   2, 2, dtInteger, dtInteger);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("2x3", opBlob2x3, dtInteger,
				   2, 2, dtTwoVal, dtThreeVal);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("3x2", opBlob3x2, dtInteger,
				   2, 2, dtThreeVal, dtTwoVal);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

  }

  return(list);
}

static void *
symregCaseInitialize(popNum, fc)
int popNum;
int fc;
{
  x = fitnessCase[fc].x;
  return(0);
}

static void
symregCaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  int fit;

  if (resultIsError(rp)) {
    *stdp = HUGE_VAL;
  } else {

    if ((resultInteger(rp) > fitnessCase[fc].answer - 1000) &&
	(resultInteger(rp) < fitnessCase[fc].answer + 1000))
      *hitp += 1;

    if (*stdp != HUGE_VAL) {
      fit = resultInteger(rp) - fitnessCase[fc].answer;
      if (fit < 0)
        fit = -fit;
      *stdp += fit;
    }
#ifdef DEBUG_BLOBSAMPLE
  printf("Case %d, X=%d: Computed=%d, Actual=%d\n", fc, x, resultInteger(rp),
	 fitnessCase[fc].answer);
  printf("     Absolute Difference=%d, Std Fitness now %f\n", fit, *stdp);
#endif /* DEBUG_BLOBSAMPLE */
  }
}

static int
symregTerminateRun(popNum, hits, raw, std)
int popNum;
int hits;
double raw;
double std;
{
  return(std == 0);
}

void
appInitialize(gp, pop, popNum)
void *gp;
population *pop;
int popNum;
{
  int i;

  /* set up blob types */
  datatypeMakeAlias(dtTwoVal, dtBlob);
  datatypeMakeAlias(dtThreeVal, dtBlob);

  /* initialize fitness cases */
  for (i = 0; i < fitnessCases; i++) {
    x = fitnessCase[i].x = irndrng(1, 255);
    fitnessCase[i].answer = x*x*x*x + x*x*x + x*x + x;
  }

  errorCodeSetMsgPtr(ErrorDivideByZero, &MsgDivideByZero);

  populationSetFitnessCases(pop, fitnessCases);
  populationSetTerminalList(pop, symregTerminals());
  populationSetFunctionList(pop, symregFunctions());
  populationSetReturnTypes(pop, dtInteger);
  populationSetCaseInitializeFunc(pop, symregCaseInitialize);
  populationSetCaseFitnessFunc(pop, symregCaseFitness);
  populationSetTerminateRunFunc(pop, symregTerminateRun);
}

#ifdef DEBUG_BLOBSAMPLE

const char *proglist[] = {
  "(+ X (* X (+ X (* X (+ X (* X X))))))",
  0
};

#endif /* DEBUG_BLOBSAMPLE */
