/*
 * To get the Los Altos Hills version, define LOS_ALTOS_HILLS
 */

#include <string.h>
#ifdef INCLUDE_MALLOC_H
#include <malloc.h>
#endif
#include "geppetto.h"
#include "proto.h"

static char **arenaNew P((NOARGS));
static void arenaInitialize P((NOARGS));
#ifdef DEBUG_ANT
static void arenaDump P((NOARGS));
#endif /* DEBUG_ANT */
static result *opRight P((int, const object **, void *));
static result *opLeft P((int, const object **, void *));
static result *opMove P((int, const object **, void *));
static result *opIfFoodAhead P((int, const object **, void *));
static result *opBlock P((int, const object **, void *));
static objectList *antTerminals P((NOARGS));
static objectList *antFunctions P((NOARGS));
static int antCaseTerminate P((result *, void *, int));
static void antCaseFitness P((result *, int, int *, double *, double *,
			      void *));
static int antTerminateRun P((int, int, double, double));
static void *antCaseInit P((int, int));
static void antDestructor P((NOARGS));

typedef enum antDirection {
  north, east, south, west,
  minDirection=north, maxDirection=west,
} antDirection;

typedef struct antInfo {
  int ticks, hits;
  antDirection direction;
  int posX, posY;
  char **arena;
} antInfo;

static antInfo antEnv;
static char **arena = 0;

#define foodAt(ra, ax, ay)	((ra)[ay][ax] == '*')
#define antPath(ra, ax, ay)	((ra)[ay][ax] = 'x')
#define dirChar(d)		((d) == north ? 'N' : \
				 ((d) == east ? 'E' : \
				  ((d) == south ? 'S' : \
				   ((d) == west ? 'W' : '?'))))

#ifdef JOHN_MUIR

#define minX			0
#define maxX			31
#define minY			0
#define maxY			31
#define maxFood			89
#define maxTicks		600

static const char *arenaTemplate[32] = {
  "o**********                     ",
  "          *                     ",
  "          *                     ",
  "          *                     ",
  "          *   o*                ",
  "****      *   o         o*******",
  "   *      *   o         *       ",
  "   *      *   *         *       ",
  "   *      *o*oo         *       ",
  "   *      *o            *       ",
  "   ********o            *       ",
  "           *      o*****o       ",
  "           ooo*o  *             ",
  "               o  *             ",
  "               *  *             ",
  "           o*ooo  *             ",
  "           o      *             ",
  "           *      *             ",
  "       o*ooo      o             ",
  "       o          o             ",
  "       o          *             ",
  "       *          *             ",
  "    o*oo          *             ",
  "    o             *             ",
  "    *             *             ",
  "    *             *             ",
  "    *             o             ",
  "    *oo****o******o             ",
  "                                ",
  "                                ",
  "                                ",
  "                                ",
};

static void
arenaInitialize()
{
  int i;

  for (i = 0; i < 32; i++)
    strcpy(arena[i], arenaTemplate[i]);
}

#else /* !JOHN_MUIR */
#ifdef SANTA_FE

#define minX			0
#define maxX			31
#define minY			0
#define maxY			31
#define maxFood			89
#define maxTicks		600

static const char *arenaTemplate[32] = {
  "o***                            ",
  "   *                            ",
  "   *                    o***oo  ",
  "   *                    *    *  ",
  "   *                    *    *  ",
  "   ****o*****       o**oo    o  ",
  "            *       o        *  ",
  "            *       *        o  ",
  "            *       *        o  ",
  "            *       *        *  ",
  "            o       *        o  ",
  "            *       o        o  ",
  "            *       o        *  ",
  "            *       *        o  ",
  "            *       *  ooo***o  ",
  "            o   o*ooo  *        ",
  "            o   o      o        ",
  "            *   o      o        ",
  "            *   *      o*ooo    ",
  "            *   *          *    ",
  "            *   *          o    ",
  "            *   *          o    ",
  "            *   o      ooo*o    ",
  "            *   o      *        ",
  " oo**oo*****o   *               ",
  " *              *               ",
  " *              *               ",
  " *     o*******oo               ",
  " *     *                        ",
  " o     *                        ",
  " o****oo                        ",
  "                                ",
};

static void
arenaInitialize()
{
  int i;

  for (i = 0; i < 32; i++)
    strcpy(arena[i], arenaTemplate[i]);
}

#else /* !SANTA_FE */
#ifdef LOS_ALTOS_HILLS

#define minX			0
#define maxX			99
#define minY			0
#define maxY			99
#define maxFood			157
#define maxTicks		3000

static const char *arenaTemplate[68] = {
  "o***                                          ",
  "   *                                          ",
  "   *                                  o***oo  ",
  "   *                                  *    *  ",
  "   *                                  *    *  ",
  "   ****o*****           o************oo    o  ",
  "            *           o                  *  ",
  "            *           *                  o  ",
  "            *           *                  o  ",
  "            *           *                  *  ",
  "            o           *                  o  ",
  "            *           o                  o  ",
  "            *           o                  *  ",
  "            *           *                  o  ",
  "            *           *            ooo***o  ",
  "            o       o*ooo            *        ",
  "            o       o                o        ",
  "            *       o                o        ",
  "            *       *                o*ooo    ",
  "            *       *                    *    ",
  "            *       *                    o    ",
  "            *       *                    o    ",
  "            *       o                ooo*o    ",
  "            *       o                *        ",
  " oo**oo*****o       *                *        ",
  " *                  *                *        ",
  " *                  *                *        ",
  " *                  *                *        ",
  " *                  *                *        ",
  " *                  *                *        ",
  " *     o*****oooo**oo                *        ",
  " *     *                             *        ",
  " o     *                             *        ",
  " o****oo                             *        ",
  "                                   *o*        ",
  "                                   *          ",
  "                                   *          ",
  "                                   *          ",
  "                                   *          ",
  "                                   *          ",
  "                                   *          ",
  "                                   *          ",
  "                              ******          ",
  "                              o               ",
  "                              *               ",
  "                              *               ",
  "                              *               ",
  "                              *               ",
  "                              *               ",
  "                              *               ",
  "                            *oo               ",
  "                            *                 ",
  "                            *                 ",
  "                            *                 ",
  "                            *                 ",
  "                            *                 ",
  "                      o******                 ",
  "                      o                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                      *                       ",
  "                                              ",
};

static void
arenaInitialize()
{
  int i;
  const char *spc54 = "                                                      ";
  const char *spc100 = "                                                                                                    ";

  for (i = 0; i < 68; i++) {
    strcpy(arena[i], arenaTemplate[i]);
    strcpy(&(arena[i][46]), spc54);
  }
  for (i = 68; i < 100; i++)
    strcpy(arena[i], spc100);
}

#endif /* LOS_ALTOS_HILLS */
#endif /* SANTA_FE */
#endif /* JOHN_MUIR */

static char **
arenaNew()
{
  char **cpp;
  char *cp;
  int i;

  cpp = (char **)malloc((sizeof(char)*(maxX+2) + sizeof(char *))*(maxY+1));
  cp = (char *)cpp + sizeof(char *)*(maxY+1);
  for (i = 0; i <= maxY; i++) {
    if (i > 0)
      cp += maxX+2;
    cpp[i] = cp;
  }

  return(cpp);
}

#ifdef DEBUG_ANT
static void
arenaDump()
{
  int i;

  for (i = 0; i <= maxY; i++)
    printf("%s\n", arena[i]);
}
#endif /* DEBUG_ANT */

static result *
opRight(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  antInfo *antp = (antInfo *)envp;

  /* take a tick */
  antp->ticks++;

  /* turn right */
  antp->direction++;
  if ((antp->direction < minDirection) || (antp->direction > maxDirection))
    antp->direction = minDirection;

#ifdef NOISY
  printf("right\n");
#endif /* NOISY */

  return(resultCreate(dtVoid));
}

static result *
opLeft(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  antInfo *antp = (antInfo *)envp;

  /* take a tick */
  antp->ticks++;

  /* turn left */
  antp->direction--;
  if ((antp->direction < minDirection) || (antp->direction > maxDirection))
    antp->direction = maxDirection;

#ifdef NOISY
  printf("left\n");
#endif /* NOISY */

  return(resultCreate(dtVoid));
}

static result *
opMove(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  antInfo *antp = (antInfo *)envp;
#ifdef NOISY
  int gotFood = 0;
#endif /* NOISY */

  /* take a tick */
  antp->ticks++;

  /* move in the proper direction */
  switch (antp->direction) {
  case west:
    antp->posX--;
    if (antp->posX < minX)
      antp->posX = maxX;
    break;
  case east:
    antp->posX++;
    if (antp->posX > maxX)
      antp->posX = minX;
    break;
  case north:
    antp->posY--;
    if (antp->posY < minY)
      antp->posY = maxY;
    break;
  case south:
    antp->posY++;
    if (antp->posY > maxY)
      antp->posY = minY;
    break;
  }

  /* see if there's any food here */
  if (foodAt(antp->arena, antp->posX, antp->posY)) {
    antp->hits++;
#ifdef NOISY
    gotFood = 1;
#endif /* NOISY */
  }
  antPath(antp->arena, antp->posX, antp->posY);

#ifdef NOISY
  printf("move\t");
  if (gotFood)
    printf("%d", antp->hits);
  printf("\t%d\n", antp->ticks);
#endif /* NOISY */

  return(resultCreate(dtVoid));
}

static result *
opIfFoodAhead(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  antInfo *antp = (antInfo *)envp;
  int posX, posY;

  posX = antp->posX;
  posY = antp->posY;

#ifdef EXTRA_NOISY
  printf("ifFoodAhead(%d,%d)->%c<%d>", posX, posY, dirChar(antp->direction),
	 antp->direction);
#endif /* EXTRA_NOISY */

  /* look in the proper direction */
  switch (antp->direction) {
  case west:
    posX--;
    if (posX < minX)
      posX = maxX;
    break;
  case east:
    posX++;
    if (posX > maxX)
      posX = minX;
    break;
  case north:
    posY--;
    if (posY < minY)
      posY = maxY;
    break;
  case south:
    posY++;
    if (posY > maxY)
      posY = minY;
    break;
  }

#ifdef EXTRA_NOISY
  printf(" : (%d,%d)%c\n", posX, posY,
	 (foodAt(antp->arena, posX, posY) ? '*' : 'o'));
#endif /* EXTRA_NOISY */

  /* evaluate the proper subexpression */
  return(foodAt(antp->arena, posX, posY) ? objectEval(argv[0], envp) :
	 objectEval(argv[1], envp));
}

static result *
opBlock(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  int i;
  result *r;


  /* evaluate arguments */
  i = 0;
  r = 0;
  do {
    if (r)
      resultFree(r);
    r = objectEval(argv[i++], envp);
  } while (!resultIsError(r) && (i < argc));

  /* return the result */
  return(r);
}

static objectList *
antTerminals()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(3);
  if (list) {

    osp = complexOperatorSrcCreate("right", opRight, dtVoid, 0, 0);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("left", opLeft, dtVoid, 0, 0);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("move", opMove, dtVoid, 0, 0);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
  }

  return(list);
}

static objectList *
antFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(4);
  if (list) {

  osp = complexOperatorSrcCreate("ifFoodAhead", opIfFoodAhead, dtVoid, 2, 2,
				 dtVoid, dtVoid);
  if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("progn2", opBlock, dtVoid, 2, 2, dtVoid,
				   dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

    osp = complexOperatorSrcCreate("progn3", opBlock, dtVoid, 3, 3, dtVoid,
				   dtVoid, dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }

#ifdef LOS_ALTOS_HILLS
    osp = complexOperatorSrcCreate("progn4", opBlock, dtVoid, 4, 4, dtVoid,
				   dtVoid, dtVoid, dtVoid);
    if (objectListAdd(list, osp)) {
      operatorSrcFree(osp);
      objectListFree(list);
      return(0);
    }
#endif /* LOS_ALTOS_HILLS */
  }

  return(list);
}

static void *
antCaseInit(popNum, fc)
int popNum;
int fc;
{
  /* (re-)initialize environment */
  arenaInitialize();
  antEnv.ticks = 0;
  antEnv.hits = 0;
  antEnv.direction = east;
  antEnv.posX = antEnv.posY = 0;
  antEnv.arena = arena;

  return((void *)&antEnv);
}

static int
antCaseTerminate(rp, envp, fc)
result *rp;
void *envp;
int fc;
{
  antInfo *aip = (antInfo *)envp;

#ifdef NOISY
  printf("------\n");
#endif /* NOISY */

  return((aip->hits >= maxFood) || (aip->ticks >= maxTicks));
}

static void
antCaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  antInfo *aip = (antInfo *)envp;

#ifdef DEBUG_ANT
{
  charString *cstr;

  cstr = charStringCreate();
  charStringSet(cstr, "Result is ");
  resultToString(rp, cstr);
  charStringPrint(cstr);
  charStringFree(cstr);

  arenaDump();

  printf("Ant Info: hits=%d, ticks=%d, posX=%d, posY=%d\n", aip->hits,
	 aip->ticks, aip->posX, aip->posY);
}
#endif /* DEBUG_ANT */

  *rawp = *hitp = aip->hits;
  if (resultIsError(rp))
    *stdp = maxFood + 1;
  else
    *stdp = maxFood - aip->hits;
}

static int
antTerminateRun(popNum, hits, raw, std)
int popNum;
int hits;
double raw;
double std;
{
  return(hits == maxFood);
}

static void
antDestructor()
{
  free(arena);
}

void
appInitialize(gp, pop, popNum)
void *gp;
population *pop;
int popNum;
{
#ifdef LOS_ALTOS_HILLS

  /* tweak some things */
  globalSetPopulationSize(gp, 2000);
  populationSetMaximumLoops(pop, 3000);
#endif /* LOS_ALTOS_HILLS */

  /* build a prototype arena for the ants to play in */
  arena = arenaNew();

  /* describe the problem to Geppetto */
  populationSetFitnessCases(pop, 1);
  populationSetTerminalList(pop, antTerminals());
  populationSetFunctionList(pop, antFunctions());
  populationSetCaseInitializeFunc(pop, antCaseInit);
  populationSetCaseTerminateFunc(pop, antCaseTerminate);
  populationSetCaseFitnessFunc(pop, antCaseFitness);
  populationSetTerminateRunFunc(pop, antTerminateRun);
  populationSetDestructorFunc(pop, antDestructor);
}

#ifdef DEBUG_ANT

const char *proglist[] = {
#if defined(JOHN_MUIR) || defined(SANTA_FE)
	"(ifFoodAhead (move) (progn2 (left) (ifFoodAhead (move) (progn3 (left) (left) (ifFoodAhead (move) (progn2 (left) (move)))))))",
	"(ifFoodAhead (move) (progn2 (right) (ifFoodAhead (move) (progn3 (right) (right) (ifFoodAhead (move) (progn2 (right) (move)))))))",
	"(ifFoodAhead (move) (progn3 (left) (progn2 (ifFoodAhead (move) (right)) (progn2 (right)  (progn2 (left) (right)))) (progn2 (ifFoodAhead (move) (left)) (move))))",
#else /* (JOHN_MUIR || SANTA_FE) */
#ifdef LOS_ALTOS_HILLS
	"(progn4 (ifFoodAhead (progn2 (progn3 (move) (progn2 (move) (move)) (right)) (ifFoodAhead (move) (ifFoodAhead (ifFoodAhead (left) (left)) (progn4 (progn2 (ifFoodAhead (move) (right)) (move)) (right) (move) (move))))) (progn4 (progn2 (ifFoodAhead (move) (right)) (move)) (right) (move) (move))) (ifFoodAhead (ifFoodAhead (move) (ifFoodAhead (ifFoodAhead (move) (left)) (ifFoodAhead (left) (right)))) (ifFoodAhead (left) (right))) (progn2 (progn3 (move) (move) (right)) (ifFoodAhead (progn2 (progn3 (move) (progn2 (move) (move)) (right)) (ifFoodAhead (ifFoodAhead (move) (move)) (move))) (move))) (move))",
#endif /* LOS_ALTOS_HILLS */
#endif /* JOHN_MUIR || SANTA_FE */
	0
};
#endif /* DEBUG_ANT */
