/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <string.h>
#include <malloc.h>
#include "geppetto.h"
#include "population.h"
#include "proto.h"

static const char *skipName P((FILE *, charString *, const char *));
static int readCreationMethod P((FILE *, charString *,
				 programCreationMethod *));
static int readSelectionMethod P((FILE *, charString *,
				  parentSelectionMethod *));
static int readBoolean P((FILE *, charString *, const char *, int *));
static int readInteger P((FILE *, charString *, const char *, int *));
static int readDouble P((FILE *, charString *, const char *, double *));
static population *readCheckpointFile P((const interface *, int *));
static void argPrint P((const char *));
static population *processArgs P((int, char *[], const interface *));
static void dumpParams P((NOARGS));
static int safeRename P((const char *, const char *));
static int writeSelectionMethod P((FILE *, parentSelectionMethod));
static int writeCreationMethod P((FILE *, programCreationMethod));
static int writeCheckpointFile P((const population *, int));
static void loop P((population *, interface *));
int main P((int, char *[]));

char *progname;

/*
 * Users might want to customize these values for their application
 */
int alwaysEval = 0;
programCreationMethod creator = pcmRamped;
int maxTotalDepth = 17;
char *checkptName = 0;
int generations = 50;
int initialDepth = 6;
int maxMutateDepth = 4;
int populationSize = 500;
int randomSeed;
parentSelectionMethod selector = psmFitnessProportionate;
int countUnique = 0;
int verbose = 0;
int checkptFreq = 0;
double parsimony = 0.0;
double pctReproduce = .095;
double pctXoverInternal = .695;
double pctXoverAny = .195;
int maxLoops = 1000;

/*
 * This is set when a checkpoint file is read
 */
static int firstGen = 0;

static const char *
skipName(in, cstr, name)
FILE *in;
charString *cstr;
const char *name;
{
  const char *cp;
  unsigned len;

  /* read string */
  if (charStringRead(cstr, in) || charStringLength(cstr) == 0)
    return(0);

  /* make sure it's the string we're looking for */
  cp = charStringBuffer(cstr);
  len = strlen(name);
  if (strncmp(cp, name, len) != 0 || !isspace(cp[len]))
    return(0);

  /* skip trailing whitespace */
  for (cp += len; *cp && isspace(*cp); cp++)
    ;

  return(cp);
}

static int
readCreationMethod(in, cstr, cmp)
FILE *in;
charString *cstr;
programCreationMethod *cmp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, "createMethod");
  if (!cp)
    return(1);

  /* find creation method */
  switch (*cp) {
  case 'f':
  case 'F':
    if (strncasecmp(cp, "Full", 4) == 0 && (cp[4] == 0 || isspace(cp[4]))) {
      *cmp = pcmFull;
      return(0);
    }
    break;
  case 'g':
  case 'G':
    if (strncasecmp(cp, "Grow", 4) == 0 && (cp[4] == 0 || isspace(cp[4]))) {
      *cmp = pcmGrow;
      return(0);
    }
    break;
  case 'r':
  case 'R':
    if (strncasecmp(cp, "Ramped", 6) == 0 && (cp[6] == 0 || isspace(cp[6]))) {
      *cmp = pcmRamped;
      return(0);
    }
    break;
  }

  /* didn't find it */
  fprintf(stderr, "%s: Bad creation method in checkpoint file!\n", progname);
  return(1);
}

static int
readSelectionMethod(in, cstr, smp)
FILE *in;
charString *cstr;
parentSelectionMethod *smp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, "selectMethod");
  if (!cp)
    return(1);

  /* find creation method */
  switch (*cp) {
  case 'f':
  case 'F':
    if (strncasecmp(cp, "FitProp", 7) == 0 && (cp[7] == 0 || isspace(cp[7]))) {
      *smp = psmFitnessProportionate;
      return(0);
    }
    break;
  case 'g':
  case 'G':
    if (strncasecmp(cp, "Greedy", 6) == 0 && (cp[6] == 0 || isspace(cp[6]))) {
      *smp = psmGreedyOverselection;
      return(0);
    }
    break;
  case 't':
  case 'T':
    if (strncasecmp(cp, "Tourn", 5) == 0 && (cp[5] == 0 || isspace(cp[5]))) {
      *smp = psmTournament;
      return(0);
    }
    break;
  }

  /* didn't find it */
  fprintf(stderr, "%s: Bad selection method in checkpoint file!\n", progname);
  return(1);
}

static int
readBoolean(in, cstr, name, blnp)
FILE *in;
charString *cstr;
const char *name;
int *blnp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, name);
  if (!cp) {
    fprintf(stderr, "%s: Looking for field \"%s\"", progname, name);
    fprintf(stderr, " in checkpoint file, got \"%s\"\n", cp);
    return(1);
  }

  /* grab the boolean value */
  if (*cp == 'T' || *cp == 't')
    *blnp = 1;
  else if (*cp == 'F' || *cp == 'f')
    *blnp = 0;
  else {
    fprintf(stderr, "%s: Bad boolean value '%s'", progname, cp);
    fprintf(stderr, " for field \"%s\" in checkpoint file\n", name);
    return(2);
  }

  return(0);
}

static int
readInteger(in, cstr, name, intp)
FILE *in;
charString *cstr;
const char *name;
int *intp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, name);
  if (!cp) {
    fprintf(stderr, "%s: Looking for field \"%s\"", progname, name);
    fprintf(stderr, " in checkpoint file, got \"%s\"\n", cp);
    return(1);
  }

  /* grab the integer */
  sscanf(cp, "%d", intp);

  return(0);
}

static int
readDouble(in, cstr, name, dblp)
FILE *in;
charString *cstr;
const char *name;
double *dblp;
{
  const char *cp;

  /* skip field name */
  cp = skipName(in, cstr, name);
  if (!cp) {
    fprintf(stderr, "%s: Looking for field \"%s\"", progname, name);
    fprintf(stderr, " in checkpoint file, got \"%s\"\n", cp);
    return(1);
  }

  /* grab the double value */
  sscanf(cp, "%lf", dblp);

  return(0);
}

static population *
readCheckpointFile(ip, genp)
const interface *ip;
int *genp;
{
  FILE *in;
  charString *cstr;
  population *pop = 0;
  int rval;

  /* try to open the checkpoint file */
  in = fopen(checkptName, "r");
  if (!in)
    return(0);

  /* create an input buffer */
  cstr = charStringCreate();
  if (!cstr) {
    fclose(in);
    return(0);
  }

  /* read checkpoint data */
  rval = (readInteger(in, cstr, "currentGen", genp) ||
	  readCreationMethod(in, cstr, &creator) ||
	  readInteger(in, cstr, "maxGen", &generations) ||
	  readInteger(in, cstr, "initDepth", &initialDepth) ||
	  readInteger(in, cstr, "maxTotDepth", &maxTotalDepth) ||
	  readInteger(in, cstr, "maxMutDepth", &maxMutateDepth) ||
	  readSelectionMethod(in, cstr, &selector) ||
	  readBoolean(in, cstr, "verbose", &verbose) ||
	  readDouble(in, cstr, "pctRepro", &pctReproduce) ||
	  readDouble(in, cstr, "pctXoverAny", &pctXoverAny) ||
	  readDouble(in, cstr, "pctXoverInt", &pctXoverInternal) ||
	  readDouble(in, cstr, "parsimony", &parsimony) ||
	  readBoolean(in, cstr, "alwaysEval", &alwaysEval) ||
	  readBoolean(in, cstr, "countUnique", &countUnique) ||
	  readInteger(in, cstr, "checkptFreq", &checkptFreq));
  if (!rval) {
    pop = populationReadCheckpoint(in, ip, maxLoops, parsimony);
    if (pop) {
      int i;

      for (i = 0; i < pop->populationSize; i++) {
	if (pop->programs[i] == 0)
	  fprintf(stderr, "Program %d is null\n", i);
      }
    }
  }

  fclose(in);

  return(pop);
}

static void
argPrint(str)
const char *str;
{
  static int lineLen = 0;
  static int current = 0;
  int len;

  /* find line length */
  if (lineLen == 0) {
    lineLen = 80;
    current = lineLen;
  }

  len = strlen(str);
  current -= len;
  if (current > 0) {
    fputs(str, stderr);
  } else {
    fprintf(stderr, "\n\t%s", str);
    current = lineLen - (8 + len);
  }
}

static population *
processArgs(argc, argv, ip)
int argc;
char *argv[];
const interface *ip;
{
  int c, errflag = 0;
  extern char *optarg;
  population *pop = 0;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

#ifdef MASSIVE_CHECKING
    /* check malloc chain every time malloc() is called */
    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
#else
    /* don't validate pointers during calls to mem*,b*,str* routines */
    moa.i = 0;
    dbmallopt(MALLOC_CKDATA, &moa);
#endif
  }
#endif

  /* process all arguments */
  while ((c = getopt(argc, argv, "aAc:d:f:F:g:i:m:o:p:r:s:t:uUvVx:z:")) != EOF)
    switch (c) {
    case 'a':
      alwaysEval = 0;
      break;
    case 'A':
      alwaysEval = 1;
      break;
    case 'c':
      switch (*optarg) {
      case 'f':
      case 'F':
	creator = pcmFull;
	break;
      case 'g':
      case 'G':
	creator = pcmGrow;
	break;
      case 'r':
      case 'R':
	creator = pcmRamped;
	break;
      default:
	fprintf(stderr, "%s: Invalid creation method '%s'\n", progname, optarg);
	fprintf(stderr, "\tCreation Method must be one of:\n");
	fprintf(stderr, "\t\tF(ull)\n");
	fprintf(stderr, "\t\tG(row)\n");
	fprintf(stderr, "\t\tR(amped half and half)\n");
	errflag = 1;
	break;
      }
      break;
    case 'd':
      maxTotalDepth = strtol(optarg, 0, 0);
      if (maxTotalDepth <= 0) {
	fprintf(stderr,
		"%s: Maximum total depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'f':
      checkptName = (char *)malloc(strlen(optarg)+1);
      if (!checkptName) {
	fprintf(stderr, "%s: Couldn't get memory to save file name!\n",
		progname);
	exit(1);
      }
      strcpy(checkptName, optarg);
      break;
    case 'F':
      checkptName = (char *)malloc(strlen(optarg)+1);
      if (!checkptName) {
	fprintf(stderr, "%s: Couldn't get memory to save file name!\n",
		progname);
	exit(1);
      }
      strcpy(checkptName, optarg);
      pop = readCheckpointFile(ip, &firstGen);
      if (!pop) {
	fprintf(stderr,	"%s: Error reading checkpoint file '%s'!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'g':
      generations = strtol(optarg, 0, 0);
      if (generations <= 0) {
	fprintf(stderr, "%s: Generation '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'i':
      initialDepth = strtol(optarg, 0, 0);
      if (initialDepth <= 0) {
	fprintf(stderr, "%s: Initial depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'm':
      maxMutateDepth = strtol(optarg, 0, 0);
      if (maxMutateDepth <= 0) {
	fprintf(stderr, "%s: Maximum mutation depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'o':
      overselectionPercentage = strtod(optarg, 0);
      if (overselectionPercentage < 0.0 || overselectionPercentage > 1.0) {
	fprintf(stderr, "%s: Invalid overselection percentage '%s'\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'p':
      populationSize = strtol(optarg, 0, 0);
      if (populationSize <= 0) {
	fprintf(stderr, "%s: Population size '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'r':
      randomSeed = strtol(optarg, 0, 0);
      if (randomSeed <= 0) {
	fprintf(stderr, "%s: Invalid random seed '%s'\n", progname, optarg);
	errflag = 1;
      }
      break;
    case 's':
      switch (*optarg) {
      case 'f':
      case 'F':
	selector = psmFitnessProportionate;
	break;
      case 'g':
      case 'G':
	selector = psmGreedyOverselection;
	break;
      case 't':
      case 'T':
	selector = psmTournament;
	break;
      default:
	fprintf(stderr, "%s: Invalid selection type '%s'\n", progname, optarg);
	fprintf(stderr, "\tSelection Type must be one of:\n");
	fprintf(stderr, "\t\tF(itness proportionate)\n");
	fprintf(stderr, "\t\tG(reedy overselection)\n");
	fprintf(stderr, "\t\tT(ournament)\n");
	errflag = 1;
	break;
      }
      break;
    case 't':
      tournamentRounds = strtol(optarg, 0, 0);
      if (tournamentRounds < 1 || tournamentRounds > 10) {
	fprintf(stderr, "%s: Invalid number of tournament rounds '%s'\n", progname, optarg);
	errflag = 1;
      }
      break;
    case 'u':
      countUnique = 1;
      break;
    case 'U':
      countUnique = 0;
      break;
    case 'v':
      verbose = 1;
      break;
    case 'V':
      verbose = 0;
      break;
    case 'x':
      checkptFreq = strtol(optarg, 0, 0);
      if (checkptFreq < 1) {
	fprintf(stderr,
		"%s: Checkpoint frequency '%s' must be greater than 0\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'z':
      parsimony = strtod(optarg, 0);
      break;
    case '?':
      fprintf(stderr, "%s: Bad argument '%c'\n", progname, c);
      errflag = 1;
      break;
    }

  /* die if there were problems */
  if (errflag) {
    argPrint("Usage: ");
    argPrint(progname);
    argPrint(" [-a(lwaysEvaluateProgram)]");
    argPrint(" [-A <opposite of -a>]");
    argPrint(" [-c creationMethod)]");
    argPrint(" [-d maximumTotalDepth]");
    argPrint(" [-f checkpointFilename]");
    argPrint(" [-F checkpointInputFilename]");
    argPrint(" [-g #generations]");
    argPrint(" [-i initialDepth]");
    argPrint(" [-m maximumMutationDepth]");
    argPrint(" [-o overselectionPercentage]");
    argPrint(" [-p populationSize]");
    argPrint(" [-r randomseed]");
    argPrint(" [-s selectionMethod]");
    argPrint(" [-t tournamentRounds]");
    argPrint(" [-u(niqueProgramStatistics)]");
    argPrint(" [-U <opposite of -u>]");
    argPrint(" [-v(erbose)]");
    argPrint(" [-V <opposite of -v>]");
    argPrint(" [-x checkpointFreq]");
    argPrint(" [-z parsimonyFactor]");
    argPrint("\n");
    exit(1);
  }

  return(pop);
}

static void
dumpParams()
{
  printf("Global:\n");
  printf("\tRandom Seed: %d\n", randomSeed);
  printf("\tNumber of Generations: %d\n\n", generations);

  printf("Creation:\n");
  printf("\tInitial Depth: %d\n", initialDepth);
  printf("\tPopulation Size: %d\n", populationSize);
  printf("\tCreationMethod: %s\n\n", (creator == pcmFull ? "Full" :
				  (creator == pcmGrow ? "Grow" :
				   (creator == pcmRamped ? "Ramped" :
				    "Unknown"))));

  printf("Evaluation:\n");
  if (alwaysEval)
    printf("\tAlways Evaluate Programs\n");
  else
    printf("\tOnly Evaluate Non-reproduced Programs\n");
  printf("\tParsimony Factor: %f\n", parsimony);

  printf("Breeding:\n");
  printf("\tMaximum Mutation Depth: %d\n", maxMutateDepth);
  printf("\tMaximum Total Depth: %d\n", maxTotalDepth);
  printf("\tSelection Method: %s\n\n", (selector == psmFitnessProportionate ?
				    "Fitness Proportionate" :
				    (selector == psmGreedyOverselection ?
				     "Greedy Overselection" :
				     (selector == psmTournament ?
				      "Tournament" : "Unknown"))));

  if ((selector == psmFitnessProportionate) ||
      (selector == psmGreedyOverselection)) {
    printf("Breeding Method Percentage:\n");
    printf("\t%% Reproduced: %5.2f\n", pctReproduce*100);
    printf("\t%% Any Point Xover: %5.2f\n", pctXoverAny*100);
    printf("\t%% Internal Xover: %5.2f\n\n", pctXoverInternal*100);
  }

  if (selector == psmGreedyOverselection)
    printf("Greedy Overselection Point: %f\n", overselectionPercentage);
  else if (selector == psmTournament)
    printf("Tournament Rounds: %d\n", tournamentRounds);

  printf("\n");
}

static int
safeRename(oldname, newname)
const char *oldname, *newname;
{
  struct stat statbuf;

  /* remove new file if it exists */
  if (stat(newname, &statbuf) == 0)
    unlink(newname);

  /* move old file to new name */
  if (rename(oldname, newname))
    return(1);

  /* success */
  return(0);
}

static int
writeCreationMethod(out, cm)
FILE *out;
programCreationMethod cm;
{
  const char *name = "Unknown";

  switch (cm) {
  case pcmFull:
    name = "Full";
    break;
  case pcmGrow:
    name = "Grow";
    break;
  case pcmRamped:
    name = "Ramped";
    break;
  }
  return(fprintf(out, "createMethod %s\n", name) == EOF);
}

static int
writeSelectionMethod(out, sm)
FILE *out;
parentSelectionMethod sm;
{
  const char *name = "Unknown";

  switch (sm) {
  case psmFitnessProportionate:
    name = "FitProp";
    break;
  case psmGreedyOverselection:
    name = "Greedy";
    break;
  case psmTournament:
    name = "Tourn";
    break;
  }
  return(fprintf(out, "selectMethod %s\n", name) == EOF);
}

#define writeBoolean(out, name, val)	\
  (fprintf(out, "%s %s\n", (name), (val) ? "T" : "F") == EOF)
#define writeInteger(out, name, val)	\
  (fprintf(out, "%s %d\n", (name), (val)) == EOF)
#define writeDouble(out, name, val)	\
  (fprintf(out, "%s %f\n", (name), (val)) == EOF)

static int
writeCheckpointFile(pop, gen)
const population *pop;
int gen;
{
  const char *ext = "bak";
  char *backupName;
  FILE *out;
  int rval;

  /* get enough space for new name */
  backupName = (char *)malloc(strlen(checkptName) + 1 + strlen(ext) + 1);
  if (!backupName)
    return(0);

  /* create new name */
  strcpy(backupName, checkptName);
  strcat(backupName, ".");
  strcat(backupName, ext);

  /* backup old checkpoint file if it exists */
  safeRename(checkptName, backupName);

  /* try to open the checkpoint file */
  out = fopen(checkptName, "w");
  if (!out)
    return(-1);

  /* write checkpoint data */
  rval = (writeInteger(out, "currentGen", gen) ||
	  writeCreationMethod(out, creator) ||
	  writeInteger(out, "maxGen", generations) ||
	  writeInteger(out, "initDepth", initialDepth) ||
	  writeInteger(out, "maxTotDepth", maxTotalDepth) ||
	  writeInteger(out, "maxMutDepth", maxMutateDepth) ||
	  writeSelectionMethod(out, selector) ||
	  writeBoolean(out, "verbose", verbose) ||
	  writeDouble(out, "pctRepro", pctReproduce) ||
	  writeDouble(out, "pctXoverAny", pctXoverAny) ||
	  writeDouble(out, "pctXoverInt", pctXoverInternal) ||
	  writeDouble(out, "parsimony", parsimony) ||
	  writeBoolean(out, "alwaysEval", alwaysEval) ||
	  writeBoolean(out, "countUnique", countUnique) ||
	  writeInteger(out, "checkptFreq", checkptFreq) ||
	  populationWriteCheckpoint(pop, out));

  fclose(out);

  /* if we failed, lose new file; otherwise, lose old file */
  if (rval) {
    unlink(checkptName);
    safeRename(backupName, checkptName);
  } else
    unlink(backupName);

  /* free allocated memory, return */
  free(backupName);
  return(rval);
}

static void
loop(pop, ip)
population *pop;
interface *ip;
{
  int g, done;
  int ckpt = 0;

  for (g = firstGen; g <= generations; g++) {
    if (g != firstGen)
      populationBreed(pop, ip, selector, pctReproduce, pctXoverAny,
		      pctXoverInternal, maxMutateDepth, maxTotalDepth);
#ifdef DEBUG_CHECKPOINT
    {
      population *nupop;

      if (writeCheckpointFile(pop, g)) {
	fprintf(stderr, "%s: Couldn't write checkpoint file \"%s\"!\n",
		progname, checkptName);
	exit(1);
      }
      nupop = readCheckpointFile(ip, &ckpt);
      if (!nupop || g != ckpt || !populationCompare(pop, nupop)) {
	fprintf(stderr, "%s: Checkpointed population is messed up!\n",
		progname);
	exit(1);
      } else
	fprintf(stderr, "%s: Ckpt gen %d OK\n", progname, g);
      if (nupop)
	populationFree(nupop);
    }
#else /* DEBUG_CHECKPOINT */
    if (++ckpt == checkptFreq) {
      if (writeCheckpointFile(pop, g))
	fprintf(stderr, "%s: Couldn't write checkpoint file \"%s\"!\n",
		progname, checkptName);
      else
	printf("\t<Checkpoint written>\n");
      ckpt = 0;
    }
#endif /* DEBUG_CHECKPOINT */
    done = populationEval(pop, ip, maxLoops, alwaysEval, parsimony, g);
    populationDump(pop, g, verbose, countUnique);
    if (done)
      break;
  }

  /* print final summary */
  printf("Best program found on generation %d\n", pop->bestGeneration);
  if (verbose)
    programDump(pop->bestOfRun, 1);
}

int
main(argc, argv)
int argc;
char *argv[];
{
  population *pop;
  interface *ip;
  int o;
  struct timeval first, second;
  struct timezone tzp;
  double elapsed;

  /* save program name */
  if ((progname = strrchr(argv[0], '/')) == 0)
    progname = argv[0];
  else
    progname++;

  /* initialize random number seed */
  randomSeed = time(0) ^ getpid();

  /* create application interface object */
  ip = interfaceCreate(appInitialize);
  if (!ip) {
    fprintf(stderr, "%s: Couldn't create application interface!\n", progname);
    exit(1);
  }

#ifdef DEBUG_CHECKPOINT
  /* initialize checkpoint file name */
  checkptName = "ckpt.debug";
#endif

  /* process the argument list */
  pop = processArgs(argc, argv, ip);

  /* set up greedy overselection percentage if it's uninitialized */
  if (overselectionPercentage < 0) {
    overselectionPercentage = .32;
    o = 1000;
    while (populationSize > o) {
      o <<= 1;
      overselectionPercentage /= 2;
    }
  }

  /* print parameters for this run */
  dumpParams();

  /* randomize things */
  srnd(randomSeed);

  /* start timing */
  gettimeofday(&first, &tzp);

  /* compute the overhead on the gettimeofday() call */
  if (!pop) {
    pop = populationCreate(ip, populationSize, initialDepth, creator);
    if (!pop) {
      fprintf(stderr, "%s: Couldn't create population!\n", progname);
      exit(1);
    }
  }

  loop(pop, ip);

  /* stop timing, print elapsed time */
  gettimeofday(&second, &tzp);
  if (first.tv_usec > second.tv_usec) {
    second.tv_sec--;
    second.tv_usec += 1000000;
  }
  elapsed = (double )(second.tv_sec - first.tv_sec) +
	(double )(second.tv_usec - first.tv_usec)/1000000.0;
  printf("Elapsed time = %f seconds\n", elapsed);

  /* clean up allocated memory */
  populationFree(pop);
  interfaceFree(ip);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
  constantSrcFreeStack();
  resultFreeStack();
  variableFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */
  errorCodeFreeStack();

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
