/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdio.h>
#include <malloc.h>
#include "variable.h"
#include "proto.h"

#ifdef KEEP_ALLOCATED_MEMORY
static variable *freeStack = 0;
#endif /* KEEP_ALLOCATED_MEMORY */

variable *
variableCreate(dt, name, varp)
datatype dt;
const char *name;
void *varp;
{
  datatype rdt;
  variable *vp;
  char *np;

  /* create a new instance of variable */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    vp = freeStack;
    freeStack = *(variable **)vp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    vp = (variable *)malloc(sizeof(variable));

  /* fill new variable instance */
  if (vp) {
    vp->otype = otVariable;
    vp->dtype = dt;

    /* try to save name */
    np = (char *)malloc(strlen(name) + 1);
    if (np) {
      strcpy(np, name);
      vp->name = np;
    } else
      vp->name = 0;

    /* save pointer to variable */
    rdt = datatypeResolve(dt);
    switch (rdt) {
    case dtBoolean:
      vp->v.bp = (bool *)varp;
      break;
    case dtShort:
      vp->v.sp = (short *)varp;
      break;
    case dtInteger:
      vp->v.ip = (int *)varp;
      break;
    case dtLong:
      vp->v.lp = (long *)varp;
      break;
    case dtFloat:
      vp->v.fp = (float *)varp;
      break;
    case dtDouble:
      vp->v.dp = (double *)varp;
      break;
    case dtBlob:
      vp->v.bbp = varp;
      break;
    case dtVoid:
    case dtList:
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
    case dtError:
      if (vp->name)
	free(vp->name);
      free(vp);
      vp = 0;
      break;
    }
  }
  return(vp);
}

variable *
variableCopy(vp)
const variable *vp;
{
  variable *nvp;
  char *name;

  /* create a new instance of variable */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    nvp = freeStack;
    freeStack = *(variable **)nvp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    nvp = (variable *)malloc(sizeof(variable));

  /* if we got some memory, copy this variable */
  if (nvp) {
    memcpy(nvp, vp, sizeof(variable));

    /* save copy of name (if possible) */
    name = (char *)malloc(strlen(vp->name) + 1);
    if (name) {
      strcpy(name, vp->name);
      nvp->name = name;
    }

    if (datatypeResolve(nvp->dtype) == dtBlob)
      nvp->v.bbp = blobCopy(vp->v.bbp, vp->dtype);
  }

  /* return copy */
  return(nvp);
}

result *
variableEval(vp)
const variable *vp;
{
  datatype rdt;
  result *rp = 0;

  rdt = datatypeResolve(vp->dtype);
  switch (rdt) {
  case dtBoolean:
    rp = resultCreate(dtBoolean, (int )(*(vp->v.bp)));
    break;
  case dtShort:
    rp = resultCreate(dtShort, (int )(*(vp->v.sp)));
    break;
  case dtInteger:
    rp = resultCreate(dtInteger, *(vp->v.ip));
    break;
  case dtLong:
    rp = resultCreate(dtLong, *(vp->v.lp));
    break;
  case dtFloat:
    rp = resultCreate(dtFloat, (double )(*(vp->v.fp)));
    break;
  case dtDouble:
    rp = resultCreate(dtDouble, *(vp->v.dp));
    break;
  case dtBlob:
    rp = resultCreate(dtBlob, *(vp->v.bbp));
    break;
  case dtVoid:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
  case dtError:
    rp = resultCreate(dtError, ErrorBadDataType);
    break;
  }
  return(rp);
}

int
variableCompare(vp1, vp2)
const variable *vp1, *vp2;
{
  datatype rdt;
  int cmp = 0;

  if (vp1->dtype == vp2->dtype)
    rdt = datatypeResolve(vp1->dtype);
    switch (rdt) {
    case dtBoolean:
      cmp = (vp1->v.bp == vp2->v.bp);
      break;
    case dtShort:
      cmp = (vp1->v.sp == vp2->v.sp);
      break;
    case dtInteger:
      cmp = (vp1->v.ip == vp2->v.ip);
      break;
    case dtLong:
      cmp = (vp1->v.lp == vp2->v.lp);
      break;
    case dtFloat:
      cmp = (vp1->v.fp == vp2->v.fp);
      break;
    case dtDouble:
      cmp = (vp1->v.dp == vp2->v.dp);
      break;
    case dtBlob:
      cmp = blobCompare(vp1->v.bbp, vp2->v.bbp, vp1->dtype);
      break;
    case dtVoid:
    case dtList:
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
    case dtError:
      break;
  }
  return(cmp);
}

const variable **
variableNodePtr(vpp, count, nodeNum, internal, typeptr)
const variable **vpp;
int *count;
int nodeNum, internal;
datatype *typeptr;
{
  /* variables are terminal points */
  if (internal)
    return(0);

  /* see if we're the node they want */
  if (((*count)++ >= nodeNum) &&
      ((*typeptr == 0) || ((*typeptr & (*vpp)->dtype) != 0)))
    return(vpp);

  /* we aren't the right node */
  return(0);
}

int
variableToString(vp, cstr)
const variable *vp;
charString *cstr;
{
  datatype rdt;
  int rval = 1;
  char buffer[128];

  rdt = datatypeResolve(vp->dtype);
  switch (rdt) {
  case dtBoolean:
  case dtShort:
  case dtInteger:
  case dtLong:
  case dtFloat:
  case dtDouble:
  case dtBlob:
    rval = charStringCatenate(cstr, vp->name);
    break;
  case dtVoid:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
  case dtError:
    sprintf(buffer, "<Illegal Variable Type '%d'>", vp->dtype);
    rval = charStringCatenate(cstr, buffer);
    break;
  }
  return(rval);
}

void
variableFree(vp)
variable *vp;
{
  /* free name */
  free(vp->name);
  if (datatypeResolve(vp->dtype) == dtBlob)
    blobFree(vp->v.bbp, vp->dtype);

  /* free everything else */
#ifdef KEEP_ALLOCATED_MEMORY
  *(variable **)vp = freeStack;
  freeStack = vp;
#else /* !KEEP_ALLOCATED_MEMORY */
  free(vp);
#endif /* KEEP_ALLOCATED_MEMORY */
}
    
#ifdef KEEP_ALLOCATED_MEMORY
void
variableFreeStack()
{
  variable *vp;

  while (freeStack) {
    vp = freeStack;
    freeStack = *(variable **)vp;
    free(vp);
  }
}
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef DEBUG_VARIABLE

int main P((NOARGS));

int
main()
{
  charString *cstr;
  variable *vp, *nvp;
  bool bval = 0;
  int ival = 0;
  float fval = 0.0;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  cstr = charStringCreate();

  vp = variableCreate(dtBoolean, "bval", &bval);
  bval = 1;
  charStringSet(cstr, "Variable is ");
  variableToString(vp, cstr);
  charStringPrint(cstr);
  variableFree(vp);

  vp = variableCreate(dtInteger, "ival", &ival);
  ival = 2;
  charStringSet(cstr, "Variable is ");
  variableToString(vp, cstr);
  charStringPrint(cstr);
  variableFree(vp);

  vp = variableCreate(dtFloat, "fval", &fval);
  fval = 1.1;
  charStringSet(cstr, "Variable is ");
  variableToString(vp, cstr);
  charStringPrint(cstr);
  variableFree(vp);

  vp = variableCreate(dtInteger, "ival", &ival); nvp = variableCopy(vp);
  ival = 12345;
  charStringSet(cstr, "Original is ");
  variableToString(vp, cstr);
  charStringCatenate(cstr, ", copy is ");
  variableToString(nvp, cstr);
  charStringPrint(cstr);
  variableFree(vp); variableFree(nvp);

  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  variableFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_VARIABLE */
