/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <malloc.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "result.h"
#include "proto.h"

#ifdef KEEP_ALLOCATED_MEMORY
static result *freeStack = 0;
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef DETECT_RESULT_LEAKS
static result *allocList = 0;
#endif /* DETECT_RESULT_LEAKS */

result *
#ifdef __STDC__
resultCreate(datatype dt, ...)
#else
resultCreate(va_alist)
va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  datatype dt;
#endif
  result *rp;
  datatype rdt;

#ifdef __STDC__
  va_start(ap, dt);
#else /* !__STDC__ */
  va_start(ap);
  dt = va_arg(ap, datatype);
#endif /* __STDC__ */

  /* create a new instance of result */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    rp = freeStack;
    freeStack = *(result **)rp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    rp = (result *)malloc(sizeof(result));

  /* fill the new result instance */
  if (rp) {

#ifdef DETECT_RESULT_LEAKS
    /* add to list of allocated results */
    rp->prev = 0;
    rp->next = allocList;
    if (allocList)
      allocList->prev = rp;
    allocList = rp;
#endif /* DETECT_RESULT_LEAKS */

    /* fill in result info */
    rp->otype = otResult;
    rp->dtype = dt;
    rdt = datatypeResolve(dt);
    switch (rdt) {
    case dtVoid:
      break;
    case dtBoolean:
      rp->v.b = (bool )(va_arg(ap, bool) ? 1 : 0);
      break;
    case dtShort:
      rp->v.s = (short )va_arg(ap, int);
      break;
    case dtInteger:
      rp->v.i = va_arg(ap, int);
      break;
    case dtLong:
      rp->v.l = (long )va_arg(ap, int);
      break;
    case dtFloat:
      rp->v.f = (float )va_arg(ap, double);
      break;
    case dtDouble:
      rp->v.d = va_arg(ap, double);
      break;
    case dtList:
      rp->v.lp = va_arg(ap, resultList *);
      break;
    case dtBlob:
      rp->v.bbp = blobCopy(va_arg(ap, blob *), dt);
      break;
    case dtError:
      rp->v.e = va_arg(ap, errorCode);
      break;
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
      free(rp);
      rp = 0;
      break;
    }
  }

  /* clean up and cruise */
  va_end(ap);
  return(rp);
}

result *
resultCopy(rp)
const result *rp;
{
  result *nrp;

  /* create a new instance of result */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    nrp = freeStack;
    freeStack = *(result **)nrp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    nrp = (result *)malloc(sizeof(result));

  /* copy the data */
  if (nrp) {
    memcpy(nrp, rp, sizeof(result));
    if (datatypeResolve(nrp->dtype) == dtList)
      nrp->v.lp = resultListCopy(rp->v.lp);
    else if (datatypeResolve(nrp->dtype) == dtBlob)
      nrp->v.bbp = blobCopy(rp->v.bbp, rp->dtype);
  }

#ifdef DETECT_RESULT_LEAKS
  /* add to list of allocated results */
  nrp->prev = 0;
  nrp->next = allocList;
  if (allocList)
    allocList->prev = nrp;
  allocList = nrp;
#endif /* DETECT_RESULT_LEAKS */

  /* return a copy of this result */
  return(nrp);
}

int
resultCompare(rp1, rp2)
const result *rp1, *rp2;
{
  datatype rdt;
  int cmp = 0;

  if (rp1->dtype == rp2->dtype)
    rdt = datatypeResolve(rp1->dtype);
    switch (rdt) {
    case dtBoolean:
      cmp = (rp1->v.b == rp2->v.b);
      break;
    case dtShort:
      cmp = (rp1->v.s == rp2->v.s);
      break;
    case dtInteger:
      cmp = (rp1->v.i == rp2->v.i);
      break;
    case dtLong:
      cmp = (rp1->v.l == rp2->v.l);
      break;
    case dtFloat:
      cmp = (rp1->v.f == rp2->v.f);
      break;
    case dtDouble:
      cmp = (rp1->v.d == rp2->v.d);
      break;
    case dtList:
      cmp = resultListCompare(rp1->v.lp, rp2->v.lp);
      break;
    case dtBlob:
      cmp = blobCompare(rp1->v.bbp, rp2->v.bbp, rp1->dtype);
      break;
    case dtError:
      cmp = (rp1->v.e == rp2->v.e);
      break;
    case dtVoid:
      cmp = 1;
      break;
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
      break;
  }
  return(cmp);
}

const result **
resultNodePtr(rpp, count, nodeNum, internal, typeptr)
const result **rpp;
int *count;
int nodeNum, internal;
datatype *typeptr;
{
  /* results are terminal points */
  if (internal)
    return(0);

  /* see if we're the node they want */
  if (((*count++) == nodeNum) &&
      ((*typeptr == 0) || ((*typeptr & (*rpp)->dtype) != 0)))
    return(rpp);

  /* we aren't the right node */
  return(0);
}

int
resultToString(rp, cstr)
const result *rp;
charString *cstr;
{
  datatype rdt;
  int rval = 1;
  char buffer[128];

  rdt = datatypeResolve(rp->dtype);
  switch (rdt) {
  case dtVoid:
    rval = charStringCatenate(cstr, "<VOID>");
    break;
  case dtBoolean:
    rval = charStringCatenate(cstr, rp->v.b ? "TRUE" : "FALSE");
    break;
  case dtShort:
    sprintf(buffer, "%d", rp->v.s);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtInteger:
    sprintf(buffer, "%d", rp->v.i);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtLong:
    sprintf(buffer, "%ld", rp->v.l);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtFloat:
    sprintf(buffer, "%f", rp->v.f);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtDouble:
    sprintf(buffer, "%f", rp->v.d);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtList:
    rval = resultListToString(rp->v.lp, cstr);
    break;
  case dtBlob:
    rval = blobToString(rp->v.bbp, rp->dtype, cstr);
    break;
  case dtError:
    sprintf(buffer, "<ERROR: %s>", errorCodeMessage(rp->v.e));
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
    sprintf(buffer, "<Illegal Result Type '%d'>", rp->dtype);
    rval = charStringCatenate(cstr, buffer);
    break;
  }

  return(rval);
}

void
resultFree(rp)
result *rp;
{
  /* free object list (if we have one) */
  if (datatypeResolve(rp->dtype) == dtList)
    resultListFree(rp->v.lp);
  else if (datatypeResolve(rp->dtype) == dtBlob)
    blobFree(rp->v.bbp, rp->dtype);

#ifdef DETECT_RESULT_LEAKS
  /* disconnect from chain of allocated results */
  if (rp->prev)
    (rp->prev)->next = rp->next;
  if (rp->next)
    (rp->next)->prev = rp->prev;
  if (allocList == rp)
    allocList = rp->next;
#endif /* DETECT_RESULT_LEAKS */

#ifdef KEEP_ALLOCATED_MEMORY
  *(result **)rp = freeStack;
  freeStack = rp;
#else /* !KEEP_ALLOCATED_MEMORY */
  free(rp);
#endif /* KEEP_ALLOCATED_MEMORY */
}

#ifdef DETECT_RESULT_LEAKS
int
resultFoundLeak()
{
  int retval = (allocList != 0);

  while (allocList)
    resultFree(allocList);

  return(retval);
}
#endif /* DETECT_RESULT_LEAKS */

#ifdef KEEP_ALLOCATED_MEMORY
void
resultFreeStack()
{
  result *rp;

  while (freeStack) {
    rp = freeStack;
    freeStack = *(result **)rp;
    free(rp);
  }
}
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef DEBUG_RESULT

#define ErrorRandomViolence	ErrorUserDefined+9

static void checkZero P((charString *));
static void checkValues P((charString *));
static void checkVariables P((charString *));
#ifdef DETECT_RESULT_LEAKS
static void checkLeakDetection P((NOARGS));
#endif /* DETECT_RESULT_LEAKS */
int main P((NOARGS));

static void
checkZero(cstr)
charString *cstr;
{
  result *voip, *blnp, *shtp, *intp, *lngp, *fltp, *dblp, *lstp, *errp;
  resultList *lval;

  charStringClear(cstr);

  voip = resultCreate(dtVoid, 0);
  charStringSet(cstr, "result is ");
  resultToString(voip, cstr);
  charStringPrint(cstr);

  blnp = resultCreate(dtBoolean, 0);
  charStringSet(cstr, "result is ");
  resultToString(blnp, cstr);
  charStringPrint(cstr);

  shtp = resultCreate(dtShort, 0);
  charStringSet(cstr, "result is ");
  resultToString(shtp, cstr);
  charStringPrint(cstr);

  intp = resultCreate(dtInteger, 0);
  charStringSet(cstr, "result is ");
  resultToString(intp, cstr);
  charStringPrint(cstr);

  lngp = resultCreate(dtLong, 0);
  charStringSet(cstr, "result is ");
  resultToString(lngp, cstr);
  charStringPrint(cstr);

  fltp = resultCreate(dtFloat, 0.0);
  charStringSet(cstr, "result is ");
  resultToString(fltp, cstr);
  charStringPrint(cstr); 

  dblp = resultCreate(dtDouble, 0.0);
  charStringSet(cstr, "result is ");
  resultToString(dblp, cstr);
  charStringPrint(cstr); 

  lval = resultListCreate(1);
  lstp = resultCreate(dtList, lval);
  charStringSet(cstr, "result is ");
  resultToString(lstp, cstr);
  charStringPrint(cstr); 

  errp = resultCreate(dtError, ErrorBadDataType);
  charStringSet(cstr, "result is ");
  resultToString(errp, cstr);
  charStringPrint(cstr);

  resultFree(blnp);
  resultFree(intp);
  resultFree(fltp);
  resultFree(lstp);
  resultFree(errp);
  resultFree(dblp);
  resultFree(lngp);
  resultFree(shtp);
  resultFree(voip);
}

static void
checkValues(cstr)
charString *cstr;
{
  result *blnp, *shtp, *intp, *lngp, *fltp, *dblp, *lstp, *errp;
  resultList *lval;

  charStringClear(cstr);

  blnp = resultCreate(dtBoolean, 1);
  charStringSet(cstr, "result is ");
  resultToString(blnp, cstr);
  charStringPrint(cstr);

  shtp = resultCreate(dtShort, 2147483647);
  charStringSet(cstr, "result is ");
  resultToString(shtp, cstr);
  charStringPrint(cstr);

  intp = resultCreate(dtInteger, 2147483647);
  charStringSet(cstr, "result is ");
  resultToString(intp, cstr);
  charStringPrint(cstr);

  lngp = resultCreate(dtLong, 2147483647);
  charStringSet(cstr, "result is ");
  resultToString(lngp, cstr);
  charStringPrint(cstr);

  fltp = resultCreate(dtFloat, 1.23456789);
  charStringSet(cstr, "result is ");
  resultToString(fltp, cstr);
  charStringPrint(cstr);

  dblp = resultCreate(dtDouble, 1.23456789);
  charStringSet(cstr, "result is ");
  resultToString(dblp, cstr);
  charStringPrint(cstr);

  resultFree(intp);

  lval = resultListCreate(2);
  lstp = resultCreate(dtList, lval);
  charStringSet(cstr, "result is ");
  resultToString(lstp, cstr);
  charStringPrint(cstr);

  errp = resultCreate(dtError, ErrorRandomViolence);
  charStringSet(cstr, "result is ");
  resultToString(errp, cstr);
  charStringPrint(cstr);

  resultFree(blnp);
  resultFree(errp);
  resultFree(lstp);
  resultFree(fltp);
  resultFree(shtp);
  resultFree(lngp);
  resultFree(dblp);
}

static void
checkVariables(cstr)
charString *cstr;
{
  result *blnp, *shtp, *intp, *lngp, *fltp, *dblp, *errp;
  bool bval;
  short sval;
  int ival;
  long lval;
  float fval;
  double dval;
  errorCode eval;

  charStringClear(cstr);

  bval = 1;
  blnp = resultCreate(dtBoolean, bval);
  charStringSet(cstr, "result is ");
  resultToString(blnp, cstr);
  charStringPrint(cstr);

  sval = 666;
  shtp = resultCreate(dtShort, sval);
  charStringSet(cstr, "result is ");
  resultToString(shtp, cstr);
  charStringPrint(cstr);

  ival = 12345;
  intp = resultCreate(dtInteger, ival);
  charStringSet(cstr, "result is ");
  resultToString(intp, cstr);
  charStringPrint(cstr);

  lval = 123456789;
  lngp = resultCreate(dtLong, lval);
  charStringSet(cstr, "result is ");
  resultToString(lngp, cstr);
  charStringPrint(cstr);

  fval = 123.456789;
  fltp = resultCreate(dtFloat, fval);
  charStringSet(cstr, "result is ");
  resultToString(fltp, cstr);
  charStringPrint(cstr); 

  dval = 12345678.9;
  dblp = resultCreate(dtDouble, dval);
  charStringSet(cstr, "result is ");
  resultToString(dblp, cstr);
  charStringPrint(cstr); 

  eval = ErrorBadDataType;
  errp = resultCreate(dtError, eval);
  charStringSet(cstr, "result is ");
  resultToString(errp, cstr);
  charStringPrint(cstr);

  resultFree(blnp);
  resultFree(intp);
  resultFree(fltp);
  resultFree(errp);
  resultFree(dblp);
  resultFree(lngp);
  resultFree(shtp);
}

#ifdef DETECT_RESULT_LEAKS
static void
checkLeakDetection()
{
  result *errp;

  errp = resultCreate(dtError, ErrorRandomViolence);
  if (!resultFoundLeak())
    printf("Didn't detect result leak!\n");
}
#endif /* DETECT_RESULT_LEAKS */

int
main()
{
  charString *cstr;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  cstr = charStringCreate();

  checkZero(cstr);
  checkValues(cstr);
  checkVariables(cstr);

#ifdef DETECT_RESULT_LEAKS
  checkLeakDetection();
#endif /* DETECT_RESULT_LEAKS */

  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  resultFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_RESULT */
