/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#ifdef DEBUG_PROGRAM
#include <stdio.h>
#endif /* DEBUG_PROGRAM */
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include "pmrand.h"
#include "constant.h"
#include "objectlist.h"
#include "operator.h"
#include "operatrsrc.h"
#include "program.h"
#include "proto.h"

static object *selectObject P((const objectList *, int));
static object *treeCreate P((const objectList *, const objectList *,
			     programCreationMethod, int, int));
static object *parseOp P((const objectList *, const objectList *,
			  const char **));
static object *parseTerm P((const objectList *, const char **));

static object *
selectObject(list, types)
const objectList *list;
int types;
{
#ifdef DEBUG_PROGRAM
  charString *cstr;
  char buffer[16];
#endif /* DEBUG_PROGRAM */
  int n, i, len, dir;
  object *ob;

#ifdef DEBUG_PROGRAM
  cstr = charStringCreate();
  charStringSet(cstr, "Looking for ");
  datatypeToString(types, cstr);
  charStringPrint(cstr);
#endif /* DEBUG_PROGRAM */

  /* choose a random member of the list */
  len = objectListLength(list);
  i = n = irndrng(0, len-1);
  ob = objectListEntry(list, n);

  /* if this isn't what we need... */
  if ((objectDataType(ob) & types) == 0) {

    /* look through the list until we find it */
    i = n;
    dir = (irndrng(0, 1) ? 1 : -1);
    do {
#ifdef DEBUG_PROGRAM
      charStringSet(cstr, "Rejected \"");
      objectToString(ob, cstr);
      charStringCatenate(cstr, "\"");
      datatypeToString(objectDataType(ob), cstr);
      sprintf(buffer, "\"#%d", i);
      charStringCatenate(cstr, buffer);
      charStringPrint(cstr);
#endif /* DEBUG_PROGRAM */
      i = (i + dir) % len;
      if (i < 0)
	i = len - 1;
      ob = objectListEntry(list, i);
    } while (((objectDataType(ob) & types) == 0) && (i != n));

    /* return NULL if we didn't find anything */
    if (i == n) {
      ob = 0;
#ifdef DEBUG_PROGRAM
      printf("None of the %d entries matched\n", len);
      fflush(stdout);
#endif /* DEBUG_PROGRAM */
    }
  }
#ifdef DEBUG_PROGRAM
  charStringFree(cstr);
#endif /* DEBUG_PROGRAM */
  return(ob);
}

static object *
treeCreate(termlist, funclist, creationMethod, rtntypes, depth)
const objectList *termlist, *funclist;
programCreationMethod creationMethod;
int rtntypes, depth;
{
  object *ob;
  const object *cob;
  const operatorSrc *osp;
  int i, n;
  operator *op;
  int argtype;

  if (creationMethod == pcmFull) {

    /* return terminal if this is the end of the tree */
    if (depth == 1) {
      ob = objectCopy(selectObject(termlist, rtntypes), 1);
#ifdef DEBUG_PROGRAM
      if (ob) {
	charString *cstr;

	cstr = charStringCreate();
	objectToString(ob, cstr);
	charStringPrint(cstr);
	charStringFree(cstr);
      } else {
	printf("<NULL>\n");
	fflush(stdout);
      }
#endif /* DEBUG_PROGRAM */
      return(ob);
    }
  } else {

    /* return terminal occasionally */
    if (irndrng(0, depth) == 0)
      return(objectCopy(selectObject(termlist, rtntypes), 1));
  }

  /* choose an object */
  cob = selectObject(funclist, rtntypes);

  /* if it's an operator, get the source */
  if (cob && (objectType(cob) == otSimpleOperator ||
	      objectType(cob) == otComplexOperator))
    cob = (const object *)operatorSource((const operator *)cob);

  if (!cob || ((objectType(cob) != otSimpleOperatorSrc) &&
	       (objectType(cob) != otComplexOperatorSrc)))
    return(cob ? objectCopy(cob, 1) : 0);

  /* choose number of arguments */
  osp = (const operatorSrc *)cob;
  if (operatorSrcMinArgs(osp) == operatorSrcMaxArgs(osp))
    n = operatorSrcMaxArgs(osp);
  else
    n = irndrng(operatorSrcMinArgs(osp), operatorSrcMaxArgs(osp));

  /* create a new operator instance */
  op = operatorCreate(osp, n);

  /* fill in arguments */
  for (i = 0; i < n; i++) {

    /* create next argument */
    argtype = operatorSrcArgType(osp, i);
    ob = treeCreate(termlist, funclist, creationMethod, argtype, depth-1);

    /* if argument creation failed, throw operator away and exit */
    if (!ob) {
      operatorFree(op);
      return(0);
    }

    /* add new argument */
    operatorAddArg(op, ob);
  }
  return((object *)op);
}

program *
programCreate(termlist, funclist, rtntypes, depth, creationMethod)
const objectList *termlist, *funclist;
datatype rtntypes;
int depth;
programCreationMethod creationMethod;
{
  object *ob;
  program *pp;
  int tries = 0;

  /* make sure we got a valid creation method */
  if (creationMethod != pcmFull && creationMethod != pcmGrow)
    return(0);

  do {
    ob = treeCreate(termlist, funclist, creationMethod, rtntypes, depth);

    /* if we got one... */
    if (ob) {
      pp = (program *)malloc(sizeof(program));
      if (pp) {
	pp->tree = ob;
	pp->internals = objectCount(ob, 1);
	pp->terminals = objectCount(ob, 0);
	pp->depth = objectDepth(ob, 0);
	pp->hits = 0;
	pp->raw = pp->standardized = pp->normalized = 0;
	pp->cumPercentage = 0;
#ifdef TRACK_BREED_TYPE
	pp->breedType = pbtCreation;
#endif /* TRACK_BREED_TYPE */
      }
    } else
      pp = 0;
  } while (!pp && tries++ < 20);

#ifdef DEBUG_PROGRAM
  /* print the final product */
  if (pp)
    programDump(pp, 1);
  else
    printf("programCreate FAILED!!!\n");
#endif /* DEBUG_PROGRAM */

  return(pp);
}

#define endChar(c)	(((c) == 0) || ((c) == ')') || isspace(c))

static object *
parseTerm(tlp, sp)
const objectList *tlp;
const char **sp;
{
  unsigned len;
  const object *ob;
  char lastchar;
  constant *cp;
  long lval;
  double dval;
  blob *bbp;
  datatype dtype;

  /* find end of name */
  len = strcspn(*sp, " ()");
  if (len == 0)
    return(0);

  /* find object */
  ob = objectListLookup(tlp, *sp, len);
  if (ob) {
    *sp += len;
    return(objectCopy(ob, 0));
  }

  /* try for a boolean */
  if (strncmp(*sp, "TRUE", 4) == 0) {
    *sp += 4;
    return((object *)booleanCreate((bool )1));
  } else if (strncmp(*sp, "FALSE", 5) == 0) {
    *sp += 5;
    return((object *)booleanCreate((bool )0));
  }

  /* try for an integer */
  len = strspn(*sp, "+-0123456789");
  lastchar = (*sp)[len];
  if (endChar(lastchar) || (((lastchar == 'S') || (lastchar == 'L')) &&
			    endChar((*sp)[len+1]))) {
    lval = strtol(*sp, sp, 0);
    if (lastchar == 'S') {
      (*sp)++;
      cp = shortCreate((short )lval);
    } else if (lastchar == 'L') {
      (*sp)++;
      cp = longCreate(lval);
    } else
      cp = integerCreate((int )lval);

    /* return object */
    return((object *)cp);
  }
    
  /* try for a double */
  if (len > 0) {
    len = strspn(*sp, "+-0123456789.Ee");
    lastchar = (*sp)[len];
    if ((lastchar == 0) || (lastchar == ')') ||	isspace(lastchar) ||
	((lastchar == 'F') &&
	 (((*sp)[len+1] == 0) || isspace((*sp)[len+1])))) {
      dval = strtod(*sp, sp);
      if (lastchar == 'F') {
	(*sp)++;
	cp = floatCreate((float )dval);
      } else
	cp = doubleCreate(dval);

      /* return object */
      return((object *)cp);
    }
  }

  /* try for a blob */
  if (strncmp(*sp, "BLOB:", 5) == 0) {
    *sp += 5;
    bbp = blobParse(sp, &dtype);

    /* if we successfully got a blob... */
    if (bbp && strncmp(*sp, ":BLOB", 5) == 0) {
      *sp += 5;

      /* return a new constant */
      cp = blobPtrCreate(bbp, dtype);
      if (cp)
	return((object *)cp);

      /* clean up if we failed to create a new Blob constant */
      blobFree(bbp, dtype);
      return((object *)0);
    }

    /* skip the blob */
    while (**sp) {
      len = strcspn(*sp, ":");
      if (strncmp((*sp)+len, ":BLOB", 5) == 0) {
	*sp += len + 5;
	break;
      }
      *sp += len;
    }
  }

  /* skip token */
  len = strcspn(*sp, " )");
  *sp += len;
  return(0);
}

static object *
parseOp(tlp, flp, sp)
const objectList *tlp, *flp;
const char **sp;
{
  char buf[256];
  unsigned len;
  object *ob;
  object *top;

  /* make sure we've got an operator */
  if (**sp != '(')
    return(0);

  /* skip whitespace */
  do {
    (*sp)++;
  } while (**sp && isspace(**sp));

  /* find end of name */
  len = strcspn(*sp, " ()");
  if (len == 0)
    return(0);

  /* get a copy of the name */
  strncpy(buf, *sp, len);
  buf[len] = 0;

  /* find an object */
  if ((*sp)[len] == ' ')
    ob = objectListLookup(flp, *sp, len);
  else
    ob = objectListLookup(tlp, *sp, len);
  if (!ob)
    return(0);

  /* skip operator */
  (*sp) += len;

  /* copy this object (but not params), make sure it's an operator */
  top = objectCopy(ob, 0);
  if (!top || (objectType(top) != otSimpleOperator &&
	       objectType(top) != otComplexOperator)) {
    if (top)
      objectFree(top);
    return(0);
  }

  /* skip arguments */
  while (**sp && **sp != ')') {

    /* skip whitespace */
    while (**sp && isspace(**sp))
      (*sp)++;

    /* get next argument */
    if (**sp == '(')
      ob = parseOp(tlp, flp, sp);
    else if (**sp && **sp != ')')
      ob = parseTerm(tlp, sp);
    else
      ob = 0;

    /* add it to the list */
    if (operatorAddArg((operator *)top, ob) && ob)
      objectFree(ob);
  }

  /* skip final ')', return program tree */
  (*sp)++;
  return(top);
}

program *
programParse(termlist, funclist, string)
const objectList *termlist, *funclist;
const char *string;
{
  const char *s = string;
  const char **sp = &s;
  object *ob;
  program *pp = 0;

  /* make sure there's something to do */
  if (!s || *s == 0)
    return(0);

  /* skip leading whitespace */
  while (*s && isspace(*s))
    s++;

  /* see if we've got an operator or a terminal */
  if (*s == '(')
    ob = parseOp(termlist, funclist, sp);
  else if (*s)
    ob = parseTerm(termlist, sp);
  else
    ob = 0;

  if (ob) {
    pp = (program *)malloc(sizeof(program));
    if (pp) {
      pp->tree = ob;
      pp->internals = objectCount(ob, 1);
      pp->terminals = objectCount(ob, 0);
      pp->depth = objectDepth(ob, 0);
      pp->hits = 0;
      pp->raw = pp->standardized = pp->normalized = 0;
      pp->cumPercentage = 0;
#ifdef TRACK_BREED_TYPE
      pp->breedType = pbtParse;
#endif /* TRACK_BREED_TYPE */
    }
  }

  return(pp);
}

program *
programCopy(pp)
const program *pp;
{
  program *npp;

  npp = (program *)malloc(sizeof(program));
  if (npp) {
    memcpy(npp, pp, sizeof(program));
    npp->tree = objectCopy(pp->tree, 1);
    npp->cumPercentage = 0;
#ifdef TRACK_BREED_TYPE
    npp->breedType = pbtReproduction;
#endif /* TRACK_BREED_TYPE */
  }
  return(npp);
}

program *
programMutate(pp, termlist, funclist, rtntypes, subtreeDepth, totalDepth)
program *pp;
const objectList *termlist, *funclist;
datatype rtntypes;
int subtreeDepth, totalDepth;
{
  int count, nodeNum, internal;
  object **nodepp, *subtree;
  int depth;
  int tries = 0;

  /* choose node at which tree will be mutated */
  count = pp->terminals + pp->internals;
  nodeNum = (count > 0 ? irndrng(0, count-1) : 0);
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
  printf("mutate: node %d (of %d)\n", nodeNum, count);
  fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */
  if (nodeNum >= pp->terminals) {
    internal = 1;
    nodeNum -= pp->terminals;
  } else
    internal = 0;

  /* get pointer to node to mutate */
  count = 0;
  nodepp = objectNodePtr((const object **)&(pp->tree), &count, nodeNum,
		       internal, &rtntypes);
  if (nodepp) {

    /* make sure program doesn't get too deep */
    depth = totalDepth - (pp->depth - objectDepth(*nodepp, 0));
    if (depth < 0)
      depth = 0;
    if (depth > subtreeDepth)
      depth = subtreeDepth;

    /* create new subtree */
    do {
      subtree = treeCreate(termlist, funclist, pcmGrow, rtntypes, depth);
    } while (!subtree && tries++ < 20);

    /* if we got something... */
    if (subtree) {

      /* lose old subtree */
      objectFree(*nodepp);
      *nodepp = subtree;

      /* update program internal data */
      pp->internals = objectCount(pp->tree, 1);
      pp->terminals = objectCount(pp->tree, 0);
      pp->depth = objectDepth(pp->tree, 0);
      pp->hits = 0;
      pp->raw = pp->standardized = pp->normalized = 0;
#ifdef TRACK_BREED_TYPE
      pp->breedType = pbtMutation;
    } else {
      pp->breedType = pbtCopyViaMutate;
#endif /* TRACK_BREED_TYPE */
    }
  }

  /* return new program */
  return(pp);
}

int
programCrossover(pp1, pp2, xotype, maxDepth, topTypes)
program *pp1, *pp2;
programCrossoverType xotype;
int maxDepth;
datatype topTypes;
{
  object **nodepp1, **nodepp2;
  int count, mod, nodeNum;
  object *obp;
  datatype type1, type2;
  int depth1 = 0, depth2 = 0;
  int fail1, fail2;

  /* cruise if we're missing a pointer */
  if (!pp1 || !pp2) {
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
      printf("xover failed: bad prog ptr: p1=%x, p2=%x\n", (unsigned int)pp1,
	     (unsigned int)pp2);
      fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */
    return(1);
  }

  /* make sure it's possible to crossover internally */
  if (xotype != pxoTerminal &&
      ((pp1->internals == 0) || (pp2->internals == 0))) {
    if (xotype == pxoAny)
      xotype = pxoTerminal;
    else if (xotype == pxoInternal) {
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
      printf("xover failed: internals: p1=%d, p2=%d\n",
	     pp1->internals, pp2->internals);
      fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */
#ifdef TRACK_BREED_TYPE
      pp1->breedType = pbtCopyViaXover;
      pp2->breedType = pbtCopyViaXover;
#endif /* TRACK_BREED_TYPE */
      return(1);
    }
  }

  /* make sure it's possible to crossover terminally */
  if (xotype != pxoInternal &&
      ((pp1->terminals == 0) || (pp2->terminals == 0))) {
    if (xotype == pxoAny)
      xotype = pxoInternal;
    else if (xotype == pxoTerminal) {
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
      printf("xover failed: terminals: p1=%d, p2=%d\n",
	     pp1->terminals, pp2->terminals);
      fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */
#ifdef TRACK_BREED_TYPE
      pp1->breedType = pbtCopyViaXover;
      pp2->breedType = pbtCopyViaXover;
#endif /* TRACK_BREED_TYPE */
      return(1);
    }
  }

  /* if we can choose Any, figure out if we want internals or terminals */
  if (xotype == pxoAny) {
    count = pp1->internals + pp2->internals;
    if (irndrng(0, count + pp1->terminals + pp2->terminals) < count)
      xotype = pxoInternal;
    else
      xotype = pxoTerminal;
  }

  /* get pointer to node from program 1 */
  count = 0;
  mod = xotype == pxoInternal ? pp1->internals : pp1->terminals;
  nodeNum = (mod > 0 ? irndrng(0, mod-1) : 0);
  type1 = 0;
  nodepp1 = objectNodePtr((const object **)&(pp1->tree), &count, nodeNum,
		     xotype == pxoInternal, &type1);
  if ((type1 == 0) && (nodeNum == 0))
    type1 = topTypes;
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
  printf("xover: p1@%d (of %d)\n", nodeNum, mod);
  fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */

  /* get pointer to node from program 2 */
  count = 0;
  mod = xotype == pxoInternal ? pp2->internals : pp2->terminals;
  nodeNum = (mod > 0 ? irndrng(0, mod-1) : 0);
  type2 = type1;
  nodepp2 = objectNodePtr((const object **)&(pp2->tree), &count, nodeNum,
		     xotype == pxoInternal, &type2);
  if (nodepp2 == 0)
nodepp2 = objectNodePtr((const object **)&(pp2->tree), &count, nodeNum,
		     xotype == pxoInternal, &type2);
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
  printf("xover: p2@%d (of %d)\n", nodeNum, mod);
  fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */

#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
  {
    charString *cstr;

    cstr = charStringCreate();

    charStringSet(cstr, "Program 1: ");
    objectToString(pp1->tree, cstr);
    charStringPrint(cstr);

    charStringSet(cstr, "Subtree 1: ");
    objectToString(*nodepp1, cstr);
    charStringPrint(cstr);

    charStringSet(cstr, "Program 2: ");
    objectToString(pp2->tree, cstr);
    charStringPrint(cstr);

    charStringSet(cstr, "Subtree 2: ");
    objectToString(*nodepp2, cstr);
    charStringPrint(cstr);

    charStringSet(cstr, "Types: Parent1");
    datatypeToString(type1, cstr);
    charStringCatenate(cstr, ", Child1");
    if (nodepp2 && *nodepp2)
      datatypeToString(objectDataType(*nodepp2), cstr);
    else
      charStringCatenate(cstr, " <BAD>");
    charStringCatenate(cstr, ", Parent2");
    datatypeToString(type2, cstr);
    charStringCatenate(cstr, ", Child2");
    if (nodepp1 && *nodepp1)
      datatypeToString(objectDataType(*nodepp1), cstr);
    else
      charStringCatenate(cstr, " <BAD>");
    charStringPrint(cstr);

    charStringFree(cstr);
  }
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */

  /* if we didn't get two crossover points... */
  if ((nodepp1 == 0) || (*nodepp1 == 0) || (nodepp2 == 0) || (*nodepp2 == 0)) {
#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
    printf("xover failed: node 1 %s, node 2 %s\n",
	   ((nodepp1 == 0) || (*nodepp1 == 0) ? "BAD" : "OK"),
	   ((nodepp2 == 0) || (*nodepp2 == 0) ? "BAD" : "OK"));
    fflush(stdout);
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */
#ifdef TRACK_BREED_TYPE
    pp1->breedType = pbtCopyViaXover;
    pp2->breedType = pbtCopyViaXover;
#endif /* TRACK_BREED_TYPE */
    return(2);
  }

  /* switch nodes */
  obp = *nodepp1;
  *nodepp1 = *nodepp2;
  *nodepp2 = obp;

  /* make sure the crossed over programs are still valid */
  fail1 = ((objectDataType(*nodepp1) & type1) == 0);
  fail2 = ((objectDataType(*nodepp2) & type2) == 0);

#ifdef DEBUG_PROGRAM
{
  charString *cstr;

  cstr = charStringCreate();
  charStringSet(cstr, "Wanted:");
  datatypeToString(type1, cstr);
  charStringCatenate(cstr, ", Got:");
  datatypeToString(objectDataType(*nodepp1), cstr);
  charStringCatenate(cstr, (fail1 ? "<FAILED>" : "<OK>"));
  charStringPrint(cstr);

  charStringClear(cstr);
  charStringSet(cstr, "Wanted:");
  datatypeToString(type2, cstr);
  charStringCatenate(cstr, ", Got:");
  datatypeToString(objectDataType(*nodepp2), cstr);
  charStringCatenate(cstr, (fail2 ? "<FAILED>" : "<OK>"));
  charStringPrint(cstr);

  charStringFree(cstr);
}
#endif /* DEBUG_PROGRAM */

  /* only check new program depth if we think the program might be valid */
  if (!fail1) {
    depth1 = objectDepth(pp1->tree, 0);
    fail1 = (depth1 > maxDepth);
  }
  if (!fail2) {
    depth2 = objectDepth(pp2->tree, 0);
    fail2 = (depth2 > maxDepth);
  }

  if (fail1) {
    if (fail2) {

      /* switch things back and return, since they're now both copies */
      obp = *nodepp1;
      *nodepp1 = *nodepp2;
      *nodepp2 = obp;
    } else {

      /* return program 1 to its original state */
      objectFree(*nodepp1);
      *nodepp1 = objectCopy(*nodepp2, 1);
    }
  } else if (fail2) {

    /* return program 2 to its original state */
    objectFree(*nodepp2);
    *nodepp2 = objectCopy(*nodepp1, 1);
  }

#if defined(DEBUG_PROGRAM) || defined(DEBUG_POP_CODE)
{
  int chk1, chk2;

  chk1 = objectDepth(pp1->tree, 0);
  chk2 = objectDepth(pp2->tree, 0);
  printf("programCrossover: pp1 %s (%d deep%s), pp2 %s (%d deep%s)\n",
	 (fail1 ? "reproduced" : "crossed over"), chk1,
	 ((fail1 && (chk1 == pp1->depth)) ||
	  (!fail1 && (chk1 == depth1)) ? "" : " !WRONG!"),
	 (fail2 ? "reproduced" : "crossed over"), chk2,
	 ((fail2 && (chk2 == pp2->depth)) ||
	  (!fail2 && (chk2 == depth2)) ? "" : " !WRONG!"));
  fflush(stdout);
}
#endif /* DEBUG_PROGRAM || DEBUG_POP_CODE */

  /* update program 1 internal data */
  if (!fail1) {
    pp1->internals = objectCount(pp1->tree, 1);
    pp1->terminals = objectCount(pp1->tree, 0);
    pp1->depth = depth1;
    pp1->hits = 0;
    pp1->raw = pp1->standardized = pp1->normalized = 0;
#ifdef TRACK_BREED_TYPE
    pp1->breedType = pbtCrossover;
  } else {
    pp1->breedType = pbtCopyViaXover;
#endif /* TRACK_BREED_TYPE */
  }

  /* update program 2 internal data */
  if (!fail2) {
    pp2->internals = objectCount(pp2->tree, 1);
    pp2->terminals = objectCount(pp2->tree, 0);
    pp2->depth = depth2;
    pp2->hits = 0;
    pp2->raw = pp2->standardized = pp2->normalized = 0;
#ifdef TRACK_BREED_TYPE
    pp2->breedType = pbtCrossover;
  } else {
    pp2->breedType = pbtCopyViaXover;
#endif /* TRACK_BREED_TYPE */
  }

  /* let 'em know we succeeded */
  return(0);
}

int
programCompare(pp1, pp2)
const program *pp1, *pp2;
{
  /* NULL pointers can match */
  if (!pp1 && !pp2)
    return(1);

  /* fail on cases where one pointer's NULL but the other isn't */
  if (!pp1 || !pp2)
    return(0);

  /* fail if there are different numbers of nodes */
  if ((pp1->internals != pp2->internals) || (pp1->terminals != pp2->terminals))
    return(0);

  /* fail if they've got different depths */
  if (pp1->depth != pp2->depth)
    return(0);

  /* check program objects */
  return(objectCompare(pp1->tree, pp2->tree));
}

void
programDump(pp, dumpProgram)
const program *pp;
int dumpProgram;
{
  charString *cstr;

  if (!pp)
    return;

  if (dumpProgram) {
    cstr = charStringCreate();
    charStringSet(cstr, "Program: ");
    objectToString(pp->tree, cstr);
    charStringPrint(cstr);
    charStringFree(cstr);
  }
  printf("Nodes: %d internal, %d terminal, %d deep\t Hits: %d\n",
	 pp->internals, pp->terminals, pp->depth, pp->hits);
  printf("Statistics: %d hits, %f raw, %f std, %f norm, %f cum%%\n",
	 pp->hits, pp->raw, pp->standardized, pp->normalized,
	 pp->cumPercentage);
  fflush(stdout);
#ifdef TRACK_BREED_TYPE
  {
    const char *bt = "UNKNOWN";

    switch(pp->breedType) {
    case pbtCreation:
      bt = "Created\n";
      break;
    case pbtParse:
      bt = "Parsed\n";
      break;
    case pbtReproduction:
      bt = "Reproduced\n";
      break;
    case pbtMutation:
      bt = "Mutated\n";
      break;
    case pbtCrossover:
      bt = "Crossed Over\n";
      break;
    case pbtCopyViaMutate:
      bt = "Copied (Failed Mutation)\n";
      break;
    case pbtCopyViaXover:
      bt = "Copied (Failed Crossover)\n";
      break;
    default:
      bt = "** INVALID TYPE **";
      break;
    }
    printf("Breeding Type: %s\n", bt);
    fflush(stdout);
  }
#endif /* TRACK_BREED_TYPE */
}

void
programFree(pp)
program *pp;
{
  if (pp) {
    objectFree(pp->tree);
    free(pp);
  }
}

#ifdef DEBUG_PROGRAM

#include <stdio.h>
#include <time.h>
#include "constant.h"
#include "optrivial.h"

static result *opWierd P((int, const object **, void *));
static objectList *appFunctions P((NOARGS));
static void checkCopy P((const objectList *, const objectList *));
static void checkCrossover P((const objectList *, const objectList *));
static void checkMutate P((const objectList *, const objectList *));
static void parseOne P((const objectList *, const objectList *, const char *));
static void createAndParseOne P((const objectList *, const objectList *,
				 const char *, const object *, charString *));
static void parseConst P((const objectList *, const objectList *));
static void checkParse P((const objectList *, const objectList *));
static void processArgs P((int, char *[]));
int main P((int, char *[]));

static long seed;

#define dtWierd	dtUserDef1

static result *
opWierd(argc, argv, envp)
int argc;
const object **argv;
void *envp;
{
  /* make sure we got the right number of arguments */
  if (argc != 1) {
    fprintf(stderr, "opWierd got %d args, needs 1\n", argc);
    exit(1);
  }

  /* evaluate argument */
  return(objectEval(argv[0], envp));
}

static objectList *
appFunctions()
{
  objectList *list;
  operatorSrc *osp;

  /* create list of operators */
  list = objectListCreate(3);

  osp = complexOperatorSrcCreate("+", opComplexAdd, dtInteger, 2, 2,
				 dtInteger|dtWierd, dtInteger);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  osp = complexOperatorSrcCreate("-", opComplexSubtract, dtInteger, 2, 2,
				 dtInteger|dtWierd, dtInteger);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  osp = complexOperatorSrcCreate("WIERD", opWierd, dtWierd, 1, 1, dtInteger);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static void
checkCopy(termlist, funclist)
const objectList *termlist, *funclist;
{
  program *pp, *npp;

  pp = programCreate(termlist, funclist, dtAll, 4, pcmFull);
  npp = programCopy(pp);
  fputs("Original ", stdout); fflush(stdout);  programDump(pp, 1);
  fputs(" Copy of ", stdout); fflush(stdout); programDump(npp, 1);
  printf("programCompare thinks they %s equal\n",
	 programCompare(pp, npp) ? "are" : "ARE NOT");
  fflush(stdout);
  programFree(pp);
  programFree(npp);
}

static void
checkCrossover(termlist, funclist)
const objectList *termlist, *funclist;
{
  program *p1, *p2;

  p1 = programCreate(termlist, funclist, dtAll, 4, pcmFull);
  p2 = programCreate(termlist, funclist, dtAll, 4, pcmGrow);
  fputs("Original(1) ", stdout); fflush(stdout); programDump(p1, 1);
  fputs("Original(2) ", stdout); fflush(stdout); programDump(p2, 1);

  if (programCrossover(p1, p2, pxoInternal, 17, dtAll) != 0) {
    printf("Internal crossover failed!\n");
    fflush(stdout);
  } else {
    fputs("XoverInt(1) ", stdout); fflush(stdout); programDump(p1, 1);
    fputs("XoverInt(2) ", stdout); fflush(stdout); programDump(p2, 1);
  }

  if (programCrossover(p1, p2, pxoTerminal, 17, dtAll) != 0) {
    printf("Terminal crossover failed!\n");
    fflush(stdout);
  } else {
    fputs("XoverTrm(1) ", stdout); fflush(stdout); programDump(p1, 1);
    fputs("XoverTrm(2) ", stdout); fflush(stdout); programDump(p2, 1);
  }

  if (programCrossover(p1, p2, pxoInternal, 2, dtAll) != 0) {
    printf("Internal crossover #2 failed!\n");
    fflush(stdout);
  } else {
    fputs("XoverInt(1) ", stdout); fflush(stdout); programDump(p1, 1);
    fputs("XoverInt(2) ", stdout); fflush(stdout); programDump(p2, 1);
  }

  programFree(p1);
  programFree(p2);
}

static void
checkMutate(termlist, funclist)
const objectList *termlist, *funclist;
{
  program *pp, *npp;
  int i;

  pp = programCreate(termlist, funclist, dtAll, 4, pcmFull);
  fputs("Original ", stdout); fflush(stdout); programDump(pp, 1);
  for (i = 0; i < 10; i++) {
    npp = programMutate(programCopy(pp), termlist, funclist, dtAll, 4, 17);
    printf("Mutate %d ", i); fflush(stdout); programDump(npp, 1);
    fflush(stdout);
    programFree(npp);
  }
  programFree(pp);
}

static void
parseOne(termlist, funclist, src)
const objectList *termlist, *funclist;
const char *src;
{
  program *pp;

  printf("Program source is \"%s\"\n", src);
  fflush(stdout);
  pp = programParse(termlist, funclist, src);
  if (pp) {
    programDump(pp, 1);
    programFree(pp);
  } else {
    printf("ERROR! program \"%s\" could not be parsed!\n", src);
    fflush(stdout);
  }
}

static void
createAndParseOne(termlist, funclist, typename, ob, cstr)
const objectList *termlist, *funclist;
const char *typename;
const object *ob;
charString *cstr;
{
  program *pp;

  charStringClear(cstr);
  objectToString(ob, cstr);
  pp = programParse(termlist, funclist, charStringBuffer(cstr));
  if (!pp) {
    printf("ERROR!  Couldn't parse %s \"%s\"!\n", typename,
	   charStringBuffer(cstr));
    fflush(stdout);
  } else {
    if (objectCompare(pp->tree, ob) == 0) {
      printf("%s \"%s\" doesn't match", typename, charStringBuffer(cstr));
      charStringClear(cstr);
      objectToString(pp->tree, cstr);
      printf(" str/prs version \"%s\"!\n", charStringBuffer(cstr));
      fflush(stdout);
    }
    programFree(pp);
  }
}

static void
parseConst(termlist, funclist)
const objectList *termlist, *funclist;
{
  charString *cstr;
  int i;
  const char *typename;
  constant *cp;

  cstr = charStringCreate();
  for (i = 0; i < 6; i++) {
    switch (i) {
    case 0:
      typename = "boolean";
      cp = booleanCreate((bool )i);
      break;
    case 1:
      typename = "short";
      cp = shortCreate((short )i);
      break;
    case 2:
      typename = "integer";
      cp = integerCreate((int )i);
      break;
    case 3:
      typename = "long";
      cp = longCreate((long )i);
      break;
    case 4:
      typename = "float";
      cp = floatCreate((float )i);
      break;
    case 5:
      typename = "double";
      cp = doubleCreate((double )i);
      break;
    }

    createAndParseOne(termlist, funclist, typename, (const object *)cp, cstr);
    constantFree(cp);
  }
  charStringFree(cstr);
}

static void
checkParse(termlist, funclist)
const objectList *termlist, *funclist;
{
  parseOne(termlist, funclist, "1234");
  parseOne(termlist, funclist, "1234.56789");
  parseOne(termlist, funclist, "(+ (- 9.8 7.6) (+ 5 4))");
  parseOne(termlist, funclist, "UNKNOWN");
  parseOne(termlist, funclist, "(+ (badfunc a b c) (badTerminal))");
  parseOne(termlist, funclist, "BLOB:UNKNOWN:BLOB");
  parseConst(termlist, funclist);
}

static void
processArgs(argc, argv)
int argc;
char *argv[];
{
  int c, errflag = 0;
  extern char *optarg;

  /* process all arguments */
  while ((c = getopt(argc, argv, "s:")) != EOF)
    switch (c) {
    case 's':
      seed = atoi(optarg);
      if (seed <= 0) {
	fprintf(stderr, "program: Invalid random seed '%s'\n", optarg);
	errflag = 1;
      }
      break;
    case '?':
      fprintf(stderr, "program: Bad argument '%c'\n", c);
      errflag = 1;
      break;
    }

  /* die if there were problems */
  if (errflag) {
    fprintf(stderr, "Usage: program [-s seed]\n");
    exit(1);
  }
}

int
main(argc, argv)
int argc;
char *argv[];
{
  objectList *termlist, *funclist;
  int i;
  program *pp;
  result *rp;
  charString *cstr;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  seed = 123456789;
  processArgs(argc, argv);
  printf("Seed = %d\n", seed);
  fflush(stdout);
  srnd(seed);

  /* create datatype alias for dtWierd */
  datatypeMakeAlias(dtWierd, dtInteger);

  /* create list of terminals */
  termlist = objectListCreate(6);
  for (i = 1; i < 6; i++)
    objectListAdd(termlist, integerCreate(i));

  /* create list of operators */
  funclist = appFunctions();

  /* create a full program */
  pp = programCreate(termlist, funclist, dtAll, 4, pcmFull);
  fputs("Full ", stdout); fflush(stdout); programDump(pp, 1);
  programFree(pp);

  /* grow a program (may not be full) */
  pp = programCreate(termlist, funclist, dtAll, 4, pcmGrow);
  fputs("Grow ", stdout); fflush(stdout); programDump(pp, 1);
  programFree(pp);

  cstr = charStringCreate();

  /* try to run a full program */
  pp = programCreate(termlist, funclist, dtAll, 4, pcmFull);
  fputs("Full ", stdout); fflush(stdout); programDump(pp, 1);
  rp = programEval(pp, 0);
  charStringSet(cstr, "Result is ");
  resultToString(rp, cstr);
  charStringPrint(cstr);
  resultFree(rp);
  programFree(pp);

  /* try to run a grow program */
  pp = programCreate(termlist, funclist, dtAll, 4, pcmGrow);
  fputs("Grow ", stdout); fflush(stdout); programDump(pp, 1);
  rp = programEval(pp, 0);
  charStringSet(cstr, "Result is ");
  resultToString(rp, cstr);
  charStringPrint(cstr);
  resultFree(rp);
  programFree(pp);

  /* make sure copy works */
  checkCopy(termlist, funclist);

  /* make sure crossover works */
  checkCrossover(termlist, funclist);

  /* make sure copy works */
  checkMutate(termlist, funclist);

  /* test parse routine */
  checkParse(termlist, funclist);

  /* free everything */
  objectListFree(termlist);
  objectListFree(funclist);
  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
  resultFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_PROGRAM */
