/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include "pmrand.h"
#include "program.h"
#include "population.h"
#include "proto.h"

#ifdef DEBUG_POPULATION
#define DEBUG_POP_CODE
#endif /* DEBUG_POPULATION */

int tournamentRounds = 3;
double overselectionPercentage = -1.0;

static int byStandard P((const void *, const void *));
static program *findCumPct P((program **, int, double));
static program *selectFitProp P((const population *));
static program *selectGreedy P((const population *));
static program *selectTourn P((const population *));

population *
populationCreate(ip, popSize, initialDepth, creationMethod)
const interface *ip;
int popSize;
int initialDepth;
programCreationMethod creationMethod;
{
  int progsize;
  population *pop;
  program *pp;
  int i, j, attempts;
  int minDepth, depth;

  progsize = sizeof(program *)*popSize;
  pop = (population *)malloc(sizeof(population) + progsize*2);
  if (pop) {

    /* initialize new program set */
    pop->populationSize = popSize;
    pop->programs = (program **)((char *)pop + sizeof(population));
    pop->workspace = (program **)((char *)pop->programs + progsize);
    pop->bestOfRun = NULL;

    /* create all the programs */
    minDepth = 2;
    for (i = 0; i < popSize; i++) {
      attempts = 0;

      do {

	/* create next program */
	switch (creationMethod) {
	case pcmFull:
	case pcmGrow:
	  pp = programCreate(interfaceTerminalList(ip),
			     interfaceFunctionList(ip),
			     interfaceReturnTypes(ip), initialDepth,
			     creationMethod);
	  break;
	case pcmRamped:
	  if (minDepth >= initialDepth)
	    depth = minDepth;
	  else
	    depth = (irnd() % (initialDepth - minDepth + 1)) + minDepth;
	  pp = programCreate(interfaceTerminalList(ip),
			     interfaceFunctionList(ip),
			     interfaceReturnTypes(ip), depth,
			     (irnd() & 1 ? pcmFull : pcmGrow));
	  break;
	default:
	  /* bad creation method ... free memory and exit */
	  free(pop);
	  return(0);
	}

	/* if we created a program */
	if (pp) {

	  /* look for a duplicate */
	  for (j = 0; j < i; j++)
	    if (programCompare(pp, pop->programs[j])) {
	      programFree(pp);
	      pp = 0;
	      break;
	    }
	}

	/* see if we should increase the minimum depth */
	if (pp == 0) {
	  attempts++;
	  if (attempts == 20) {
	    minDepth++;
	    if (minDepth > initialDepth)
	      initialDepth = minDepth;
	  }
	}
      } while (pp == 0);

      /* add new program */
      pop->programs[i] = pp;
    }
  }

  return(pop);
}

static int
byStandard(a, b)
const void *a, *b;
{
  const program * const *p1 = (const program * const *)a;
  const program * const *p2 = (const program * const *)b;

  if (programStandardizedFitness(*p1) < programStandardizedFitness(*p2))
    return(-1);
  if (programStandardizedFitness(*p1) > programStandardizedFitness(*p2))
    return(1);
  return(0);
}

int
populationEval(pop, ip, maxLoops, alwaysEval, parsimony, genNum)
population *pop;
const interface *ip;
int maxLoops, alwaysEval;
double parsimony;
int genNum;
{
  int i;
  program *pp;
  double adjtot = 0, pct = 0;
#ifdef DETECT_RESULT_LEAKS
  int leaked = 0;
#endif /* DETECT_RESULT_LEAKS */

  for (i = 0; i < pop->populationSize; i++) {
    pp = pop->programs[i];

    /* if this was reproduced, we don't have to reevaluate it */
    if (alwaysEval || ((programHits(pp) == 0) &&
		       (programStandardizedFitness(pp) == 0.0))) {
      interfaceEval(ip, pp, maxLoops, parsimony);
#ifdef DETECT_RESULT_LEAKS
      if (resultFoundLeak())
	leaked++;
#endif /* DETECT_RESULT_LEAKS */
    }

    /* compute adjusted fitness */
    programSetAdjustedFitness(pp, 1.0/(1.0 + programStandardizedFitness(pp)));

    /* total all adjusted fitnesses */
    adjtot += programAdjustedFitness(pp);
  }

  /* now go through and compute normalized fitness */
  for (i = 0; i < pop->populationSize; i++) {
    pp = pop->programs[i];
    programSetNormalizedFitness(pp, programAdjustedFitness(pp) / adjtot);
  }

  /* sort by standardized fitness */
  qsort((char *)pop->programs, (unsigned )pop->populationSize,
	sizeof(program *), byStandard);

  /* set cumulative percentage */
  pct = 0;
  for (i = 0; i < pop->populationSize; i++) {
    pp = pop->programs[i];
    pct += programNormalizedFitness(pp);
    programSetCumulativePercentage(pp, pct);
  }

  /* see if there's a new best-of-run program */
  if ((pop->bestOfRun == NULL) ||
      (programStandardizedFitness(pop->programs[0]) <
       programStandardizedFitness(pop->bestOfRun))) {
    if (pop->bestOfRun)
      programFree(pop->bestOfRun);
    pop->bestOfRun = pop->programs[0];
    pop->bestGeneration = genNum;
  }

#ifdef DETECT_RESULT_LEAKS
  /* report number of leaks found */
  if (leaked > 0)
    fprintf(stderr,
	    "WARNING\nWARNING: Some result objects were not freed!\nWARNING\n\n");
#endif /* DETECT_RESULT_LEAKS */

  /* see if this program is good enough to terminate the run */
  return(interfaceTerminateRun(ip, pop->bestOfRun));
}

static program *
findCumPct(ppp, plen, val)
program **ppp;
int plen;
double val;
{
  int len, base, mid;
  double pval;

  /* do a binary search to find a particular cumulative percentage */
  base = 0;
  for (len = plen; len != 0; len >>= 1) {
    mid = base + (len >> 1);

    /* if we found a match, we're done */
    pval = programCumulativePercentage(ppp[mid]);
    if (val == pval)
      return(ppp[mid]);

    /* move base to the right */
    if (val > pval) {
      base = mid + 1;
      len--;
    }
  }

  /* return the correct program */
  if (base >= 0 && base < plen)
    return(ppp[base]);

  /* didn't find it */
  return(0);
}

static program *
selectFitProp(pop)
const population *pop;
{
  return(findCumPct(pop->programs, pop->populationSize, drnd()));
}

static program *
selectGreedy(pop)
const population *pop;
{
  double r;

  /* choose group to select from */
  if (drnd() < 0.8)
    r = drnd();
  else
    r = 1.0 - drndrng(overselectionPercentage, 1.0);

  /* return program */
  return(findCumPct(pop->programs, pop->populationSize, r));
}

static program *
selectTourn(pop)
const population *pop;
{
  int i, r, n;

  r = irndrng(0, pop->populationSize-1);
  for (i = 1; i < tournamentRounds; i++) {
    n = irndrng(0, pop->populationSize-1);
    if (n < r)
      r = n;
  }

#ifdef DEBUG_POP_CODE
  printf("selectTourn: chose %d\n", r);
#endif /* DEBUG_POP_CODE */
  return(pop->programs[r]);
}

void
populationBreed(pop, ip, selector, pctReproduce, pctXoverAny, pctXoverInternal,
		maxMutateDepth, maxTotalDepth)
population *pop;
const interface *ip;
parentSelectionMethod selector;
double pctReproduce, pctXoverAny, pctXoverInternal;
int maxMutateDepth, maxTotalDepth;
{
  parentSelectionFunction *selectFunc = NULL;
  double pctMutation;
  double r;
  int n;
  program *p1, *p2;
  program **tmp;

  /* get routine to use for parent selection */
  switch (selector) {
  case psmFitnessProportionate:
    selectFunc = selectFitProp;
    break;
  case psmGreedyOverselection:
    selectFunc = selectGreedy;
    break;
  case psmTournament:
    selectFunc = selectTourn;
    break;
  default:
    fprintf(stderr, "Invalid selection method %d\n", selector);
    return;
  }

  /* make these into cumulative totals */
  pctMutation = 1.0 - (pctXoverAny + pctXoverInternal);
  pctXoverAny += pctMutation;
  pctXoverInternal = 1.0;

  /* generate new programs */
  n = pop->populationSize;
  while (n > 0) {

    /* see if we only have room for a single program */
    if (n == 1)
      r = drndrng(0.0, pctMutation);
    else
      r = drnd();

    /* generate new program(s) */
    p1 = programCopy(selectFunc(pop));
    if (r < pctReproduce) {
      pop->workspace[--n] = p1;
#ifdef DEBUG_POP_CODE
      printf("Program %d copied\n", n);
#endif /* DEBUG_POP_CODE */
    } else if (r < pctMutation) {
      pop->workspace[--n] = programMutate(p1, interfaceTerminalList(ip),
					  interfaceFunctionList(ip), dtAll,
					  maxMutateDepth, maxTotalDepth);
      programResetFitness(pop->workspace[n]);
#ifdef DEBUG_POP_CODE
      printf("Program %d mutated\n", n);
#endif /* DEBUG_POP_CODE */
    } else {
      p2 = programCopy(selectFunc(pop));

      programResetFitness(p1);
      programResetFitness(p2);

      programCrossover(p1, p2, (r < pctXoverAny ? pxoAny : pxoInternal),
		       maxTotalDepth, interfaceReturnTypes(ip));
#ifdef DEBUG_POP_CODE
      printf("Programs %d and %d crossed over (%s)\n", n-1, n-2,
	     (r < pctXoverAny ? "any" : "internal"));
#endif /* DEBUG_POP_CODE */

      pop->workspace[--n] = p1;
      pop->workspace[--n] = p2;
    }
  }

  /* switch workspace and program list */
  tmp = pop->workspace;
  pop->workspace = pop->programs;
  pop->programs = tmp;

  /* don't free first program if it's the best of run */
  if (pop->workspace[0] != pop->bestOfRun)
    programFree(pop->workspace[0]);

  /* free old programs */
  for (n = 1; n < pop->populationSize; n++)
    programFree(pop->workspace[n]);
}

int
populationCompare(pop1, pop2)
const population *pop1, *pop2;
{
  int p;

  /* NULL pointers can match */
  if (!pop1 && !pop2)
    return(1);

  /* fail on cases where one pointer is NULL but the other isn't */
  if (!pop1 || !pop2)
    return(0);

  /* fail if there are different numbers of programs */
  if (pop1->populationSize != pop2->populationSize)
    return(0);

  /* fail if the best-of-generation programs differ */
  if (pop1->bestGeneration != pop2->bestGeneration)
    return(0);
  if (programCompare(pop1->bestOfRun, pop2->bestOfRun) == 0)
    return(0);

  /* compare programs */
  for (p = 0; p < pop1->populationSize; p++)
    if (!programCompare(pop1->programs[p], pop2->programs[p]))
      return(0);

  /* must be identical */
  return(1);
}

void
populationDump(pop, genNum, dumpPrograms, countUnique)
const population *pop;
int genNum, dumpPrograms, countUnique;
{
  int i, j;
  const program *pp;
  int hits, hitsTotal = 0, hitsBest = 0, hitsWorst = 0;
  double std, stdTotal = 0.0, stdBest = 0, stdWorst = 0;
  int stdSamples = 0;
  double norm, normTotal = 0.0, normBest = 0, normWorst = 0;
  int depth, depthTotal = 0, deepest = 0, shallowest = 0;
  int complexity = 0;
  int unique = 0;
  const char *bestName;
#ifdef TRACK_BREED_TYPE
  int breedTotal[MAX_BREED_VALUES];
  static const char *breedName[MAX_BREED_VALUES] = {
    "Creation", "Parse", "Reproduction", "Mutation", "Crossover",
    "Copy (Failed Mutation)", "Copy (Failed Xover)"
  };
#endif /* TRACK_BREED_TYPE */

#ifdef TRACK_BREED_TYPE
  /* clear array of breed types */
  for (i = 0; i < MAX_BREED_VALUES; i++)
    breedTotal[i] = 0;
#endif /* TRACK_BREED_TYPE */

  for (i = 0; i < pop->populationSize; i++) {
    pp = pop->programs[i];

    /* print program (if desired) */
    if (dumpPrograms) {
      printf("%d ", i);
      programDump(pp, 1);
    }

    /* gather hits statistics for this generation */
    hits = programHits(pp);
    hitsTotal += hits;
    if (i == 0) {
      hitsBest = hitsWorst = hits;
    } else {
      if (hits > hitsBest)
	hitsBest = hits;
      if (hits < hitsWorst)
	hitsWorst = hits;
    }

    /* gather standardized statistics for this generation */
    std = programStandardizedFitness(pp);
    if (stdSamples == 0) {
      stdBest = stdWorst = std;
    } else if (std != HUGE_VAL) {
      if (std < stdBest)
	stdBest = std;
      if (std > stdWorst)
	stdWorst = std;
    }
    if (std != HUGE_VAL) {
      stdTotal += std;
      stdSamples++;
    }

    /* gather normalized statistics for this generation */
    norm = programNormalizedFitness(pp);
    normTotal += norm;
    if (i == 0) {
      normBest = normWorst = norm;
    } else {
      if (norm < normBest)
	normBest = norm;
      if (norm > normWorst)
	normWorst = norm;
    }

    /* gather depth statistics for this generation */
    depth = programDepth(pp);
    depthTotal += depth;
    if (i == 0) {
      deepest = shallowest = depth;
    } else {
      if (depth > deepest)
	deepest = depth;
      if (depth < shallowest)
	shallowest = depth;
    }

    /* total number of nodes for all programs */
    complexity += programInternalNodes(pp) + programTerminalNodes(pp);

    if (countUnique) {
      /* see if this program is unique */
      for (j = 0; j < i; j++)
	if (programCompare(pp, pop->programs[j]))
	  break;
      if (j >= i)
	unique++;
    }

#ifdef TRACK_BREED_TYPE
    /* track this program's source */
    breedTotal[programBreedSource(pp)]++;
#endif /* TRACK_BREED_TYPE */
  }

  hitsTotal /= pop->populationSize;
  if (stdSamples > 0)
    stdTotal /= stdSamples;
  normTotal /= pop->populationSize;
  depthTotal /= pop->populationSize;
  complexity /= pop->populationSize;

  bestName = pop->bestOfRun == pop->programs[0] ? "Run" : "Generation";

  printf("Generation %d Summary:\n", genNum);
  printf("Hits: Best=%d, Average=%d, Worst=%d\n", hitsBest, hitsTotal,
	 hitsWorst);
  printf("Standardized Fitness: Best=%f, Average=%f, Worst=%f\n", stdBest,
	 stdTotal, stdWorst);
  printf("Normalized Fitness: Best=%f, Average=%f, Worst=%f\n", normBest,
	 normTotal, normWorst);
  printf("Depth: Deepest=%d, Average=%d, Most Shallow=%d\n", deepest,
	 depthTotal, shallowest);
  printf("Structural Complexity: ");
  printf("Best Of %s=%d", bestName, programInternalNodes(pop->programs[0]) +
	 programTerminalNodes(pop->programs[0]));
  printf(", Average=%d\n", complexity);
  if (countUnique)
    printf("%% of Unique Programs: %f\n",
	   (unique * 100.0)/(double )pop->populationSize);
#ifdef TRACK_BREED_TYPE
  printf("Percentage of programs generated by:\n");
  for (i = 0; i < MAX_BREED_VALUES; i++)
    if (breedTotal[i] > 0)
      printf("  %25s %5.2f%% (%d total)\n", breedName[i],
	     ((double )breedTotal[i]*100)/pop->populationSize, breedTotal[i]);
#endif /* TRACK_BREED_TYPE */
  printf("Best Of %s Program:\n", bestName);
  programDump(pop->programs[0], 1);
  printf("\n");
}

population *
populationReadCheckpoint(infile, ip, maxLoops, parsimony)
FILE *infile;
const interface *ip;
int maxLoops;
double parsimony;
{
  charString *cstr;
  const char *cp;
  int popSize, bestGen = 0;
  program *bestProg = 0;
  int progsize;
  population *pop;
  int i, notEOF;

  /* create an input buffer */
  cstr = charStringCreate();
  if (!cstr)
    return(0);

  /* read first string */
  if (charStringRead(cstr, infile) || charStringLength(cstr) == 0)
    return(0);

  /* make sure it's the population size */
  cp = charStringBuffer(cstr);
  if (strncmp(cp, "PopSize ", 8) != 0) {
    charStringFree(cstr);
    return(0);
  }
  sscanf(cp+8, "%d", &popSize);

  /* get the best of run program (if there is one) */
  if (charStringRead(cstr, infile) || charStringLength(cstr) == 0) {
    charStringFree(cstr);
    return(0);
  }
  cp = charStringBuffer(cstr);
  if (strncmp(cp, "BestGen ", 8) == 0) {

    /* get best generation */
    sscanf(cp+8, "%d", &bestGen);

    /* find start of program */
    for (cp += 8; *cp && !isspace(*cp); cp++)
      ;

    /* parse best-of-run program */
    charStringChop(cstr);
    bestProg = programParse(interfaceTerminalList(ip),
			    interfaceFunctionList(ip), cp);
    if (bestProg == 0) {
      charStringFree(cstr);
      return(0);
    }

    /* get next string */
    if (charStringRead(cstr, infile) || charStringLength(cstr) == 0) {
      charStringFree(cstr);
      programFree(bestProg);
      return(0);
    }
    cp = charStringBuffer(cstr);
  }

  /* make sure we're ready to read the programs */
  if (strncmp(cp, "Programs", 8) != 0) {
    charStringFree(cstr);
    programFree(bestProg);
    return(0);
  }

  /* read programs */
  progsize = sizeof(program *)*popSize;
  pop = (population *)malloc(sizeof(population) + progsize*2);
  if (pop) {

    /* initialize new program set */
    pop->populationSize = popSize;
    pop->programs = (program **)((char *)pop + sizeof(population));
    pop->workspace = (program **)((char *)pop->programs + progsize);
    pop->bestOfRun = bestProg;
    pop->bestGeneration = bestGen;

    /* parse all the programs */
    notEOF = 1;
    for (i = 0; i < popSize; i++) {

      /* read next program */
      if (notEOF)
	notEOF = !(charStringRead(cstr, infile) ||
		   charStringLength(cstr) == 0);

      /* parse program */
      if (notEOF) {
	charStringChop(cstr);
	pop->programs[i] = programParse(interfaceTerminalList(ip),
					interfaceFunctionList(ip),
					charStringBuffer(cstr));
	if (pop->programs[i] == 0)
	  fprintf(stderr, "Failed to parse \"%s\"\n", charStringBuffer(cstr));
      } else
	pop->programs[i] = 0;
    }
  }

  /* evaluate best-of-run program to get correct hits/fitness */
  if (pop->bestOfRun)
    interfaceEval(ip, pop->bestOfRun, maxLoops, parsimony);

  /* clean up */
  charStringFree(cstr);
  return(pop);
}

int
populationWriteCheckpoint(pop, outfile)
const population *pop;
FILE *outfile;
{
  charString *cstr;
  int i;

  cstr = charStringCreate();

  /* write general info */
  fprintf(outfile, "PopSize %d\n", pop->populationSize);

  /* write best of run program (if it exists) */
  if (pop->bestOfRun) {

    fprintf(outfile, "BestGen %d ", pop->bestGeneration);
    programCheckpoint(pop->bestOfRun, cstr);
    charStringCatenate(cstr, "\n");
    charStringWrite(cstr, outfile);
  }

  /* write programs */
  fprintf(outfile, "Programs\n");
  for (i = 0; i < pop->populationSize; i++) {
    charStringClear(cstr);
    programCheckpoint(pop->programs[i], cstr);
    charStringCatenate(cstr, "\n");
    charStringWrite(cstr, outfile);
  } 

  /* clean up */
  charStringFree(cstr);
  return(0);
}

void
populationLoop(pop, ip, generations, maxLoops, parsimony, alwaysEval,
	       maxMutateDepth, maxTotalDepth, selector, pctReproduce,
	       pctXoverAny, pctXoverInternal, verbose, countUnique)
population *pop;
const interface *ip;
int generations, maxLoops;
double parsimony;
int alwaysEval, maxMutateDepth, maxTotalDepth;
parentSelectionMethod selector;
double pctReproduce, pctXoverAny, pctXoverInternal;
int verbose, countUnique;
{
  int g, done;

  for (g = 0; g < generations; g++) {
    if (g)
      populationBreed(pop, ip, selector, pctReproduce, pctXoverAny,
		      pctXoverInternal, maxMutateDepth, maxTotalDepth);
    done = populationEval(pop, ip, maxLoops, alwaysEval, parsimony, g);
    populationDump(pop, g, verbose, countUnique);
    if (done)
      break;
  }

  /* print final summary */
  printf("Best program found on generation %d\n", pop->bestGeneration);
  if (verbose)
    programDump(pop->bestOfRun, 1);
}

void
populationFree(pop)
population *pop;
{
  int i;

  /* free best-of-run if it's not from this generation */
  if (pop->bestOfRun != pop->programs[0])
    programFree(pop->bestOfRun);

  /* free all programs */
  for (i = 0; i < pop->populationSize; i++)
    programFree(pop->programs[i]);

  free(pop);
}

#ifdef DEBUG_POPULATION

#include <string.h>
#include "constntsrc.h"
#include "operatrsrc.h"
#include "optrivial.h"

static objectList *tstTerminals P((NOARGS));
static objectList *tstFunctions P((NOARGS));
static void tstInitialize P((interface *));
static void tstCaseFitness P((result *, int, int *, double *, double *,
			      void *));
static int tstTerminateRun P((int, double, double));
static void processArgs P((int, char *[]));
static void checkCkpt P((const population *, const interface *));
int main P((int, char *[]));

char *progname;
int initialDepth = 3;
int mutateDepth = 6;
int xoverDepth = 6;
int populationSize = 16;
long seed;
int verbose = 1;
int generations = 3;

const int fitnessCases = 1;

static objectList *
tstTerminals()
{
  objectList *list;
  constantSrc *csp;

  list = objectListCreate(1);
  csp = integerSrcCreate(0, 100);
  if (!csp || objectListAdd(list, csp)) {
    constantSrcFree(csp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static objectList *
tstFunctions()
{
  objectList *list;
  operatorSrc *osp;

  list = objectListCreate(2);

  osp = simpleOperatorSrcCreate("+", opAdd, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  osp = simpleOperatorSrcCreate("-", opSubtract, 2);
  if (objectListAdd(list, osp)) {
    operatorSrcFree(osp);
    objectListFree(list);
    return(0);
  }

  return(list);
}

static void
tstCaseFitness(rp, fc, hitp, rawp, stdp, envp)
result *rp;
int fc;
int *hitp;
double *rawp;
double *stdp;
void *envp;
{
  int val;

  val = resultInteger(rp);
  if (val < 0)
    val = 0;
  *hitp = val;
  *stdp = 100 - val/10;
}

static int
tstTerminateRun(hits, raw, std)
int hits;
double raw;
double std;
{
  return(hits >= 1000);
}

static void
tstInitialize(ip)
interface *ip;
{
  interfaceSetFitnessCases(ip, fitnessCases);
  interfaceSetTerminalList(ip, tstTerminals());
  interfaceSetFunctionList(ip, tstFunctions());
  interfaceCaseFitnessFunc(ip, tstCaseFitness);
  interfaceTerminateRunFunc(ip, tstTerminateRun);
}

static void
processArgs(argc, argv)
int argc;
char *argv[];
{
  int c, errflag = 0;
  extern char *optarg;

  /* process all arguments */
  while ((c = getopt(argc, argv, "c:g:i:m:p:qs:v")) != EOF)
    switch (c) {
    case 'c':
      xoverDepth = atoi(optarg);
      if (xoverDepth <= 0) {
	fprintf(stderr, "%s: Crossover depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'g':
      generations = atoi(optarg);
      if (generations <= 0) {
	fprintf(stderr, "%s: Generation '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'i':
      initialDepth = atoi(optarg);
      if (initialDepth <= 0) {
	fprintf(stderr, "%s: Initial depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'm':
      mutateDepth = atoi(optarg);
      if (mutateDepth <= 0) {
	fprintf(stderr, "%s: Mutation depth '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'p':
      populationSize = atoi(optarg);
      populationSize = atoi(optarg);
      if (populationSize <= 0) {
	fprintf(stderr, "%s: Population size '%s' must be greater than zero!\n",
		progname, optarg);
	errflag = 1;
      }
      break;
    case 'q':
      verbose = 0;
      break;
    case 's':
      seed = atoi(optarg);
      if (seed <= 0) {
	fprintf(stderr, "%s: Invalid random seed '%s'\n", progname, optarg);
	errflag = 1;
      }
      break;
    case 'v':
      verbose = 1;
      break;
    default:
      if (c != '?')
	fprintf(stderr, "%s: Bad argument '%c'\n", progname, c);
      errflag = 1;
      break;
    }

  /* die if there were problems */
  if (errflag) {
    fprintf(stderr, "Usage: %s", progname);
    fputs(" [-c crossoverDepth]", stderr);
    fputs(" [-g #generations]", stderr);
    fputs(" [-i initialDepth]", stderr);
    fputs(" [-m mutationDepth]", stderr);
    fputs(" [-p populationSize]", stderr);
    fputs(" [-q(uiet)]", stderr);
    fputs(" [-s seed]", stderr);
    fputs(" [-v(erbose)]", stderr);
    fputc('\n', stderr);
    exit(1);
  }
}

static void
checkCkpt(pop, ip)
const population *pop;
const interface *ip;
{
  char filename[L_tmpnam];
  FILE *ckpt;
  int rtnval;
  population *new;

  /* create a temporary file */
  tmpnam(filename);
  ckpt = fopen(filename, "w");
  if (!ckpt) {
    fprintf(stderr, "%s: Couldn't open temporary file!\n", progname);
    return;
  }

  /* write checkpoint info, close file */
  rtnval = populationWriteCheckpoint(pop, ckpt);
  fclose(ckpt);
  if (rtnval) {
    fprintf(stderr, "%s: Problems encountered while writing ckpt file\n",
	    progname);
  } else {

    /* reopen the checkpoint file for reading */
    ckpt = fopen(filename, "r");
    if (!ckpt) {
      fprintf(stderr, "%s: Couldn't reopen checkpoint file!\n", progname);
    } else {

      /* read checkpoint file we just created */
      new = populationReadCheckpoint(ckpt, ip, 1000, 0.0);
      if (!new) {
	fprintf(stderr, "%s: Problems encountered while reading ckpt file!\n",
		progname);
	return;
      } else {

	/* compare the populations */
	if (!populationCompare(pop, new))
	  fprintf(stderr, "%s: Populations didn't match!\n", progname);

	/* get rid of new population */
	populationFree(new);
	printf("Population checkpoint works\n");
      }

      /* close checkpoint file */
      fclose(ckpt);
    }
  }

  /* get rid of temporary file */
  unlink(filename);
}

int
main(argc, argv)
int argc;
char *argv[];
{
  population *pop;
  interface *ip;
  int g, done;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
/*    dbmallopt(MALLOC_CKCHAIN, &moa); */
  }
#endif

  /* save program name */
  if ((progname = strrchr(argv[0], '/')) == 0)
    progname = argv[0];
  else
    progname++;

  /* initialize random number seed */
  seed = 123456789;

  /* process the argument list */
  processArgs(argc, argv);

  /* randomize things */
  srnd(seed);
  printf("Seed = %d\n", seed);

  /* create application interface */
  ip = interfaceCreate(tstInitialize);

  /* create population */
  pop = populationCreate(ip, populationSize, initialDepth, pcmRamped);

  /* evolve population */
  for (done = g = 0; !done && g < generations; g++) {
    if (g)
      populationBreed(pop, ip, psmFitnessProportionate, .30, .30, .30,
		      mutateDepth, xoverDepth);
    done = populationEval(pop, ip, 1000, 0, 0.0, g);
    populationDump(pop, g, verbose, 1);
  }

  checkCkpt(pop, ip);

  populationFree(pop);
  interfaceFree(ip);

#ifdef KEEP_ALLOCATED_MEMORY
  resultFreeStack();
  constantFreeStack();
  constantSrcFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_POPULATION */
