/*
SGPC: Simple Genetic Programming in C
(c) 1993 by Walter Alden Tackett and Aviram Carmi

 This code and documentation is copyrighted and is not in the public domain.
 All rights reserved.

 - This notice may not be removed or altered.

 - You may not try to make money by distributing the package or by using the
   process that the code creates.

 - You may not distribute modified versions without clearly documenting your
   changes and notifying the principal author.

 - The origin of this software must not be misrepresented, either by
   explicit claim or by omission.  Since few users ever read sources,
   credits must appear in the documentation.

 - Altered versions must be plainly marked as such, and must not be
   misrepresented as being the original software.  Since few users ever read
   sources, credits must appear in the documentation.

 - The authors are not responsible for the consequences of use of this
   software, no matter how awful, even if they arise from flaws in it.

If you make changes to the code, or have suggestions for changes,
let us know!  (gpc@ipld01.hac.com)
*/

#include <math.h>
#include "pmrand.h"

/* definitions for the random number generator (seed * GEN % MOD) */
#define GEN 16807
#define MOD 2147483647
#define QUO 127773
#define RES 2836

static int seed = 19920412;

static int PMrand P((int *));
static double park_miller_randomizer P((NOARGS));

/*
 * Park-Miller "minimal standard" pseudo random number generator
 * Implementation by Jan Jannink (c) 1992
 */
static int
PMrand(s)
int *s;
{
  int   tmp;

  tmp = (*s >> 16) * GEN;
  *s = (*s & 65535) * GEN + ((tmp & 32767) << 16) + (tmp >> 15);
  return (*s -= (*s < 0 ? MOD: 0));
}

static double
park_miller_randomizer()
{
  return (((double) PMrand(&seed)) / ((double) MOD));
}

void
srnd(s)
rndint s;
{
  seed = (int )s;
}

urndint
grnd()
{
  return((urndint )seed);
}

double
drnd()
{
  return (1.0-park_miller_randomizer());
}

rndint
irnd()
{
  return(PMrand(&seed));
}

/* Pick a random integer between low and high */
rndint
irndrng(low, high)
rndint low, high;
{
  rndint r;

  if (low >= high)
    r = low;
  else {
    r = (PMrand(&seed) % (high - low + 1)) + low;
    if (r > high)
      r = high;
  }
  return(r);
}

/* real random number between specified limits */
double
drndrng(lo ,hi)
double lo, hi;
{
  return ((1.0-park_miller_randomizer()) * (hi - lo)) + lo;
}

#ifdef DEBUG_PMRAND

#include "proto.h"

#define TESTMAX	20

int main P((NOARGS));

int
main()
{
  int i, j;
  rndint totals[TESTMAX+2];

  for (i = 2; i < TESTMAX; i++) {

    /* clear totals */
    for (j = 0; j <= i+1; j++)
      totals[j] = 0;

    /* create totals */
    for (j = 0; j < 1000*i; j++)
      totals[irndrng(1, i)]++;

    /* print totals */
    printf("For 0..%d\n", i);
    for (j = 0; j <= i+1; j++)
      printf("%5d: %d\n", j, totals[j]-1000);
  }

  for (i = 0; i < 20; i++)
    printf("%f\t%f\t%f\n", drnd(), drndrng(0.0, 0.5), drndrng(0.5, 1.0));

  exit(0);
}
#endif /* DEBUG_PMRAND */
