/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <malloc.h>
#include "genlist.h"
#include "proto.h"

genericList *
genericListCreate(elemsize, maxlen)
int elemsize;
int maxlen;
{
  genericList *glp;
  void *list;

  /* make sure params are valid */
  if (elemsize <= 0 || maxlen < 0)
    return(0);

  glp = (genericList *)malloc(sizeof(genericList));
  if (glp) {

    /* create a list if they wanted to preallocate some entries */
    if (maxlen > 0) {
      list = (void *)malloc((unsigned )(elemsize * maxlen));
      if (!list)
	maxlen = 0;
    }

    /* initialize everything else */
    glp->list = list;
    glp->used = 0;
    glp->allocated = maxlen;
    glp->elementSize = elemsize;
  }
  return(glp);
}

int
genericListExtend(glp, newlen)
genericList *glp;
int newlen;
{
  void *tmp;

  /* if we don't have a list yet... */
  if (!glp->list) {
    glp->list = (void *)malloc((unsigned )(glp->elementSize*newlen));
    if (!glp->list)
      return(-1);

    /* remember how many elements were allocated */
    glp->allocated = 16;
  }

  /* if the list needs to be extended... */
  if (newlen >= glp->allocated) {

    tmp = (void *)realloc(glp->list, (unsigned )(glp->elementSize * newlen));
    if (!tmp)
      return(-2);

    /* remember that we've got more room */
    glp->allocated = newlen;
    glp->list = tmp;
  }

  /* extended the list */
  return(0);
}

int
genericListAdd(glp, vp)
genericList *glp;
const void *vp;
{
  void *el;

  /* make sure there's enough room */
  if (!glp->list || glp->allocated <= glp->used)
    if (genericListExtend(glp, glp->used + 1) != 0)
      return(-1);

  /* add element */
  el = (void *)((unsigned long )glp->list + (glp->elementSize * glp->used++));
  bcopy(vp, el, (unsigned )glp->elementSize);

  /* success */
  return(0);
}

int
genericListSet(glp, index, vp)
genericList *glp;
int index;
const void *vp;
{
  void *el;

  /* make the array is long enough to include 'index' */
  if (!glp->list || glp->allocated <= index)
    if (genericListExtend(glp, index) != 0)
      return(-1);

  if (glp->used >= index)
    el = (void *)((unsigned long )glp->list + (glp->elementSize * index));
  else {

    /* clear preceding elements */
    el = (void *)((unsigned long )glp->list + (glp->elementSize * glp->used));
    while (glp->used < index) {
      bzero(el, (unsigned )glp->elementSize);
      (glp->used)++;
      el = (void *)((unsigned long )el + glp->elementSize);
    }
  }

  /* set element */
  bcopy(vp, el, (unsigned )glp->elementSize);

  /* success */
  return(0);
}

genericList *
genericListCopy(glp)
const genericList *glp;
{
  genericList *nglp;

  nglp = genericListCreate(glp->elementSize, glp->allocated);
  if (nglp) {
    nglp->elementSize = glp->elementSize;
    bcopy(glp->list, nglp->list, (unsigned )(glp->used * glp->elementSize));
    nglp->used = glp->used;
  }

  return(nglp);
}

void
genericListFree(glp)
genericList *glp;
{
  free(glp->list);
  free(glp);
}

#ifdef DEBUG_GENERICLIST

#define TEST_LENGTH	10

int main P((NOARGS));

int
main()
{
  genericList *glp, *nglp;
  int i, ival[TEST_LENGTH];

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  glp = genericListCreate(sizeof(int *), TEST_LENGTH - 1);
  if (!glp) {
    printf("Create failed!\n");
    exit(1);
  }

  for (i = 0; i < TEST_LENGTH; i++) {
    ival[i] = TEST_LENGTH - i;
    if (genericListAdd(glp, (void *)&(ival[i])))
      printf("Add failed for %d\n", i);
  }

  printf("%d elements in list\n", genericListLength(glp));
  for (i = 0; i < genericListLength(glp); i++)
    printf("Element %d = %d\n", i, *(int *)genericListEntry(glp, i));

  nglp = genericListCopy(glp);
  printf("%d elements in copied list\n", genericListLength(nglp));
  for (i = 0; i < genericListLength(nglp); i++)
    printf("Element %d = %d\n", i, *(int *)genericListEntry(nglp, i));

  genericListFree(nglp);
  genericListFree(glp);

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_GENERICLIST */
