/*
 * Copyright (C) 1993 by Dave Glowacki
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdio.h>
#include <malloc.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "constant.h"
#include "proto.h"

#ifdef KEEP_ALLOCATED_MEMORY
static constant *freeStack = 0;
#endif /* KEEP_ALLOCATED_MEMORY */

constant *
#ifdef __STDC__
constantCreate(datatype dt, ...)
#else
constantCreate(va_alist)
va_dcl
#endif
{
  va_list ap;
#ifndef __STDC__
  datatype dt;
#endif
  constant *cp;
  datatype rdt;

#ifdef __STDC__
  va_start(ap, dt);
#else /* !__STDC__ */
  va_start(ap);
  dt = va_arg(ap, datatype);
#endif /* __STDC__ */

  /* create a new instance of constant */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    cp = freeStack;
    freeStack = *(constant **)cp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    cp = (constant *)malloc(sizeof(constant));

  /* fill the new constant instance */
  if (cp) {
    cp->otype = otConstant;
    cp->dtype = dt;

    /* make sure they've specified a valid datatype */
    rdt = datatypeResolve(dt);
    switch (rdt) {
    case dtBoolean:
      cp->v.b = (bool )(va_arg(ap, int) ? 1 : 0);
      break;
    case dtShort:
      cp->v.s = (short )va_arg(ap, int);
      break;
    case dtInteger:
      cp->v.i = va_arg(ap, int);
      break;
    case dtLong:
      cp->v.l = (long )va_arg(ap, int);
      break;
    case dtFloat:
      cp->v.f = (float )va_arg(ap, double);
      break;
    case dtDouble:
      cp->v.d = va_arg(ap, double);
      break;
    case dtBlob:
      cp->v.bbp = va_arg(ap, blob *);
      break;
    case dtVoid:
    case dtList:
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
    case dtError:
      free(cp);
      cp = 0;
      break;
    }
  }

  /* clean up and cruise */
  va_end(ap);
  return(cp);
}

constant *
constantCopy(cp)
const constant *cp;
{
  constant *ncp;

  /* create a new instance of constant */
#ifdef KEEP_ALLOCATED_MEMORY
  if (freeStack) {
    ncp = freeStack;
    freeStack = *(constant **)ncp;
  } else
#endif /* KEEP_ALLOCATED_MEMORY */
    ncp = (constant *)malloc(sizeof(constant));

  /* copy this constant */
  if (ncp) {
    memcpy(ncp, cp, sizeof(constant));
    if (datatypeResolve(ncp->dtype) == dtBlob)
      ncp->v.bbp = blobCopy(cp->v.bbp, cp->dtype);
  }

  return(ncp);
}

int
constantCompare(cp1, cp2)
const constant *cp1, *cp2;
{
  datatype rdt;
  int cmp = 0;

  if (cp1->dtype == cp2->dtype)
    rdt = datatypeResolve(cp1->dtype);
    switch (rdt) {
    case dtBoolean:
      cmp = (cp1->v.b == cp2->v.b);
      break;
    case dtShort:
      cmp = (cp1->v.s == cp2->v.s);
      break;
    case dtInteger:
      cmp = (cp1->v.i == cp2->v.i);
      break;
    case dtLong:
      cmp = (cp1->v.l == cp2->v.l);
      break;
    case dtFloat:
      cmp = (cp1->v.f >= (cp2->v.f - 0.0000000001) &&
	     cp1->v.f <= (cp2->v.f + 0.0000000001));
      break;
    case dtDouble:
      cmp = (cp1->v.d >= (cp2->v.d - 0.0000000001) &&
	     cp1->v.d <= (cp2->v.d + 0.0000000001));
      break;
    case dtBlob:
      cmp = blobCompare(cp1->v.bbp, cp2->v.bbp, cp1->dtype);
      break;
    case dtVoid:
    case dtList:
    case dtUserDef0:
    case dtUserDef1:
    case dtUserDef2:
    case dtUserDef3:
    case dtUserDef4:
    case dtUserDef5:
    case dtUserDef6:
    case dtUserDef7:
    case dtUserDef8:
    case dtUserDef9:
    case dtError:
      break;
  }
  return(cmp);
}

result *
constantEval(cp)
const constant *cp;
{
  datatype rdt;
  result *rp = 0;

  rdt = datatypeResolve(cp->dtype);
  switch (rdt) {
  case dtBoolean:
    rp = resultCreate(dtBoolean, (int )(cp->v.b));
    break;
  case dtShort:
    rp = resultCreate(dtShort, (int )(cp->v.s));
    break;
  case dtInteger:
    rp = resultCreate(dtInteger, cp->v.i);
    break;
  case dtLong:
    rp = resultCreate(dtLong, (int )(cp->v.l));
    break;
  case dtFloat:
    rp = resultCreate(dtFloat, (double )(cp->v.f));
    break;
  case dtDouble:
    rp = resultCreate(dtDouble, cp->v.d);
    break;
  case dtBlob:
    rp = resultCreate(cp->dtype, cp->v.bbp);
    break;
  case dtVoid:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
  case dtError:
    rp = resultCreate(dtError, ErrorBadDataType);
    break;
  }
  return(rp);
}

const constant **
constantNodePtr(cpp, count, nodeNum, internal, typeptr)
const constant **cpp;
int *count;
int nodeNum, internal;
datatype *typeptr;
{
  /* constants are terminal points */
  if (internal)
    return(0);

  /* see if we're the node they want */
  if (((*count)++ >= nodeNum) &&
      ((*typeptr == 0) || ((*typeptr & (*cpp)->dtype) != 0)))
    return(cpp);

  /* we aren't the right node */
  return(0);
}

int
constantToString(cp, cstr)
const constant *cp;
charString *cstr;
{
  datatype rdt;
  int rval = 1;
  char buffer[128];
  int i, j;

  rdt = datatypeResolve(cp->dtype);
  switch (rdt) {
  case dtBoolean:
    rval = charStringCatenate(cstr, cp->v.b ? "TRUE" : "FALSE");
    break;
  case dtShort:
    sprintf(buffer, "%dS", cp->v.s);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtInteger:
    sprintf(buffer, "%d", cp->v.i);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtLong:
    sprintf(buffer, "%dL", cp->v.l);
    rval = charStringCatenate(cstr, buffer);
    break;
  case dtFloat:
  case dtDouble:
    if (rdt == dtFloat)
      sprintf(buffer, "%21.10fF", cp->v.f);
    else
      sprintf(buffer, "%21.10f", cp->v.d);

    /* lose blanks */
    for (i = j = 0; buffer[i]; i++)
      if (!isspace(buffer[i]))
	buffer[j++] = buffer[i];
    buffer[j] = 0;

    rval = charStringCatenate(cstr, buffer);
    break;
  case dtBlob:
    rval = charStringCatenate(cstr, "BLOB:");
    if (!rval) {
      rval = blobToString(cp->v.bbp, cp->dtype, cstr);
      if (!rval)
	rval = charStringCatenate(cstr, ":BLOB");
    }
    break;
  case dtVoid:
  case dtList:
  case dtUserDef0:
  case dtUserDef1:
  case dtUserDef2:
  case dtUserDef3:
  case dtUserDef4:
  case dtUserDef5:
  case dtUserDef6:
  case dtUserDef7:
  case dtUserDef8:
  case dtUserDef9:
  case dtError:
    sprintf(buffer, "<Illegal Constant Type '%d'>", cp->dtype);
    rval = charStringCatenate(cstr, buffer);
    break;
  }
  return(rval);
}

void
constantFree(cp)
constant *cp;
{
  /* free blob if we've got one */
  if (datatypeResolve(cp->dtype) == dtBlob)
    blobFree(cp->v.bbp, cp->dtype);

#ifdef KEEP_ALLOCATED_MEMORY
  *(constant **)cp = freeStack;
  freeStack = cp;
#else /* !KEEP_ALLOCATED_MEMORY */
  free(cp);
#endif /* KEEP_ALLOCATED_MEMORY */
}
    
#ifdef KEEP_ALLOCATED_MEMORY
void
constantFreeStack()
{
  constant *cp;

  while (freeStack) {
    cp = freeStack;
    freeStack = *(constant **)cp;
    free(cp);
  }
}
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef DEBUG_CONSTANT

int main P((NOARGS));

int
main()
{
  charString *cstr;
  constant *cp, *ncp;
  bool bval;
  short sval;
  int ival;
  long lval;
  float fval;
  double dval;

#ifdef _DEBUG_MALLOC_INC
  {
    union dbmalloptarg	  moa;

    moa.i = 1;
    dbmallopt(MALLOC_CKCHAIN, &moa);
  }
#endif

  cstr = charStringCreate();

  cp = constantCreate(dtBoolean, 1);
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = constantCreate(dtShort, 2);
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = constantCreate(dtInteger, 3);
  ival = 3;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = constantCreate(dtLong, 4);
  lval = 4;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = constantCreate(dtFloat, 5.0);
  fval = 5.6;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = constantCreate(dtDouble, 6.0);
  fval = 7.8;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = booleanCreate(0);
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = shortCreate(0);
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = integerCreate(0);
  ival = 3;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = longCreate(0);
  lval = 4;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = floatCreate(0.0);
  fval = 5.6;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = doubleCreate(0.0);
  fval = 7.8;
  charStringSet(cstr, "Constant is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  bval = 0;
  cp = booleanCreate(bval);
  charStringSet(cstr, "Boolean is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  sval = 333;
  cp = shortCreate(sval);
  charStringSet(cstr, "Short is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  ival = 666;
  cp = integerCreate(ival);
  charStringSet(cstr, "Integer is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  lval = 999;
  cp = longCreate(lval);
  charStringSet(cstr, "Long is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  fval = 3.141593;
  cp = floatCreate(fval);
  charStringSet(cstr, "Float is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  dval = 3.141593;
  cp = doubleCreate(dval);
  charStringSet(cstr, "Double is ");
  constantToString(cp, cstr);
  charStringPrint(cstr);
  constantFree(cp);

  cp = integerCreate(12345); ncp = constantCopy(cp);
  charStringSet(cstr, "Original is ");
  constantToString(cp, cstr);
  charStringCatenate(cstr, ", copy is ");
  constantToString(ncp, cstr);
  charStringPrint(cstr);
  constantFree(cp);
  constantFree(ncp);

  charStringFree(cstr);

#ifdef KEEP_ALLOCATED_MEMORY
  constantFreeStack();
#endif /* KEEP_ALLOCATED_MEMORY */

#ifdef _DEBUG_MALLOC_INC
  malloc_dump(1);
#endif

  return(0);
}
#endif /* DEBUG_CONSTANT */
