;;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: GECO -*-

(in-package :GECO)
#|
Genetic Evolution through Combination of Objects (GECO)

Copyright (C) 1992  George P. W. Williams, Jr.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
|#

;;; Selection methods

;; This first routine doesn't even pick based on fitness!!

(defmethod PICK-RANDOM-ORGANISM-INDEX ((p population))
  "Returns the index number of a random organism from the population."
  (random (size p)))

(defmethod PICK-RANDOM-ORGANISM ((p population))
  "Returns a random organism from the population."
  (aref (organisms p) (pick-random-organism-index p)))

;;=======================================================

(defun ROULETTE-PICK-RANDOM-WEIGHT-INDEX (weights-table size)
  "Select a random index into a table of weights, using the roulette wheel approach."
  ;; assumes that the weights-table has been normalized to sum to 1.0
  (do ((rand (random size))
       (partial-sum 0.0)
       (i 0 (1+ i)))
      ((or (>= i (1- size))
           (>= partial-sum rand))
       i)
    (incf partial-sum (aref weights-table i))))

(defmethod ROULETTE-PICK-RANDOM-ORGANISM-INDEX ((pop population))
  "Select a random organism number from the population, weighted by fitness, using the roulette wheel approach,
as used in DeJong's R1; also known as Brindle's stochastic sampling with replacement."
  ;; assumes that the population has been initialized,
  ;; and that (sum-normalized-fitness (statistics pop)) is valid [and all that implies]
  (do ((rand (random (size pop)))
       (partial-sum 0.0)(i 0 (1+ i)))
      ((or (>= i (1- (size pop)))
           (>= partial-sum rand))
       i)
    (incf partial-sum (normalized-fitness (aref (organisms pop) i)))))

(defmethod ROULETTE-PICK-RANDOM-ORGANISM ((pop population))
  "Select a random organism from the population, weighted by fitness, using the roulette wheel approach,
as used in DeJong's R1; also known as Brindle's stochastic sampling with replacement."
  (aref (organisms pop)
        (roulette-pick-random-organism-index pop)))

;;=======================================================

#| The following technique is generally considered to be one of the best (most error free) selection techniques. Booker's
(1982) investigation of machine learning demonstrated it's superiority over DeJong's expected-value model (stochastic
sampling without replacement). |#

(defmethod STOCHASTIC-REMAINDER-PRESELECT ((pop population) &KEY (multiplier 1) &AUX
                                           (pop-size (size pop)))
  "Prepare and return a function of no arguments which will select random organisms
from the population, weighted by fitness, using a technique referred to by Brindle as
stochastic remainder selection without replacement."
  (let ((choices (make-array pop-size :element-type 'fixnum))
        (choice# 0)
        (fractions (make-array pop-size :element-type 'short-float))
        (organisms (organisms pop))
        (avg-normalized-fitness (avg-normalized-fitness (statistics pop))))
    (declare (dynamic-extent fractions))
    ;; First, assign whole numbers:
    (dotimes (organism# pop-size)
      (multiple-value-bind (expected frac)
                           (truncate (normalized-fitness (aref (organisms pop) organism#))
                                     avg-normalized-fitness)
        (setf (aref fractions organism#) (float frac))
        (dotimes (i expected)
          (setf (aref choices choice#) organism#)
          (incf choice#))))
    ;; then assign fractional parts:
    (do ((organism# 0 (mod (1+ organism#) pop-size))
         (mult (/ 1.0 multiplier)))     ; used when the user consumes more organisms per use than it produces
        ((>= choice# pop-size))
      (when (and (> (aref fractions organism#) 0)
                 (< (random 1.0) (aref fractions organism#)))
        (setf (aref choices choice#) organism#)
        (incf choice#)
        (decf (aref fractions organism#) mult)))
    ;; Return a function which will return organisms from the chosen, one at a time, until they are all gone
    ;; and will return NIL thereafter.
    #'(lambda (&AUX (choice# (if (> pop-size 0) (random pop-size))))
        (when choice#
          (decf pop-size)
          (prog1
            (aref organisms (aref choices choice#))   ; the function's result
            (setf (aref choices choice#) (aref choices pop-size)))))))

#| -- an example of using this function:
(let ((selector (stochastic-remainder-preselect some-population)))
  (do ((organism (funcall selector) (funcall selector)))
      ((null organism))
    (do-something-with organism)))

-- Note that all the storage held by the closure can be reclaimed as garbage
when the scope of the let binding the selector function is exited. 
|#
