;;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: GECO -*-

(in-package :GECO)
#|
Genetic Evolution through Combination of Objects (GECO)

Copyright (C) 1992  George P. W. Williams, Jr.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
|# #|
This file illustrates use of GECO to implement two different genetic algorithms to solve a simple problem.
The problem is often called 'count ones,' it simply maximizes the one-bits in the chromosome.
The two genetic algorithms (genetic plans in Holland's terminology) are similar to Holland's classic algorithm
(though they both use uniform crossover), and a variation on Holland's algorithm which uses a different selection
procedure discussed by Shaffer in his paper "Some Effects of Selection Procedures on Hyperplane Sampling by
Genetic Algorithms"  (in Davis' 1987 collection Genetic Algorithms and Simulated Annealing). 
|#

(defclass BINARY-CHROMOSOME-10 (binary-chromosome)
  ((SIZE
    :reader size
    :type fixnum
    :initform 10
    :allocation :class)))

(defclass SIMPLE-BINARY-10-ORGANISM (organism)
  ((CHROMOSOME-CLASSES
    :initform (list 'binary-chromosome-10)
    :reader chromosome-classes
    :allocation :class)))

(defclass BINARY-POPULATION-STATISTICS (population-statistics)
  ((ALLELE-COUNTS
    :accessor allele-counts
    :initform nil
    :type (or null (vector fixnum 10)))))

(defclass SIMPLE-BINARY-POPULATION (generational-population)
  ((ORGANISM-CLASS
    :initform 'simple-binary-10-organism
    :reader organism-class
    :allocation :class)
   (POPULATION-STATISTICS-CLASS
    :initform 'binary-population-statistics
    :reader population-statistics-class
    :allocation :class)))

(defmethod COMPUTE-STATISTICS :AFTER ((pop-stats binary-population-statistics))
  (setf (allele-counts pop-stats)
        (compute-binary-allele-statistics (population pop-stats))))

(defclass SIMPLE-PLAN (genetic-plan)
  ()
  (:documentation
   "Abstract class to allow method sharing for initialization & regeneration."))

(defmethod SHARED-INITIALIZE :AFTER ((self simple-plan) slots-for-initform &REST initargs)
  (declare (ignore slots-for-initform initargs))
  (setf (statistics self) '()))         ; so we can push instances

(defmethod EVALUATE ((self simple-binary-10-organism) (plan simple-plan)
                     &AUX (chromosome (first (genotype self))))
  (declare (ignore plan))
  (setf (fitness self)
        (count-allele-values chromosome 0 (size chromosome) 1)))

(defmethod REGENERATE ((plan simple-plan) (old-pop simple-binary-population) &AUX
                       (pop-size (size old-pop))
                       (new-pop (make-population (ecosystem old-pop)
                                                 (class-of old-pop)
                                                 :size pop-size)))
  (setf (previous-population new-pop) old-pop)
  (setf (ecosystem new-pop) (ecosystem old-pop))
  ;; selectively reproduce, crossover, and mutate
  (operate-on-population plan old-pop new-pop)
  ;; record old-pop's statistics
  (push (statistics old-pop)     ; impractical for real-world problems
        (statistics plan))
  new-pop)

(defclass SIMPLE-PLAN-1 (simple-plan)
  ((PROB-MUTATE
    :reader prob-mutate
    :initform 0.03
    :type short-float
    :allocation :class
    :documentation
    "This is the probability of mutating an organism, not a single locus as is often used.")
   (PROB-CROSS
    :reader prob-cross
    :type short-float
    :initform 0.7
    :allocation :class)))

(defmethod OPERATE-ON-POPULATION
           ((plan simple-plan-1) old-population new-population &AUX
            (new-organisms (organisms new-population))
            (p-cross (prob-cross plan))
            (p-mutate (+ p-cross (prob-mutate plan))))
  (let ((selector (stochastic-remainder-preselect old-population)))
    (do ((org1 (funcall selector) (funcall selector))
         org2
         (random# (random 1.0) (random 1.0))
         (i 0 (1+ i)))
        ((null org1))
      (cond
       ((> p-cross random#)
        (if (< 1 (hamming-distance (first (genotype org1))
                                   (first (genotype (setf org2 (pick-random-organism
                                                                old-population))))))
            (uniform-cross-organisms
             org1 org2
             (setf (aref new-organisms i)
                   (copy-organism org1 :new-population new-population))
             (copy-organism (aref (organisms old-population) 0)))
          ;; hamming distances < 2 will produce eidetic offspring anyway, so bypass crossover & evaluation
          (setf (aref new-organisms i)
                (copy-organism-with-fitness org1 :new-population new-population))))
       ((> p-mutate random#)
        (mutate-organism
         (setf (aref new-organisms i)
               (copy-organism org1 :new-population new-population))))
       (T ;; copying the fitness bypasses the need for a redundant evaluate
        (setf (aref new-organisms i)
              (copy-organism-with-fitness org1 :new-population new-population)))))))

(defclass SIMPLE-PLAN-2 (simple-plan)
  ((PROB-MUTATE
    :reader prob-mutate
    :initform 0.03
    :type short-float
    :allocation :class
    :documentation
    "This is the probability of mutating an organism, not a single locus as is often used.")
   (PROB-CROSS
    :reader prob-cross
    :initform 0.7
    :type short-float
    :allocation :class)))

(defmethod OPERATE-ON-POPULATION
           ((plan simple-plan-2) old-population new-population &AUX
            (new-organisms (organisms new-population))
            (p-cross (prob-cross plan))
            (p-mutate (+ p-cross (prob-mutate plan))))
  (let ((selector (stochastic-remainder-preselect old-population)))
    (do* ((org1 (funcall selector) (funcall selector))
          org2
          (random# (random 1.0) (random 1.0))
          (i 0 (1+ i)))
        ((null org1))
      (cond
       ((> p-cross random#)
        (if (and (setq org2 (funcall selector))
                 (< 1 (hamming-distance (first (genotype org1))
                                        (first (genotype org2)))))
            (uniform-cross-organisms
             org1 org2
             (setf (aref new-organisms i)
                   (copy-organism org1 :new-population new-population))
             (setf (aref new-organisms (1+ i))
                   (copy-organism org2 :new-population new-population)))
          (progn ;; hamming distances < 2 will produce eidetic offspring anyway, so bypass crossover & evaluation
            (setf (aref new-organisms i)
                  (copy-organism-with-fitness org1 :new-population new-population))
            (when org2
              (setf (aref new-organisms (1+ i))
                    (copy-organism-with-fitness org2 :new-population new-population)))))
        (incf i))                       ;  because we just added an extra to the new population
       ((> p-mutate random#)
        (mutate-organism (aref new-organisms i)))
       (T ;; copying the fitness bypasses the need for a redundant evaluate
        (setf (aref new-organisms i)
              (copy-organism-with-fitness org1 :new-population new-population)))))))

;;; Some stuff to test the algorithms:

(defvar *SBE* nil "a simple binary ecosystem")

(defun TEST-PLAN (times plan &KEY
                        (stream t)
                        (pop-size 20)
                        (evaluation-limit 400))
  (let (maxs avgs gens evals)
    (flet ((test ()
             (dotimes (i times)
               (setq *sbe* (make-instance 'ecosystem
                             :pop-class 'simple-binary-population
                             :pop-size pop-size
                             :plan-class plan
                             :evaluation-limit evaluation-limit))
               (evolve *sbe*)
               (format t "~& -- Max=~F, Avg=~F, #gens=~D, #evals=~D."
                       (max-fitness (statistics (population *sbe*)))
                       (avg-fitness (statistics (population *sbe*)))
                       (generation-number *sbe*)
                       (evaluation-number *sbe*))
               (push (max-fitness (statistics (population *sbe*))) maxs)
               (push (avg-fitness (statistics (population *sbe*))) avgs)
               (push (generation-number *sbe*) gens)
               (push (evaluation-number *sbe*) evals))))
      (format stream "~&~A:" plan)
      (time (test)))
    (format stream "~& ==> Avg max=~F, Avg avg=~F, Avg #gens=~F, Avg #evals=~F"
            (/ (float (reduce #'+ maxs)) times)
            (/ (float (reduce #'+ avgs)) times)
            (/ (float (reduce #'+ gens)) times)
            (/ (float (reduce #'+ evals)) times))))

#|
(evaluate (population *sbe*) (plan *sbe*))
(inspect *sbe*)
(evolve *sbe*)
(dbg :converge)
(undbg)
(progn (test-plan 1 'simple-plan-1)
       (inspect *sbe*))
(progn (test-plan 1 'simple-plan-2)
       (inspect *sbe*))
(test-plan 10 'simple-plan-1)
(test-plan 10 'simple-plan-2)
|#
