;;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: GECO -*-

(in-package :GECO)
#|
Genetic Evolution through Combination of Objects (GECO)

Copyright (C) 1992  George P. W. Williams, Jr.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
|#

;;; population methods

(defmethod SHARED-INITIALIZE :AFTER ((pop population) slots-for-initform
                                     &REST initargs)
  (declare (ignore slots-for-initform initargs))
  (when (size pop)
    (setf (organisms pop) (make-population-vector pop (size pop)))
    (make-organisms pop)))

(defmethod MAKE-POPULATION-VECTOR ((self population) size)
  (make-array size))

(defmethod MAKE-ORGANISM ((pop population))
  (make-instance (organism-class pop)
    :population pop))

(defmethod MAKE-ORGANISMS ((pop population) &AUX
                             (organisms (organisms pop)))
  (dotimes (i (size pop))
    (setf (aref organisms i)
          (make-organism pop))))

(defmethod COPY-POPULATION ((self population)
                            &AUX (pop-size (size self))
                            (new-self (make-population (ecosystem self)
                                                       (class-of self)
                                                       :size pop-size))
                            (new-organisms (organisms new-self))
                            (old-organisms (organisms self)))
  (setf (ecosystem new-self) (ecosystem self))
  (dotimes (i pop-size)
    (setf (aref new-organisms i) (copy-organism (aref old-organisms i)
                                                :new-population new-self)))
  new-self)

(defmethod EVALUATE ((pop population) (plan genetic-plan) &AUX
                     (orgs (organisms pop))
                     (ecosys (ecosystem pop)))
  (dotimes (i (length orgs))
    (let ((org  (aref orgs i)))
      (if (dbg-p :fitness)
          (if (fitness org)
              (let ((orig-fitness (fitness org)))
                (evaluate org plan)
                (incf (evaluation-number ecosys))
                (when (/= orig-fitness (fitness org))
                  (dbgo "~%**** bad fitness: orig=~F, now=~F"
                        orig-fitness (fitness org))
                  (break)))))
      (unless (fitness org)
        (evaluate org plan)
        (incf (evaluation-number ecosys)))))
  (normalize-fitness pop
                     (setf (statistics pop) (make-population-statistics pop))
                     plan))

(defmethod MAKE-POPULATION-STATISTICS ((pop population))
  (make-instance (population-statistics-class pop)
    :population pop))

(defmethod COMPUTE-STATISTICS ((pop population))
  "This method should only be used if the statistics need to be recomputed.
They are initially computed when the population-statistics instance is first created for the population."
  (compute-statistics (statistics pop)))

(defmethod COMPUTE-BINARY-ALLELE-STATISTICS ((population population))
  "Returns a list of vectors (one per chromosome in the organisms of the population) of counts by locus of non-zero alleles."
  (with-accessors ((orgs organisms))
                  population
    (let ((counts-list (mapcar #'(lambda (chr)
                                   (make-array (size chr)
                                               :element-type 'fixnum
                                               :initial-element 0))
                               ;; assume chromosomes of all organisms are the same size
                               (genotype (aref orgs 0)))))
      (dotimes (org# (size population))
        (do* ((chromosome-ptr (genotype (aref orgs org#))
                              (rest chromosome-ptr))
              (chromosome (first chromosome-ptr)
                          (first chromosome-ptr))
              (counts-ptr counts-list (rest counts-ptr))
              (counts (first counts-ptr)
                      (first counts-ptr)))
             ((null chromosome-ptr))
          (dotimes (locus# (size chromosome))
            (if (/= 0 (locus chromosome locus#))
                (incf (aref counts locus#))))))
      counts-list)))

(defmethod NORMALIZE-FITNESS ((pop population) stats (plan genetic-plan)
                              &AUX (orgs (organisms pop)))
  "Normalize all the fitness values for each organism in the population, according to the
genetic plan plan, and update the population-statistics with normalized values
(using compute-normalized-statistics)."
  (unless (= (max-fitness stats) (min-fitness stats))
    ;; don't normalize if the population is completely converged, or we'll get arithmetic exceptions
    (dotimes (i (length orgs))
      (normalize-fitness (aref orgs i) stats plan))
    (compute-normalized-statistics stats)))

#| There are a number of different ways to normalize the fitness values. With some plans and evaluation
functions, it may not even be necessary, though beware that the fitness should always be >= 0. See Chapter
4 of Goldberg's book, under the sections on Scaling Mechanisms and Ranking Procedures. Note that some
selection procedures are also based on ranking. |#

(defmethod CONVERGED-P ((pop population) &AUX
                        (pop-size (size pop))
                        (organisms (organisms pop))
                        (stats (statistics pop))
                        threshold
                        (passing-count 0))
  "Return non-NIL if the population has converged.
This method requires the convergence-fraction and convergence-threshold-margin
functions on the population."
  (if (dbg-p :converge)
      (dbgo "~&---CONVERGE-P: generation=~4D, threshold=~,3F"
            (generation-number (ecosystem pop)) threshold))
  (if (= (max-fitness stats) (min-fitness stats))
      (progn (if (dbg-p :converge)
                 (dbgo "~&   population is completely converged, fitness=~F"
                       (max-fitness stats)))
             T)
    (progn 
      (setq threshold (convergence-threshold-margin pop))
      (dotimes (i pop-size)
        (if (>= (normalized-fitness (aref organisms i)) threshold)
            (incf passing-count)))
      (if (dbg-p :converge)
          (dbgo "~&   passing-count=~3D and passing-fraction=~,3F"
                passing-count (float (/ passing-count pop-size))))
      (>= (float (/ passing-count pop-size))
          (convergence-fraction pop)))))

;;; The following could be :allocation :per-class slots if/when this can be implemented efficiently & portably:

(defmethod POPULATION-STATISTICS-CLASS ((pop population))
  'population-statistics)

(defmethod CONVERGENCE-FRACTION ((pop population))
  0.95)

(defmethod CONVERGENCE-THRESHOLD-MARGIN ((pop population))
  0.95)
