;;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: GECO -*-

(in-package :GECO)
#|
Genetic Evolution through Combination of Objects (GECO)

Copyright (C) 1992  George P. W. Williams, Jr.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
|#

(defmethod SHARED-INITIALIZE :AFTER ((organism organism) slots-for-initform &REST initargs)
  (declare (ignore slots-for-initform initargs))
  (if (null (genotype organism))
      (make-random-chromosomes organism)))

(defmethod PRINT-OBJECT ((self organism) stream)
  (print-unreadable-object (self stream :type t :identity t)
    (format stream "~@[Fit=~F~]~@[, Norm=~,2F~] ~A"
            (fitness self)
            (normalized-fitness self)
            (genotype-printable-form self))))

(defmethod SIZE ((self organism))
  (reduce #'+ (mapcar #'size (genotype self))))

(defmethod COPY-ORGANISM ((self organism) &KEY (new-population (population self))
                          &AUX (new-self (make-organism new-population))
                          result)
  (setf (genotype new-self)
        (dolist (chromosome (genotype self)
                            (nreverse result))
          (push (copy-chromosome chromosome new-self)
                result)))
  new-self)

(defmethod COPY-ORGANISM-WITH-FITNESS ((self organism)
                                       &KEY (new-population (population self)) &AUX
                                       (new-self (copy-organism
                                                  self :new-population new-population)))
  (setf (fitness new-self) (fitness self))
  new-self)

(defmethod MAKE-CHROMOSOME ((org organism) chromosome-class &KEY
                            size random)
  (make-instance chromosome-class
    :organism org
    :size size
    :random random))

(defmethod MAKE-CHROMOSOMES ((self organism) &KEY random
                             &AUX result)
  (dolist (chromosome-class (chromosome-classes self)
                            (nreverse result))    ; returned from dolist
    (push (make-chromosome self chromosome-class
                           :random random)
          result)))

(defmethod MAKE-RANDOM-CHROMOSOMES ((self organism))
  (setf (genotype self)
        (make-chromosomes self :random t)))

(defmethod RANDOMIZE-CHROMOSOMES ((self organism))
  (dolist (chromosome (genotype self))
    (pick-random-alleles chromosome)))

(defmethod GENOTYPE-PRINTABLE-FORM ((self organism))
  (let ((chromosomes (when (genotype self)
                       (genotype self))))
    (format nil "[~A~{ ~A~}]" (first chromosomes) (rest chromosomes))))

#| Since the target implementor will be defining a primary method for evaluate, perform the decoding
automatically via a :before method. |#

(defmethod EVALUATE :BEFORE ((self organism) (plan genetic-plan))
  (declare (ignore plan))
  (decode self))

(defmethod DECODE ((self organism))
  )                                     ; do nothing

(defmethod NORMALIZE-FITNESS ((self organism) stats (plan genetic-plan))
  (let ((delta (- (max-fitness stats) (min-fitness stats)))
        (range (- (fitness self) (min-fitness stats))))
    (declare (short-float delta range))
    (setf (normalized-fitness self)
          (the short-float (/ range delta)))))

(defmethod EIDETIC ((org1 organism) (org2 organism) &AUX
                    (genes-1 (genotype org1))
                    (genes-2 (genotype org2)))
  "Predicate, true if the organisms are of the same class and have the same chromosomes."
 (and (eq (class-of org1) (class-of org2))
      (or (and (null genes-1) (null genes-2))
          (do* ((g1-ptr genes-1 (rest g1-ptr))
                (g2-ptr genes-2 (rest g2-ptr))
                (chromosome-1 (first g1-ptr) (first g1-ptr))
                (chromosome-2 (first g2-ptr) (first g2-ptr))
                (same (eidetic chromosome-1 chromosome-2)
                      (eidetic chromosome-1 chromosome-2)))
               ((or (null g1-ptr) (not same))
                same)))))

(defmethod PICK-RANDOM-CHROMOSOME-INDEX ((self organism))
  (do* ((sizes (mapcar #'size (genotype self)))
        (n-genes (reduce #'+ sizes))
        (s-list sizes (rest s-list))
        (c# 0)
        (sum 0))
       ((or (null s-list)
            (>= (random n-genes) sum))
        c#)
    (incf sum (first s-list))
    (incf c#)))

(defmethod PICK-RANDOM-CHROMOSOME ((self organism))
  (nth (pick-random-chromosome-index self) (genotype self)))

;;; ========= Genetic Operators =========

(defmethod MUTATE-ORGANISM ((self organism) &AUX
                            (chromosome (nth (pick-random-chromosome-index self)
                                             (genotype self)))
                            (locus-index (pick-locus-index chromosome)))
  "Mutate a random locus on an organism"
  (mutate-chromosomes chromosome locus-index))

(defmethod CROSS-ORGANISMS ((parent1 organism) (parent2 organism)
                            (child1 organism) (child2 organism)
                            &AUX
                            (chromosome# (pick-random-chromosome-index parent1))
                            (parent1-chromosome (nth chromosome#
                                                     (genotype parent1)))
                            (locus-index (pick-locus-index parent1-chromosome)))
  "Crossover random chromosomes between two organisms."
  (cross-chromosomes parent1-chromosome
                     (nth chromosome# (genotype parent2))
                     (nth chromosome# (genotype child1))
                     (nth chromosome# (genotype child2))
                     locus-index))

(defmethod UNIFORM-CROSS-ORGANISMS ((parent1 organism) (parent2 organism)
                                    (child1 organism) (child2 organism)
                                    &AUX
                                    (chromosome# (pick-random-chromosome-index parent1))
                                    (parent1-chromosome (nth chromosome#
                                                             (genotype parent1))))
  "Uniform crossover of random chromosomes between two organisms."
  (uniform-cross-chromosomes parent1-chromosome
                             (nth chromosome# (genotype parent2))
                             (nth chromosome# (genotype child1))
                             (nth chromosome# (genotype child2))))
