;;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: GECO -*-

(in-package :GECO)
#|
Genetic Evolution through Combination of Objects (GECO)

Copyright (C) 1992  George P. W. Williams, Jr.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
|#

(defmethod SHARED-INITIALIZE :AFTER ((self chromosome) slots-for-initform
                                     &REST initargs
                                     &KEY size organism random)
  (declare (ignore slots-for-initform initargs))
  (setf (organism self) organism)
  (setf (loci self) (make-loci-vector self (if size size (size self))))
  (if random (pick-random-alleles self)))

(defmethod MAKE-LOCI-VECTOR ((self chromosome) size)
  (make-array size :element-type 'fixnum
              :initial-element 0))

(defmethod PICK-RANDOM-ALLELES ((self chromosome))
  (dotimes (i (size self))
    (setf (locus self i) (pick-random-allele self i))))

(defmethod PICK-RANDOM-ALLELE ((self chromosome) locus-index)
  "Returns a random allele for the locus."
  (random (locus-arity self locus-index)))

(defmethod ALLELE-CODE-TO-VALUE ((self chromosome) locus-index allele-index)
  (aref (printable-allele-set self locus-index) allele-index))

(defmethod COPY-CHROMOSOME ((self chromosome) owner-organism)
  (let ((new-self (make-chromosome owner-organism (class-of self))))
    (dotimes (i (size self))
      (setf (locus new-self i) (locus self i)))
    new-self))

(defmethod PRINT-OBJECT ((self chromosome) stream)
  (print-unreadable-object (self stream :type t :identity t)
    (princ (loci-printable-form self) stream)))

(defmethod LOCI-PRINTABLE-FORM ((self chromosome)
                                &AUX (s (make-string (size self))))
  (dotimes (i (size self))
    (setf (aref s i) (locus-printable-form self i)))
  s)

(defmethod LOCUS-PRINTABLE-FORM ((self chromosome) locus-index)
  "Returns a character which represents the value of the allele at the indicated locus of the chromosome."
  (aref (printable-allele-set self locus-index)
        (locus self locus-index)))

(defmethod EIDETIC ((chr1 chromosome) (chr2 chromosome)
                    &AUX (size1 (size chr1)))
  "Predicate, true if the chromosomes are of the same class and have the same alleles."
  (and (eq (class-of chr1) (class-of chr2))
       (= size1 (size chr2))      ; some day this may be important
       (do ((locus# 0 (1+ locus#))
            (same t))
           ((or (not same)
                (>= locus# size1))
            same)
         (setq same (= (locus chr1 locus#)
                       (locus chr2 locus#))))))

(defmethod SIZE ((self chromosome))
  (length (loci self)))

(defmethod LOCUS ((self chromosome) index)
  (aref (loci self) index))

(defmethod (SETF LOCUS) (new-value (self chromosome) index)
  (setf (aref (loci self) index) new-value))

(defmethod PICK-LOCUS-INDEX ((self chromosome))
  "Returns a random locus index into the chromosome."
  (random (size self)))

(defmethod COUNT-ALLELE-VALUES
           ((chromosome chromosome) from-index loci-to-count allele-value)
  "Returns the number of loci in the loci-to-count loci of chromosome, starting at from-index,
wich have allele-value."
  (do* ((locus# from-index (1+ locus#))
        (counter 1 (1+ counter))
        (result (if (= allele-value (locus chromosome locus#)) 1 0)
                (+ result (if (= allele-value (locus chromosome locus#)) 1 0))))
      ((>= counter loci-to-count)
       result)))

(defmethod HAMMING-DISTANCE ((c1 chromosome) (c2 chromosome)
                             &AUX (distance 0))
  (dotimes (i (size c1))
    (if (/= (locus c1 i) (locus c2 i))
        (incf distance)))
  distance)

;;; ========= Genetic Operators =========

(defmethod MUTATE-CHROMOSOMES ((self chromosome) locus-index)
  (setf (locus self locus-index)
        (pick-random-allele self locus-index)))

(defmethod CROSS-CHROMOSOMES ((parent1 chromosome) (parent2 chromosome)
                              (child1 chromosome) (child2 chromosome)
                              locus-index)
  "Cross a chromosome segment between two parents at a specified location in the children."
  ;; Assume all chromosomes are the same size.
  ;; First, move the unswapped portions
  (dotimes (i locus-index)
    (setf (locus child1 i) (locus parent1 i))
    (setf (locus child2 i) (locus parent2 i)))
  ;; Then, move the swapped portions
  (do ((i (1+ locus-index) (1+ i)))
      ((>= i (size parent1)))
    (setf (locus child1 i) (locus parent2 i))
    (setf (locus child2 i) (locus parent1 i))))

(defmethod UNIFORM-CROSS-CHROMOSOMES ((parent1 chromosome) (parent2 chromosome)
                                      (child1 chromosome) (child2 chromosome)
                                      &OPTIONAL (bias 0.5))
  "Uniform crossover of chromosomes between two parents, with optional bias in (0.0:1.0].
Default bias is 0.5; larger bias indicates larger number of crossed alleles."
  ;; Assume all chromosomes are the same size.
  (dotimes (i (size parent1))
    (if (< (random 1.0) bias)
        (progn (setf (locus child1 i) (locus parent2 i))
               (setf (locus child2 i) (locus parent1 i)))
      (progn (setf (locus child1 i) (locus parent1 i))
             (setf (locus child2 i) (locus parent2 i))))))


;;; ========= Methods for class BINARY-CHROMOSOME =========

(defmethod LOCUS-ARITY ((self binary-chromosome) locus-index)
  (declare (ignore locus-index))
  2)

(defmethod PRINTABLE-ALLELE-SET ((self binary-chromosome) locus-index)
  (declare (ignore locus-index))
  #(#\0 #\1))

(defmethod MAKE-LOCI-VECTOR ((self binary-chromosome) size)
  (make-array size :element-type 'bit))


;;; ========= Binary-Chromosome Decoding Routines =========

(defmethod DECODE-BINARY-LOCI-VALUE
                               ((chromosome binary-chromosome) from-index loci-to-decode)
  "Returns the binary value represented by the loci-to-decode loci of chromosome, starting at from-index.
chromosome is assumed to be an array of bit."
  (do* ((locus# from-index (1+ locus#))
        (counter 1 (1+ counter))
        (result (locus chromosome locus#)
                (+ (* 2 result) (aref chromosome locus#))))
       ((>= counter loci-to-decode)
        result)))

;;; ========= Methods for class GRAY-CODE-TRANSLATION =========
;; The following gray-code translation code is based on an implementation in C by Larry Yaeger
;; <larryy@apple.com>, which was published in the GA-List v6n5 (GA-List@AIC.NRL.Navy.Mil).

(defmethod SHARED-INITIALIZE :AFTER
                               ((self gray-code-translation) slots-for-initform
                                  &REST initargs
                                    &KEY (number-of-bits nil number-of-bits-supplied-p))
  (declare (ignore slots-for-initform initargs))
  (if (not number-of-bits-supplied-p)
      (slot-makunbound self 'number-of-bits)
    (progn
      (let ((b2g (make-array (list number-of-bits number-of-bits)
                             :element-type 'bit))
            (g2b (make-array (list number-of-bits number-of-bits)
                             :element-type 'bit)))
        (dotimes (i number-of-bits)
          (dotimes (j number-of-bits)
            (setf (aref b2g i j) 0)
            (setf (aref g2b i j)
                  (if (<= j i) 1 0)))
          (setf (aref b2g i i) 1)
          (if (> i 0)
              (setf (aref b2g i (1- i) )1)))
        (setf (b2g-map self) b2g
              (g2b-map self) g2b
              (number-of-bits self) number-of-bits)))))
;;(defvar *gct*)
;;(setq *gct* (make-instance 'gray-code-translation :number-of-bits 5))

(defmethod GRAY2BIN ((self gray-code-translation) gray &AUX
                     (bin 0) bit
                     (g2b (g2b-map self))
                     (numbits (number-of-bits self))
                     (numbits-1 (1- numbits))
                     (ii numbits-1)
                     jj)
  (dotimes (i numbits)
    (setq bit 0)
    (setq jj numbits-1)
    (dotimes (j numbits)
      (setq bit (boole boole-xor bit
                       (if (logbitp jj gray)
                           (aref g2b i j)
                         0)))
      ;;(dbgo "~&~5Tjj=~D, (logbitp jj gray)=~S, (aref g2b i j)=~D"
      ;;      jj (logbitp jj gray) (aref g2b i j))
      (decf jj))
    ;;(dbgo "~&bin=~D, bit=~D, ii=~D" bin bit ii)
    (setq bin (boole boole-ior bin (ash bit ii)))
    (decf ii))
  bin)
;;(gray2bin *gct* 3)
;;(dotimes (i (expt 2 (number-of-bits *gct*))) (dbgo "~& ~2D ~2D" i (gray2bin *gct* i)))

(defmethod BIN2GRAY ((self gray-code-translation) bin &AUX
                     (gray 0) bit
                     (b2g (b2g-map self))
                     (numbits (number-of-bits self))
                     (numbits-1 (1- numbits))
                     (ii numbits-1)
                     jj)
  (dotimes (i numbits)
    (setq bit 0)
    (setq jj numbits-1)
    (dotimes (j numbits)
      (setq bit (boole boole-xor bit
                       (if (logbitp jj bin)
                           (aref b2g i j)
                         0)))
      ;;(dbgo "~&~5Tjj=~D, (logbitp jj bin)=~S, (aref g2b i j)=~D"
      ;;      jj (logbitp jj bin) (aref b2g i j))
      (decf jj))
    ;;(dbgo "~&gray=~D, bit=~D, ii=~D" gray bit ii)
    (setq gray (boole boole-ior gray (ash bit ii)))
    (decf ii))
  gray)
;(bin2gray *gct* 2)
#|
(progn
  (dbgo "~&Int ~7TBinary ~19TGray ~23TGrayInt  RecoveredInt")
  (dotimes (i (expt 2 (number-of-bits *gct*)))
    (let ((g (bin2gray *gct* i)))
      (dbgo "~&~3D  ~8B  ~8B  ~4D  ~8D"
            i i g g (gray2bin *gct* g)))))
|#
