/*		DSCLEARN for the CFS-C Classifier System

This file, DSCLEARN.C, is part of the CFS-C classifier system (Copyright 1986,1988 Rick L. Riolo).

The subroutines in this file implement some of the discovery learning algorithms for
the CFS-C classifier system, including the main entry point for all the
discovery learning mechansims.
See the file DSC-OPS.C for definitions of the subroutines that implement
the discovery "operators" (e.g., crossover, mutation, etc.).

The subroutines in this file:
	Discover	The main entry point for the CFS-C discovery mechanisms.
	ReCalcFitness Recalculate ``fitness'' values for use by discovery algorithms.
	PckPrnts	pick parent classifiers by strength
	PkPrPool	pick pool of classifiers for strong parents
	PkBPrnts	pick high bidding cfs for parents
	PkBPPool	Pick Bidding Parents, i.e., classifier that bid this step,

	RplcCfs 	replace classifiers with new classifiers produced this step
	PckRPool	pick pool of classifiers from which those to be replaced will be picked.
	PckRndRp	pick replaceable pool using uniform random distribution
	PckBiaRp	pick replaceable pool using inverted strength biased distribution
    PickExtraBiddersToReplace
	DemoRpl 	demonstration display of replacable pool of classifiers
	PckRCf		pick classifier from pool of replacable classifiers
	GenTypeM	return similarity measure for genotypes of two classifiers
	WrtNCf		write NEW classifier in a buffer.
**/ 

#include	"compiler.h"
#include	<math.h>
#include	<stdio.h>
#if ( LATTICEC || CBELLMTS || SUN3 || MPWC )
#include	<ctype.h>
#endif
#include	"utility.h"
#include	"core.ext"
#include	"cfops.ext"
#include	"dsclearn.def"

extern	unsigned int	DemoLev;
extern	short	 int	GenFlg;
extern	char			*GOutBuff;

float	URand01();

VOID Discover(), ReCalcFitness(), PkPrPool(), DemoPool(), DemoRpl();
VOID PckRndRp(), PckBiaRp(), WrtNCf();
VOID DscCDM(), DscCEff(), DscACPC(), DscCSS(), DscTLB();

unsigned int RplcCfs(), PckRPool();
struct CfNode *PckRCf();

#if MPWC
#define __SEG__ CFSCDSC
#endif

/*****************************

Discover	Call routines to 'discover' new classifiers.

General plan is to check for trigger conditions, and for those found,
apply special "disovery" learning algoritms to generate new classifiers.

New classifiers generated by all operators are stored in an array of
new-classifier nodes, up to a maximum number = NmCfs * FrNewCfs.
NmNewCfs is the array entry into which the next new classifier is to be put.

If there are new classifiers produced during a step, all the new classifiers
are used to replace others at one time, at the end of this subroutine, by
the call to RplcCfs(). This is turn calls other subroutines to actually choose
classifiers to be replaced.

******/

VOID Discover()
{

#if CDSC_ALL
	FitnessCalc = FALSE;			 /* Not calculated yet, so no totals for picking parents/replaceables */

	MxNewCfs = Round(NmCfs*FrNewCfs);	/* get max number new classifiers allowed this step */
	NmNewCfs = PrPoolSz = 0;			/* haven't got any new yet! */

	if ( URand01() < CDMsgsRt )
		DscCDM();						/* Cover Detector Messages, if necessary */

	if ( MadeMstk && URand01() < CEffsRt )
		DscCEff();

	if ( URand01() < ACPCRt )			/* Asynchronously couple profitable classifier */
		DscACPC();

	if ( CSSRt != 0 )       			/* Couple Stage Setting classifiers */
		DscCSS();

#ifdef THIS  /* turned off---see DscTLB1() in DSCOPS2.C */
	if ( URand01() < TLBRt )			/* Triggered on Low average Bid */
		DscTLB();
#endif

	if ( URand01() < TLBRt && NmCandCf > 0 && NmCandCf < NmIMsgMx/2 )	/* Triggered on Low number of bidders */
		DscTLB1();

	if ( URand01() < BkgGARt )
		DscBkgGA(); 					/* Apply Background operators */

	if ( NmNewCfs > 0 )
		RplcCfs();						/* Replace some classifiers with NewCfs */

#endif	/* CDSC_ALL */

} /* Discover */

#if CDSC_ALL

/*****************************

ReCalcFitness   Recalculate ``fitness'' values for classifiers.

Recalculate special ``fitness'' values (primarily for Discovery):
	TotCfStr and AveCfStr (see CORE.DEF)
	PBiasStr (fitness for reproduction) |- in each CfNode 
	RBiasStr (fitness for replacement)  |- in each CfNode
	TPrtBStr = sum ( PBiasStr )
	TRplBStr = sum ( RBiasStr )

Also  EligRpl ("eligible for replacement") is set in each CfNode.
NmEligRp is sum of those with EligRpl set TRUE,
and EligRpl is set to FALSE for classifiers to be protected from replacement.

ALWAYS protect rules with:
	- NoRplFlg TRUE	(these are specially marked, e.g., at when loaded from a file)

ALWAYS pick rules with strength <= 0, EXCEPT when NoRplFlg is TRUE.

If  RandRplc = 0 (pick inverse to "fitness")
	RandRplc = 2 (pick uniform random from unprotected rules),
 then Protect some classifiers from replacement:
	- Str > RplCfUBd * AveCfStr		 All above this bound are protected
	- if RplACSBd not 0, then		   special treatment for those active recently
		StpLPrd > CycleStp - RplACSBd   defines active for these purposes
		Str > RplACUBd * AveCfStr	  protect above this for active rules

If RandRplc = 1, i.e., pick uniform random from (almost) all rules,
  subject to the two ALWAYS rules, above.

If RplcXBid > 0, bias replacement to rules that did not post messages
the last time they bid.

NOTE: If classifiers are being protected from replacement if above a certain
  strength, this assumes AveCfStr is close enough on entry (from Bucket Brigade Routines
  applied just before the Discovery routines that may call this).
  This does recalculate it, though.

If RandRplc is TRUE, don't bother to calculate the biased-replacement fitness values.

Set FitnessCalc = TRUE when done.

******/

VOID ReCalcFitness ( )
{
	register int			cfcnt, rplcxbid;
	register float			str, pfitness, rfitness; /* strength, parent-fitness and replacement-fitness */
    register float          rand, cummul;
	register float		    protectBound, protectActiveBound;
	register struct CfNode  *cfptr;

#if CTEST
    if ( DscDemo > 3 )
        WriteStd( "\nReCalcFitness..." );
#endif

	FitnessCalc = TRUE;			 /* set this marker */

	protectBound = AveCfStr * RplCfUBd;	/* Won't replace above this, so don't include */

	if ( RplACSBd != 0 )
		protectActiveBound = AveCfStr * RplACUBd;   /* protect active classifiers */

	TotCfStr	= 0;				/* zero the totals to be accumulated */
	TotCfBR		= 0;
	TPrtBStr	= TRplBStr	= 0;
   	NmEligRp	= 0;

        /* Now do normal fitness calculations for all rules. */
		/* work through the list, in no particular order */

	for ( cfcnt = 0, cfptr = CurCfs; cfcnt < NmCfs; ++cfcnt, ++cfptr ) {
	 	str = cfptr->Strength;			/* get strength into local var */
		TotCfStr += str;				/* add to total */
		TotCfBR += cfptr->BidRatio;

		if ( str <= (float) 0.0 ) {	/* strength <= 0: not a parent, and will be replaced */
			cfptr->EligRpl = TRUE;
			++NmEligRp;
		}

		else  {	 /* str > 0, so calculate fitnesses */

			if ( BidFit ) {	   					/* if we're using "bid" (support ignored) for fitness, */
				pfitness = str * cfptr->BidRatio; 			/* parent fitness */
				rfitness = 1.0 / (str * cfptr->BidRatio); 	/* replacement fitness is inverse */
			}
			else {		  /* just use strength (or inverse) */
				pfitness = str;					 /* parent fitness */
				rfitness = 1.0 / str;			   /* replacement fitness is inverse */
			}

				/* now for parenting... */

			if ( PrntBPow != (float) 1.0 )		  /* bias it more! */
				pfitness = pow( pfitness, PrntBPow );

			cfptr->PBiasStr = pfitness;
			TPrtBStr += pfitness;

				/* now for replacement... */

			if ( cfptr->NoRplFlg ) {
				cfptr->EligRpl = FALSE;	 /* these ALWAYS are protected from replacement */
				continue;
			}

			if ( RandRplc == 1 ) {		  /* replace uniform random, so */
				cfptr->EligRpl = TRUE;	  /* mark as eligible, and continue */
				++NmEligRp;
				continue;
			}

				/* RandRplc is 0 (inverse to fitness) or 2 (uniform random from unprotected) */

			if ( str > protectBound ) {
				cfptr->EligRpl = FALSE;	 /* these are protected from replacement */
				continue;
			}

			if ( RplACSBd != 0 ) {		  /* check for active classifiers to protect */
				if ( cfptr->StpLPrd > (CycleStp - RplACSBd) && str > protectActiveBound ) { 
					cfptr->EligRpl = FALSE; /* these recent active also protected */
					continue;
				}
			}

				/* must be replacable...*/

			cfptr->EligRpl = TRUE;	  /* mark as eligible */
			++NmEligRp; 				/* increment this counter */

			if ( RplcBPow != (float) 1.0 )	  /* bias it more! */
				rfitness  = pow( rfitness, RplcBPow );

                /* if it didn't win last bid, make it more likely to be replaced */
            if ( RplcXBid > 1 && cfptr->StpLBid != cfptr->StpLPst ) {
                rfitness *= RplcXBid;
            }

			cfptr->RBiasStr = rfitness; /* store it in the node */
			TRplBStr += rfitness;		/* increment the total */

		}	/* end str > 0 */

	 } /* endfor classifiers */

        /* need the cast for SUN3 3 compiler bug */
	AveCfStr = TotCfStr / (int) NmCfs;
	AveCfBR	= TotCfBR / (int) NmCfs;

} /* ReCalcFitness */




/******************************

PckPrnts	Pick parent classifiers.

	Number	Number of parents (0..2) to return in Parents[] array entries.

	Return	Number actually picked.
			Put pointers to them in the first Number entries of Parents[] array.

The scheme used is to get a pool of parents on the first call to the subroutine
during a given step, and then on that and subsequent calls allocate parents
from that pool as requested.	

The array PrntPool[] contains pointers to the pool of parents.
The array Parents[] is used to return the parents picked from the pool.

In general, high strength classifiers are chosen (see below for details).
However: THIS DOES NOT reproduce any classifiers that have strengths <= 0.

******/

int PckPrnts ( Number )
	int		Number;
{

	if ( PrPoolSz == 0 )	{
		PkPrPool(); 					/* Get a pool of parents all at once */
	}

	if ( Number > 2 ) {
		WriteStd("\n**PckPrnts: Asked for > 2.");
		Number = 2;
	}

	if ( Number > PrPoolSz - NxtPPool ) 
		return( 0 );

	else if ( Number != 0 ) {
		Parents[0] = PrntPool[NxtPPool++];
		if ( Number == 2 )
			Parents[1] = PrntPool[NxtPPool++];
	}

	return( Number );

} /* PckPrnts */


/*****************************

PkPrPool	Pick pool of classifiers to be parents.

The basic idea is to pick parents with probability proportional to
strength (in fact equal to PBiasStr, a possible biased function of strength).

All classifiers with Strength <= 0 are not taken (i.e., prob = 0).
So skip those as we try to pick parents.

If PkPrntWOR (pick without replacement) is on, as parents are chosen
the existing pool must be checked to be sure the candidate is not
already in the pool.

If need be this calls ReCalcFitness() to calculate fitnesses:
	PBiasStr  - the fitness value stored with each eligible classifier.
	TPrtBSt - sum of PBiasStr for all eligible parents.
where "eligible" means strength > 0.
See ReCalcFitness() for details on the fitness calculations.

******/

VOID PkPrPool ( )
{
	register int 			retries, i, cfcnt;
	register float			randnum, cummul, totbias;
	int					 numnew;
	struct CfNode	*cfptr;

  	if ( !FitnessCalc )		 /* Calc TPrnBStr from PrntBStr for each cf */
		ReCalcFitness();

	NxtPPool = retries = 0;
	totbias = TPrtBStr;
	numnew = MxNewCfs - NmNewCfs;

	if ( numnew > NEWCFMX ) {
		WriteStd( "\n**PkPrPool: numnew too big!\n" );
		numnew = NEWCFMX;
	}

	while ( PrPoolSz < numnew )	{
		randnum = URandF( totbias );
#if CTEST
		if ( DscDemo >= 3 ) {
			sprintf(GOutBuff,"\ntotbiasS %f, randnum %f", totbias, randnum );
			WriteStd( GOutBuff );
		}
#endif
		for ( cfptr = CurCfs, cfcnt = cummul = 0; cfcnt < NmCfs; ++cfptr, ++cfcnt ) {
			if ( cfptr->Strength > 0 ) { 			/* If strength > 0 then give it a chance */
				if ( PkPrnWOR ) {
					for ( i = 0; i < PrPoolSz; ++i )	/* see if we picked this one yet */
						if ( cfptr == PrntPool[i] )
							break;
					if ( i < PrPoolSz ) {			/* we did use it, so skip it */
						continue;
					}
				}
				cummul += cfptr->PBiasStr;
#if CTEST
				if ( DscDemo >= 3 ) {
					sprintf(GOutBuff,"\ncf %d, cummul now %f", cfptr->Cf_Id, cummul );
					WriteStd( GOutBuff );
				}
#endif
				if	( randnum <= cummul )
					break;								/* drew this one */ 
			}
		}

		if ( cfptr == NULL ) {				/* Whoops! This must be a rounding error with randnum near 1. */
			WriteStd( "\n**PkPrPool: Rand draw->NULL!\n" );
			++retries;
			if ( retries == 20 ) {
				WriteStd("\n**PkPrPool: 20 retries\n");
				break;
			}
		}

		else if ( cfptr->NoPrnFlg ) {
			++retries;
			if ( retries == 40 ) {
				WriteStd("\n**PkPrPool: 40 retries\n");
				break;
			}
		}

		else {
#if CTEST
			if ( DscDemo >= 3)
				WriteStd( "<= Parent" );
#endif
			
			PrntPool[PrPoolSz++] = cfptr;		/* This is one! */
			totbias -= cfptr->PBiasStr;
		}
	}

} /* PkPrPool */


/******************************

PkBPrnts	Pick parent classifiers from bidders, then from hi strength.

	Number	Number of offspring to return in Parents[] entries (1 or 2).

	Max		'e' = just need Number parents; otherwise establish pool
			of MaxNewCfs parents.

	Return	Number actually picked.
			Put pointers to them in the first Number entries of Parents[] array.

This basically picks from bidders, and if it needs more it pick high
strength classifiers. It acts much as PckPrnts() and PkPrPool().

******/

int PkBPrnts ( Number, Max )
	int		Number;
	char	Max;
{

	if ( PrPoolSz == 0 ) {
		if ( Max == 'e' )
			PkBPPool( Number );
		else
			PkBPPool( (MxNewCfs - NmNewCfs) );
	}

	if ( Number > 2 ) {
		WriteStd("\n***PkBPrnts: Asked for > 2.");
		Number = 2;
	}

	if ( Number > PrPoolSz - NxtPPool ) 
		return( 0 );

	else if ( Number != 0 ) {
		Parents[0] = PrntPool[NxtPPool++];
		if ( Number == 2 )
			Parents[1] = PrntPool[NxtPPool++];
	}

	if ( Max == 'e' )				/* If call for exactly number needed, assume they will all */
		PrPoolSz = 0;				/* be used, so there will be no pool left. */

	return( Number );

} /* PkBPrnts */


/******************************

PkBPPool	Pick Bidding Parents, i.e., classifier that bid this step,
			and if need be pick hi strength classifiers.

	Number	Number to pick.

NOTE:	Pick proportional to biased bid.
		If that doesn't return enough, fill the rest in with
		classifiers chosen probabilistically proportional to strength.

******/

int PkBPPool ( Number )
	int		Number;
{
	register int 	i, retries, cfcnt;
	register float	randnum, cummul, totbbid, fractot;
	struct CfNode	*cfptr;

	if ( !FitnessCalc )	 /* Calc TPrtBStr from PrntBStr for each cf */
		ReCalcFitness();	/* (Only used if too few bidders */

	NxtPPool = 0;			/* Where to get next parent from pool */

	if ( Number > NEWCFMX ) {
		WriteStd( "\n***PkBPPool: Number too big!\n" );
		Number = NEWCFMX;
	}

#if CTEST
	if ( DscDemo >= 1 ) {
		sprintf( GOutBuff, "\nPkBPPool: pick %d parents (NmCandCf %d).", Number, NmCandCf );
		WriteStd( GOutBuff );
	}
#endif

	if ( NmCandCf <= Number ) {					/* Not more than we need, so ... */
		for ( cfptr=CandCfs; cfptr != NULL; cfptr=cfptr->NxtCand )
			PrntPool[PrPoolSz++] = cfptr;			/* Get all the bidders */
	}
	else {										/* get only some of the bidders */
		retries = 0;
		totbbid = TotBBid;							/* Total 'biased' bids for candidates not yet winners */
		fractot = totbbid * 0.01;					/* keep record of 1/100 of total, to see when to re-sum. */

		while ( PrPoolSz < Number )	{
			randnum = URandF( totbbid );			/* randnum is uniform in 0...TotBBid: use it for the draw */
#if CTEST
			if ( DscDemo >= 3 ) {
				sprintf( GOutBuff, "\nPick winner (rnum is %.4f, totbiasbids %7.4f):", randnum, totbbid );
				WriteStd( GOutBuff );
			}
#endif
			for ( cfptr=CandCfs, cummul=0; cfptr != NULL; cfptr=cfptr->NxtCand ) {
				if ( PkPrnWOR ) {
					for ( i = 0; i < PrPoolSz; ++i )	/* see if we picked this one yet */
						if ( cfptr == PrntPool[i] )
							break;
					if ( i < PrPoolSz ) {			/* we did use it, so skip it */
						continue;
					}
				}
				cummul += cfptr->CfBBid;			/* cummuilative prob = sum of biased bids */
#if CTEST
				if ( DscDemo >= 3 ) {
					sprintf( GOutBuff, "\n   Cand Cf %u: cummul %.4f ", cfptr->Cf_Id, cummul );
					WriteStd( GOutBuff );
				}
#endif
				if	( randnum <= cummul )			/* cf wins! */
					break;							/* so stop the 'draw' */
			}

			if ( cfptr == NULL ) {				/* Whoops! This must be a rounding error with randnum near 1. */
				sprintf( GOutBuff, "\n***PkBPPool: Rdraw->NULL; step %u, rand %f, cumm %f, totbbid %f, TotBBid %f.\n",
						CycleStp, randnum, cummul, totbbid, TotBBid );
				WriteStd( GOutBuff );
				++retries;
				if ( retries == 10 ) {
					WriteStd("\n***PkBPPool: 10 retries\n");
					break;
				}
			}

			else if ( cfptr->NoPrnFlg ) {
				++retries;
				if ( retries == 20 ) {
					WriteStd("\n***PkBPPool: 20 retries\n");
					break;
				}
			}

			else {
#if CTEST
				if ( DscDemo >= 3 )
					WriteStd( "<= Pick parent" );
#endif
				PrntPool[PrPoolSz++] = cfptr;			/* This is one! */
				if ( PrPoolSz == Number )
					break;								/* thats it lets quit */
				totbbid -= cfptr->CfBBid;				/* adjust total biased-bids--one less candidate now*/
				if ( totbbid < fractot ) {			/* its time to recalculate the total biased bid */
#if CTEST
					if ( DscDemo >= 3 ) {
						WriteStd( "\n\n***PkBPPool: Calc. tot bias-bid for rand draw" );
						sprintf( GOutBuff, "\nCycleStp %d: totbbid %f, fractot %f", CycleStp, totbbid, fractot );
						WriteStd( GOutBuff );
					}
#endif
					for ( totbbid=0.0, cfptr = CandCfs; cfptr != NULL; cfptr = cfptr->NxtCand ) {
						for ( i = 0; i < PrPoolSz; ++i ) /* see if we picked this one yet */
							if ( cfptr == PrntPool[i] )
								break;
						if ( i == PrPoolSz )			/* we didn't use it, so */
							totbbid += cfptr->CfBBid;	/* add biased bid to the total biased bid */
					}
					fractot = totbbid * 0.01;	/* recalculate the fraction total for future checks */
#if CTEST
					if ( DscDemo >= 3 ) {
						sprintf( GOutBuff, "  (New totbbid %f, fractot %f).\n\n", totbbid, fractot );
						WriteStd( GOutBuff );
					}
#endif
				}	/* endif we had to re-calculate totbbid */
			}
		}
	}

		/* Now if we need more parents, get from hi strength list */

#if CTEST
	if ( DscDemo > 0 ) {
		sprintf( GOutBuff, "\nGot %d from bidders, get %d from HiS:", PrPoolSz, Number-PrPoolSz );
		WriteStd( GOutBuff );
	}
#endif

	totbbid = TPrtBStr; 	/* strength total to start */

	while ( PrPoolSz < Number )	{
		randnum = URandF( totbbid );
#if CTEST
		if ( DscDemo >= 3 ) {
			sprintf(GOutBuff,"\ntotbiasS %f, randnum %f", totbbid, randnum );
			WriteStd( GOutBuff );
		}
#endif
		for ( cfptr = CurCfs, cfcnt = cummul = 0; cfcnt < NmCfs; ++cfptr, ++cfcnt ) {
			if ( cfptr->Strength > 0 ) { 			/* If strength > 0 then give it a chance */
				if ( PkPrnWOR ) {
					for ( i = 0; i < PrPoolSz; ++i )	/* see if we picked this one yet */
						if ( cfptr == PrntPool[i] )
							break;
					if ( i < PrPoolSz ) {			/* we did use it, so skip it */
						continue;
					}
				}
				cummul += cfptr->PBiasStr;
#if CTEST
				if ( DscDemo >= 3 ) {
					sprintf(GOutBuff,"\ncf %d, cummul now %f", cfptr->Cf_Id, cummul );
					WriteStd( GOutBuff );
				}
#endif
				if	( randnum <= cummul )
					break;								/* drew this one */ 
			}
		}

		if ( cfptr == NULL ) {				/* Whoops! This must be a rounding error with randnum near 1. */
			WriteStd( "\n***PkBPPool(s): Rand draw->NULL!\n" );
			++retries;
			if ( retries == 20 ) {
				WriteStd("\n***PkBPPool(s): 20 retries\n");
				break;
			}
		}

		else if ( cfptr->NoPrnFlg ) {
			++retries;
			if ( retries == 40 ) {
				WriteStd("\n***PkBPPool(s): 40 retries\n");
				break;
			}
		}

		else {
#if CTEST
			if ( DscDemo >= 3)
				WriteStd( "<= Parent" );
#endif
			
			PrntPool[PrPoolSz++] = cfptr;			/* This is one! */
			totbbid -= cfptr->PBiasStr;
		}
	}

} /* PkBPPool */


/****************************

RplcCfs 	Replace classifiers with new classifiers in NewCfs[] array.

	For each new classifier, pick CrowdFac classifiers as pool of
	classifiers to be replaces, then pick one of those with genotype
	most similar to new classifier.

NOTE: MxCfCopy, if not 0 (i.e. if being used),
	  it overrides any protection of classifiers from
	  replacement (from RplCfUBd, etc.: see ReCalcFitness() above), and
	  it overides any CrowdFac.

	  If there are too many of one type, the lowest strength is replaced.

******/

unsigned int RplcCfs ( )
{
	int				cfi, i;
	register int	copies, cfcnt;
	register float	lowstr;
	register struct CfNode	*old, *testcf;
	struct NCfNode	*new;

	if ( !FitnessCalc )
		ReCalcFitness();

	for ( cfi = 0; cfi < NmNewCfs; ++cfi ) {

		new = &NewCfs[cfi]; 				/* get the new one */

		if ( MxCfCopy != 0 ) { 	/* if `unlimited' copies of a genotype are NOT allowed... */
				/*
					Look through whole cflist for classifiers identical to new 
				*/
#if CTEST
			if ( DscDemo > 0 ) {
				sprintf( GOutBuff, "\nLooking for MxCfCopy %d copies of cf:\n", MxCfCopy );
				WriteStd( GOutBuff );
				WrtNCf( new, GOutBuff, 1 );
				WriteStd( GOutBuff );
			}
#endif

			for ( old = NULL, testcf = CurCfs, copies = cfcnt = 0; cfcnt < NmCfs; ++testcf, ++cfcnt ) {

				i = 0;

				if ( new->NewCnd2T == testcf->Cnd2Type && new->NewActT == testcf->ActType )
					for ( ; i < INTPRSTR; ++i ) {
						if ( new->NewCnd1B[i] != testcf->Cnd1Bits[i] ||
							 new->NewCnd1D[i] != testcf->Cnd1DCs[i] ||
							 new->NewCnd2B[i] != testcf->Cnd2Bits[i] ||
							 new->NewCnd2D[i] != testcf->Cnd2DCs[i] ||
							 new->NewActB[i]  != testcf->ActBits[i] ||
							 new->NewActD[i]  != testcf->ActDCs[i] )
							break;	 /* something not equal, so quit */ 
					}

				if ( i == INTPRSTR ) {  		/* classifiers are equal (got through all int-sized parsts) */

					++copies;					/* increment copy count */

					if ( old == NULL )		  /* first duplicate, so save it */
						old = testcf;
					else if ( testcf->Strength < old->Strength )
						old = testcf;			/* save the lowest strength in old */

#if CTEST
					if ( DscDemo >= 3 ) {
						sprintf( GOutBuff, "\n ==> cf %d is copy %d.", testcf->Cf_Id, copies );
						WriteStd( GOutBuff );
					}
#endif

					if ( copies >= MxCfCopy ) {	/* got too many, so... */
						if ( old == NULL ) {	/* oh-o, should have one... */
							sprintf( GOutBuff, "ERR (RplcCfs): copies (%d) >= MxCfCopy (%d) & old is NULL [step %u]!",
								copies, MxCfCopy, CycleStp );
							WriteStd( GOutBuff );
							WriteStd( "\n	Pick replacements standard way.\n" );
						}

						else {		  /* old is the one to replace... */
					   		TRplBStr -= old->RBiasStr;	/* adjust total to reflect absence of this cf */
							--NmEligRp;			 /* one fewer eligible */
						}

						break;						/* quit looking in any case */
					}

				} /* endif testcf = new classifier */

			} /* endfor compare new cf to all current ones */

			if ( copies < MxCfCopy || old == NULL ) {  		/* not too many copies, so pick as usual */

#if CTEST
				if ( DscDemo > 0 ) {
					sprintf( GOutBuff, "====> Found %d <= MxCfCopy, so call PckRPool.\n", copies );
					WriteStd( GOutBuff );
				}
#endif

				if ( (i = PckRPool()) != CrowdFac ) {	/* get pool of classifiers that can be replaced */
					sprintf( GOutBuff, "\n**RplcCfs: Pool (%d) not right size (%d). No replacement! (Step %d)\n",
						 i, CrowdFac, CycleStp );
					WriteStd( GOutBuff );
					break;
				}

				old = PckRCf( new );			/* replace one in pool most like it */

			} /* endif not too many duplicates */

		} /* endif MxCfCopy in force */

		else {					 /* unlimited copies allowed, so... */

			if ( (i = PckRPool()) != CrowdFac ) {	/* get pool of classifiers that can be replaced */
				sprintf( GOutBuff, "\n**RplcCfs: Pool (%d) not right size (%d). No replacement! (Step %d)\n",
					 i, CrowdFac, CycleStp );
				WriteStd( GOutBuff );
				break;
			}

			old = PckRCf( new );			/* replace one in pool most like it */
		}

			/* Now old should point to one to replace */

#if CTEST
		if ( DscDemo > 0 ) {
			sprintf( GOutBuff, "\n  Replace %d.\n", old->Cf_Id );
			WriteStd( GOutBuff );
		}
#endif

#if GENLOG
		if ( GenFlg ) {
			sprintf( GOutBuff, "   (repl %d)", old->Cf_Id );
			WriteGen( GOutBuff );
		}
#endif

		old->Cf_Id = new->NewCf_Id; 			/* now copy it all from new to old */
		old->StpCrtd = CycleStp;				/* created this step--so don't put it in pools this step */
		old->EligRpl = FALSE;				   /* not eligible for replacement this step anymore */	

		for ( i = 0; i < INTPRSTR; ++i ) {
			old->Cnd1Bits[i] = new->NewCnd1B[i];
			old->Cnd1DCs[i]  = new->NewCnd1D[i];
			old->Cnd2Bits[i] = new->NewCnd2B[i];
			old->Cnd2DCs[i]  = new->NewCnd2D[i];
			old->ActBits[i]  = new->NewActB[i];
			old->ActDCs[i]	 = new->NewActD[i];
		}

		old->Cnd1Type = new->NewCnd1T;
		old->Cnd2Type = new->NewCnd2T;
		old->ActType = new->NewActT;

		TotCfStr -= old->Strength;
		TotCfBR -= old->BidRatio;

		old->Strength = new->NewStr;
		old->BidRatio = new->NewBR;

		TotCfStr += new->NewStr;
		TotCfBR += new->NewBR;

		old->TotNmBid = old->TotMtch = old->TotProd = old->TotPost = 0;
		old->TotEMtch = old->TotEAct = old->TotPosRw = old->TotNegRw = 0;
		old->StpLPrd = old->StpLPst = old->TotNmOfs = 0;
		old->ChngStr = 0.0;

        if ( new->NewBR < 0.1 ) {
            sprintf( GOutBuff, "\n\n=====> RplcCfs (stp %u): new->Id %u, BR %f.\n\n", CycleStp, new->NewCf_Id, new->NewBR );
            WriteStd( GOutBuff );
        }
	}

#if CTEST
	if ( DscDemo > 0 )
		WriteStd( "Done replacing classifiers.\n");
#endif
#if GENLOG
	if ( GenFlg )
		WriteGen( "\n\n" );
#endif

	return( NmNewCfs );

} /* RplcCfs */

/**************************************

PckRPool	Pick pool of classifiers from which classifiers to be replaced will be chosen.

This sets RPoolSz to number of classifiers actually put in pool.

PckRPool() picks classifiers from the current list that will be a "pool" of
classifiers from which classifiers to be replaced will be chosen.
The number picked is CrowdFac, a number >= 1.
Pointers to those chosen are put in the array RpCfPool[] defined in DSCLEARN.DEF .

Classifiers are chosen randomly under a distribution defined by:
	RandRplc = 0	Inverse to fitness
	RandRplc = 1	Uniform random
	RandRplc = 2	Uniform random from those with strength less than bound.

HOWEVER, classifiers with strength <= 0 (BUT WITHOUT NoRplFlg TRUE in CfNode)
ALWAYS are chosen for replacement first. This is done here. If more need to
be picked, PckBiaRp() is called to pick inverse to fitness, and
PckRndRp() is called to pick at (uniform) random.

NOTES:

Selection is without replacement, obviously. To enforce this, set the
ReplFlg TRUE when one is chosen for the pool, so we don't pick it again. 

HOWEVER, if CrowdFac > 1 (i.e., more than one chosen for the pool):
ReplFlg must then be set FALSE for any in the pool.
That is done by PckRCf(), which makes the pick from the pool.

When a classifier is actually replaced, the one that replaces it has
StpCrtd set to CycleStp, the current step, AND it has
EligRpl set FALSE, so it won't be picked for replacement later in this step!

Some Classifiers are NOT considered 'eligible' for replacement, e.g.,
classifiers with NoRplFlg (No Replacement Flag) set TRUE in the CfNode.
Those ELIGIBLE have the EligRpl set TRUE by ReCalcFitness().
NmEligRp is count of all eligible, including those with Str <= 0.
See ReCalcFitness() for details on fitness calculation and check for eligibility.

Because some classifiers may be ineligible for replacement
this subroutine may not pick the Number of classifiers requested.

*******/

unsigned int PckRPool ( )
{
	register unsigned int	nmtopick, cfcnt;
	register struct CfNode	*cfptr;

#if CTEST
	if ( DscDemo > 0 ) {
		sprintf( GOutBuff, "\nPick Replacable CfPool from %d Eligible (+ Str<= 0); CrowdFac %d).", NmEligRp, CrowdFac );
		WriteStd( GOutBuff );
	}
#endif

	RCPoolSz = 0;								/* haven't chosen any yet! */
	nmtopick = CrowdFac;

	if ( nmtopick > NmEligRp ) {	 			/* Pick more than eligible could be tiresome... */
		sprintf(GOutBuff,"\n\n***PckRPool: nmtopick (%d) is >= NmEligRp (%d)--step %d.",
			nmtopick, NmEligRp, CycleStp );
		WriteStd( GOutBuff );
		WriteStd( "\nPick NmEligRp\n\n");
		if ( NmEligRp > 0 )
			nmtopick = NmEligRp;
		else {
			WriteStd( "NmEligRp is 0! None Picked!!!\n\n");
			return ( 0 );
		}
	}

		/* look for classifiers with strength below zero if needed... */

	if ( CfStrMin <= 0 ) {
	 	for ( cfptr = CurCfs, cfcnt = 0; cfcnt < NmCfs && RCPoolSz < nmtopick; ++cfptr, ++cfcnt ) {
		 	if ( cfptr->Strength <= 0 && !cfptr->NoRplFlg ) {
				RPoolCfs[RCPoolSz++] = cfptr;	/* it gets chosen to be replaced */
				cfptr->ReplFlg = TRUE;			/* mark it as in pool */
				--NmEligRp;					 /* one fewer eligible (these are not counted in TRplBStr, so don't decrement */
#if CTEST
				if ( DscDemo >= 3 ) {
					sprintf( GOutBuff, "\nPut cf %d into RPoolCfs (str %f is <= 0).", cfptr->Cf_Id, cfptr->Strength );
					WriteStd( GOutBuff );
				}
#endif
			}
		}
	} /* endif there are strengths < 0 */

	if ( RCPoolSz < nmtopick )	{			/* We need to get some more */
  		if ( RandRplc ) 
    		PckRndRp( nmtopick );			/* Pick them at random */
	    else
		    PckBiaRp( nmtopick );			/* pick using biased inverse strength */
	}

#if CTEST
	if ( DscDemo > 0 )
		DemoRpl( );
#endif

	return( RCPoolSz );

} /* PckRPool */


VOID DemoRpl ( )
{
	register int	i;

	WriteStd( "  The pool:\n" );
	for ( i = 0; i < RCPoolSz; ++i ) {
		PutCNd( RPoolCfs[i], GOutBuff, DEMOFMT1 );
		WriteStd( GOutBuff );
	}

} /* DemoRpl */


/**********************************

PckRndRp	Pick classifiers at random (uniform) to be in the Replacable pool.

Pick until RCPoolSz, the number picked, is equal to NmToPick, the number we need.
Put pointers to classifiers chosen in the RPoolCfs array, at the RCPoolSz
entry, and then increment RCPoolSz.

NOTES:
If RandRplc == 1, then pick at uniform random from all without NoRplFlg set TRUE.
If RandRplc == 2, then pick at uniform random ONLY from those that
are not protected by:

a) strength > AveCfStr * RplCfUBd

b) are active in last RplACSBd steps and have strength > AveCfStr * RplACUBd

This assumes ReCalcFitness() has been called this step,
to calculate fitnesses and to mark classifiers as eligible for replacement.
Those eligible are marked EligRpl = TRUE.
The number of those is NmEligRp.
(EligRpl also is set FALSE for those created this step (StpCrtd == CycleStp),
so they won't be chosen later in the step.)

Set ReplFlg TRUE for each chosen, so not to choose again for this pool.
Also decrement NmEligRp for each rule chosen here.
(If CrowdFac > 1, PckRCf() will set undo these changes for those
in the pool not actually picked, so they could be in another pool this step.)

This subroutine is designed to be used when the CrowdFac is not a
significant fraction of the classifier list size. (e.g., < 1/4 or so). If you plan to use
big CrowdFac's, I'd use a modified version of the algorithm used in PckBiaRp().

******/

VOID PckRndRp ( NmToPick )
	unsigned int	NmToPick;
{
	register unsigned int	ri, hitProtected, hitProtectedBound;
	struct CfNode			*cfptr;

	hitProtected = 0;					   /* count number of hits on protected rules */
	if ( NmEligRp != 0 )					/* (check just in case) */
		hitProtectedBound = 10.0 * ( (float) NmCfs / (float) NmEligRp ); /* Bound is inverse to proportion eligible */
	else
		hitProtectedBound = 0;

#if CTEST
		if ( DscDemo >= 3 ) {
			if ( RandRplc == 1 )
				WriteStd( "\nPckRndRp: Pick at uniform random..." );
			else {
				sprintf( GOutBuff, "\nPckRndRp: Protect Str > %5.1f, for StpLPrd > %d, str > %5.1f", 
					AveCfStr * RplCfUBd, CycleStp-RplACSBd, AveCfStr * RplACUBd );
				WriteStd( GOutBuff );
			}
		}
#endif

	while ( RCPoolSz < NmToPick ) { 		/* while we need to pick more... */

		if ( NmToPick > NmEligRp )  {
			sprintf(GOutBuff,"\n\n***PckRndRp: NmToPick (%d) is > NmEligRp (%d), step %d.\nDon't pick any more.\n",
				NmToPick, NmEligRp, CycleStp );
			WriteStd( GOutBuff );
			break;
		}

		if ( (ri = URandN( NmCfs-1 )) >= NmCfs ) {
			sprintf( GOutBuff, "\n\n**PckRndRp: URandN (%d) >= NmCfs (%d).\nTry again (CycleStp %d).\n",
					 ri, NmCfs, CycleStp );
			WriteStd( GOutBuff );
			continue;
		}
		cfptr = CfLst + ri; 					/* get pointer to ri-th CfNode in list */

#if CTEST
		if ( DscDemo >= 3 || CycleStp == 4123 ) {
			sprintf( GOutBuff, "\n  NmEligRp %d, Test cf %d (ri %d, str %5.1f): ", NmEligRp, cfptr->Cf_Id, ri, cfptr->Strength );
			WriteStd( GOutBuff );
		}
#endif

			/* Don't use if picked just now or if Not Eligible */

		if ( cfptr->ReplFlg || !cfptr->EligRpl ) {

#if CTEST
			if ( DscDemo >= 3 || CycleStp == 4123 ) {
   				sprintf( GOutBuff, "  skip it (ReplFlg %d, StpCrtd %d, NoRplFlg %d, EligRpl %d).",
					 cfptr->ReplFlg, cfptr->StpCrtd, cfptr->NoRplFlg, cfptr->EligRpl );
				WriteStd( GOutBuff );
				if ( RandRplc == 2 ) {
					sprintf( GOutBuff, " (str %5.1f)", cfptr->Strength );
					WriteStd( GOutBuff );
				}
			}
#endif

			if ( ++hitProtected > hitProtectedBound ) {
				sprintf( GOutBuff, "\n**PckRndRp: hit protected rules %d times (Stp %u)! Don't try anymore this step.",
					 hitProtected, CycleStp );
				WriteStd( GOutBuff );
				sprintf( GOutBuff, "\n			NmEligRp %d, RCPoolSz %d, NmToPick %d.\n", 
					 NmEligRp, RCPoolSz, NmToPick );
				WriteStd( GOutBuff );
				return;
			}

			continue;
		}

		else {  								/* Pick this one! */
			RPoolCfs[RCPoolSz++] = cfptr;		/* store pointer to it, increment pool index */
			cfptr->ReplFlg = TRUE;				/* mark it */
			--NmEligRp;						 /* One fewer eligible */
			hitProtected = 0;				   /* start count over */

#if CTEST
			if ( DscDemo >= 3 || CycleStp == 4123 )
				WriteStd( "  <- into the pool!" );
#endif

		}

	}

} /* PckRndRp */


/**********************************

PckBiaRp	Pick classifiers to be in the Replacable pool using a biased probability distribution.

Pick until RCPoolSz, the number picked, is equal to NmToPick, the number we need.
Put pointers to classifiers chosen in the RPoolCfs array, at the RCPoolSz
entry, and then increment RCPoolSz.

Set ReplFlg TRUE for each chosen, so not to choose again for this pool.
Also decrement NmEligRp for each rule chosen here.
(If CrowdFac > 1, PckRCf() will undo these changes for those
in the pool not actually picked, so they could be in another pool this step.)

Pick classifiers with probability equal to RBiasStr, a fitness
value calculated in ReCalcFitness(), divided by TRplBStr, 
the sum of RBiasStr over ELIGIBLE classifiers with Strength > 0.

Note that a local value for TRplBStr, totbstr, is kept and updated
as classifiers are chosen for the pool.  However, TRplBStr is only
updated when a specific classifier is chosen from the pool by PckRCf().

Some classifiers are marked as INELIGIBLE for replacement (EligRpl==FALSE);
the number eligible is NmEligRp.  See ReCalcFitness for details.
EligRpl is set FALSE for those created this step (StpCrtd == CycleStp),
so they won't be chosen later in the step.

******/

VOID PckBiaRp ( NmToPick )
	unsigned int	NmToPick;
{
	register int	cfcnt;
	register struct CfNode	*cfptr;
	register float	randnum, cummul;
	float			totbstr, frtotbs;

	totbstr	= TRplBStr;
	frtotbs	= totbstr * 0.01;				/* recalc (to correct round errors) when totbstr < this number */

	while ( RCPoolSz < NmToPick )  {	  	/* while we need to pick more... */

		if ( NmToPick > NmEligRp )  {
			sprintf(GOutBuff,"\n\n***PckBiaRp: NmToPick (%d) is > NmEligRp (%d), step %d.\nDon't pick any more.\n",
				NmToPick, NmEligRp, CycleStp );
			WriteStd( GOutBuff );
			break;
		}
		
		randnum = URandF( totbstr );		/* get random number 0 .. total biased strength */

#if CTEST
		if ( DscDemo >= 3 ) {
			sprintf( GOutBuff, "\nPckBiaRp @ %d: totbstr %8.3e, randnum %8.3e, Protect Str > %5.1f; StpLPrd > %d, %5.1f", 
				CycleStp, totbstr, randnum, AveCfStr * RplCfUBd, CycleStp-RplACSBd, AveCfStr * RplACUBd );
			WriteStd( GOutBuff );
		}
#endif

			/* skip if replaced already, not eligible, or created this step */

		for ( cfptr = CurCfs, cummul = cfcnt = 0; cfcnt < NmCfs; ++cfptr, ++cfcnt ) {

#if CTEST
			if ( DscDemo >= 3 ) {
				sprintf( GOutBuff, "\ncf %d: str %5.1f, StpCrtd %d, EligRpl %d, ReplFlg %d, StpLPrd %d.",
					 cfptr->Cf_Id, cfptr->Strength, cfptr->StpCrtd, cfptr->EligRpl, cfptr->ReplFlg, cfptr->StpLPrd );
				WriteStd( GOutBuff );
			}
#endif

			if ( cfptr->ReplFlg || !cfptr->EligRpl )
				continue;						/* skip it if replaced, not eligible (e.g., new this step) */

			cummul += cfptr->RBiasStr;			/* increment cummulative bias str. */

#if CTEST
			if ( DscDemo >= 3 ) {
				sprintf( GOutBuff, ": cummul now %8.3e.", cummul );
				WriteStd( GOutBuff );
			}
#endif

			if ( cummul >= randnum )
				break;							/* This is the one drawn. */

		}   /* endfor all classifiers */

		if ( cfcnt >= NmCfs ) {			/* do this test first, just in case... */
			sprintf( GOutBuff,"\n**PckBiaRp: Rand draw NULL. randnum %f, cummul %f, totbstr %f, TRplBStr %f (step %u)!",
				randnum, cummul, totbstr, TRplBStr, CycleStp );
			WriteStd( GOutBuff );
		}

		else {  								/* This is it! */
#if CTEST
			if ( DscDemo >= 3 ) 
				WriteStd( "  <= into the pool.\n");
#endif
			RPoolCfs[RCPoolSz++] = cfptr;		/* store pointer to it */
			cfptr->ReplFlg = TRUE;				/* mark it */
			--NmEligRp;						 /* fewer eligible */
			totbstr -= cfptr->RBiasStr; 		/* adjust total for next draw without replacement */

			if ( totbstr < frtotbs ) {			/* its time to re-sum the total */
#if CTEST
				if ( DscDemo >= 3 ) {
					WriteStd("\n**PckBiaRp: Recalc totbiastr for rand draw.");
					sprintf(GOutBuff,"\nTRplBStr %8.3e totbstr %8.3e, frtotbs %8.3e,",TRplBStr,totbstr,frtotbs);
					WriteStd( GOutBuff );
				}
#endif

					 /* include only if not replaced, Eligible, and if not new this step */

				for ( cfptr = CurCfs, cfcnt = totbstr = 0; cfcnt < NmCfs; ++cfptr, ++cfcnt )
					if ( !cfptr->ReplFlg && cfptr->EligRpl && cfptr->StpCrtd != CycleStp )
						totbstr += cfptr->RBiasStr;

				frtotbs = totbstr * 0.01;
#if CTEST
				if ( DscDemo >= 3 ) {
					sprintf( GOutBuff,"\n  New totbstr %8.3e frtotbs %8.3e.\n", totbstr, frtotbs );
					WriteStd( GOutBuff );
				}
#endif
			}	 /* endif had to recalc totbstr */

		} /* endelse this is one to put in pool */

	} /* } we have more to pick */

} /* PckBiaRp */



/**********************************

PckRCf		Pick a classifier to replace, from a 'crowd' if there is one.

	New		Pointer to NEW classifier that will replace one chosen.

	Return	Pointer to classifier to be replaced.

If CrowdFac is 1, just take first cf from the pool.
If CrowdFac > 1, look through the RPoolCfs array for a classifier with the
most similar 'genotype' and then return that as one to replace.

NOTES:

If !RandRplc, decrement TRplBStr by RBiasStr for the classifier chosen:
its no longer eligible, so it won't figure in future probability calcuations.
So, set EligRpl FALSE to note that.

If CrowdFac > 1, for each rule not chosen to be replaced,
increment NmEligRp (its eligible again).

For all rules, set ReplFlg FALSE. Its not in the pool anymore.

******/

struct CfNode *PckRCf ( New )
	struct NCfNode	*New;
{
	unsigned int	crowdi, oldi;
	float			oldscore, tscore, GenTypeM();
	struct CfNode *old;

	if ( CrowdFac == 1 ) {					/* Just take the first and only one */
		old = RPoolCfs[0];
		old->ReplFlg = FALSE;
	}

	else {									/* CrowdFac > 1, so look through the pool */

#if CTEST
		if ( DscDemo > 0 ) {
			sprintf( GOutBuff, "  Checking Crowd for nearest to new cf %d...", New->NewCf_Id );
			WriteStd( GOutBuff );
		}
#endif

		oldi = 0;							   /* assume the first in crowd is one to use */
		old  = RPoolCfs[oldi];
		oldscore = GenTypeM( old, New );		/* get its score */

#if CTEST
		if ( DscDemo > 0 ) {
			sprintf( GOutBuff, "\n	Assume %d is closest (score %f).", old->Cf_Id, oldscore );
			WriteStd( GOutBuff );
		}
#endif

		for ( crowdi = 1; crowdi < CrowdFac; ++crowdi ) {	 /* look at the rest and replace old by better ones */

			if ( oldscore == (float) 1.0 )			/* perfect match, so... */
				break;								/* just use this 'old' */

			else {
				tscore = GenTypeM( RPoolCfs[crowdi], New );

#if CTEST
				if ( DscDemo > 0 ) {
					sprintf( GOutBuff, "\n	Check %d (score %f)...", RPoolCfs[crowdi]->Cf_Id, tscore );
					WriteStd( GOutBuff );
				}
#endif

				if ( tscore > oldscore ) {
					oldi = crowdi;					/* this one is better, so save it. */
					old = RPoolCfs[oldi];
					oldscore = tscore;				/* save the score, too. */
#if CTEST
					if ( DscDemo > 0 )
						WriteStd( "its closer--assume its best." );
#endif

				}
			}

		}

		NmEligRp += CrowdFac - 1;				   /* Any not chosen are again eligible */
		for ( crowdi = 0; crowdi < CrowdFac; ++crowdi ) 
			RPoolCfs[crowdi]->ReplFlg = FALSE;		/* reset ALL these flags */

	} /* endelse CrowdFac > 1 */

	if ( !RandRplc )								/* if were using a bias distribution... */
		TRplBStr -= old->RBiasStr;					/* adjust total to reflect absence of this cf */

#if CTEST
	if ( DscDemo >= 3 ) {
		WriteStd( "\nCf Id, ReplFlg now:\n" );
		for ( crowdi = 0; crowdi < CrowdFac; ++crowdi ) {
			sprintf( GOutBuff, "%4d  %1d\n", RPoolCfs[crowdi]->Cf_Id, RPoolCfs[crowdi]->ReplFlg );
			WriteStd( GOutBuff );
		}
	}
#endif

	return( old );	

} /* end PckRCf */


/**********************************

GenTypeM	Calculate degree to which genotypes of two classifiers match.

	Cf		Pointer to CfNode of one classifier.

	NCf		Pointer to NCfNode of the other.  <=== Note this is a NCfNode ====

	Return	Match score, from 0.0 (completely different) to 1.0 (exactly the same).

Just look at string parts of two conditions and the action.
(Currently this does NOT count condition types or action-type.)
Score is just sum of loci at which the two classifiers are the same
divided by the total number of loci compared. 
Note that 'the same' means 0=0, 1=1, or #=#; the #'s (don't cares) don't
have a special status when looking at the classifiers genotype.

******/

float GenTypeM ( Cf, NCf )
	struct	CfNode	*Cf;
	struct NCfNode	*NCf;
{
	float	ret;
	int 	i;
	char	cfa[STRNGSZ+1], ncfa[STRNGSZ+1];

	cfa[STRNGSZ] = ncfa[STRNGSZ] = '\0';

	ret = 0.0;

	BCndtoA( Cf->Cnd1Bits, Cf->Cnd1DCs, cfa );		/* get ascii form of first conditions */
	BCndtoA( NCf->NewCnd1B, NCf->NewCnd1D, ncfa );
	for ( i = 0; i < STRNGSZ; ++i )
		if ( cfa[i] == ncfa[i] )
			ret += 1;

	BCndtoA( Cf->Cnd2Bits, Cf->Cnd2DCs, cfa );		/* get ascii form of second conditions */
	BCndtoA( NCf->NewCnd2B, NCf->NewCnd2D, ncfa );
	for ( i = 0; i < STRNGSZ; ++i )
		if ( cfa[i] == ncfa[i] )
			ret += 1;

	BActtoA( Cf->ActBits, Cf->ActDCs, cfa );		/* get ascii form of actions */
	BActtoA( NCf->NewActB, NCf->NewActD, ncfa );
	for ( i = 0; i < STRNGSZ; ++i )
		if ( cfa[i] == ncfa[i] )
			ret += 1;

	ret = ret / (3.0 * STRNGSZ);					/* return a fraction from 0.0 to 1.0 */

	return( ret );

} /* GenTypeM */


/**********************************

WrtNCf		Write ascii form of NEW classifier (from NCfNode) to buffer.

	NewCf 	Pointer to NCfNode containing cf to write.
	Buff	Char Buff into which cf is to be written.
	Format	If 1, display with Id, Strength, and BidRatio.
			Else display for use by the CrFullCf() subroutine (see DSC-OPS.C).

******/

VOID WrtNCf ( NewCf, Buff, Format )
	struct	NCfNode	*NewCf;
	char			Buff[];
	int 			Format;
{
	int		retval;
	char	cfbuf[3*(STRNGSZ)+32],			/* assemble the classifier itself here */
			tbuf[STRNGSZ+10]; 				/* temporary buffer */
	struct CfOpNode *coptr, *GetCOpCd();	/* pointer to classifier operator (ActType) node. */

	Buff[0] = cfbuf[0] = '\0';
	if ( NewCf->NewCnd1T == 0 )
		cfbuf[0] = 'm';
	else
		cfbuf[0] = '~';
	BCndtoA( NewCf->NewCnd1B, NewCf->NewCnd1D, tbuf );
	tbuf[STRNGSZ] = ',';
	tbuf[STRNGSZ+1] = '\0';
	CopyChar( &cfbuf[1], tbuf, STRNGSZ/2 );
	cfbuf[(STRNGSZ/2)+1] = ' ';
	cfbuf[(STRNGSZ/2)+2] = '\0';
	strcat( cfbuf, &tbuf[STRNGSZ/2] );

	if ( NewCf->NewCnd2T == 0 )
		tbuf[0] = 'm';
	else
		tbuf[0] = '~';
	BCndtoA( NewCf->NewCnd2B, NewCf->NewCnd2D, &tbuf[1]);
	tbuf[STRNGSZ+1] = '/';
	tbuf[STRNGSZ+2] = '\0';
	strcat( cfbuf, tbuf );

	coptr = GetCOpCd( NewCf->NewActT, &retval );
	if ( retval == ERROR ) { 
		sprintf(GOutBuff,"\n\n***WrtNCf: cf %d has bad ActType %d.\n",
				NewCf->NewCf_Id, NewCf->NewActT );
		WriteStd( GOutBuff );
	}
	strcat( cfbuf, coptr->CfOpName );
	strcat( cfbuf, "/" );
	BActtoA( NewCf->NewActB, NewCf->NewActD, tbuf );
	tbuf[STRNGSZ] = '\0';
	strcat( cfbuf, tbuf );
	if ( Format == 1 )
		sprintf( Buff, "%-4u %s  %6.0f  %4.2f\n", NewCf->NewCf_Id, cfbuf, NewCf->NewStr, NewCf->NewBR );
	else
		sprintf( Buff, "%s", cfbuf );

} /* WrtNCf */
#endif	/* CDSC_ALL */

