/*
 * pendemo.c	Non-fuzzy code for the inverted pendulum simulation.
 *
 * Copyright (c) Togai InfraLogic, Inc. 1989, 1990, 1991, 1992.
 * All rights reserved.
 *
 * $Header:   Z:/vcs/common/pendemo.c_v   1.13   13 Oct 1992 11:38:10   eah  $
 *
 * LANGUAGE : Microsoft C, Version 6.00A
 *
 * Main loop
 *
 *     while (not quitting) do begin
 *
 *	   scale program vars to pendulum universe of discourse
 *	   call pendulum fuzzy-routine
 *	   scale pendulum output and calculate new inputs
 *
 *	   update all dynamic windows that are open
 *	   check keyboard for event and process it
 *
 *	   swap video pages for next cycle
 *
 *     end
 *
 * Animation technique:
 *
 *     The simulation of the pendulum and the updating of all
 *     dynamic windows (i.e., a window containing a fuzzy-rule
 *     and its associated membership functions) is accomplished
 *     by the following steps:
 *
 *	   1 - To start, page flipping is accomplished by
 *	       a call to the following routines:
 *
 *		     _setvisualpage(page_number);
 *				 and
 *		     _setactivepage(page_number);
 *
 *	       Animation is simulated by setting the
 *	       visual page to the opposite of the active
 *	       page.  All drawing is then done on the
 *	       active page.   When the drawing
 *	       is complete, and the new variables have
 *	       been calculated, you then switch pages
 *	       and repeat the process.
 *
 *	   2 - Initializing both pages (0 and 1) to a
 *	       known state.  In the case of this program,
 *	       all static text and graphics, the outline
 *	       of the windows, their frames and any other
 *	       static information, is drawn to both pages.
 *
 *	       Usually, the same routine is called twice,
 *	       with a call to a page flipping routine
 *	       in between for this type of drawing.
 *
 *	   3 - While the visual page is being viewed by the
 *	       user, we are scaling the input variables for
 *	       the Fuzzy-C routine call.  After this we
 *	       call the actual externally linked Fuzzy-C
 *	       routine, and then upon return, scale the
 *	       returned output to our graphical scale.
 *
 *	   4 - Updating all dynamic information on the active
 *	       page.  All graphs are updated, and the pendulum
 *	       is redrawn.
 *
 *	   5 - The last step is to switch pages and repeat
 *	       the process from step 3.
 *
 * Miscellaneous notes:
 *
 *     You should note in the demo.til file the structure and loc-
 *     ation of the embedded C code.  Any rules you might add
 *     must follow the rules outlined at the beginning of that
 *     file.
 */

/*
 * include files
 */

#include <stdio.h>
#include <graph.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <memory.h>
#include <stdlib.h>
#include <conio.h>
#include <malloc.h>

#include <tilcomp.h>			/* portability definitions */
#ifdef	FC110
#include "drvr110.h"			/* FC110 interface stuff */
#endif
#include "pendemo.h"			/* structures and constants */
#include "demo.h"			/* knowledge base routines */

/*
 * global variables
 */

int ThetaScaleFactor;			/* Theta scaler factor */
int OmegaScaleFactor;			/* Omega scaler factor */

int motorscalefactor	    = 4;
int motorsize		    = 30;	/* Motor current output */
int motormax		    = 45;	/* Motor maximum limit */
int motormin		    = 15;	/* Motor minimum limit */
int motorstep		    = 1;	/* Motor increse/decrease step */
int motorsteps		    = 30;

int sticksize		    = 10;	/* distance of the bob to pivot */
int stickfactor 	    = -1;
int sticktop		    = 13;
int stickbottom 	    = 7;
short stickstop 	    = TRUE;

int bobmass		    = 19;	/* mass of the bob */
int bobmax		    = 35;	/* maximun mass of the bob */
int bobmin		    = 5;
int bobstep		    = 1;	/* Bob increase/decrease step */
int bobsteps		    = 30;
int bopthreshold	    = 4;
int bopstartthresh	    = 220;

int Error;				/* Unscaled input to rulebase */
int dError;				/* Unscaled input to rulebase */
int Current;				/* Output from fuzzy rulebase */
SBYTE tempcurrent;			/* scaled output for our system */

int lastError;
int lastdError;

int bopticks		    = 0;
short animatebop	    = FALSE;
short shooting		    = FALSE;
short retractbopper	    = FALSE;
short allowbopping	    = TRUE;
int shotpos		    = 0;
int extendbopper	    = 0;
int bopperimpactpt	    = 0;

int timelinedata[100];
int timelinemax 	    = 65;

struct window		    windows[12];  /* Windows array */
struct rulelistelem	    rulelist[11]; /* Rule list */
int s4data[5][255];
int s6data[5][255];
short done;				/* Program terminated flag */
short visualPage;			/* Page flipping var */

int helpwin		    = 0;	/* help window index */
int timewin		    = 1;	/* time value window index */
int pendwin		    = 2;	/* pendulum window index */
int pickwin		    = 3;	/* pick list matrix index */
int currentwin		    = 4;	/* current window index */
int mbfwin		    = 5;	/* rule window index */
int titlewin		    = 6;	/* titlebar window index */
int messagewin		    = 7;	/* message window index */
int aboutwin		    = 11;	/* about window index */
int numwinopen		    = 5;	/* number of windows open */
int curmbfx		    = 2;	/* current matrix x index */
int curmbfy		    = 2;	/* current matrix y index */
int pendball		    = RED;	/* pendball var for flashing */
int curpage		    = 1;	/* Help current page */
int maxpage		    = 2;	/* Help maximun pages */
int currule		    = 0;	/* currently selected rule */
int oldrule		    = 1;	/* previously selected rule */
int ruletodisable	    = -1;	/* rule that will be disabled */

char title1[50] 	    = " Fuzzy-C Pendulum Demonstration, Ver. 3.0 ";
char title2[50] 	    = "          Togai InfraLogic, Inc.          ";
char title3[50] 	    = "    Artificial Intelligence On A Chip     ";

char huge *helpmaps[5];
char huge *rulemaps[11];		/* Rule text images	    */
char huge *miscmaps[13];
char huge *screensave;			/* Screen buffer for info   */
char huge *helpedwin;			/* Screen buffer for help   */
char huge *titlebarmap[3];		/* Screen buffer for title  */
char huge *minstrmap;			/* Screen buffer for minstr */

UBYTE *errorlist[5] = {
    _Theta_PM_map,
    _Theta_PS_map,
    _Theta_Z_map,
    _Theta_NS_map,
    _Theta_NM_map,
};

UBYTE *derrorlist[5] = {
    _dTheta_PM_map,
    _dTheta_PS_map,
    _dTheta_Z_map,
    _dTheta_NS_map,
    _dTheta_NM_map,
};

char	listnames[5][3] = {
    "PM",
    "PS",
    "Z ",
    "NS",
    "NM",
};

float Inertia	   = (float)2.0;  /* sticksize ** 2 * bobmass */
float Theta	   = (float)0.0;  /* actual scaled bob angle */
float DesiredTheta = (float)1.570796; /* desired bob angle PI/2 */
float Omega	   = (float)0.0;  /* actual scaled bob angular vel. */
float Alpha	   = (float)0.0;  /* actual bob angular acceleration */
float I 	   = (float)0.0;  /* real unscaled Current -1 to 1 */
float limitLeft;		  /* Left pendulum limiter PI */
float limitRight;		  /* Right pendulum limiter 0 */

#ifdef TRACE
FILE *output_fp;
#endif

#ifdef FC110
#include "demo110.h"
int fc110 = 0;
#endif

/*
 * main routine
 */

void main ()
{
#ifdef FC110
    printf ("Initializing FC110 knowledge base memory...\n");
    demo110_init ("demo110.hex", 0);
#endif

    initialize ();

    while (!done) {

	updatePendulum ();
	updateDisplay ();
	updateTimeLine ();
	swapvideopages ();

	updatePendulum ();
	updateDisplay ();
	updateTimeLine ();
	swapvideopages ();

	if (currule != oldrule)
	    oldrule = currule;

	if (ruletodisable == -1)
	    ruletodisable = -1;

	checkkeyboard ();
    }

    closeup ();
}

/*
 * The 'done' flag was set, and now its time to close up and go home
 */

void closeup ()
{
    _setvisualpage (0);
    _setvideomode (_TEXTC80);
}

/*
 * We are ready to cycle through the pendulum fuzzy call. First we must
 * scale our input vars from our graphical usages (i.e., -1.6 to 1.6 for
 * Omega and PI to 0 for Theta) to the universe of discourse of the fuzzy
 * inputs (Error and dError which are 0 to 255). After calling Pendulum()
 * we update Omega and Theta for the next cycle.
 */

void updatePendulum()
{
    scaleerrors ();

    tempcurrent = 0;

#ifdef FC110
    if (fc110) {
	Pend_FC110 ((SBYTE) Error, (SBYTE) dError, &tempcurrent);
	get_alphas ();
    } else
#endif
	Pend ((SBYTE) Error, (SBYTE) dError, &tempcurrent);

#ifdef TRACE
    fprintf (output_fp, "%5.d %6.d %7.d\n", Error, dError, tempcurrent);
#endif

    cyclesystem (tempcurrent);
}

#ifdef	FC110

/*
 * Pick up the rule alpha values when operating in FC110 mode.
 */

void get_alphas ()
{
    int i;
    GLOBALS

    SET_GLOBALS(0)

    i = 5;

    rulelist[PM_Z].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[Z_NM].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[Z_PM].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[NM_Z].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[PS_Z].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[Z_NS].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[NS_PS].DOM = (FUBYTE) LOAD(i);   i++;
    rulelist[Z_PS].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[NS_Z].DOM =  (FUBYTE) LOAD(i);   i++;
    rulelist[PS_NS].DOM = (FUBYTE) LOAD(i);   i++;
    rulelist[Z_Z].DOM =   (FUBYTE) LOAD(i);   i++;
}

#endif

/*
 * Update all of the dynamic windows.
 */

void updateDisplay ()
{
    drawcurrent (currentwin);
    drawpendulum (2, Theta);
    updatecurmbf (0, 0);
    drawrule ();
    drawActiveRules ();
}

/*
 * check the keyboard and see if the user has entered a command
 * which we need to process.  An invalid command brings up the help.
 */

void checkkeyboard ()
{
    int key;

    if ((bopticks > bopstartthresh) && (!animatebop)) {
	animatebop = TRUE;
	retractbopper = FALSE;
	extendbopper = 0;
	bopticks = 0;
    }

    if (kbhit ()) {

	key = getch ();
	key = toupper (key);

	switch (key) {

	    case 27:
	    case 'Q':
		/*ESCAPE get me outta here */
		done = TRUE;
#ifdef TRACE
		fclose (output_fp);
#endif
		break;

	    case '?':
		curpage = 1;
		showhelp ();
		break;

#ifdef FC110
	    case 'f':
	    case 'F':
		fc110 = !fc110;
		break;
#endif

#ifdef NEC_PC
	    case F1:	  /* F1  disable/enable rule  */
		mbfdisable();
		break;

	    case F2:	  /* F2  freeze pendulum      */
		key=getch ();

		if (key == 0)
		    getch ();
		break;

	    case F3:	  /* F3  bump to the left     */
		Omega = (float) Omega + 3.0;
		pendball = bumpball;
		break;

	    case F4:	  /* F4  bump to the right    */
		Omega = (float) Omega - 3.0;
		pendball = bumpball;
		break;

	    case F5:	  /* F5  Increase Motor Size  */
		motorsize = motorsize + motorstep;
		if (motorsize > motormax)
		    motorsize = motormax;
		break;

	    case F6:	  /* F6  Increase Bob Size    */
		bobmass = bobmass + bobstep;

		if (bobmass > bobmax)
		    bobmass = bobmax;
		break;

	    case F7:	  /* F7  Stop bob movement    */
		stickstop = !stickstop;
		break;

	    case F8:	  /* F8  Stop bopping *hah*   */
		allowbopping = !allowbopping;
		bopticks = 0;
		extendbopper = 0;
		retractbopper = FALSE;
		animatebop = FALSE;
		break;

	    case F9:	  /* F9  Start Pendulum Left  */
		Theta = (float) PI;
		Omega = (float) 0.0;
		Alpha = (float) 0.0;
		break;

	    case F10:	  /* F10 Start Pendulum Right */
		Theta = (float) 0.0;
		Omega = (float) 0.0;
		Alpha = (float) 0.0;
		break;

	    case SF5:	  /* Shift F5 Decrease Motor  */
		motorsize=motorsize-motorstep;

		if (motorsize < motormin)
		    motorsize = motormin;
		break;

	    case SF6:	  /* Shift F6 Decrase Bob     */
		bobmass = bobmass - bobstep;

		if (bobmass < bobmin)
		    bobmass = bobmin;
		break;

	    case 11:	  /* Move Up in Rule List     */
		if (currule > 0)
		    currule = currule - 1;
		else
		    currule = numrules;
		break;

	    case 10:	  /* Move Down in Rule List   */
		if (currule < numrules)
		    currule = currule + 1;
		else
		    currule = 0;
		break;
#else
	    case 0:
		/*Hey, he entered a function key... which one	 */

		key = getch ();

		switch (key) {

		    case 59:	  /* F1  disable/enable rule  */
			mbfdisable();
			break;

		    case 60:	  /* F2  freeze pendulum      */
			key=getch ();

			if (key == 0)
			    getch ();
			break;

		    case 61:	  /* F3  bump to the left     */
			Omega = (float) Omega + 3.0;
			pendball = bumpball;
			break;

		    case 62:	  /* F4  bump to the right    */
			Omega = (float) Omega - 3.0;
			pendball = bumpball;
			break;

		    case 63:	  /* F5  Increase Motor Size  */
			motorsize = motorsize + motorstep;
			if (motorsize > motormax)
			    motorsize = motormax;
			break;

		    case 64:	  /* F6  Increase Bob Size    */
			bobmass = bobmass + bobstep;

			if (bobmass > bobmax)
			    bobmass = bobmax;
			break;

		    case 65:	  /* F7  Stop bob movement    */
			stickstop = !stickstop;
			break;

		    case 66:	  /* F8  Stop bopping *hah*   */
			allowbopping = !allowbopping;
			bopticks = 0;
			extendbopper = 0;
			retractbopper = FALSE;
			animatebop = FALSE;
			break;

		    case 67:	  /* F9  Start Pendulum Left  */
			Theta = (float) PI;
			Omega = (float) 0.0;
			Alpha = (float) 0.0;
			break;

		    case 68:	  /* F10 Start Pendulum Right */
			Theta = (float) 0.0;
			Omega = (float) 0.0;
			Alpha = (float) 0.0;
			break;

		    case 88:	  /* Shift F5 Decrease Motor  */
			motorsize=motorsize-motorstep;

			if (motorsize < motormin)
			    motorsize = motormin;
			break;

		    case 89:	  /* Shift F6 Decrase Bob     */
			bobmass = bobmass - bobstep;

			if (bobmass < bobmin)
			    bobmass = bobmin;
			break;

		    case 72:	  /* Move Up in Rule List     */
			if (currule > 0)
			    currule = currule - 1;
			else
			    currule = numrules;
			break;

		    case 80:	  /* Move Down in Rule List   */
			if (currule < numrules)
			    currule = currule + 1;
			else
			    currule = 0;
			break;
		};
		break;
#endif

	    default:
		curpage = 1;
		showhelp ();

	}
    }
}

/*
 * Disable or Enable a rule.  This will toggle the flag that is embedded in
 * the demo.fuz file, thus turning that rule on or off.
 */

void mbfdisable ()
{
    rulelist[currule].disabled = !rulelist[currule].disabled;
    ruletodisable = currule;
}

/*
 *  Initialize everything...
 */

void initialize ()
{
    int i;
    struct videoconfig vc;

#ifdef NEC_PC
    if (_setvideomode (_98RESSCOLOR) == 0 ) {
#else
    if (_setvideomode (_ERESCOLOR) == 0 ) {
#endif
	printf ("Video configuration not successful. Exiting...\n");
	exit (1);
    }

    _getvideoconfig(&vc);

#ifdef TRACE
    if ((output_fp = fopen ("pend.dat", "w+")) == NULL) {
	printf ("cannot create pend.dat, aborting\n");
	exit (1);
    }

    fprintf (output_fp, "Theta dTheta Current\n");
#endif

    for (i = 0; i < timelinemax+1; i++)
	timelinedata[i] = 0;

    swapvideopages ();

#ifdef	NEC_PC
    _setcolor (BLACK);
    _rectangle (_GFILLINTERIOR, 0, 0, 639, 399);
    _setcolor (LTYELLOW);
    strcenter (14, "Initializing ...");
    swapvideopages ();

    _setcolor (BLACK);
    _rectangle (_GFILLINTERIOR, 0, 0, 639, 399);
    _setcolor (LTYELLOW);
    strcenter (14, "Initializing ...");
    swapvideopages ();
#endif

    createbitmaps ();

    _setcolor (mainscreen);
    _rectangle (_GFILLINTERIOR, 0, 0, 639, 349);
    swapvideopages ();

    _setcolor (mainscreen);
    _rectangle (_GFILLINTERIOR, 0, 0, 639, 349);
    swapvideopages ();

    setwindow (helpwin, 100, 50, 420, 250, BLACK);	  /* Help Window		      */
    setwindow (timewin, 10, 56, 175, 131, BLACK);	  /* Value Time-Plot Window*/
    setwindow (pendwin, 225, 56, 175, 131, BLACK);	  /* Pendulum Window	   */
    setwindow (pickwin, 435, 56, 195, 131, BLACK);	  /* Pick List Window	   */
    setwindow (currentwin, 10, 197, 175, 147, BLACK);	  /* Current Window	   */
    setwindow (mbfwin, 225, 197, 405, 147, BLACK);	  /* Mbf Window 		      */
    setwindow (titlewin, 10, 5, 620, 42, BLACK);	  /* Title bar info window */
    setwindow (messagewin, 140, 95, 340, 35, BLACK);	  /* Message Window	   */
    setwindow (aboutwin, 100, 50, 420, 250, BLACK);	  /* About Window	      */

    initMbfpoints ();

    ThetaScaleFactor = 50;
    OmegaScaleFactor = 15;
    limitLeft = (float) 0.0;
    limitRight = (float) PI;

    for (i = 1; i <= numwinopen; i++)
       openwindow (i);

    drawRuleMatrix (3);
    drawcurrent (currentwin);
    drawpendulum (2, DesiredTheta);
    drawtimeline (1);
    updateTimeLine ();
    swapvideopages ();

    drawRuleMatrix (3);
    drawcurrent (currentwin);
    drawpendulum (2, DesiredTheta);
    drawtimeline (1);
    updateTimeLine ();
    swapvideopages ();

    animateheading ();

    done = FALSE;
}

int findelem (x, y)
int x,y;
{
    register int i;

    for (i = 0; i < numrules + 1; i++)
	if (rulelist[i].thetaptr == x && rulelist[i].dthetaptr == y)
	   return i;

    return 99;
}

void showabout(void)
{
    int key;
    struct window w;

    _setvisualpage (0);
    _setactivepage (0);

    w = windows[aboutwin];

    screensave = new_bitmap (
		    w.horiz-3,
		    w.vert-3,
		    w.horiz+w.width+3,
		    w.vert+w.height+3
		 );

    _getimage (w.horiz-3,w.vert-3,w.horiz+w.width+3,w.vert+w.height+3,screensave);

    frame (0,LTYELLOW);
    _setcolor (BLACK);
    _rectangle (_GFILLINTERIOR,w.horiz,w.vert,w.horiz+w.width,w.vert+w.height);

    textcolor (LTBLUE);
    strcenter (5, "Here is where the about stuff will go");


    for (;;) {
	key=getch();
	if (key==27)
	    break;
    }

    _putimage (w.horiz-3,w.vert-3,screensave,_GPSET);

    _setvisualpage (!visualPage);
    _setactivepage (visualPage);

    hfree ((void huge*)screensave);
}

/*
 *  Pop up the help window and call the paging routine.. upon return, restore
 *  the screen to its original state
 */

void showhelp()
{
     struct window w;
     int    key;

     _setvisualpage(0);
     _setactivepage(0);

     w=windows[0];

     screensave = new_bitmap (
		    w.horiz-3,
		    w.vert-3,
		    w.horiz+w.width+3,
		    w.vert+w.height+3
		  );

     _getimage(w.horiz-3,w.vert-3,w.horiz+w.width+3,w.vert+w.height+3,screensave);

     frame(0,LTYELLOW);

     helppage(curpage);

     for (;;) {
	     key=getch();
#ifdef NEC_PC
	     if (key=='P' || key=='p')
	       if (curpage>1) {
		       curpage--;
		       helppage(curpage);
	       }
	     if (key=='N' || key=='n')
	       if (curpage<maxpage) {
		       curpage++;
		       helppage(curpage);
	       }
#else
	     if (key==0) key=getch();
	     if (key==73)
	       if (curpage>1) {
		       curpage--;
		       helppage(curpage);
	       }
	     if (key==81)
	       if (curpage<maxpage) {
		       curpage++;
		       helppage(curpage);
	       }
#endif
	     if (key==27)
	       break;
     }

     _putimage(w.horiz-3,w.vert-3,screensave,_GPSET);

     _setvisualpage(!visualPage);
     _setactivepage(visualPage);

     hfree((void huge*)screensave);
}

/*
 * Give the user some help..allow them to page through back and forth
 */
void helppage (page)
int page;
{
    struct window w;

    w = windows[0];

    _setcolor (BLACK);
    _rectangle (_GFILLINTERIOR, w.horiz, w.vert, w.horiz+w.width, w.vert+w.height);

    textcolor (LTBLUE);

    switch (page) {

	case 1:
	    strcenter (5,"Valid Commands");

	    textcolor (WHITE);

#ifdef NEC_PC
	    textmode (_GOR);
	    strcenter ( 7, "1 - Toggles Current Fuzzy-Rule On or Off     ");
	    strcenter ( 8, "2 - Freezes the Pendulum                     ");
	    strcenter ( 9, "3 - Bumps the Pendulum to the Left           ");
	    strcenter (10, "4 - Bumps the Pendulum to the Right          ");
	    strcenter (11, "5 - Increases Motor size  (Shift-5 to Lower) ");
	    strcenter (12, "6 - Increases Bob Mass    (Shift-6 to Lower) ");
	    strcenter (13, "7 - Toggles Bob Movement On or Off           ");
	    strcenter (14, "8 - Toggles Bopper On or Off                 ");
	    strcenter (15, "9 - Starts the Pendulum to the Left          ");
	    strcenter (16, "0 - Starts the Pendulum to the Right         ");
	    strcenter (17, "ESC - Exit the Pendulum Demonstration        ");
	    textmode (_GPSET);
#else
	    strcenter ( 7, "F1  - Toggles Current Fuzzy-Rule On or Off     ");
	    strcenter ( 8, "F2  - Freezes the Pendulum                     ");
	    strcenter ( 9, "F3  - Bumps the Pendulum to the Left           ");
	    strcenter (10, "F4  - Bumps the Pendulum to the Right          ");
	    strcenter (11, "F5  - Increases Motor size  (Shift-F5 to Lower)");
	    strcenter (12, "F6  - Increases Bob Mass    (Shift-F6 to Lower)");
	    strcenter (13, "F7  - Toggles Bob Movement On or Off           ");
	    strcenter (14, "F8  - Toggles Bopper On or Off                 ");
	    strcenter (15, "F9  - Starts the Pendulum to the Left          ");
	    strcenter (16, "F10 - Starts the Pendulum to the Right         ");
	    strcenter (17, "ESC - Exit the Pendulum Demonstration          ");
#endif
	    textcolor (LTYELLOW);
	    strcenter (19, "Up and Down arrows select current fuzzy rule");
	    break;

      case 2:
	    strcenter (5, "Togai InfraLogic Fuzzy-C Pendulum Demonstration");
	    strcenter (6, "Version 3.0");

#ifdef	NEC_PC
	    textmode (_GOR);
#endif
	    textcolor (WHITE);
	    strcenter (7, "Copyright (c) 1989, 1990, Togai InfraLogic, Inc.");
	    strcenter (8, "All rights reserved.                            ");
	    strcenter (10, "Written by Gary M. Teichrow");

	    textcolor (GREEN);
	    strcenter (12,"Address all inquiries to:   ");

	    textcolor (WHITE);
	    strcenter (13, "Togai InfraLogic, Inc.      ");
	    strcenter (14, "30 Corporate Park, Suite 107");
	    strcenter (15, "Irvine, CA 92714   U.S.A.   ");
	    strcenter (17, "Phone: (714) 975-8522       ");
	    strcenter (18, "F A X: (714) 975-8524       ");
#ifdef	NEC_PC
	    textmode (_GPSET);
#endif
	    break;
    }

    textcolor (GREEN);

    if (curpage == 1)
#ifdef NEC_PC
	strcenter (21, "N=Next Page  or  ESC=Exit");
#else
	strcenter (21, "PageDown  or  ESC=Exit");
#endif
    else if (curpage == maxpage)
#ifdef NEC_PC
	strcenter (21, "P=Previous Page  or  ESC=Exit");
#else
	strcenter (21, "PageUp  or  ESC=Exit");
#endif
}

/*
 *  Initialize our rule matrix...
 */

void initMbfpoints()
{
    int i;

    for (i = 0; i < numrules+1; i++) {
	rulelist[i].open    = FALSE;
	rulelist[i].disabled= FALSE;
    }

/*
 *			THETA
 *
 *	       ||NM | NS |  Z | PS | PM |
 *	  =====||=======================|
 *   D	   NM  ||   |	 | PM |    |	|
 *   T	  -----||-----------------------+
 *   H	   NS  ||   |	 | PS | NS |	|
 *   E	  -----||-----------------------+
 *   T	    Z  ||PM | PS |  Z | NS | NM |
 *   A	  -----||-----------------------+
 *	   PS  ||   | PS | NS |    |	|
 *	  -----||-----------------------+
 *	   PM  ||   |	 | NM |    |	|
 *	  ------------------------------+
 */

    defineMBFpoints (Z_NM, Z, NM, PM);
    defineMBFpoints (Z_NS, Z, NS, PS);
    defineMBFpoints (Z_Z , Z, Z , Z);
    defineMBFpoints (Z_PS, Z, PS, NS);
    defineMBFpoints (Z_PM, Z, PM, NM);

    defineMBFpoints (NM_Z, NM, Z, PM);
    defineMBFpoints (NS_Z, NS, Z, PS);
    defineMBFpoints (PS_Z, PS, Z, NS);
    defineMBFpoints (PM_Z, PM, Z, NM);

    defineMBFpoints (PS_NS, PS, NS, Z);
    defineMBFpoints (NS_PS, NS, PS, Z);

    buildscaleddata (PM, _Theta_PM_map);
    buildscaleddata (PS, _Theta_PS_map);
    buildscaleddata (Z,  _Theta_Z_map);
    buildscaleddata (NS, _Theta_NS_map);
    buildscaleddata (NM, _Theta_NM_map);
}

void buildscaleddata (elem, mbf)
int elem;
UBYTE mbf[];
{
    register int i;

    for (i=0;i<256;i++) {
	s4data[elem][i] = (int) (mbf[i]/4);
	s6data[elem][i] = (int) (mbf[i]/6);
    }
}

/*
 * For a given rule(listx,listy), define the pointlists for the two inputs
 * and the output.  Also do some initialization stuff.
 */

void defineMBFpoints (rule, tp, dtp, cur)
int rule;
int tp;
int dtp;
int cur;
{
    rulelist[rule].thetaptr = tp;
    rulelist[rule].dthetaptr = dtp;
    rulelist[rule].currentptr = cur;
    buildlist (rule, 0, errorlist[tp]);
    buildlist (rule, 1, derrorlist[dtp]);
    buildlist (rule, 2, errorlist[cur]);
}

/*
 * Build a pointlist from a given mbf array generated by fuzzy-c...
 * Basically, we parse the array, find a high point or two, then search to
 * the left and find a low point or the end of the array, and search to the
 * right and find a low point or an end.
 */

void buildlist (rule, list, mbf)
int rule;
int list;
UBYTE mbf[];
{
    register int i;
    register int sp;
    register int ep;

    sp = 0;
    ep = 0;

    for (i = 0; i < 256; i++) {
	if ((sp == 0) && (mbf[i] != 0))
	    sp = i;

	if ((ep == 0) && (sp != 0) && (mbf[i] == 0))
	    ep = i - 1;
    }

    if (ep == 0)
	ep = 255;

    setpoint (&rulelist[rule].pointlists[list][0], sp, ep);
}

/*
 * Since our Theta varies from -pi to 0 and Omega varies from -1.6 to 1.6
 * and the universe of discourse for our inputs are 0 to 255,  we have to
 * scale 'em.
 */

void scaleerrors ()
{
    register int i;

    for (i = 0; i < numrules + 1; i++)
	rulelist[i].DOM=0;

    dError = (int) (Omega * OmegaScaleFactor);
    dError = 0 - dError;

    Error = (int) ((DesiredTheta-Theta) * ThetaScaleFactor);

    if (dError < -128)
	dError = -128;
    else if (dError > 127)
	dError = 127;

    if (Error < -128)
	Error = -128;
    else if (Error > 127)
	Error = 127;

    if ((lastError > Error-bopthreshold) && (lastError < Error+bopthreshold))
	bopticks++;
    else if (bopticks)
	retractbopper = TRUE;

    lastError = Error;
    lastdError = dError;
}

/*
 * Calculate new Theta and Omega based on output from fuzzy pendulum call.
 * Also check resulting Theta to see if it went past the limitors
 */

void cyclesystem (outcurrent)
SBYTE outcurrent;
{
    int MainTorque;			    /* torque exerted by gravity */
    int CoilTorque;			    /* torque exerted by the coil */
    int TotalTorque;			    /* total summed torques */

    if (outcurrent) {

	Current = (int) outcurrent;
	I = (float) (Current / 127.0);

    } else if (!outcurrent && Error == 0) {

	Current = (int) outcurrent;
	I = (float) (Current / 127.0);
    }

    if (!stickstop) {
	sticksize = sticksize + stickfactor;

	if (sticksize < stickbottom)
	    stickfactor = 1;

	if (sticksize > sticktop)
	    stickfactor = -1;
    }

    MainTorque = (int) (-(sticksize/10) * (bobmass/10) * g * cos (Theta));
    CoilTorque = (int) ((motorsize * motorscalefactor) * -I);
    TotalTorque = MainTorque + CoilTorque;

    Inertia = ((float) sticksize / 10.0) * ((float) sticksize / 10.0)
					   * ((float) bobmass / 10.0);

    Alpha = (float) TotalTorque / Inertia;
    Omega = (float) Omega + Alpha * dT;
    Theta = (float) (Theta + Omega * dT + Alpha * dT2);

    if (Theta > limitRight) {
	Theta = limitRight;
	Omega = (float) 0.0;
	Alpha = (float) 0.0;
    }

    if (Theta < limitLeft) {
	Theta = limitLeft;
	Omega = (float) 0.0;
	Alpha = (float) 0.0;
    }
}
