/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#ifndef wissensbank_h
#define wissensbank_h

#include "global.h"
#include "lingvar.h"
#include "regel.h"
#include "anwendung.h"
#include "controller.h"

// Definitions eines Namens fuer die linguistische Variable, die den
// Systemfehler beschreibt.
#define FehlerVarName "fehler"

/*
 *------------------------------------------------------------------------------
 * Klasse : Wissensbank
 * Zweck  : Enthaelt alle Daten, die fuer die Steuerung einer Anwendung von
 *          Bedeutung sind :
 *            Steuer-Variablen - ling. Variablen, die die Ein/Ausgabe-Variablen
 *                               einer Anwendung modellieren.
 *            Fehler-Variablen - ling. Variablen zur Definition eines System-
 *                               fehlers.
 *            Steuer-Regeln    - Regeln, die auf den Steuer-Variablen basieren
 *                               und der Berechnung von Steuerwerten dienen.
 *            Fehler-Regeln    - Regeln, die auf den Fehler-Variablen basieren
 *                               und der Berechnung eines Systemfehlers dienen.
 *            Anwendung        - Anwendung, die gesteuert werden soll.
 *            Controller       - Controller, der Anwendung steuert.
 *
 *          Ausserdem werden diverse Funktionen zur Ueberpruefung der Konsistenz
 *          zur Verfuegung gestellt.
 *------------------------------------------------------------------------------
 */
class Wissensbank
{
  public : Wissensbank();
           ~Wissensbank();

           const LingVarSatz* steuer_var() const;
           LingVarSatz* steuer_var_ref();
           void steuer_var(LingVarSatz*);

           const LingVarSatz* fehler_var() const;
           LingVarSatz* fehler_var_ref();
           void fehler_var(LingVarSatz*);

           const RegelBasis* steuer_regeln() const;
           RegelBasis* steuer_regeln_ref();
           void steuer_regeln(RegelBasis*);

           const RegelBasis* fehler_regeln() const;
           RegelBasis* fehler_regeln_ref();
           void fehler_regeln(RegelBasis*);

           Anwendung* anwendung();
           void anwendung(Anwendung*);

           boolean steuer_var_ok(Ausgabe*);
           boolean fehler_var_ok(Ausgabe*);

           boolean steuer_regeln_ok(Ausgabe*);
           boolean fehler_regeln_ok(Ausgabe*);

           boolean anwendung_ok(Ausgabe*);
           boolean konsistent(boolean neuronal, Ausgabe*);

           FuzController* controller();
           void controller(FuzController*);
           LingVarSatz* mach_fehler_var();
  private: boolean _teste_lingvarsatz(LingVarSatz*, char*, Ausgabe*);
           boolean _kompatibel(LingVarSatz*, RegelBasis*);
           LingVarSatz _steuer_var,
                       _fehler_var;
           RegelBasis _steuer_regeln,
                      _fehler_regeln;
           Anwendung* _anwendung;
           FuzController* _controller;
};

/*
 *--------------- einfachere Funktionen als inline-Definition ------------------
 */

inline void Wissensbank::steuer_var(LingVarSatz* lvs) { _steuer_var = *lvs; }

inline LingVarSatz* Wissensbank::steuer_var_ref() { return &_steuer_var; }

inline const LingVarSatz* Wissensbank::steuer_var() const
{
  return &_steuer_var;
}


inline void Wissensbank::fehler_var(LingVarSatz* lvs) { _fehler_var = *lvs; }

inline LingVarSatz* Wissensbank::fehler_var_ref() { return &_fehler_var; }

inline const LingVarSatz* Wissensbank::fehler_var() const
{
  return &_fehler_var;
}


inline void Wissensbank::steuer_regeln(RegelBasis* rb) { _steuer_regeln = *rb; }

inline RegelBasis* Wissensbank::steuer_regeln_ref() { return &_steuer_regeln; }

inline const RegelBasis* Wissensbank::steuer_regeln() const
{
  return &_steuer_regeln;
}


inline void Wissensbank::fehler_regeln(RegelBasis* rb) { _fehler_regeln = *rb; }

inline RegelBasis* Wissensbank::fehler_regeln_ref() { return &_fehler_regeln; }

inline const RegelBasis* Wissensbank::fehler_regeln() const
{
  return &_fehler_regeln;
}


inline Anwendung* Wissensbank::anwendung() { return _anwendung; }

inline FuzController* Wissensbank::controller() { return _controller; }

#endif
